<?php
/**
 * Handles all calls to external APIs (WooCommerce and Telegram).
 * This class uses static methods so that it can be accessed anywhere
 * without needing to be instantiated.
 */
class WASCT_API_Handler {

    /**
     * Builds authentication headers for the WooCommerce API.
     *
     * @return array|WP_Error An array containing the authorisation header, or an error if the keys are not configured.
     */
    private static function get_wc_auth_headers() {
        static $headers = null;
        if ($headers !== null) {
            return $headers;
        }
    
        $options = wasct_get_settings();
        $consumer_key    = $options['wc_consumer_key'] ?? '';
        $consumer_secret = $options['wc_consumer_secret'] ?? '';
    
        if ( empty($consumer_key) || empty($consumer_secret) ) {
            return new WP_Error('wc_keys_missing', __( 'The WooCommerce API keys are not configured.', 'web-in-air-shop-connect-for-telegram' ));
        }
    
        $headers = [ 
            'Authorization' => 'Basic ' . base64_encode( $consumer_key . ':' . $consumer_secret ) 
        ];
    
        return $headers;
    }

    /**
     * Sends a request to the Telegram API.
     * This is the central function for all communications with Telegram.
     *
     * @param string $method The Telegram API method to call (e.g. “sendMessage”).
     * @param array  $data   The data to send with the request.
     * @return array|WP_Error The result of the request if successful, or an error.
     */
    private static function send_telegram_request( $method, $data ) {
        $options = wasct_get_settings();
        $token = $options['telegram_token'] ?? '';

        $url = "https://api.telegram.org/bot{$token}/{$method}";
        $request_args = [
            'body'      => $data,
            'timeout'   => 20,
        ];

        $response = wp_remote_post( $url, $request_args );


        $body = wp_remote_retrieve_body( $response );
        $decoded_body = json_decode( $body, true );

        if ( isset($decoded_body['ok']) && $decoded_body['ok'] === true ) {
            return $decoded_body['result'];
        } else {
            $error_message = $decoded_body['description'] ?? __( 'Unknown error from the Telegram API', 'web-in-air-shop-connect-for-telegram' );
            return new WP_Error('telegram_api_error', $error_message);
        }
    }

    /**
     * Retrieves a list of WooCommerce products with caching.
     *
     * @param array $args Arguments for the WC API (per_page, page, search, etc.).
     * @return array|WP_Error An array with products and pagination, or an error.
     */
    public static function get_woocommerce_products( $args = [] ) {
        $lang_code = WASCT_Webhook_Handler::get_current_lang_for_api();
        if ($lang_code) {
            $args['lang'] = $lang_code;
        }
    
        $args['_fields'] = ['id', 'name', 'images'];
    
        $cache_version = get_option( 'wasct_cache_version', 1 );
        $transient_key = 'wasct_products_v' . $cache_version . '_' . md5( wp_json_encode($args) );
        
        $cached_data = get_transient( $transient_key );
        if ( false !== $cached_data ) {
            return $cached_data; 
        }
    
        $headers = self::get_wc_auth_headers();
        if ( is_wp_error( $headers ) ) return $headers;
    
        $api_url = get_home_url() . '/wp-json/wc/v3/products';
        $default_args = [ 'status' => 'publish', 'stock_status' => 'instock', 'orderby' => 'title', 'order' => 'asc' ];
        $args = wp_parse_args( $args, $default_args );
        $request_url = $api_url . '?' . http_build_query( $args );
        $response = wp_remote_get( $request_url, [ 'headers' => $headers, 'timeout' => 10 ]);
    
        if ( is_wp_error( $response ) ) return $response;
    
        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );
        
        if ( wp_remote_retrieve_response_code( $response ) !== 200 ) { 
            $error_message = $data['message'] ?? __( 'WooCommerce API error', 'web-in-air-shop-connect-for-telegram' );
            return new WP_Error( 'wc_api_error', $error_message, ['status' => wp_remote_retrieve_response_code( $response )] ); 
        }
        
        $result = [
            'products'       => $data,
            'total_pages'    => (int) wp_remote_retrieve_header( $response, 'X-WP-TotalPages' ),
            'total_products' => (int) wp_remote_retrieve_header( $response, 'X-WP-Total' ),
        ];
    
        set_transient( $transient_key, $result, 10 * MINUTE_IN_SECONDS );
        
        return $result;
    }

    /**
     * Retrieves data for a single product by its ID, with caching.
     *
     * @param int $product_id The WooCommerce product ID.
     * @return array|WP_Error The product data, or an error.
     */
    public static function get_woocommerce_product_by_id( $product_id ) {
    if ( ! is_numeric( $product_id ) || $product_id <= 0 ) {
        return new WP_Error( 'invalid_product_id', __( 'Invalid product ID.', 'web-in-air-shop-connect-for-telegram' ) );
    }
    
    $lang_code = WASCT_Webhook_Handler::get_current_lang_for_api();
    
    $cache_version = get_option('wasct_cache_version', 1); 

    $transient_key = 'wasct_product_v' . $cache_version . '_' . $product_id . '_' . $lang_code;
    
    $cached_product = get_transient($transient_key);
    if (false !== $cached_product) {
        return $cached_product;
    }

    $headers = self::get_wc_auth_headers();
    if ( is_wp_error( $headers ) ) return $headers;
    
    $lang_query = $lang_code ? '?lang=' . $lang_code : '';
    
    $api_url = get_home_url() . "/wp-json/wc/v3/products/{$product_id}" . $lang_query;
    $response = wp_remote_get( $api_url, [ 'headers' => $headers, 'timeout' => 10 ] );
    if ( is_wp_error( $response ) ) return $response;
    
    $body = wp_remote_retrieve_body( $response );
    $data = json_decode( $body, true );
    if ( wp_remote_retrieve_response_code( $response ) !== 200 ) { 
        return new WP_Error( 'wc_api_error', $data['message'] ?? __( 'WooCommerce API error', 'web-in-air-shop-connect-for-telegram' ), ['status' => wp_remote_retrieve_response_code( $response )] ); 
    }
    
    set_transient($transient_key, $data, 15 * MINUTE_IN_SECONDS);
    
    return $data;
}

    /**
     * Retrieves all variations of a variable product.
     * Handles API pagination to ensure all variations are retrieved.
     *
     * @param int $product_id The parent product ID.
     * @return array The list of variation objects.
     */
    public static function get_woocommerce_product_variations($product_id) {
        $cache_version = get_option('wasct_cache_version', 1); 
        $transient_key = 'wasct_variations_v' . $cache_version . '_' . $product_id;
    
        $cached_variations = get_transient($transient_key);
        if (false !== $cached_variations) {
            return $cached_variations;
        }

        $headers = self::get_wc_auth_headers();
        if (is_wp_error($headers)) return [];
        
        $all_variations = [];
        $page = 1;
        $lang_code = WASCT_Webhook_Handler::get_current_lang_for_api();
        $lang_param = $lang_code ? '&lang=' . $lang_code : '';
        $is_successful = true;  

        do {
            $api_url = get_home_url() . "/wp-json/wc/v3/products/{$product_id}/variations?per_page=100&page={$page}" . $lang_param;
            $response = wp_remote_get($api_url, ['headers' => $headers, 'timeout' => 15]);

            if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
                $is_successful = false;
                break;
            }
            $variations = json_decode(wp_remote_retrieve_body($response), true);
            if (empty($variations)) {
                break;
            }
            $all_variations = array_merge($all_variations, $variations);
            $page++;
        } while (count($variations) === 100);
        
        if ($is_successful) {
            set_transient($transient_key, $all_variations, HOUR_IN_SECONDS);
        }

        return $all_variations;
    }

    /**
     * Modifies the caption of an existing media message (photo).
     *
     * @param int    $chat_id    The chat ID.
     * @param int    $message_id The ID of the message to be modified.
     * @param string $caption    The new caption.
     * @param array|null $keyboard   The new inline keyboard (if needed).
     * @return array|WP_Error
     */
    public static function edit_telegram_message_caption( $chat_id, $message_id, $caption, $keyboard = null ) {
        $data = [
            'chat_id'      => $chat_id,
            'message_id'   => $message_id,
            'caption'      => $caption,
            'parse_mode'   => 'Markdown'
        ];
        if ($keyboard) {
            $data['reply_markup'] = json_encode( ['inline_keyboard' => $keyboard] );
        }
        return self::send_telegram_request('editMessageCaption', $data);
    }

    /**
     * Deletes a message in a Telegram conversation.
     *
     * @param int $chat_id    The chat ID.
     * @param int $message_id The ID of the message to be deleted.
     * @return array|WP_Error
     */
    public static function delete_telegram_message( $chat_id, $message_id ) {
        $data = [ 'chat_id' => $chat_id, 'message_id' => $message_id ];
        return self::send_telegram_request('deleteMessage', $data);
    }

    /**
     * Sends a photo in a Telegram conversation.
     * If the URL is invalid or empty, a replacement image is used.
     *
     * @param int        $chat_id   The chat ID.
     * @param string     $photo_url The URL of the image to send.
     * @param string     $caption   The image caption.
     * @param array|null $keyboard  The inline keyboard to attach.
     * @return array|WP_Error
     */
	public static function send_telegram_photo( $chat_id, $photo_url, $caption = '', $keyboard = null ) {
        $data = [
            'chat_id'    => $chat_id,
            'caption'    => $caption,
            'parse_mode' => 'Markdown'
        ];
        if ( $keyboard ) {
            $data['reply_markup'] = json_encode( ['inline_keyboard' => $keyboard] );
        }
    
        if ( empty($photo_url) ) {
            $photo_url = plugins_url( 'assets/images/placeholder.jpg', WASCT_PLUGIN_FILE );
        }
    
        return self::_send_photo_as_upload('sendPhoto', $data, $photo_url);
    }

    /**
     * Sends a simple text message in a Telegram conversation.
     *
     * @param int        $chat_id The chat ID.
     * @param string     $message The message text.
     * @param array|null $keyboard The online keyboard to attach.
     * @return array|WP_Error
     */
    public static function send_telegram_message( $chat_id, $message, $keyboard = null ) {
        $data = [
            'chat_id'    => $chat_id,
            'text'       => $message,
            'parse_mode' => 'Markdown'
        ];
        if ( $keyboard ) {
            $data['reply_markup'] = json_encode( ['inline_keyboard' => $keyboard] );
        }
        return self::send_telegram_request('sendMessage', $data);
    }
    

    /**
     * Generic function to retrieve terms from a WooCommerce taxonomy (categories, tags, etc.).
     *
     * @param string $endpoint The API endpoint (e.g., “products/categories”).
     * @param array  $args     Arguments to filter the query.
     * @return array|WP_Error An array with the terms and pagination, or an error.
     */
    private static function get_wc_taxonomy_terms( $endpoint, $args = [] ) {
        $lang_code = WASCT_Webhook_Handler::get_current_lang_for_api();
        if ($lang_code) {
            $args['lang'] = $lang_code;
        }
        $cache_version = get_option( 'wasct_cache_version', 1 );
        ksort($args); 
        $args_hash = md5( http_build_query($args) );
        $transient_key = 'wasct_tax_' . $endpoint . '_v' . $cache_version . '_' . $args_hash;
        
        $cached_data = get_transient( $transient_key );
        if ( false !== $cached_data ) {
            return $cached_data; 
        }

        $headers = self::get_wc_auth_headers();
        if ( is_wp_error( $headers ) ) {
            return $headers;
        }
        $api_url = get_home_url() . '/wp-json/wc/v3/' . $endpoint;
        
        $default_args = [ 'per_page' => 100, 'hide_empty' => true, 'orderby' => 'name', 'order' => 'asc' ];
        $args = wp_parse_args( $args, $default_args );
        $request_url = $api_url . '?' . http_build_query( $args );

        $response = wp_remote_get( $request_url, [ 'headers' => $headers, 'timeout' => 10 ] );
        
        if ( is_wp_error( $response ) ) {
            return $response;
        }

        $response_code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );

        if ( $response_code !== 200 ) {
            return new WP_Error( 'wc_api_error', $data['message'] ?? 'WooCommerce API error', ['status' => $response_code] );
        }

        $result = [
            'terms'       => $data,
            'total_pages' => (int) wp_remote_retrieve_header( $response, 'X-WP-TotalPages' ),
            'total_terms' => (int) wp_remote_retrieve_header( $response, 'X-WP-Total' ),
        ];
        set_transient( $transient_key, $result, HOUR_IN_SECONDS );
        return $result;
    }

    /**
     * Retrieves WooCommerce product categories.
     * @param array $args API arguments.
     * @return array|WP_Error
     */
    public static function get_woocommerce_categories( $args = [] ) {
        return self::get_wc_taxonomy_terms( 'products/categories', $args );
    }

    /**
     * Retrieves WooCommerce product tags.
     * @param array $args API arguments.
     * @return array|WP_Error
     */
    public static function get_woocommerce_tags( $args = [] ) {
        return self::get_wc_taxonomy_terms( 'products/tags', $args );
    }

    /**
     * Retrieves product brands (if the taxonomy exists).
     * @param array $args API arguments.
     * @return array|WP_Error
     */
    public static function get_woocommerce_brands( $args = [] ) {
        return self::get_wc_taxonomy_terms( 'products/brands', $args );
    }

    /**
     * Modifies the media (photo), caption, and keyboard of an existing message.
     * Ideal for changing images without sending a new message.
     *
     * @param int    $chat_id    The chat ID.
     * @param int    $message_id The ID of the message to be modified.
     * @param string $photo_url  The new URL of the photo.
     * @param string $caption    The new caption.
     * @param array|null $keyboard   The new keyboard.
     * @return array|WP_Error
     */
    public static function edit_telegram_message_media( $chat_id, $message_id, $photo_url, $caption = '', $keyboard = null ) {
        $data = [
            'chat_id'      => $chat_id,
            'message_id'   => $message_id,
            'caption'      => $caption,
        ];
        if ( $keyboard ) {
            $data['reply_markup'] = json_encode( ['inline_keyboard' => $keyboard] );
        }
    
        if ( empty($photo_url) ) {
            $photo_url = plugins_url( 'assets/images/placeholder.jpg', WASCT_PLUGIN_FILE );
        }
    
        return self::_send_photo_as_upload('editMessageMedia', $data, $photo_url);
    }
    
    /**
     * Sends a photo by uploading the file directly to Telegram.
     * This is a more robust method than sending a URL.
     *
     * @param string     $method     The Telegram method (“sendPhoto” or “editMessageMedia”).
     * @param array      $base_data  The base data (chat_id, message_id, etc.).
     * @param string     $photo_url  The URL of the image to be processed.
     * @return array|WP_Error
     */
    private static function _send_photo_as_upload($method, $base_data, $photo_url) {
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        $temp_downloaded_path = download_url($photo_url, 15);
        $is_invalid_image = false;
        
        global $wp_filesystem;
        if (empty($wp_filesystem)) {
            require_once (ABSPATH . '/wp-admin/includes/file.php');
            WP_Filesystem();
        }

        if (is_wp_error($temp_downloaded_path)) {
            $is_invalid_image = true;
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log
            error_log('WASCT Image Download Error: Failed to download URL: ' . $photo_url . ' - Error: ' . $temp_downloaded_path->get_error_message());
        } else {
            $mime_type = mime_content_type($temp_downloaded_path);
            if (strpos($mime_type, 'image/') !== 0) {
                $is_invalid_image = true;
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log
                error_log('WASCT Image Type Error: Downloaded file is not an image. MIME type: ' . $mime_type . ' for URL: ' . $photo_url);
                $wp_filesystem->delete($temp_downloaded_path);
            }
        }
        
        if ($is_invalid_image) {
            $placeholder_url = plugins_url('assets/images/placeholder.jpg', WASCT_PLUGIN_FILE);
            $keyboard = isset($base_data['reply_markup']) ? json_decode($base_data['reply_markup'], true)['inline_keyboard'] : null;

            if ($method === 'editMessageMedia') {
                return self::edit_telegram_message_media($base_data['chat_id'], $base_data['message_id'], $placeholder_url, $base_data['caption'], $keyboard);
            } else { 
                return self::send_telegram_photo($base_data['chat_id'], $placeholder_url, $base_data['caption'], $keyboard);
            }
        }

        $options = wasct_get_settings();
        $token = $options['telegram_token'] ?? '';
        if (empty($token)) { return new WP_Error('no_token', __('The Telegram token is not configured.', 'web-in-air-shop-connect-for-telegram')); }

        $upload_dir = wp_upload_dir();
        $cache_dir = $upload_dir['basedir'] . '/wasct-image-cache';
        if (!$wp_filesystem->is_dir($cache_dir)) { wp_mkdir_p($cache_dir); }
        $filename = md5($photo_url) . '.jpg';
        $cached_file_path = $cache_dir . '/' . $filename;
        $wp_filesystem->move($temp_downloaded_path, $cached_file_path, true);

        $api_url = "https://api.telegram.org/bot{$token}/{$method}";
        $boundary = '----' . wp_generate_password(24, false);
        $payload = '';
        $post_fields = $base_data;

        if ($method === 'editMessageMedia') {
            $media_info = [
                'type' => 'photo', 'media' => 'attach://file_to_upload',
                'caption' => $base_data['caption'] ?? '', 'parse_mode' => 'Markdown'
            ];
            $post_fields['media'] = json_encode($media_info);
            unset($post_fields['caption']);
        }

        foreach ($post_fields as $key => $value) {
            $payload .= '--' . $boundary . "\r\n";
            $payload .= 'Content-Disposition: form-data; name="' . esc_attr($key) . "\"\r\n\r\n";
            $payload .= $value . "\r\n";
        }
        
        $file_field_name = ($method === 'editMessageMedia') ? 'file_to_upload' : 'photo';

        $payload .= '--' . $boundary . "\r\n";
        $payload .= 'Content-Disposition: form-data; name="' . esc_attr($file_field_name) . '"; filename="' . basename($cached_file_path) . '"' . "\r\n";
        $payload .= 'Content-Type: ' . mime_content_type($cached_file_path) . "\r\n\r\n";
        $payload .= $wp_filesystem->get_contents($cached_file_path);
        $payload .= "\r\n";
        $payload .= '--' . $boundary . '--';

        $request_args = [
            'body'    => $payload,
            'headers' => [ 'Content-Type' => 'multipart/form-data; boundary=' . $boundary ],
            'timeout' => 30,
        ];
        
        $response = wp_remote_post($api_url, $request_args);

        if (is_wp_error($response)) { return $response; }

        $decoded_body = json_decode(wp_remote_retrieve_body($response), true);
        if (isset($decoded_body['ok']) && $decoded_body['ok'] === true) {
            return $decoded_body['result'];
        } else {
            $error_message = $decoded_body['description'] ?? __('Unknown error from the Telegram API', 'web-in-air-shop-connect-for-telegram');
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log
            error_log("Telegram API Error ({$method} upload): " . $error_message);
            return new WP_Error('telegram_api_upload_error', $error_message);
        }
    }
}