<?php
/**
 * Manages the plugin's administration page.
 */
class WASCT_Admin {
    private $options;

    public function __construct() {
        add_action( 'admin_menu', [ $this, 'add_admin_menu' ] );
        add_action( 'admin_init', [ $this, 'settings_init' ] );
        add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_admin_styles' ] );
        add_action( 'admin_post_wasct_reset_data', [ $this, 'handle_reset_action' ] );
        add_action( 'admin_notices', [ $this, 'show_reset_success_notice' ] );
        $this->options = wasct_get_settings();
    }
    
    /**
     * Prepares language data by determining the status of each translation file.
     *
     * @return array The language array enriched with status information.
     */
    private function _prepare_language_statuses() {
    $available_languages = WASCT_Utils::get_site_languages();
    $prepared_languages = [];
    $text_domain = 'web-in-air-shop-connect-for-telegram';

    foreach ($available_languages as $lang_code => $lang_data) {
        $locale = $lang_data['locale'];
        $mo_file = $text_domain . '-' . $locale . '.mo';

        $lang_data['status_key'] = 'missing';
        $lang_data['status_text'] = __('Missing', 'web-in-air-shop-connect-for-telegram');
        $lang_data['status_desc'] = __('Will be downloaded from WordPress.org', 'web-in-air-shop-connect-for-telegram');

        if (file_exists(WASCT_PLUGIN_DIR . 'languages/' . $mo_file)) {
            $lang_data['status_key'] = 'plugin';
            $lang_data['status_text'] = __('Present', 'web-in-air-shop-connect-for-telegram');
            $lang_data['status_desc'] = __('Provided by the plugin', 'web-in-air-shop-connect-for-telegram');
        } elseif ($lang_code === 'en') { 
            $lang_data['status_key'] = 'native';
            $lang_data['status_text'] = __('Native', 'web-in-air-shop-connect-for-telegram');
            $lang_data['status_desc'] = __('Source language', 'web-in-air-shop-connect-for-telegram');
        }

        $prepared_languages[$lang_code] = $lang_data;
    }
    return $prepared_languages;
}

    public function enqueue_admin_styles($hook_suffix) {
		if (strpos($hook_suffix, 'web-in-air-shop-connect-for-telegram') === false && strpos($hook_suffix, 'wasct-') === false) {
			return;
		}
		$css_file_url = plugin_dir_url( WASCT_PLUGIN_FILE ) . 'admin/css/wasct-admin-styles.css';
		wp_enqueue_style( 'wasct-admin-styles', $css_file_url, [], filemtime(WASCT_PLUGIN_DIR . 'admin/css/wasct-admin-styles.css') );
		wp_enqueue_media();
		$js_file_url = plugin_dir_url( WASCT_PLUGIN_FILE ) . 'admin/js/wasct-admin-uploader.js';
		wp_enqueue_script( 'wasct-admin-uploader', $js_file_url, [ 'jquery' ], filemtime(WASCT_PLUGIN_DIR . 'admin/js/wasct-admin-uploader.js'), true );
	}

    public function add_admin_menu() {
        $menu_icon_svg = 'data:image/svg+xml;base64,' . base64_encode('<svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M12 22C17.5228 22 22 17.5228 22 12C22 6.47715 17.5228 2 12 2C6.47715 2 2 6.47715 2 12C2 17.5228 6.47715 22 12 22Z" fill="#a0a5aa" fill-opacity="0.5"/><path d="M10 8.33333L16.6667 6L14 16L7.33333 14L10 8.33333Z" stroke="white" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/><path d="M10 8.33333L12 12" stroke="white" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/></svg>');
        add_menu_page(
            __( 'Telegram Shop', 'web-in-air-shop-connect-for-telegram' ),
            __( 'Telegram Shop', 'web-in-air-shop-connect-for-telegram' ),
            'manage_options',
            'web-in-air-shop-connect-for-telegram',
            [ $this, 'render_welcome_page' ],
            $menu_icon_svg,
            58
        );
        add_submenu_page(
            'web-in-air-shop-connect-for-telegram',
            __( 'Home - Telegram Shop', 'web-in-air-shop-connect-for-telegram' ),
            __( 'Home', 'web-in-air-shop-connect-for-telegram' ),
            'manage_options',
            'web-in-air-shop-connect-for-telegram',
            [ $this, 'render_welcome_page' ]
        );
        add_submenu_page(
            'web-in-air-shop-connect-for-telegram',
            __( 'General Settings - Telegram Shop', 'web-in-air-shop-connect-for-telegram' ),
            __( 'General', 'web-in-air-shop-connect-for-telegram' ),
            'manage_options',
            'wasct-general-settings',
            [ $this, 'render_settings_page' ]
        );
        add_submenu_page(
            'web-in-air-shop-connect-for-telegram',
            __( 'Visual & Texts Settings - Telegram Shop', 'web-in-air-shop-connect-for-telegram' ),
            __( 'Visuals & Texts', 'web-in-air-shop-connect-for-telegram' ),
            'manage_options',
            'wasct-visual-settings',
            [ $this, 'render_settings_page' ]
        );
    }
    
    public function render_welcome_page() {
        $logo_url = plugin_dir_url( WASCT_PLUGIN_FILE ) . 'assets/images/logo.webp';
        ?>
        <div class="wrap wasct-admin-wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            <div class="wasct-header-wrap wasct-welcome-header">
                <div class="wasct-header-title">
                    <h2><?php esc_html_e( 'Welcome to Web-in-Air: Shop Connect for Telegram!', 'web-in-air-shop-connect-for-telegram' ); ?></h2>
                    <p class="wasct-subtitle"><?php esc_html_e( 'Your shop, right in your customers pockets on Telegram.', 'web-in-air-shop-connect-for-telegram' ); ?></p>
                </div>
                <div class="wasct-logo"><img src="<?php echo esc_url($logo_url); ?>" alt="<?php esc_attr_e( 'Logo', 'web-in-air-shop-connect-for-telegram' ); ?>"></div>
            </div>
            <div class="wasct-welcome-panel">
                <h3><?php esc_html_e( 'To begin with', 'web-in-air-shop-connect-for-telegram' ); ?></h3>
                <p><?php esc_html_e( "This plugin connects your shop to a Telegram bot, allowing your customers to browse products, add to basket, and pay, all without leaving the Telegram app. Here are the steps to set up your Telegram shop:", 'web-in-air-shop-connect-for-telegram' ); ?></p>
                <div class="wasct-cards-container">
                    <div class="wasct-card">
                        <span class="dashicons dashicons-admin-generic"></span>
                        <h4><?php esc_html_e( '1. General Settings', 'web-in-air-shop-connect-for-telegram' ); ?></h4>
                        <p><?php esc_html_e( "Activate the bot and enter your Telegram and WooCommerce API keys. This is the most important step.", 'web-in-air-shop-connect-for-telegram' ); ?></p>
                        <a href="<?php echo esc_url(admin_url('admin.php?page=wasct-general-settings')); ?>" class="button button-primary"><?php esc_html_e( 'Configure', 'web-in-air-shop-connect-for-telegram' ); ?></a>
                    </div>
                    <div class="wasct-card">
                        <span class="dashicons dashicons-admin-appearance"></span>
                        <h4><?php esc_html_e( '2. Visual Settings & Texts', 'web-in-air-shop-connect-for-telegram' ); ?></h4>
                        <p><?php esc_html_e( "Customise the appearance of your shop, the welcome message and the menu buttons.", 'web-in-air-shop-connect-for-telegram' ); ?></p>
                        <a href="<?php echo esc_url(admin_url('admin.php?page=wasct-visual-settings')); ?>" class="button button-secondary"><?php esc_html_e( 'Customise', 'web-in-air-shop-connect-for-telegram' ); ?></a>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
	
	/**
     * Cleans and validates settings before saving them to the database.
     * This function is automatically called by the WordPress Settings API.
     *
     * @param array $input Raw data sent by the form.
     * @return array Cleaned data, ready to be saved.
     */
    public function sanitize_settings( $input ) {
        $sanitized_input = get_option( 'wasct_settings', [] );
    
        $current_tab = $input['active_tab'] ?? 'general';
    
        switch ($current_tab) {
            case 'visual':
                $sanitized_input['product_display_mode'] = isset($input['product_display_mode']) && in_array($input['product_display_mode'], ['text', 'visual']) ? $input['product_display_mode'] : 'text';
                $sanitized_input['variation_columns'] = isset($input['variation_columns']) ? sanitize_text_field( $input['variation_columns'] ) : '5';
                if (isset($input['custom_logo_url'])) {
                    $logo_url = esc_url_raw($input['custom_logo_url']);
                    $sanitized_input['custom_logo_url'] = str_replace('http://', 'https://', $logo_url);
                }
                if(isset($input['welcome_text'])) $sanitized_input['welcome_text'] = sanitize_textarea_field($input['welcome_text']);
                $sanitized_input['enable_tags_button'] = isset($input['enable_tags_button']) ? '1' : '0';
                $sanitized_input['enable_brands_button'] = isset($input['enable_brands_button']) ? '1' : '0';
                $sanitized_input['enable_contact_button'] = isset($input['enable_contact_button']) ? '1' : '0';
                break;
    
            case 'general':
            default:
                $sanitized_input['enable_bot'] = isset( $input['enable_bot'] ) ? '1' : '0';
                if(isset($input['telegram_token'])) $sanitized_input['telegram_token'] = sanitize_text_field( $input['telegram_token'] );
                if(isset($input['wc_consumer_key'])) $sanitized_input['wc_consumer_key'] = sanitize_text_field( $input['wc_consumer_key'] );
                if(isset($input['wc_consumer_secret'])) $sanitized_input['wc_consumer_secret'] = sanitize_text_field( $input['wc_consumer_secret'] );
                if(isset($input['support_username'])) $sanitized_input['support_username'] = sanitize_text_field($input['support_username']);
                $sanitized_input['enabled_languages'] = isset($input['enabled_languages']) ? array_map('sanitize_text_field', $input['enabled_languages']) : [];
                break;
        }
    
    
        if ( function_exists('wasct_setup_telegram_webhook') ) { wasct_setup_telegram_webhook(); }
        add_settings_error('wasct_options','wasct_message',__( 'Settings saved.', 'web-in-air-shop-connect-for-telegram' ),'success');
    
        return $sanitized_input;
    }
	
    public function render_settings_page() {
        ?>
        <div class="wrap wasct-admin-wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            <?php $logo_url = plugin_dir_url( WASCT_PLUGIN_FILE ) . 'assets/images/logo.webp'; ?>
            <div class="wasct-header-wrap">
                <div class="wasct-header-title">
                     <span class="wasct-version"><?php
                        /* translators: %s: The plugin version number. */
                        printf( esc_html__( 'Version %s', 'web-in-air-shop-connect-for-telegram' ), esc_html( get_plugin_data( WASCT_PLUGIN_FILE )['Version'] ) ); 
                    ?></span>
                </div>
                <div class="wasct-logo"><img src="<?php echo esc_url($logo_url); ?>" alt="<?php esc_attr_e( 'Logo', 'web-in-air-shop-connect-for-telegram' ); ?>"></div>
            </div>
            <div class="wasct-settings-content">
                <form action="options.php" method="post" enctype="multipart/form-data">
                    <?php
                    settings_fields( 'wasct_settings_group' );
                    $current_page = isset($_GET['page']) ? sanitize_key(wp_unslash($_GET['page'])) : 'wasct-general-settings';
                    switch ($current_page) {
                        case 'wasct-visual-settings':
                            do_settings_sections( 'wasct_visual_tab' );
                            echo '<input type="hidden" name="wasct_settings[active_tab]" value="visual" />';
                            break;
                        default:
                            do_settings_sections( 'wasct_general_tab' );
                            echo '<input type="hidden" name="wasct_settings[active_tab]" value="general" />';
                            break;
                    }
                    submit_button( __( 'Save changes', 'web-in-air-shop-connect-for-telegram' ) );
                    ?>
                </form>
                
                <?php
                if ($current_page === 'wasct-general-settings') {
                    echo '<div class="wasct-maintenance-section" style="margin-top: 40px;">';
                    do_settings_sections('wasct_maintenance_section');
                    echo '</div>';
                }
                ?>
            </div>
        </div>
        <?php
    }

    public function settings_init() {
        register_setting( 'wasct_settings_group', 'wasct_settings', [ 'sanitize_callback' => [ $this, 'sanitize_settings' ] ] );

        add_settings_section( 'wasct_general_section', __( 'General Settings', 'web-in-air-shop-connect-for-telegram' ), null, 'wasct_general_tab' );
        add_settings_field('wasct_enable_bot', __( 'Activate the bot', 'web-in-air-shop-connect-for-telegram' ), [ $this, 'render_field_enable_switch' ], 'wasct_general_tab', 'wasct_general_section');
        add_settings_field('wasct_support_username', __( 'Support user name', 'web-in-air-shop-connect-for-telegram' ), [ $this, 'render_field_support_username' ], 'wasct_general_tab', 'wasct_general_section');
        
        add_settings_section( 'wasct_api_keys_section', __( 'API keys', 'web-in-air-shop-connect-for-telegram' ), null, 'wasct_general_tab' );
        add_settings_field('wasct_telegram_token', __( 'Telegram bot token', 'web-in-air-shop-connect-for-telegram' ), [ $this, 'render_field_telegram_token' ], 'wasct_general_tab', 'wasct_api_keys_section');
        add_settings_field('wasct_wc_consumer_key', __( 'WooCommerce Consumer Key', 'web-in-air-shop-connect-for-telegram' ), [ $this, 'render_field_wc_consumer_key' ], 'wasct_general_tab', 'wasct_api_keys_section');
        add_settings_field('wasct_wc_consumer_secret', __( 'WooCommerce Consumer Privacy Policy', 'web-in-air-shop-connect-for-telegram' ), [ $this, 'render_field_wc_consumer_secret' ], 'wasct_general_tab', 'wasct_api_keys_section');
        
        add_settings_section( 'wasct_language_section', __( 'Language Management', 'web-in-air-shop-connect-for-telegram' ), null, 'wasct_general_tab' );
        add_settings_field('wasct_language_management', __( 'Languages available in the bot', 'web-in-air-shop-connect-for-telegram' ), [ $this, 'render_field_language_management' ], 'wasct_general_tab', 'wasct_language_section');
        
        add_settings_section( 'wasct_maintenance_section', __( 'Maintenance', 'web-in-air-shop-connect-for-telegram' ), null, 'wasct_maintenance_section' );
        add_settings_field('wasct_reset_button', __( 'Reset the plugin', 'web-in-air-shop-connect-for-telegram' ), [ $this, 'render_field_reset_button' ], 'wasct_maintenance_section', 'wasct_maintenance_section');

        add_settings_section( 'wasct_visual_section', __( 'Appearance of the bot', 'web-in-air-shop-connect-for-telegram' ), null, 'wasct_visual_tab');
        add_settings_field('wasct_product_display_mode', __( 'Product display mode', 'web-in-air-shop-connect-for-telegram' ), [ $this, 'render_field_product_display_mode' ], 'wasct_visual_tab', 'wasct_visual_section');
        add_settings_field('wasct_variation_columns', __( 'Columns for variations', 'web-in-air-shop-connect-for-telegram' ), [ $this, 'render_field_variation_columns' ], 'wasct_visual_tab', 'wasct_visual_section');
        add_settings_field('wasct_custom_logo_url', __( 'Customised bot logo', 'web-in-air-shop-connect-for-telegram' ), [ $this, 'render_field_custom_logo' ], 'wasct_visual_tab', 'wasct_visual_section');
		add_settings_field('wasct_welcome_text', __( 'Welcome message (/start)', 'web-in-air-shop-connect-for-telegram' ), [ $this, 'render_field_welcome_text' ], 'wasct_visual_tab', 'wasct_visual_section');

        add_settings_section( 'wasct_menu_visibility_section', __( 'Visibility of menu buttons', 'web-in-air-shop-connect-for-telegram' ), null, 'wasct_visual_tab');
        add_settings_field('wasct_enable_tags_button', __( 'Display the "Labels" button', 'web-in-air-shop-connect-for-telegram' ), [ $this, 'render_field_enable_tags_button' ], 'wasct_visual_tab', 'wasct_menu_visibility_section');
        add_settings_field('wasct_enable_brands_button', __( 'Display the "Brands" button', 'web-in-air-shop-connect-for-telegram' ), [ $this, 'render_field_enable_brands_button' ], 'wasct_visual_tab', 'wasct_menu_visibility_section');
        add_settings_field('wasct_enable_contact_button', __( 'Display the "Contact us" button', 'web-in-air-shop-connect-for-telegram' ), [ $this, 'render_field_enable_contact_button' ], 'wasct_visual_tab', 'wasct_menu_visibility_section');
    }
    
    

    public function render_field_reset_button() {
        ?>
        <p class="description">
            <?php esc_html_e( 'Clicking this button will clear all cached data from the plugin and reset the visual and behavioural settings to their default values.', 'web-in-air-shop-connect-for-telegram' ); ?><br>
            <strong><?php esc_html_e( 'Your Telegram and WooCommerce API keys will be retained.', 'web-in-air-shop-connect-for-telegram' ); ?></strong>
        </p>
        <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
            <input type="hidden" name="action" value="wasct_reset_data">
            <?php wp_nonce_field( 'wasct_reset_nonce_action', 'wasct_reset_nonce_field' ); ?>
            <button type="submit" class="button button-danger" onclick="return confirm('<?php echo esc_attr__( 'Are you sure you want to reset the plugin? This action cannot be undone.', 'web-in-air-shop-connect-for-telegram' ); ?>');">
                <?php esc_html_e( 'Reset data and settings', 'web-in-air-shop-connect-for-telegram' ); ?>
            </button>
        </form>
        <?php
    }

    public function handle_reset_action() {
        if ( ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['wasct_reset_nonce_field'] ?? '' ) ), 'wasct_reset_nonce_action' ) ) {
            wp_die( 'Security check failed.' );
        }
    
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( 'You do not have permission to perform this action.' );
        }
    
        $current_settings = wasct_get_settings();
        $settings_to_keep = [
            'telegram_token'     => $current_settings['telegram_token'] ?? '',
            'wc_consumer_key'    => $current_settings['wc_consumer_key'] ?? '',
            'wc_consumer_secret' => $current_settings['wc_consumer_secret'] ?? '',
        ];
        
        delete_option('wasct_settings');
        update_option('wasct_settings', $settings_to_keep);
    
        wasct_invalidate_all_data_caches();
    
        wp_redirect( admin_url( 'admin.php?page=wasct-general-settings&reset_success=1' ) );
        exit;
    }

    public function show_reset_success_notice() {
        if ( isset($_GET['page']) && $_GET['page'] === 'wasct-general-settings' && isset($_GET['reset_success']) && $_GET['reset_success'] == '1' ) {
            ?>
            <div class="notice notice-success is-dismissible">
                <p><strong><?php esc_html_e( 'The plugin has been successfully reset. All cached data has been purged.', 'web-in-air-shop-connect-for-telegram' ); ?></strong></p>
            </div>
            <?php
        }
    }

    public function render_field_enable_switch() {
        $value = $this->options['enable_bot'] ?? '0';
        echo '<label><input type="checkbox" name="wasct_settings[enable_bot]" value="1" ' . checked( '1', $value, false ) . '> ' . esc_html__( 'Enable the Telegram gateway', 'web-in-air-shop-connect-for-telegram' ) . '</label>';
    }
    public function render_field_telegram_token() {
        $value = $this->options['telegram_token'] ?? '';
        echo '<input type="text" name="wasct_settings[telegram_token]" value="' . esc_attr( $value ) . '" class="regular-text">';
    }
    public function render_field_wc_consumer_key() {
        $value = $this->options['wc_consumer_key'] ?? '';
        echo '<input type="text" name="wasct_settings[wc_consumer_key]" value="' . esc_attr( $value ) . '" class="regular-text">';
    }
    public function render_field_wc_consumer_secret() {
        $value = $this->options['wc_consumer_secret'] ?? '';
        echo '<input type="password" name="wasct_settings[wc_consumer_secret]" value="' . esc_attr( $value ) . '" class="regular-text">';
    }
    public function render_field_variation_columns() {
        $value = $this->options['variation_columns'] ?? '5';
        ?>
        <select name="wasct_settings[variation_columns]">
            <?php for ($i = 1; $i <= 5; $i++): ?>
                <option value="<?php echo esc_attr($i); ?>" <?php selected($value, $i); ?>>
                    <?php
                    /* translators: %s: The number of columns. */
                    echo esc_html(sprintf( __( '%s column(s)', 'web-in-air-shop-connect-for-telegram' ), $i ));
                    ?>
                </option>
            <?php endfor; ?>
        </select>
        <p class="description"><?php esc_html_e( 'Select the number of variation buttons displayed per line.', 'web-in-air-shop-connect-for-telegram' ); ?></p>
        <?php
    }
    public function render_field_custom_logo() {
        $value = $this->options['custom_logo_url'] ?? '';
        ?>
        <div class="wasct-logo-uploader">
            <div class="wasct-logo-preview" style="max-width: 225px; margin-bottom: 10px; border: 1px dashed #ccc; padding: 5px; min-height: 50px;">
                <?php if ($value): ?>
                    <img src="<?php echo esc_url($value); ?>" alt="<?php esc_attr_e('Logo preview', 'web-in-air-shop-connect-for-telegram'); ?>" style="max-width:100%;">
                <?php endif; ?>
            </div>
            <input type="text" name="wasct_settings[custom_logo_url]" value="<?php echo esc_attr($value); ?>" class="regular-text wasct-logo-url-input">
            <button type="button" class="button wasct-upload-logo-button"><?php esc_html_e('Upload an image', 'web-in-air-shop-connect-for-telegram'); ?></button>
            <button type="button" class="button button-secondary wasct-remove-logo-button"><?php esc_html_e('Remove', 'web-in-air-shop-connect-for-telegram'); ?></button>
            <p class="description"><?php esc_html_e('Upload a custom logo for your bot. Recommended size: 450x150 pixels.', 'web-in-air-shop-connect-for-telegram'); ?></p>
        </div>
        <?php
    }
    public function render_field_welcome_text() {
        $value = $this->options['welcome_text'] ?? '';
        $placeholder_text = __( "👋 Welcome to the shop of *{site_name}* !\n\n✨ _{site_slogan}_", 'web-in-air-shop-connect-for-telegram' );
        echo '<textarea name="wasct_settings[welcome_text]" rows="4" class="large-text code" placeholder="' . esc_attr($placeholder_text) . '">' . esc_textarea($value) . '</textarea>';
        echo '<p class="description">' . esc_html__( 'Customise the message sent when the <code>/start</code> command is used. Leave blank to use the default message. Available variables: <code>{site_name}</code>, <code>{site_slogan}</code>.', 'web-in-air-shop-connect-for-telegram' ) . '</p>';
    }
    public function render_field_support_username() {
        $value = $this->options['support_username'] ?? '';
        echo '<input type="text" name="wasct_settings[support_username]" value="' . esc_attr( $value ) . '" class="regular-text" placeholder="' . esc_attr__( 'Telegram Username', 'web-in-air-shop-connect-for-telegram' ) . '">';
        echo '<p class="description">' . esc_html__( "Enter the Telegram username (without the @) for your support account.", 'web-in-air-shop-connect-for-telegram' ) . '</p>';
    }
    public function render_field_enable_tags_button() {
        $value = $this->options['enable_tags_button'] ?? '1';
        echo '<label><input type="checkbox" name="wasct_settings[enable_tags_button]" value="1" ' . checked( '1', $value, false ) . '> ' . esc_html__( 'Yes', 'web-in-air-shop-connect-for-telegram' ) . '</label>';
    }
    public function render_field_enable_brands_button() {
        $value = $this->options['enable_brands_button'] ?? '1';
        echo '<label><input type="checkbox" name="wasct_settings[enable_brands_button]" value="1" ' . checked( '1', $value, false ) . '> ' . esc_html__( 'Yes', 'web-in-air-shop-connect-for-telegram' ) . '</label>';
    }
    public function render_field_enable_contact_button() {
        $value = $this->options['enable_contact_button'] ?? '1';
        echo '<label><input type="checkbox" name="wasct_settings[enable_contact_button]" value="1" ' . checked( '1', $value, false ) . '> ' . esc_html__( 'Yes', 'web-in-air-shop-connect-for-telegram' ) . '</label>';
    }
    public function render_field_product_display_mode() {
        $value = $this->options['product_display_mode'] ?? 'text';
        ?>
        <fieldset>
            <label>
                <input type="radio" name="wasct_settings[product_display_mode]" value="text" <?php checked('text', $value); ?>>
                <?php esc_html_e('List (List of 10 products per page, text only)', 'web-in-air-shop-connect-for-telegram'); ?>
            </label><br>
            <label>
                <input type="radio" name="wasct_settings[product_display_mode]" value="visual" <?php checked('visual', $value); ?>>
                <?php esc_html_e('Visual (product by product with photo)', 'web-in-air-shop-connect-for-telegram'); ?>
            </label>
            <p class="description"><?php esc_html_e('Choose how product lists should be displayed in the bot.', 'web-in-air-shop-connect-for-telegram'); ?></p>
        </fieldset>
        <?php
    }
    public function render_field_language_management() {
        $enabled_languages = $this->options['enabled_languages'] ?? [substr(get_locale(), 0, 2)];
        $languages_with_status = $this->_prepare_language_statuses();
    
        ?>
        <div class="notice notice-info notice-alt inline">
            <p><?php
                echo 'Translations for this plugin are managed by the community. You can contribute by visiting translate.wordpress.org.';
            ?></p>
        </div>
        <?php
    
        echo '<table class="wp-list-table widefat striped" style="margin-top: 15px;">';
        echo '<thead><tr><th>' . esc_html__('Active', 'web-in-air-shop-connect-for-telegram') . '</th><th>' . esc_html__('Language', 'web-in-air-shop-connect-for-telegram') . '</th><th>' . esc_html__('Status', 'web-in-air-shop-connect-for-telegram') . '</th></tr></thead>';
        echo '<tbody>';
    
        foreach ($languages_with_status as $lang_code => $lang_data) {
            echo '<tr>';
            echo '<td><input type="checkbox" name="wasct_settings[enabled_languages][]" value="' . esc_attr($lang_code) . '" ' . checked(in_array($lang_code, $enabled_languages), true, false) . '></td>';
            echo '<td><strong>' . esc_html($lang_data['name']) . '</strong> (' . esc_html($lang_code) . ')</td>';
            echo '<td>';
            $status_color = '#d63638';
            if ($lang_data['status_key'] === 'native') $status_color = '#00732e';
            if ($lang_data['status_key'] === 'plugin') $status_color = '#2271b1';
            $status_icon = ($lang_data['status_key'] === 'missing') ? '❌' : '✔️';
            echo '<span style="color: ' . esc_attr($status_color) . '; font-weight: bold;">' . esc_html($status_icon) . ' ' . esc_html($lang_data['status_text']) . '</span>';
            if (!empty($lang_data['status_desc'])) {
                echo '<p class="description">(' . esc_html($lang_data['status_desc']) . ')</p>';
            }
            echo '</td>';
            echo '</tr>';
        }
    
        echo '</tbody></table>';
    }
}