<?php
/**
 * Schedule the Renewal and Expiration of the subscription
 *
 * @package    we-subscription
 * @subpackage we-subscription/includess
 */

if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

add_action('init', 'wesub_action_schedulers');
add_action('wesub_do_pending_renewals', 'wesub_do_pending_renewals_callback');
add_action('wesub_do_pending_expiration', 'wesub_do_pending_expiration_callback');
add_action('wesub_do_retry_failed_renewals', 'wesub_do_retry_failed_renewals_callback');

/**
 * This function is used to  schedule actions to perfrom subscription event like renewal, expiration.
 *
 * @name wesub_action_schedulers
 * @since 1.0.0
 */
function wesub_action_schedulers()
{
	if (class_exists('ActionScheduler')) {
		if (function_exists('as_next_scheduled_action') && false === as_next_scheduled_action('wesub_do_pending_renewals')) {
			as_schedule_recurring_action(strtotime('hourly'), 3600, 'wesub_do_pending_renewals');
		}
		if (function_exists('as_next_scheduled_action') && false === as_next_scheduled_action('wesub_do_pending_expiration')) {
			as_schedule_recurring_action(strtotime('hourly'), 1800, 'wesub_do_pending_expiration');
		}
		$settings = get_option('wesub_settings');
		$is_retry_payment = isset($settings['retry_failed_renewals']) ? $settings['retry_failed_renewals'] : 'no';

		if ('on' === $is_retry_payment) {
			if (function_exists('as_next_scheduled_action') && false === as_next_scheduled_action('wesub_do_retry_failed_renewals')) {
				as_schedule_recurring_action(strtotime('hourly'), 86400, 'wesub_do_retry_failed_renewals');
			}
		}
		do_action('wesub_do_after_action_scheduled');
	}
}

/**
 * Do_pending_renewals_callback
 *
 * @return void
 */
function wesub_do_pending_renewals_callback()
{
	$current_timestamp = time();

	// Common meta query for both cases.
	$meta_query = array(
		'relation' => 'AND',
		array(
			'key' => 'wesub_status',
			'value' => 'active',
			'compare' => '=', // Ensure the comparison is equality.
		),
		array(
			'key' => 'wesub_next_renewal',
			'value' => $current_timestamp,
			'type' => 'NUMERIC',
			'compare' => '<',
		),
	);

	// Define query args.
	$wesub_args = array(
		'meta_query' => $meta_query,
	);

	if (wesub_is_hpos_enabled()) {
		$wesub_args['type'] = 'wesub_subscriptions';
		$wesub_args['limit'] = -1;
		$wesub_args['return'] = 'ids';
		$subscriptions = wc_get_orders($wesub_args);
	} else {
		$wesub_args['post_type'] = 'wesub_subscriptions';
		$wesub_args['numberposts'] = -1;
		$subscriptions = get_posts($wesub_args);
	}

	WESUB_LOG::log('WES Pending Renewals', $subscriptions, 'info');
	if (is_array($subscriptions) && !empty($subscriptions)) {
		foreach ($subscriptions as $key => $value) {

			if (wesub_is_hpos_enabled()) {
				$wesub_subscription_id = $value;
			} else {
				$wesub_subscription_id = $value->ID;
			}
			if (wesub_check_valid_subscription($wesub_subscription_id)) {
				$wesub_subscription = new Wesub_Subscription($wesub_subscription_id);

				// Create a new renewal order.
				$new_order = wc_create_order();

				// Clone line items.
				foreach ($wesub_subscription->get_items('line_item') as $wesub_item_id => $wesub_item) {
					$new_item = new WC_Order_Item_Product();
					$new_item->set_name($wesub_item->get_name());
					$new_item->set_product_id($wesub_item->get_product_id());
					$new_item->set_variation_id($wesub_item->get_variation_id());
					$new_item->set_quantity($wesub_item->get_quantity());
					$new_item->set_tax_class($wesub_item->get_tax_class());
					$new_item->set_subtotal($wesub_item->get_subtotal());
					$new_item->set_total($wesub_item->get_total());
					$new_item->set_taxes($wesub_item->get_taxes());
					$new_item->set_meta_data($wesub_item->get_meta_data());
					$new_order->add_item($new_item);
				}

				// Clone shipping items.
				foreach ($wesub_subscription->get_items('shipping') as $wesub_item_id => $wesub_item) {
					$new_item = new WC_Order_Item_Shipping();
					$new_item->set_method_title($wesub_item->get_method_title());
					$new_item->set_method_id($wesub_item->get_method_id());
					$new_item->set_total($wesub_item->get_total());
					$new_item->set_taxes($wesub_item->get_taxes());
					$new_order->add_item($new_item);
				}

				// Clone tax items.
				foreach ($wesub_subscription->get_items('tax') as $wesub_item_id => $wesub_item) {
					$new_item = new WC_Order_Item_Tax();
					$new_item->set_rate_id($wesub_item->get_rate_id());
					$new_item->set_label($wesub_item->get_label());
					$new_item->set_compound($wesub_item->get_compound());
					$new_item->set_tax_total($wesub_item->get_tax_total());
					$new_item->set_shipping_tax_total($wesub_item->get_shipping_tax_total());
					$new_order->add_item($new_item);
				}

				// Clone fee items.
				foreach ($wesub_subscription->get_items('fee') as $wesub_item_id => $wesub_item) {
					$new_item = new WC_Order_Item_Fee();
					$new_item->set_name($wesub_item->get_name());
					$new_item->set_total($wesub_item->get_total());
					$new_item->set_taxes($wesub_item->get_taxes());
					$new_item->set_tax_class($wesub_item->get_tax_class());
					$new_order->add_item($new_item);
				}

				// Clone coupon items.
				foreach ($wesub_subscription->get_items('coupon') as $wesub_item_id => $wesub_item) {
					$new_item = new WC_Order_Item_Coupon();
					$new_item->set_code($wesub_item->get_code());
					$new_item->set_discount($wesub_item->get_discount());
					$new_item->set_discount_tax($wesub_item->get_discount_tax());
					$new_order->add_item($new_item);
				}

				$new_order->set_status('pending');
				// Set billing and shipping addresses.
				$new_order->set_address($wesub_subscription->get_address('billing'), 'billing');
				$new_order->set_address($wesub_subscription->get_address('shipping'), 'shipping');

				// Set other order data.
				$new_order->set_payment_method($wesub_subscription->get_payment_method());
				$new_order->set_payment_method_title($wesub_subscription->get_payment_method_title());
				$new_order->set_customer_id($wesub_subscription->get_customer_id());
				$new_order->set_customer_ip_address($wesub_subscription->get_customer_ip_address());
				$new_order->set_customer_user_agent($wesub_subscription->get_customer_user_agent());
				$new_order->set_created_via('duplicate');
				$new_order->update_meta_data('wesub_is_renew', 'yes');
				$new_order->update_meta_data('wesub_subscription_id', $wesub_subscription_id);

				// Calculate and set totals.
				$new_order->calculate_totals();

				// Save the new order.
				$new_order->save();
				/* translators: subscription id */
				$new_order->add_order_note(sprintf(esc_attr__('This renewal order belongs to subscription #%s', 'we-subscription'), $wesub_subscription_id));

				$interval_type = $wesub_subscription->get_meta('wesub_interval_type');
				$interval_period = $wesub_subscription->get_meta('wesub_interval_period');
				$next_renewal = wesub_calculate_timestamp(time(), $interval_period, $interval_type);

				// Update next renewal.
				$wesub_subscription->update_meta_data('wesub_next_renewal', $next_renewal);

				$get_renewals = $wesub_subscription->get_data('wesub_renewals') ? $wesub_subscription->get_data('wesub_renewals') : array();
				$get_renewals[] = $new_order->get_id();
				$wesub_subscription->update_meta_data('wesub_renewals', $get_renewals);
				$wesub_subscription->update_meta_data('wesub_status', 'on-hold');
				$wesub_subscription->save();

				do_action('wesub_renewal_created', $new_order, $wesub_subscription_id);

				WESUB_LOG::log('WES Initiate Payment For Renewal Order #' . $new_order->get_id() . ' -:- Payment Method: ' . $new_order->get_payment_method(), null, 'info');
				do_action('wesub_do_renewal_payment_' . $new_order->get_payment_method(), $new_order, $wesub_subscription_id);
			}
		}
	}
}

/**
 * Wesub_do_pending_expiration_callback
 */
function wesub_do_pending_expiration_callback()
{
	$current_timestamp = time();

	// Common meta query for both cases.
	$meta_query = array(
		'relation' => 'AND',
		array(
			'key' => 'wesub_status',
			'value' => 'active',
			'compare' => '=', // Ensure the comparison is equality.
		),
		array(
			'key' => 'wesub_expire_date',
			'value' => $current_timestamp,
			'type' => 'NUMERIC',
			'compare' => '<',
		),
	);

	// Define query args.
	$wesub_args = array(
		'meta_query' => $meta_query,
	);

	if (wesub_is_hpos_enabled()) {
		$wesub_args['type'] = 'wesub_subscriptions';
		$wesub_args['limit'] = -1;
		$wesub_args['return'] = 'ids';
		$subscriptions = wc_get_orders($wesub_args);
	} else {
		$wesub_args['post_type'] = 'wesub_subscriptions';
		$wesub_args['numberposts'] = -1;
		$subscriptions = get_posts($wesub_args);
	}
	WESUB_LOG::log('WES Expiration Renewals', $subscriptions, 'info');
	if (is_array($subscriptions) && !empty($subscriptions)) {
		foreach ($subscriptions as $key => $value) {

			if (wesub_is_hpos_enabled()) {
				$wesub_subscription_id = $value;
			} else {
				$wesub_subscription_id = $value->ID;
			}
			$wesub_subscription = new Wesub_Subscription($wesub_subscription_id);

			$wesub_subscription->update_meta_data('wesub_status', 'expired');
			$wesub_subscription->update_meta_data('wesub_expired_time', $current_timestamp);
			$wesub_subscription->save();

			do_action('wesub_subscription_expired', $wesub_subscription_id);

			$mailer = WC()->mailer()->get_emails();
			if (isset($mailer['wesub_expired_email'])) {
				$mailer['wesub_expired_email']->trigger($wesub_subscription);
			}

			$assigned_role = $wesub_subscription->get_meta('wesub_assigned_user_role');
			$settings = get_option('wesub_settings');

			$remove_assigned_role = isset($settings['remove_assign_user_role']) ? $settings['remove_assign_user_role'] : null;
			if ($assigned_role && 'on' == $remove_assigned_role) {
				wesub_remove_user_role($wesub_subscription->get_user_id(), $assigned_role);
			}
		}
	}
}

/**
 * Wesub_do_retry_failed_renewals_callback
 */
function wesub_do_retry_failed_renewals_callback()
{
	$wesub_args = array(
		'status' => 'pending',
		'limit' => 50,
		'meta_query' => array(
			'relation' => 'AND',
			array(
				'key' => 'wesub_is_renew',
				'value' => 'yes',
				'compare' => '=',
			),
			array(
				'key' => 'wesub_subscription_id',
				'value' => '',
				'compare' => '!=',
			),
			array(
				'relation' => 'OR',
				array(
					'key' => 'wesub_total_retry_payment',
					'value' => 3,
					'type' => 'NUMERIC',
					'compare' => '<=',
				),
				array(
					'key' => 'wesub_total_retry_payment',
					'compare' => 'NOT EXISTS',
				),
			),
		),
	);
	$wesub_renewal_orders = wc_get_orders($wesub_args);

	foreach ($wesub_renewal_orders as $renewal_order) {
		$total_retry = $renewal_order->get_meta('wesub_total_retry_payment') ? $renewal_order->get_meta('wesub_total_retry_payment') : 0;
		$renewal_order->update_meta_data('wesub_total_retry_payment', $total_retry + 1);

		$wesub_subscription_id = $renewal_order->get_meta('wesub_subscription_id');

		WESUB_LOG::log('WES Initiate Retry Payment For Renewal Order #' . $renewal_order->get_id() . ' -:- Payment Method: ' . $renewal_order->get_payment_method() . ' -:- For Times: ' . $total_retry, null, 'info');

		do_action('wesub_do_renewal_payment_' . $renewal_order->get_payment_method(), $renewal_order, $wesub_subscription_id);
	}
}
