<?php
/**
 * This file handle the subscription
 *
 * @link  https://webeffortless.com/
 * @since 1.0.0
 *
 * @package    we-subscription
 * @subpackage we-subscription/includes
 */

if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

// Create the subscription after order checkout and manage the subscription statuses and renewal.
add_action('woocommerce_checkout_order_processed', 'wesub_create_subscription', 99, 3);
add_action('woocommerce_store_api_checkout_order_processed', 'wesub_create_subscription_store_api', 99);
add_action('woocommerce_order_status_changed', 'wesub_handle_subscription_after_payment', 10, 4);
add_action('woocommerce_cart_calculate_fees', 'wesub_signup_as_cart_fee');
// Register a new menu and content in the section.
add_filter('woocommerce_account_menu_items', 'wesub_add_my_account_menu_item');
add_action('init', 'wesub_add_my_subscriptions_endpoint');
add_action('after_switch_theme', 'wesub_flush_rewrite_rules_on_activation');
add_action('woocommerce_account_my-subscriptions_endpoint', 'wesub_endpoint_list');
add_action('woocommerce_account_view-subscription_endpoint', 'wesub_endpoint_view');
// Manage the user cancellation.
add_action('init', 'wesub_user_cancel_susbcription', 10);
// Gateways and validations.
add_filter('woocommerce_available_payment_gateways', 'wesub_disable_specific_payment_gateways');
add_filter('woocommerce_is_sold_individually', 'wesub_hide_quantity_fields_for_subscription', 10, 2);
add_filter('woocommerce_add_to_cart_validation', 'wesub_add_to_cart_validation', 10, 5);
// Manage the button texts.
add_filter('woocommerce_product_single_add_to_cart_text', 'wesub_add_to_cart_text', 10, 2);
add_filter('woocommerce_product_add_to_cart_text', 'wesub_add_to_cart_text', 10, 2);
add_filter('woocommerce_order_button_text', 'wesub_place_order_button_text');
// Add extra column in the order list table to show type of order like parent, renewal.
add_filter('manage_woocommerce_page_wc-orders_columns', 'wesub_add_order_custom_column');
add_action('manage_woocommerce_page_wc-orders_custom_column', 'wesub_display_custom_column_content', 10, 2);
add_filter('manage_edit-shop_order_columns', 'wesub_add_order_custom_column', 20);
add_action('manage_shop_order_posts_custom_column', 'wesub_display_custom_column_content', 20, 2);


/**
 * This function is used to set single quantity for susbcription product.
 *
 * @name wesub_hide_quantity_fields_for_subscription
 * @param bool   $allow .
 * @param object $product product.
 * @since 1.0.0
 */
function wesub_hide_quantity_fields_for_subscription($allow, $product)
{

	if (wesub_is_subscription_product($product->get_id())) {
		$allow = true;
	}
	return apply_filters('wesub_hide_quantity_fields_for_subscription', $allow, $product);
}

/**
 * This function is used to create subscription.
 *
 * @param object $order.
 * @since    1.0.0
 */
function wesub_create_subscription_store_api($order)
{
	wesub_create_subscription($order->get_id(), null, $order);
}

/**
 * This function is used to create subscription.
 *
 * @param int $order_id .
 * @param array $posted_data .
 * @param object $order .
 * @since    1.0.0
 */
function wesub_create_subscription($order_id, $posted_data = null, $order = null)
{
	if (is_object($order)) {
		// 3. Persistent Deduplication: Check meta in case of race conditions or previous failures
		if ('yes' === $order->get_meta('wesub_has_subscriptions')) {
			return;
		}

		$subscription_ids = array();
		foreach ($order->get_items() as $wesub_item_id => $wesub_item) {
			$wesub_product_id = $wesub_item->get_product_id();
			$variation_id = $wesub_item->get_variation_id();
			if ($variation_id) {
				$wesub_product_id = $variation_id;
			}
			if (wesub_is_subscription_product($wesub_product_id)) {
				// Get the item details.
				$quantity = $wesub_item->get_quantity();
				$subtotal = $wesub_item->get_subtotal();

				$new_subscription = new Wesub_Subscription();

				$subtotal = $wesub_item->get_subtotal();
				$total = $wesub_item->get_total();
				$subtotal_tax = $wesub_item->get_subtotal_tax();
				$total_tax = $wesub_item->get_total_tax();

				$wesub_args = array(
					'variation' => array(),
					'totals' => array(
						'subtotal' => $subtotal,
						'subtotal_tax' => $subtotal_tax,
						'total' => $total,
						'tax' => $total_tax,
						'tax_data' => array(
							'subtotal' => array($subtotal_tax),
							'total' => array($total_tax),
						),
					),
				);
				$wesub_args = apply_filters('wesub_add_product_args', $wesub_args, $order);

				// Add item to the new order.
				$product = wc_get_product($wesub_product_id);
				$new_subscription->add_product($product, $quantity, $wesub_args);

				if ($product->needs_shipping() && $order->get_items('shipping') && apply_filters('wesub_add_shipping_to_subscription', true, $order)) {
					// Add shipping to the subscription
					foreach ($order->get_items('shipping') as $wesub_item_id => $wesub_item) {
						$new_item = new WC_Order_Item_Shipping();
						$new_item->set_method_title($wesub_item->get_method_title());
						$new_item->set_method_id($wesub_item->get_method_id());
						$new_item->set_total($wesub_item->get_total());
						$new_item->set_taxes($wesub_item->get_taxes());
						$new_subscription->add_item($new_item);
					}
				}

				// Set customer details for the subscription order.
				$new_subscription->set_customer_id($order->get_customer_id());
				$billing_details = $order->get_address('billing');
				$shipping_details = $order->get_address('shipping');
				$new_subscription->set_address($billing_details, 'billing');
				$new_subscription->set_address($shipping_details, 'shipping');
				$new_subscription->set_payment_method($order->get_payment_method());
				$new_subscription->set_payment_method_title($order->get_payment_method_title());
				$new_subscription->set_currency($order->get_currency());

				// Add the custom meta.
				$new_subscription->update_meta_data('wesub_next_renewal', null);
				$new_subscription->update_meta_data('wesub_interval_period', wesub_get_interval_period($wesub_product_id));
				$new_subscription->update_meta_data('wesub_interval_type', wesub_get_interval_type($wesub_product_id));
				$new_subscription->update_meta_data('wesub_expire_period', wesub_get_expire_period($wesub_product_id));
				$new_subscription->update_meta_data('wesub_expire_type', wesub_get_interval_type($wesub_product_id));
				$new_subscription->update_meta_data('wesub_signup_fee', wesub_get_signup_fee($wesub_product_id));
				$new_subscription->update_meta_data('wesub_parent_order_id', $order_id);
				$new_subscription->update_meta_data('wesub_expire_date', null);
				$new_subscription->update_meta_data('wesub_status', 'on-hold');

				// Calculate totals for the new order.
				$new_subscription->calculate_totals();

				// Save the new order.
				$new_subscription->save();

				/* translators: %s: subscription id */
				$order->add_order_note(sprintf(__('A new subscription #%s is created', 'we-subscription'), $new_subscription->get_id()));

				$subscription_ids[] = $new_subscription->get_id();
				do_action('wesub_new_created', $new_subscription, $order);
			}
		}
		if (!empty($subscription_ids)) {
			$order->update_meta_data('wesub_subscription_ids', $subscription_ids);
			$order->update_meta_data('wesub_has_subscriptions', 'yes');
			$order->save();
		}
	}
}

/**
 * This function is used to handle the next renewal, status of subscription after the parent and renewal paid.
 *
 * @param int    $order_id .
 * @param array  $old_status .
 * @param array  $new_status .
 * @param object $order .
 *
 * @since    1.0.0
 */
function wesub_handle_subscription_after_payment($order_id, $old_status, $new_status, $order)
{
	// Check if the order status has changed.
	if ($old_status !== $new_status && ('processing' === $new_status || 'completed' === $new_status)) {
		$has_subscription = $order->get_meta('wesub_has_subscriptions');
		$is_order_paid = $order->get_meta('wesub_order_paid');
		$is_renewal = $order->get_meta('wesub_is_renew');

		// Parent order proceed.
		if ('yes' === $has_subscription && 'yes' !== $is_order_paid) {
			$subscription_ids = $order->get_meta('wesub_subscription_ids');
			if (!empty($subscription_ids)) {
				foreach ($subscription_ids as $wesub_subscription_id) {
					$wesub_subscription = new Wesub_Subscription($wesub_subscription_id);

					$interval_type = $wesub_subscription->get_meta('wesub_interval_type');
					$interval_period = $wesub_subscription->get_meta('wesub_interval_period');
					$expiry_period = $wesub_subscription->get_meta('wesub_expire_period');
					$expiry_type = $wesub_subscription->get_meta('wesub_expire_type');
					$next_renewal = wesub_calculate_timestamp(time(), $interval_period, $interval_type);
					if ($expiry_period && $expiry_type) {
						$expiry_end = wesub_calculate_timestamp(time(), $expiry_period, $expiry_type);
					}

					$next_renewal = apply_filters('wesub_next_renewal_timestamp', $next_renewal, $wesub_subscription);
					$expiry_end = apply_filters('wesub_subscription_end_timestamp', $expiry_end, $wesub_subscription);
					$wesub_subscription->update_meta_data('wesub_next_renewal', $next_renewal);
					$wesub_subscription->update_meta_data('wesub_expire_date', $expiry_end);

					$wesub_subscription->update_meta_data('wesub_status', 'active');

					$wesub_subscription->save();

					$order->update_meta_data('wesub_order_paid', 'yes');
					$order->save();

					do_action('wesub_subscription_activated', $wesub_subscription);

					$settings = get_option('wesub_settings');

					$assign_user_role = isset($settings['assign_user_role']) ? $settings['assign_user_role'] : null;
					if ($assign_user_role) {
						$wesub_subscription->update_meta_data('wesub_assigned_user_role', $assign_user_role);
						$wesub_subscription->save();
						wesub_assign_user_role($order->get_user_id(), $assign_user_role);
					}
				}
			}
		} elseif ('yes' == $is_renewal && 'yes' !== $is_order_paid) { // Renewal order proceed.
			$wesub_subscription_id = $order->get_meta('wesub_subscription_id');

			$wesub_subscription = new Wesub_Subscription($wesub_subscription_id);

			$interval_type = $wesub_subscription->get_meta('wesub_interval_type');
			$interval_period = $wesub_subscription->get_meta('wesub_interval_period');
			$next_renewal = wesub_calculate_timestamp(time(), $interval_period, $interval_type);

			// Update next Renewal.
			$wesub_subscription->update_meta_data('wesub_next_renewal', $next_renewal);
			$wesub_subscription->update_meta_data('wesub_status', 'active');
			$wesub_subscription->save();

			$order->update_meta_data('wesub_order_paid', 'yes');
			$order->save();

			do_action('wesub_renewal_paid', $order);

			$mailer = WC()->mailer()->get_emails();
			if (isset($mailer['wesub_renewal_email'])) {
				$mailer['wesub_renewal_email']->trigger($order);
			}
		}
	}
}
/**
 * Add the signfee as a cart fee.
 */
function wesub_signup_as_cart_fee()
{
	if (is_admin() && !defined('DOING_AJAX')) {
		return;
	}
	foreach (WC()->cart->get_cart() as $cart_item) {
		$wesub_product_id = $cart_item['data']->get_id();
		$signup_fee = wesub_get_signup_fee($wesub_product_id);
		if ($signup_fee) {
			$name = $cart_item['data']->get_name();
			/* translators: signup fee */
			WC()->cart->add_fee(sprintf(esc_html__('SignUp Fee - %s', 'we-subscription'), $name), $signup_fee);
		}
	}
}

/**
 * Add a custom menu item to My Account menu
 *
 * @param mixed $items .
 * @return array
 */
function wesub_add_my_account_menu_item($items)
{
	$new_menu = array('my-subscriptions' => __('My Subscriptions', 'we-subscription'));

	$first_part = array_slice($items, 0, count($items) - 1, true);
	$second_part = array_slice($items, count($items) - 1, null, true);

	return apply_filters('wesub_my_account_menus', $first_part + $new_menu + $second_part, $items);
}

/**
 * Register the custom endpoint
 */
function wesub_add_my_subscriptions_endpoint()
{
	add_rewrite_endpoint('my-subscriptions', EP_ROOT | EP_PAGES);
	add_rewrite_endpoint('view-subscription', EP_ROOT | EP_PAGES);
}

/**
 * Flush rewrite rules to avoid 404 errors
 */
function wesub_flush_rewrite_rules_on_activation()
{
	wesub_add_my_subscriptions_endpoint();
	flush_rewrite_rules();
}

/**
 * Content for the custom section
 */
function wesub_endpoint_list()
{

	$current_page = 1;
	$total_pages = 2;
	wc_get_template(
		'we-subscriptions-list.php',
		array(
			'page' => $current_page,
			'total_pages' => $total_pages,
		),
		'',
		WESUB_PATH . 'templates/'
	);
}
/**
 * Content for the view subscription
 *
 * @param integer $wesub_subscription_id .
 */
function wesub_endpoint_view($wesub_subscription_id)
{
	wc_get_template(
		'we-subscription-details.php',
		array(
			'subscription_id' => $wesub_subscription_id,
		),
		'',
		WESUB_PATH . 'templates/'
	);
}

/**
 * This function is used to cancel susbcription.
 *
 * @name wesub_user_cancel_susbcription
 * @since 1.0.0
 */
function wesub_user_cancel_susbcription()
{

	if (isset($_GET['action']) && isset($_GET['wesub_subscription_id']) && isset($_GET['_wpnonce']) && !empty($_GET['_wpnonce']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), sanitize_text_field(wp_unslash($_GET['wesub_subscription_id'])))) {
		$action = sanitize_text_field(wp_unslash($_GET['action']));
		$wesub_subscription_id = sanitize_text_field(wp_unslash($_GET['wesub_subscription_id']));
		if (wesub_check_valid_subscription($wesub_subscription_id) && 'cancel' === $action) {
			$wesub_subscription = new Wesub_Subscription($wesub_subscription_id);
			$wesub_subscription->update_meta_data('wesub_status', 'cancelled');
			$wesub_subscription->save();

			$mailer = WC()->mailer()->get_emails();
			if (isset($mailer['wesub_cancelled_email'])) {
				$mailer['wesub_cancelled_email']->trigger($wesub_subscription);
			}

			do_action('wesub_subscription_cancelled', $wesub_subscription_id);

			$assigned_role = $wesub_subscription->get_meta('wesub_assigned_user_role');

			$settings = get_option('wesub_settings');

			$remove_assigned_role = isset($settings['remove_assign_user_role']) ? $settings['remove_assign_user_role'] : null;
			if ($assigned_role && 'on' == $remove_assigned_role) {
				wesub_remove_user_role($wesub_subscription->get_user_id(), $assigned_role);
			}

			$redirect_url = wc_get_endpoint_url('view-subscription', $wesub_subscription_id, wc_get_page_permalink('myaccount'));
			wp_safe_redirect($redirect_url);
			exit;
		}
	}
}

/**
 * Disable_specific_payment_gateways
 *
 * @param mixed $available_gateways .
 * @return array
 */
function wesub_disable_specific_payment_gateways($available_gateways)
{
	if (is_admin()) {
		return $available_gateways;
	}

	$settings = get_option('wesub_settings');

	$is_manual_payment = isset($settings['manual_payment']) ? $settings['manual_payment'] : 'no';

	if ('on' !== $is_manual_payment) {
		if (wesub_cart_has_subscription()) {
			if (!empty($available_gateways) && is_array($available_gateways)) {
				foreach ($available_gateways as $gateway_id => $gateways) {

					$supported_payment_method = array('stripe', 'stripe_sepa', 'stripe_sepa_debit');
					// Supported paymnet gateway.
					$wps_payment_method = apply_filters('wesub_supported_payment_gateway', $supported_payment_method, $gateway_id);

					if (!in_array($gateway_id, $wps_payment_method)) {
						unset($available_gateways[$gateway_id]);
					}
				}
				return $available_gateways;
			}
		}
	}
	return $available_gateways;
}


/**
 * This function is used to restrict multiple susbcription product purchase.
 *
 * @name wesub_add_to_cart_validation
 * @param bool $validate validate.
 * @param int  $wesub_product_id product_id.
 * @param int  $quantity quantity.
 * @param int  $variation_id as variation_id.
 * @param bool $variations as variations.
 * @since 1.0.0
 */
function wesub_add_to_cart_validation($validate, $wesub_product_id, $quantity, $variation_id = 0, $variations = null)
{

	$product = wc_get_product($wesub_product_id);
	if (is_object($product) && 'variable' === $product->get_type()) {
		$product = wc_get_product($variation_id);
		$wesub_product_id = $variation_id;
	}
	if (wesub_cart_has_subscription() && wesub_is_subscription_product($wesub_product_id)) {

		$settings = get_option('wesub_settings');

		$can_multiple_purchase = isset($settings['multiple_product_purchases']) ? $settings['multiple_product_purchases'] : 'no';

		if ('on' !== $can_multiple_purchase) {
			$validate = apply_filters('wesub_add_to_cart_validation_notice', false, $wesub_product_id, $quantity);
			wc_add_notice(__('You can not add multiple products in cart', 'we-subscription'), 'error');
		}
	}
	return apply_filters('wesub_add_to_cart_validation', $validate, $wesub_product_id, $quantity);
}

/**
 * This function is used to get add to cart button text.
 *
 * @name wesub_add_to_cart_text
 * @param string $text Add to cart button text.
 * @param object $product .
 * @since    1.0.0
 */
function wesub_add_to_cart_text($text, $product)
{

	if (wesub_is_subscription_product($product->get_id())) {
		$settings = get_option('wesub_settings');
		$button_text = isset($settings['add_to_cart_button_text']) ? $settings['add_to_cart_button_text'] : null;
		if ($button_text) {
			return $button_text;
		}
	}
	return $text;
}

/**
 * This function is used to change place order button text.
 *
 * @name wesub_place_order_button_text
 * @param string $text Place order text.
 * @since    1.0.0
 */
function wesub_place_order_button_text($text)
{
	if (wesub_cart_has_subscription()) {
		$settings = get_option('wesub_settings');
		$button_text = isset($settings['place_order_text']) ? $settings['place_order_text'] : null;
		if ($button_text) {
			return $button_text;
		}
	}
	return $text;
}

/**
 * Wesub_add_order_custom_column
 *
 * @param mixed $columns .
 * @return array
 */
function wesub_add_order_custom_column($columns)
{
	$reordered_columns = array();

	// Inserting columns to a specific location.
	foreach ($columns as $key => $column) {
		$reordered_columns[$key] = $column;

		if ('order_total' === $key) {
			// Inserting after "Status" column.
			$reordered_columns['wesub-order-type'] = __('Type', 'we-subscription');
		}
	}
	return $reordered_columns;
}
/**
 * Wesub_display_custom_column_content
 *
 * @param mixed $column .
 * @param mixed $order .
 * @return void
 */
function wesub_display_custom_column_content($column, $order)
{
	switch ($column) {
		case 'wesub-order-type':
			if (!wesub_is_hpos_enabled()) {
				$order = wc_get_order($order);
			}
			// Get custom order metadata.
			if ('yes' === $order->get_meta('wesub_has_subscriptions')) {
				$subscription_ids = $order->get_meta('wesub_subscription_ids');
				$subscription_ids = implode(', #', array_values($subscription_ids));
				?>
																<div class="wesub-tooltip"><?php esc_attr_e('Parent', 'we-subscription'); ?>
																	<span class="wesub-tooltiptext"><?php esc_attr_e('Subscriptions', 'we-subscription'); ?>
																		#<?php echo esc_attr($subscription_ids); ?></span>
																</div>
																<?php
			} elseif ('yes' === $order->get_meta('wesub_is_renew')) {
				$wesub_subscription_id = $order->get_meta('wesub_subscription_id');
				?>
																<div class="wesub-tooltip"><?php esc_attr_e('Renewal', 'we-subscription'); ?>
																	<span class="wesub-tooltiptext"><?php esc_attr_e('Subscription', 'we-subscription'); ?>
																		#<?php echo esc_attr($wesub_subscription_id); ?></span>
																</div>
																<?php
			}
			break;
	}
}



