<?php
/**
 * Common Functions
 *
 * @package    we-subscription
 * @subpackage we-subscription/includess
 */

if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

if (!function_exists('wesub_get_product_type')) {
	/**
	 * Get the type of product.
	 *
	 * @param mixed $id is product_id.
	 * @return string
	 */
	function wesub_get_product_type($id)
	{
		$product = wc_get_product($id);
		if ($product) {
			if ($product->is_type('variation') && wesub_is_subscription_product($id)) {
				return 'variation_subscription';
			} elseif ($product->is_type('variation')) {
				return 'variation';
			} elseif ($product->is_type('variable') && wesub_is_subscription_product($id)) {
				return 'variable_subscription';
			} elseif ($product->is_type('variable')) {
				return 'variable';
			} elseif ($product->is_type('simple') && wesub_is_subscription_product($id)) {
				return 'simple_subscription';
			} elseif ($product->is_type('simple')) {
				return 'simple';
			} else {
				return 'unknown_product';
			}
		} else {
			return 'product_not_found';
		}
	}
}
if (!function_exists('wesub_is_subscription_product')) {
	/**
	 * Identify if subscription enable for the product.
	 *
	 * @param mixed $id is product_id.
	 * @return bool
	 */
	function wesub_is_subscription_product($id)
	{
		$product = wc_get_product($id);
		if ($product) {
			if ($product->is_type('variation')) {
				$is_sub = get_post_meta($id, 'wesub_variation_subscription', true);
				if ('yes' === $is_sub) {
					return true;
				}
			} elseif ($product->is_type('simple')) {
				$is_sub = get_post_meta($id, '_wes_simple_subscription', true);
				if ('yes' === $is_sub) {
					return true;
				}
			}
		}
		return false;
	}
}

if (!function_exists('wesub_is_order_contain_subscription_product')) {
	/**
	 * Identify if order contain subscription product.
	 *
	 * @param mixed $order .
	 * @return void
	 */
	function wesub_is_order_contain_subscription_product($order)
	{
	}
}
if (!function_exists('wesub_get_interval_period')) {
	/**
	 * Return the interval period.
	 *
	 * @param mixed $id is product_id.
	 * @return mixed
	 */
	function wesub_get_interval_period($id)
	{
		$saved_settings = get_post_meta($id, 'wesub_product_setting', true);
		if (is_array($saved_settings) && !empty($saved_settings) && isset($saved_settings['interval_period'])) {
			return $saved_settings['interval_period'];
		}
		return 0;
	}
}

if (!function_exists('wesub_get_interval_type')) {
	/**
	 * Return the interval type.
	 *
	 * @param mixed $id is product_id.
	 * @return mixed
	 */
	function wesub_get_interval_type($id)
	{
		$saved_settings = get_post_meta($id, 'wesub_product_setting', true);
		if (is_array($saved_settings) && !empty($saved_settings) && isset($saved_settings['interval_type'])) {
			return $saved_settings['interval_type'];
		}
		return null;
	}
}

if (!function_exists('wesub_get_expire_period')) {
	/**
	 * Return the expire interval period.
	 *
	 * @param mixed $id is product_id.
	 * @return mixed
	 */
	function wesub_get_expire_period($id)
	{
		$saved_settings = get_post_meta($id, 'wesub_product_setting', true);
		if (is_array($saved_settings) && !empty($saved_settings) && isset($saved_settings['expire_after'])) {
			return $saved_settings['expire_after'];
		}
		return 0;
	}
}
if (!function_exists('wesub_get_expire_type')) {
	/**
	 * Return the expiry interval type.
	 *
	 * @param mixed $id is product_id.
	 * @return mixed
	 */
	function wesub_get_expire_type($id)
	{
		$saved_settings = get_post_meta($id, 'wesub_product_setting', true);
		if (is_array($saved_settings) && !empty($saved_settings) && isset($saved_settings['expire_type'])) {
			return $saved_settings['expire_type'];
		}
		return null;
	}
}
if (!function_exists('wesub_get_signup_fee')) {
	/**
	 * Return the signup fee.
	 *
	 * @param mixed $id is product_id.
	 * @return mixed
	 */
	function wesub_get_signup_fee($id)
	{
		$saved_settings = get_post_meta($id, 'wesub_product_setting', true);
		if (is_array($saved_settings) && !empty($saved_settings) && isset($saved_settings['signup_fee'])) {
			return $saved_settings['signup_fee'];
		}
		return 0;
	}
}
if (!function_exists('wesub_product_price_html')) {
	/**
	 * Wesub_product_price_html
	 *
	 * @param mixed $default_price .
	 * @param mixed $wesub_product_id .
	 * @return mixed
	 */
	function wesub_product_price_html($default_price, $wesub_product_id)
	{

		if (wesub_is_subscription_product($wesub_product_id)) {
			// Ensure we have a formatted price if it's not already HTML
			if (is_numeric($default_price)) {
				$default_price = wc_price($default_price);
			}

			$period = wesub_get_interval_period($wesub_product_id);
			$type = wesub_get_interval_type($wesub_product_id);

			$price_suffix = ' / ';

			if ($period && '1' !== $period) {
				$price_suffix .= esc_html($period) . ' ';
			}
			$price_suffix .= esc_html($type);

			$expire_period = wesub_get_expire_period($wesub_product_id);

			if ($expire_period) {
				/* translators: 1: expire period 2: interval type */
				$price_suffix .= sprintf(' ' . esc_html__('for %1$s %2$s', 'we-subscription'), esc_html($expire_period), esc_html($type));
			}

			$default_price .= '<span class="wesub-subscription-price-suffix">' . $price_suffix . '</span>';
		}
		return apply_filters('wesub_product_price_html', $default_price, $wesub_product_id);
	}
}

if (!function_exists('wesub_product_price_html_hpos')) {
	/**
	 * Wesub_product_price_html_hpos
	 *
	 * @param mixed $wesub_product_id .
	 * @return mixed
	 */
	function wesub_product_price_html_hpos($wesub_product_id)
	{
		$price_html = '';
		if (wesub_is_subscription_product($wesub_product_id)) {

			$period = wesub_get_interval_period($wesub_product_id);
			$type = wesub_get_interval_type($wesub_product_id);

			$price_suffix = ' / ';

			if ($period && '1' != $period) {
				$price_suffix .= $period . ' ';
			}
			$price_suffix .= $type;

			$expire_period = wesub_get_expire_period($wesub_product_id);

			if ($expire_period) {
				/* translators: 1: expire period 2: interval type */
				$price_suffix .= sprintf(' ' . esc_html__('for %1$s %2$s', 'we-subscription'), $expire_period, $type);
			}
			$price_html .= '<span class="wesub-subscription-price-suffix">' . $price_suffix . '</span>';
		}
		return apply_filters('wesub_product_price_html_hpos', $price_html, $wesub_product_id);
	}
}

if (!function_exists('wesub_cart_renewal_data')) {
	/**
	 * Wesub_cart_renewal_data
	 *
	 * @param mixed $cart_item data.
	 * @return mixed
	 */
	function wesub_cart_renewal_data($cart_item)
	{
		$wesub_product_id = empty($cart_item['variation_id']) ? $cart_item['product_id'] : $cart_item['variation_id'];

		$product = wc_get_product($wesub_product_id);

		$qty = $cart_item['quantity'];
		$line_subtotal = $cart_item['line_subtotal'];
		$line_total = $cart_item['line_total'];

		$line_subtotal_tax = $cart_item['line_subtotal_tax'];
		$line_tax = $cart_item['line_tax'];

		$wc_tax = new WC_Tax();
		$billing_country = WC()->customer->get_billing_country();
		$tax_class = apply_filters('wesub_cart_item_tax', $product->get_tax_class(), $cart_item, $cart_item['key']);
		$line_tax_data = $wc_tax->find_rates(
			array(
				'country' => $billing_country,
				'tax_class' => $tax_class,
			)
		);
		$tax_options = get_option('woocommerce_calc_taxes');
		$tax_type = get_option('woocommerce_tax_display_cart');


		$shipping_fee = 0;
		$shipping_tax = 0;
		if ($product->needs_shipping()) {
			$shipping_fee = WC()->cart->shipping_total;
			$shipping_tax = WC()->cart->shipping_tax_total;
		}

		// Calcucate the product total incase of inclusive tax, for the tax calcuations.
		if ('yes' === $tax_options && 'incl' === $tax_type) {
			$line_subtotal = $line_subtotal + $line_subtotal_tax;
			$line_total = $line_total + $line_tax;
		}
		if ('yes' === $tax_options) {
			if ('incl' === $tax_type) {
				$line_subtotal_tax = WC_Tax::get_tax_total(WC_Tax::calc_inclusive_tax($line_subtotal, $line_tax_data));
				$line_tax = WC_Tax::get_tax_total(WC_Tax::calc_inclusive_tax($line_total, $line_tax_data));

				$line_total = $line_total - $line_tax;
				$line_subtotal = $line_subtotal - $line_subtotal_tax;
			} elseif ('excl' === $tax_type) {
				$line_subtotal_tax = WC_Tax::get_tax_total(WC_Tax::calc_exclusive_tax($line_subtotal, $line_tax_data));
				$line_tax = WC_Tax::get_tax_total(WC_Tax::calc_exclusive_tax($line_total, $line_tax_data));
			}
		}
		$coupons = WC()->cart->get_applied_coupons();
		if (!empty($coupons)) {
			$line_total = $line_subtotal;
			$line_tax = $line_subtotal_tax;
		}

		$data = array(
			'quantity' => $qty,
			'line_subtotal' => $line_subtotal,
			'line_total' => $line_total,
			'line_subtotal_tax' => $line_subtotal_tax,
			'line_tax' => $line_tax,
			'product_id' => $wesub_product_id,
			'shipping_fee' => $shipping_fee,
			'shipping_tax' => $shipping_tax,
		);
		return apply_filters('wesub_renewal_item_data', $data, $cart_item);
	}
}

if (!function_exists('wesub_calculate_timestamp')) {
	/**
	 * This function is used to calculate time in timestamp.
	 *
	 * @name wesub_calculate_timestamp
	 * @since 1.0.0
	 * @param int    $current_timestamp current_timestamp.
	 * @param int    $interval_period interval_period.
	 * @param string $interval_type interval_type.
	 */
	function wesub_calculate_timestamp($current_timestamp, $interval_period, $interval_type)
	{

		$calculated_timestamp = 0;
		switch ($interval_type) {
			case 'day':
				$calculated_timestamp = strtotime('+' . $interval_period . ' days', $current_timestamp);
				break;
			case 'week':
				$calculated_timestamp = strtotime('+' . $interval_period * 7 . ' days', $current_timestamp);
				break;
			case 'month':
				$calculated_timestamp = strtotime('+' . $interval_period . ' month', $current_timestamp);
				break;
			case 'year':
				$calculated_timestamp = strtotime('+' . $interval_period . ' year', $current_timestamp);
				break;
			default:
		}
		return $calculated_timestamp;
	}
}

if (!function_exists('wesub_check_valid_subscription')) {
	/**
	 * This function is used to check susbcription post type.
	 *
	 * @param int $wesub_subscription_id subscription_id.
	 */
	function wesub_check_valid_subscription($wesub_subscription_id)
	{
		if (isset($wesub_subscription_id) && !empty($wesub_subscription_id)) {
			if ('shop_order_placehold' === get_post_type(absint($wesub_subscription_id)) || 'wesub_subscriptions' === get_post_type(absint($wesub_subscription_id))) {
				return true;
			}
		}
		return false;
	}
}

if (!function_exists('wesub_cart_has_subscription')) {
	/**
	 * This function is used to check if the cart has the wes subscription.
	 */
	function wesub_cart_has_subscription()
	{

		$cart = WC()->cart;
		if (!$cart || $cart->is_empty()) {
			return false;
		}

		$has_subscription = false;

		foreach ($cart->get_cart_contents() as $key => $values) {

			if (wesub_is_subscription_product($values['data']->get_id())) {
				$has_subscription = true;
				break;
			}
		}
		return apply_filters('wesub_cart_has_subscription', $has_subscription);
	}
}

if (!function_exists('wesub_order_has_subscription')) {
	/**
	 * This function is used to check if the order has the wes subscription.
	 *
	 * @param int $order_id .
	 */
	function wesub_order_has_subscription($order_id)
	{

		$order = wc_get_order($order_id);

		foreach ($order->get_items('line_item') as $wesub_item_id => $wesub_item) {
			$wesub_product_id = $wesub_item->get_product_id();
			if ($wesub_item->get_variation_id()) {
				$wesub_product_id = $wesub_item->get_variation_id();
			}
			if (wesub_is_subscription_product($wesub_product_id)) {
				return true;
			}
		}
		return apply_filters('wesub_order_has_subscription', false, $order_id);
	}
}
if (!function_exists('wesub_assign_user_role')) {
	/**
	 * Assign_user_role
	 *
	 * @param mixed $wesub_user_id .
	 * @param mixed $role .
	 * @return void
	 */
	function wesub_assign_user_role($wesub_user_id, $role)
	{
		$user = new WP_User($wesub_user_id);
		$user->add_role($role);
	}
}
if (!function_exists('wesub_remove_user_role')) {
	/**
	 * Remove_user_role
	 *
	 * @param mixed $wesub_user_id .
	 * @param mixed $role .
	 * @return void
	 */
	function wesub_remove_user_role($wesub_user_id, $role)
	{
		// Check if the user has any other active subscriptions.
		if (wesub_get_user_active_subscription_count($wesub_user_id) > 0) {
			return;
		}
		$user = new WP_User($wesub_user_id);
		$user->remove_role($role);
	}
}

if (!function_exists('wesub_is_hpos_enabled')) {
	/**
	 * Identify if the hpos is enabled or not
	 */
	function wesub_is_hpos_enabled()
	{
		$hpos_status = get_option('woocommerce_custom_orders_table_enabled', 'no');

		return 'yes' === $hpos_status ? true : false;
	}
}

if (!function_exists('wesub_get_user_active_subscription_count')) {
	/**
	 * Get the number of active subscriptions for a user.
	 *
	 * @param int $wesub_user_id User ID.
	 * @return int Number of active subscriptions.
	 */
	function wesub_get_user_active_subscription_count($wesub_user_id)
	{
		if (empty($wesub_user_id)) {
			return 0;
		}
		$wesub_args = array(
			'type' => 'wesub_subscriptions',
			'customer' => $wesub_user_id,
			'limit' => -1,
			'return' => 'ids',
			'meta_query' => array(
				array(
					'key' => 'wesub_status',
					'value' => 'active',
					'compare' => '=',
				),
			),
		);
		$subscriptions = wc_get_orders($wesub_args);

		return count($subscriptions);
	}
}
