<?php
/**
 * Manage the logs created througout the plugin
 *
 * @class WESUB_LOG
 * @package    we-subscription
 * @subpackage we-subscription/includes
 */

if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

if (!class_exists('WESUB_LOG')) {
	/**
	 * WESUB_LOG
	 */
	class WESUB_LOG
	{

		/**
		 * Logger Object
		 *
		 * @var string
		 */
		private static $logger;
		/**
		 * Slug for the log
		 *
		 * @var string
		 */
		private static $log_source = 'wesub-log';

		/**
		 * Initialize Logger
		 *
		 * @var object
		 */
		private static function init()
		{
			if (is_null(self::$logger)) {
				self::$logger = wc_get_logger();
			}
		}

		/**
		 * Log a message to WooCommerce logs with date and time
		 *
		 * @param string $title The log message.
		 * @param string $data The log data.
		 * @param string $level Log level (default: 'info').
		 */
		public static function log($title, $data, $level = 'info')
		{
			self::init();
			$timestamp = current_time('Y-m-d H:i:s');

			// Convert data to readable format without JSON encoding.
			if (is_array($data)) {
				$data = wc_print_r($data, true);
			} elseif (is_object($data)) {
				$data = wc_print_r((array) $data, true);
			} elseif (!is_string($data)) {
				$data = strval($data);
			}

			$formatted_message = "[$timestamp] $title:\n$data";

			$settings = get_option('wesub_settings');
			$is_logging = isset($settings['wesub_log']) ? $settings['wesub_log'] : 'no';

			if ('on' === $is_logging) {
				if (method_exists(self::$logger, $level)) {
					self::$logger->$level($formatted_message, array('source' => self::$log_source));
				} else {
					self::$logger->info($formatted_message, array('source' => self::$log_source));
				}
			}
		}

		/**
		 * Delete old logs older than a given number of days
		 *
		 * @param int $days Number of days before logs are deleted.
		 */
		public static function delete_old_logs($days = 30)
		{
			$upload_dir = wp_upload_dir();
			$log_dir = trailingslashit($upload_dir['basedir']) . 'wc-logs/';
			$log_files = glob($log_dir . 'wesub-log-*.log');
			$time_limit = time() - ($days * DAY_IN_SECONDS);

			foreach ($log_files as $file) {
				if (filemtime($file) < $time_limit) {
					wp_delete_file($file); // Delete file.
				}
			}
		}
	}
}
