<?php
/**
 * Handle the subscription Product and it's setting
 *
 * @class Wesub_Handle_Pricing
 * @package    we-subscription
 * @subpackage we-subscription/includes
 */

defined('ABSPATH') || exit; // Exit if accessed directly.

if (!class_exists('Wesub_Handle_Pricing')) {
	/**
	 * Wesub_Handle_Pricing
	 */
	class Wesub_Handle_Pricing
	{
		/**
		 * Initiate the current class instance
		 */
		public static function instance()
		{
			return new Wesub_Handle_Pricing();
		}

		/**
		 * Contructor
		 */
		public function __construct()
		{
			add_filter('woocommerce_get_price_html', array($this, 'wesub_handle_pricing'), 100, 2);
			add_filter('woocommerce_variable_price_html', array($this, 'wesub_handle_variation_between_label'), 100, 2);
			add_filter('woocommerce_cart_item_price', array($this, 'wesub_modify_cart_item_price_html'), 10, 3);
			add_action('woocommerce_before_calculate_totals', array($this, 'wesub_modify_cart_item_total'), 100, 1);
			add_action('woocommerce_review_order_after_order_total', array($this, 'wesub_renewal_bifurcation'));
			// Wc_block.
			add_filter('woocommerce_get_item_data', array($this, 'wesub_get_item_data'), 10, 2);
			add_action('template_redirect', array($this, 'wesub_add_checkout_blocks_hook'));
		}

		/**
		 * Wesub_handle_pricing
		 *
		 * @param mixed $default_price .
		 * @param mixed $product .
		 * @return mixed
		 */
		public function wesub_handle_pricing($default_price, $product)
		{

			$default_price = wesub_product_price_html($default_price, $product->get_id());

			return apply_filters('wesub_product_price_html', $default_price, $product->get_id());
		}

		/**
		 * Wesub_handle_variation_between_label
		 *
		 * @param mixed $default_price .
		 * @param mixed $product .
		 * @return mixed
		 */
		public function wesub_handle_variation_between_label($default_price, $product)
		{
			if (!is_admin()) {
				return $default_price;
			}
			$variations = $product->get_available_variations();

			$first_element = reset($variations); // Get the first element of the array.
			$variation_id = $first_element['variation_id'];

			if (wesub_is_subscription_product($variation_id)) {
				$display_price1 = $first_element['display_price'];
				$display_price1 = wesub_product_price_html($display_price1, $variation_id);
			}

			$last_element = end($variations); // Get the last element of the array.
			$variation_id = $last_element['variation_id'];

			if (wesub_is_subscription_product($variation_id)) {
				$display_price2 = $last_element['display_price'];
				$display_price2 = wesub_product_price_html($display_price2, $variation_id);
			}
			if (isset($display_price1) && isset($display_price2)) {
				$default_price = $display_price1 . ' - ' . $display_price2;
			}
			return apply_filters('wesub_variation_between_price', $default_price, $product);
		}

		/**
		 * Modify the displayed price HTML in the cart
		 *
		 * @param mixed $default_price .
		 * @param mixed $cart_item .
		 * @param mixed $cart_item_key .
		 * @return mixed
		 */
		public function wesub_modify_cart_item_price_html($default_price, $cart_item, $cart_item_key)
		{

			$default_price = wesub_product_price_html($default_price, $cart_item['data']->get_id());

			return apply_filters('wesub_modify_cart_item_price_html', $default_price, $cart_item);
		}

		/**
		 * Removed wesub_modify_cart_item_total to avoid double counting fees.
		 */
		public function wesub_modify_cart_item_total($cart)
		{
			// This function previously added signup fee to cart item price, 
			// but we are now handling it via add_fee to avoid issues.
			return;
		}

		/**
		 * Wesub_renewal_bifurcation
		 */
		public function wesub_renewal_bifurcation()
		{
			if (!empty(WC()->cart->cart_contents)) {
				foreach (WC()->cart->cart_contents as $cart_item) {
					if (wesub_is_subscription_product($cart_item['data']->get_id())) {
						echo wp_kses_post($this->wesub_get_renewal_display_html($cart_item));
					}
				}
			}
		}

		/**
		 * Add_item_subtotal
		 *
		 * @param mixed $cart_item_subtotal .
		 * @param mixed $cart_item .
		 * @param mixed $cart_item_key .
		 * @return string
		 */
		public function wesub_item_subtotal($cart_item_subtotal, $cart_item, $cart_item_key)
		{

			$wesub_product_id = $cart_item['data']->get_id();

			// Use wesub_product_price_html to append suffix to the existing subtotal
			return wesub_product_price_html($cart_item_subtotal, $wesub_product_id);
		}

		/**
		 * Add custom cart item data to the subscription products.
		 *
		 * @param array $data Cart item data.
		 * @param array $cart_item .
		 * @return array
		 */
		public function wesub_get_item_data($data = array(), $cart_item = array())
		{

			if (isset($cart_item['data'])) {
				$wesub_product_id = $cart_item['data']->get_id();
				if (wesub_is_subscription_product($wesub_product_id)) {
					$price_html = wesub_product_price_html_hpos($wesub_product_id);
					$data[] = apply_filters(
						'wesub_block_cart_price',
						array(
							'name' => 'wesub-price-html',
							'hidden' => true,
							'value' => html_entity_decode($price_html),
						),
						$cart_item
					);
				}
			}
			return $data;
		}

		/**
		 * Add the block hook.
		 */
		public function wesub_add_checkout_blocks_hook()
		{
			if (has_block('woocommerce/checkout-totals-block')) {
				add_filter('render_block_woocommerce/checkout-order-summary-block', array($this, 'wesub_renewal_bifurcation_block'), 999);
			}
		}

		/**
		 * Wesub_renewal_bifurcation_block
		 *
		 * @param string $html is the content.
		 */
		public function wesub_renewal_bifurcation_block($html)
		{
			if (!empty(WC()->cart->cart_contents)) {
				foreach (WC()->cart->cart_contents as $cart_item) {
					if (wesub_is_subscription_product($cart_item['data']->get_id())) {
						$html .= $this->wesub_get_renewal_display_html($cart_item);
					}
				}
			}
			return $html;
		}

		/**
		 * Get the HTML for the renewal bifurcation display.
		 *
		 * @param array $cart_item The cart item.
		 * @return string The HTML content.
		 */
		public function wesub_get_renewal_display_html($cart_item)
		{
			$item_renewal_data = wesub_cart_renewal_data($cart_item);
			$wesub_product_id = $cart_item['data']->get_id();

			$interval_period = wesub_get_interval_period($wesub_product_id);
			$interval_type = wesub_get_interval_type($wesub_product_id);
			$next_renewal = wesub_calculate_timestamp(time(), $interval_period, $interval_type);
			$first_renewal = date_i18n(wc_date_format(), $next_renewal);

			$shipping_fee = isset($item_renewal_data['shipping_fee']) ? $item_renewal_data['shipping_fee'] : 0;
			$shipping_tax = isset($item_renewal_data['shipping_tax']) ? $item_renewal_data['shipping_tax'] : 0;
			$tax = isset($item_renewal_data['line_tax']) ? $item_renewal_data['line_tax'] : 0;
			$total = $item_renewal_data['line_total'] + $tax + $shipping_fee + $shipping_tax;

			ob_start();
			?>
			<tr class="wesub_renewal_bifurcation_wrapper">
				<td colspan="2">
					<div class="wesub-renewal-card">
						<div class="wesub-renewal-header">
							<h4><?php echo esc_html__('Renewal For', 'we-subscription') . ' ' . wp_kses_post($cart_item['data']->get_name()); ?>
							</h4>
						</div>
						<ul class="wesub-renewal-details">
							<li>
								<span class="wesub-label"><?php esc_html_e('Subtotal', 'we-subscription'); ?>:</span>
								<span
									class="wesub-value"><?php echo wp_kses_post(wc_price($item_renewal_data['line_subtotal'])); ?></span>
							</li>
							<?php if ($tax > 0): ?>
								<li>
									<span class="wesub-label"><?php esc_html_e('Tax', 'we-subscription'); ?>:</span>
									<span class="wesub-value"><?php echo wp_kses_post(wc_price($tax)); ?></span>
								</li>
							<?php endif; ?>
							<?php if ($shipping_fee > 0): ?>
								<li>
									<span class="wesub-label"><?php esc_html_e('Shipping', 'we-subscription'); ?>:</span>
									<span class="wesub-value"><?php echo wp_kses_post(wc_price($shipping_fee)); ?></span>
								</li>
							<?php endif; ?>
							<li class="wesub-total">
								<span class="wesub-label"><?php esc_html_e('Total', 'we-subscription'); ?>:</span>
								<span
									class="wesub-value"><?php echo wp_kses_post(wesub_product_price_html(wc_price($total), $wesub_product_id)); ?></span>
							</li>
							<li class="wesub-date">
								<span class="wesub-label"><?php esc_html_e('First Renewal Date', 'we-subscription'); ?>:</span>
								<span class="wesub-value"><?php echo wp_kses_post($first_renewal); ?></span>
							</li>
						</ul>
					</div>
				</td>
			</tr>
			<?php
			return ob_get_clean();
		}
	}
}