<?php
/**
 * Load all of the files
 *
 * @class Wesub_Autoloader
 *
 * @package    we-subscription
 * @subpackage we-subscription/includes/admin
 */

defined('ABSPATH') || exit; // Exit if accessed directly.

if (!class_exists('Wesub_Autoloader')) {
	/**
	 * Wesub_Autoloader
	 */
	class Wesub_Autoloader
	{

		/**
		 * Constructor
		 */
		public function __construct()
		{
			$this->autoload();
			Wesub_Settings::instance();
			Wesub_Product_Settings::instance();

			Wesub_Handle_Pricing::instance();

			add_action('admin_enqueue_scripts', array($this, 'wesub_enqueue_admin_scripts'));
			add_action('wp_enqueue_scripts', array($this, 'wesub_enqueue_scripts'));

			add_action(
				'woocommerce_init',
				function () {
					require_once WESUB_PATH . 'includes/class-we-subscription.php';
				}
			);

			// Register the email class.
			add_filter('woocommerce_email_classes', array($this, 'wesub_subscription_emails'));

			// Payment gateway Compatibility.
			add_action('init', array($this, 'wesub_add_gateways_compatibility'));
		}

		/**
		 * Autoload callback
		 *
		 * @since  1.0.0
		 */
		public function autoload()
		{

			$path = WESUB_PATH . 'includes';
			$admin_files = array(
				'class-wesub-product-settings.php',
				'class-wesub-settings.php',
				'class-wesub-table-list.php',
			);
			$admin_files = apply_filters('wesub_subscription_admin_files', $admin_files);
			foreach ($admin_files as $path) {
				include_once WESUB_PATH . 'includes/admin/' . $path;
			}
			$files = array(
				'class-wesub-handle-pricing.php',
				'functions-wesub-common.php',
				'wesub-order-type.php',
				'wesub-handle-subscription.php',
				'wesub-scheduled-actions.php',
				'class-wesub-log.php',
			);
			$files = apply_filters('wesub_subscription_files', $files);
			foreach ($files as $path) {
				include_once WESUB_PATH . 'includes/' . $path;
			}
		}

		/**
		 * Enqueue the js scripts and css callback
		 *
		 * @since  1.0.0
		 */
		public function wesub_enqueue_admin_scripts()
		{
			$admin_js_files = array(
				'we-subscription-product-edit-page',
				'we-subscription-setting-page',
			);
			foreach ($admin_js_files as $file_name) {

				// Register your script.
				wp_register_script($file_name, WESUB_URL . 'assets/admin/js/' . $file_name . '.js', array('jquery'), WESUB_VERSION, true);

				// Enqueue the script.
				wp_enqueue_script($file_name);

				// Localize the script.
				if ('we-subscription-product-edit-page' === $file_name) {
					wp_localize_script(
						$file_name,
						'wesub_product_edit_object',
						array(
							'day' => __('Day', 'we-subscription'),
							'week' => __('Week', 'we-subscription'),
							'month' => __('Month', 'we-subscription'),
							'year' => __('Year', 'we-subscription'),
						)
					);
				} else {
					wp_localize_script(
						$file_name,
						'wesub_localize_object',
						array(
							'ajaxurl' => admin_url('admin-ajax.php'),
							'nonce' => wp_create_nonce('we-backend-nonce'),
						)
					);
				}
			}
			// Enqueue the css files.
			$admin_css_files = array(
				'we-subscription-product-edit-page',
				'we-subscription-setting-page',
			);
			foreach ($admin_css_files as $file_name) {
				wp_enqueue_style($file_name, WESUB_URL . 'assets/admin/css/' . $file_name . '.css', WESUB_VERSION, true);
			}
		}

		/**
		 * Include wesub_subscription_emails files
		 *
		 * @param mixed $email_classes .
		 * @return array
		 */
		public function wesub_subscription_emails($email_classes)
		{
			require WESUB_PATH . 'emails/class-wesub-cancelled-email.php';
			require WESUB_PATH . 'emails/class-wesub-expired-email.php';
			require WESUB_PATH . 'emails/class-wesub-renewal-email.php';
			$email_classes['wesub_cancelled_email'] = new WESUB_Cancelled_Email();
			$email_classes['wesub_expired_email'] = new WESUB_Expired_Email();
			$email_classes['wesub_renewal_email'] = new WESUB_Renewal_Email();

			return $email_classes;
		}

		/**
		 * Add_gateways_compatibility
		 */
		public function wesub_add_gateways_compatibility()
		{
			if (class_exists('WC_Gateway_Stripe')) {
				include_once WESUB_PATH . 'gateways/woocommerce-gateway-stripe/class-wesub-woocommerce-gateway-stripe.php';
				WESUB_Woocommerce_Gateway_Stripe::get_instance();
			}
		}

		/**
		 * Enqueue the js scripts and css callback
		 *
		 * @since  1.0.0
		 */
		public function wesub_enqueue_scripts()
		{
			if (true || is_cart() || is_checkout()) {
				wp_register_script('wesub-wc-block', WESUB_URL . 'wc-block/wesub-wc-block.js', array('jquery', 'wp-data'), WESUB_VERSION, true);
				$settings = get_option('wesub_settings');
				$button_text = isset($settings['place_order_text']) ? $settings['place_order_text'] : null;

				wp_localize_script(
					'wesub-wc-block',
					'wesub_block',
					array(
						'button_text' => $button_text,
					)
				);
				wp_enqueue_script('wesub-wc-block');
			}

			$css_files = array(
				'we-subscription-my-account',
			);
			foreach ($css_files as $file_name) {
				wp_enqueue_style($file_name, WESUB_URL . 'assets/css/' . $file_name . '.css', WESUB_VERSION, true);
			}
		}
	}
}
new Wesub_Autoloader();
