<?php
/**
 * This function will register subscription list table.
 *
 * @since 1.0.0
 * @package    we-subscription
 * @subpackage we-subscription/includes
 */

if (!class_exists('WP_List_Table')) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * Add the Subscription List Table
 */
class WESUB_Table_List extends WP_List_Table
{
	/**
	 * Define table columns
	 */
	public function get_columns()
	{
		$columns = array();
		$columns['cb'] = '<input type="checkbox" />';
		$columns['wesub_id'] = __('ID', 'we-subscription');
		$columns['wesub_parent_id'] = __('Parent ID', 'we-subscription');
		$columns['wesub_status'] = __('Status', 'we-subscription');
		$columns['wesub_username'] = __('Username', 'we-subscription');
		$columns['wesub_payment_method'] = __('Payment Method', 'we-subscription');
		$columns['wesub_amount'] = __('Amount', 'we-subscription');
		$columns['wesub_next_renewal'] = __('Next Renewal', 'we-subscription');
		$columns['wesub_renewals'] = __('Renewals', 'we-subscription');
		$columns['wesub_expiry_date'] = __('Expiry', 'we-subscription');
		$columns['wesub_created_date'] = __('Created', 'we-subscription');
		return apply_filters('wesub_post_column_labels', $columns);
	}

	/**
	 * Fetch data for the table
	 */
	public function prepare_items()
	{
		$columns = $this->get_columns();
		$hidden = array();
		$sortable = $this->get_sortable_columns();

		$this->_column_headers = array($columns, $hidden, $sortable);
		$this->process_bulk_action();

		$this->items = $this->get_data();

		$per_page = 10;
		$current_page = $this->get_pagenum();
		$total_items = count($this->items);

		$this->set_pagination_args(
			array(
				'total_items' => $total_items,
				'per_page' => $per_page,
			)
		);

		// Paginate your data.
		$this->items = array_slice($this->items, (($current_page - 1) * $per_page), $per_page);
	}

	/**
	 * Get the data
	 */
	public function get_data()
	{

		$per_page = 10;
		$current_page = $this->get_pagenum();
		$offset = ($current_page - 1) * $per_page;

		$wesub_args = array(
			'type' => 'wesub_subscriptions',
			'limit' => $per_page,
			'offset' => $offset,
			'orderby' => 'date',
			'order' => 'DESC',
			'return' => 'ids',
		);
		$subscriptions = wc_get_orders($wesub_args);
		// Search.
		$search = isset($_REQUEST['wesub_search']) ? sanitize_text_field(wp_unslash($_REQUEST['wesub_search'])) : '';
		if (!empty($search)) {
			// Verify nonce for search action.
			if (!isset($_REQUEST['wesub_search_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_REQUEST['wesub_search_nonce'])), 'wesub_search_action')) {
				$search = '';
			} else {
				$subscriptions = new Wesub_Subscription($search);
				if (empty($subscriptions)) {
					$subscriptions = array();
				} else {
					$subscriptions = array($search);
				}
			}
		}

		$total_items = count($subscriptions);

		$data = array();
		foreach ($subscriptions as $id) {
			$wesub_subscription = new Wesub_Subscription($id);

			$parent_id = $wesub_subscription->get_meta('wesub_parent_order_id');
			$parent_order = wc_get_order($parent_id);
			$parent_order_url = null;
			if ($parent_order) {
				$parent_order_url = $parent_order->get_edit_order_url();
			}
			$wesub_user_id = $wesub_subscription->get_user_id();
			$user_info = get_userdata($wesub_user_id);
			$username = null;
			if ($user_info) {
				$username = $user_info->user_login;
			}
			$wesub_date_created = $wesub_subscription->get_date_created();
			if ($wesub_date_created) {
				$wesub_date_created = $wesub_date_created->date(wc_date_format());
			}

			if ($wesub_subscription->get_meta('wesub_parent_order_id')) {
				$data[] = array(
					'wesub_id' => $wesub_subscription->get_id(),
					'wesub_parent_id' => $wesub_subscription->get_meta('wesub_parent_order_id'),
					'wesub_parent_edit_url' => $parent_order_url,
					'wesub_username' => $username,
					'wesub_payment_method' => $wesub_subscription->get_payment_method_title(),
					'wesub_status' => $wesub_subscription->get_meta('wesub_status'),
					'wesub_amount' => wc_price($wesub_subscription->get_total()),
					'wesub_next_renewal' => ($wesub_subscription->get_meta('wesub_next_renewal') ? date_i18n(wc_date_format(), $wesub_subscription->get_meta('wesub_next_renewal')) : '---'),
					'wesub_renewals' => $wesub_subscription->get_meta('wesub_renewals'),
					'wesub_expiry_date' => ($wesub_subscription->get_meta('wesub_expire_date') ? date_i18n(wc_date_format(), $wesub_subscription->get_meta('wesub_expire_date')) : '---'),
					'wesub_created_date' => $wesub_date_created,
				);
			}
		}
		return apply_filters('wesub_fetch_list_data', $data);
	}

	/**
	 * Render a column
	 *
	 * @param mixed $wesub_item .
	 * @param mixed $column_name .
	 * @return void
	 */
	public function column_default($wesub_item, $column_name)
	{
		switch ($column_name) {
			case 'wesub_id':
				echo esc_attr($wesub_item['wesub_id']);
				break;
			case 'wesub_parent_id':
				echo wp_kses_post('<a target="__blank" href="' . $wesub_item['wesub_parent_edit_url'] . '">' . $wesub_item['wesub_parent_id'] . '</a>');
				break;
			case 'wesub_username':
				echo esc_attr($wesub_item['wesub_username']);
				break;
			case 'wesub_payment_method':
				echo esc_attr($wesub_item['wesub_payment_method']);
				break;
			case 'wesub_status':
				$status = $wesub_item['wesub_status'];
				echo '<span class="wesub-status-badge status-' . esc_attr($status) . '">' . esc_html(ucfirst($status)) . '</span>';
				break;
			case 'wesub_amount':
				echo wp_kses_post($wesub_item['wesub_amount']);
				break;
			case 'wesub_next_renewal':
				echo esc_attr($wesub_item['wesub_next_renewal']);
				break;
			case 'wesub_renewals':
				if (!empty($wesub_item['wesub_renewals']) && is_array($wesub_item['wesub_renewals'])) {
					echo '<button type="button" class="button wesub-view-renewals" data-id="' . esc_attr($wesub_item['wesub_id']) . '">' . esc_html__('View Renewals', 'we-subscription') . '</button>';
					echo '<div id="wesub-renewals-content-' . esc_attr($wesub_item['wesub_id']) . '" style="display:none;">';
					echo '<table class="widefat fixed striped">';
					echo '<thead><tr>';
					echo '<th>' . esc_html__('ID', 'we-subscription') . '</th>';
					echo '<th>' . esc_html__('Status', 'we-subscription') . '</th>';
					echo '<th>' . esc_html__('Total', 'we-subscription') . '</th>';
					echo '<th>' . esc_html__('Date Created', 'we-subscription') . '</th>';
					echo '</tr></thead>';
					echo '<tbody>';
					foreach (array_reverse($wesub_item['wesub_renewals']) as $wesub_renewal_id) {
						$order = wc_get_order($wesub_renewal_id);
						if ($order) {
							echo '<tr>';
							echo '<td><a href="' . esc_url($order->get_edit_order_url()) . '" target="_blank">#' . esc_html($wesub_renewal_id) . '</a></td>';
							echo '<td><span class="wesub-status-badge status-' . esc_attr($order->get_status()) . '">' . esc_html(wc_get_order_status_name($order->get_status())) . '</span></td>';
							echo '<td>' . wp_kses_post($order->get_formatted_order_total()) . '</td>';
							echo '<td>' . esc_html($order->get_date_created()->date_i18n(wc_date_format())) . '</td>';
							echo '</tr>';
						}
					}
					echo '</tbody>';
					echo '</table>';
					echo '</div>';
				} else {
					echo '---';
				}
				break;
			case 'wesub_expiry_date':
				echo esc_attr($wesub_item['wesub_expiry_date']);
				break;
			case 'wesub_created_date':
				echo esc_attr($wesub_item['wesub_created_date']);
				break;
		}
		do_action('wesub_post_column_label_content', $column_name, $wesub_item);
	}

	/**
	 * Render checkbox column
	 *
	 * @param mixed $wesub_item .
	 * @return string
	 */
	public function column_cb($wesub_item)
	{
		return sprintf('<input type="checkbox" name="wesub_id[]" value="%s" />', $wesub_item['wesub_id']);
	}

	/**
	 * Define bulk actions
	 *
	 * @return mixed
	 */
	public function get_bulk_actions()
	{
		$actions = array(
			'wesub-delete' => 'Delete',
			'wesub-cancel' => 'Cancel',
		);
		return apply_filters('wesub_bulk_actions', $actions);
	}

	/**
	 * Process_bulk_action
	 *
	 * @return void
	 */
	public function process_bulk_action()
	{
		$action = $this->current_action();
		if (!$action) {
			return;
		}

		if ('wesub-delete' === $action || 'wesub-cancel' === $action) {
			// Verify nonce for bulk actions.
			if (!isset($_POST['wesub_bulk_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['wesub_bulk_nonce'])), 'wesub_bulk_action')) {
				wp_die(esc_html__('Security check failed. Please try again.', 'we-subscription'));
			}
		}

		if ('wesub-delete' === $action) {
			if (isset($_POST['wesub_id']) && !empty($_POST['wesub_id'])) {
				$all_ids = map_deep(wp_unslash($_POST['wesub_id']), 'sanitize_text_field');
				foreach ($all_ids as $key => $id) {
					$wesub_subscription = new Wesub_Subscription($id);
					$wesub_subscription->delete(true);
					do_action('wesub_subscription_deleted', $id);
				}
				?>
								<div class="notice notice-warning is-dismissible">
									<p><strong><?php esc_html_e('Subscriptions Deleted Successfully', 'we-subscription'); ?></strong></p>
								</div>
								<?php
			}
		} elseif ('wesub-cancel' === $action) {
			if (isset($_POST['wesub_id']) && !empty($_POST['wesub_id'])) {
				$all_ids = map_deep(wp_unslash($_POST['wesub_id']), 'sanitize_text_field');
				foreach ($all_ids as $key => $id) {
					$wesub_subscription = new Wesub_Subscription($id);
					$wesub_subscription->update_meta_data('wesub_status', 'cancelled');
					$wesub_subscription->save();

					$mailer = WC()->mailer()->get_emails();
					if (isset($mailer['wesub_cancelled_email'])) {
						$mailer['wesub_cancelled_email']->trigger($wesub_subscription);
					}
					do_action('wesub_subscription_cancelled', $id);

					$assigned_role = $wesub_subscription->get_meta('wesub_assigned_user_role');
					$settings = get_option('wesub_settings');

					$remove_assigned_role = isset($settings['remove_assign_user_role']) ? $settings['remove_assign_user_role'] : null;
					if ($assigned_role && 'on' == $remove_assigned_role) {
						wesub_remove_user_role($wesub_subscription->get_user_id(), $assigned_role);
					}
				}
				?>
								<div class="notice notice-success is-dismissible">
									<p><strong><?php esc_html_e('Subscriptions Cancelled Successfully', 'we-subscription'); ?></strong></p>
								</div>
								<?php
			}
		}
		$sanitized_post = map_deep(wp_unslash($_POST), 'sanitize_text_field');
		do_action('wesub_bulk_actions', $this->current_action(), $sanitized_post);
	}

	/**
	 * Extra_tablenav
	 *
	 * @param mixed $which .
	 * @return void
	 */
	public function extra_tablenav($which)
	{
		if ('top' === $which) {
			do_action('wesub_table_filter');
			?>
					<?php
		}
	}

	/**
	 * Add search box
	 *
	 * @param mixed $text .
	 * @param mixed $input_id .
	 * @return void
	 */
	public function search_box($text, $input_id)
	{
		if (empty($_REQUEST['s']) && !$this->has_items()) {
			return;
		}
		?>
				<p class="search-box">
					<?php wp_nonce_field('wesub_search_action', 'wesub_search_nonce'); ?>
					<input type="search" id="<?php echo esc_attr($input_id); ?>" name="wesub_search"
						value="<?php esc_attr(_admin_search_query()); ?>" />
					<?php do_action('wesub_add_more_search_fields'); ?>
					<input type="submit" id="search-submit" class="button" value="<?php echo esc_attr($text); ?>" />
				</p>
				<?php
	}
}