<?php
/**
 * Handle the subscription Product and it's setting
 *
 * @class Wesub_Product_Settings
 * @package    we-subscription
 * @subpackage we-subscription/includes/admin
 */

defined('ABSPATH') || exit; // Exit if accessed directly.

if (!class_exists('Wesub_Product_Settings')) {
	/**
	 * Wesub_Product_Settings
	 */
	class Wesub_Product_Settings
	{
		/**
		 * The single instance of the class.
		 *
		 * @var Wesub_Product_Settings
		 */
		private static $instance;

		/**
		 * Current Class instance
		 */
		public static function instance()
		{
			return !is_null(self::$instance) ? self::$instance : self::$instance = new self();
		}

		/**
		 * Contructor
		 */
		public function __construct()
		{
			// Subscription for the Simple Product.
			add_action('product_type_options', array($this, 'wesub_simple_subscription_checkbox'));
			add_action('woocommerce_product_options_general_product_data', array($this, 'wesub_simple_subscription_settings'));
			add_action('save_post', array($this, 'wesub_simple_subscription_settings_save'));

			// Subscription for the variable product.
			add_action('woocommerce_variation_options', array($this, 'wesub_variation_subscription_checkbox'), 10, 3);
			add_action('woocommerce_variation_options_pricing', array($this, 'wesub_variation_subscription_settings'), 10, 3);
			add_action('woocommerce_save_product_variation', array($this, 'wesub_variation_subscription_settings_save'), 10, 2);
		}

		/**
		 * Function to add subscription checkbox
		 *
		 * @name wesub_simple_subscription_checkbox
		 * @since    1.0.0
		 * @param    Array $products_type Products type.
		 * @return   Array  $products_type.
		 */
		public function wesub_simple_subscription_checkbox($products_type)
		{
			$products_type['wesub_simple_subscription'] = array(
				'id' => '_wes_simple_subscription',
				'wrapper_class' => 'show_if_simple',
				'label' => __('Enable Subscription', 'we-subscription'),
				'description' => __('Enable Subscriptions for simple product.', 'we-subscription'),
				'default' => 'no',
			);
			return apply_filters('wesub_simple_product_checkbox', $products_type);
		}

		/**
		 * Add checkbox field to product data meta box
		 */
		public function wesub_simple_subscription_settings()
		{
			global $post;
			$product_settings = get_post_meta($post->ID, 'wesub_product_setting', true);
			$enabled = get_post_meta($post->ID, 'wesub_product_enabled', true);

			$signup_fee = isset($product_settings['signup_fee']) ? $product_settings['signup_fee'] : null;
			$expire_after = isset($product_settings['expire_after']) ? $product_settings['expire_after'] : null;
			$interval_type = isset($product_settings['interval_type']) ? $product_settings['interval_type'] : null;
			$interval_period = isset($product_settings['interval_period']) ? $product_settings['interval_period'] : 1;
			$expire_type = isset($interval_type) ? $interval_type : esc_html__('Day', 'we-subscription');

			echo '<section class="wesub-product-settings-wrapper" style="display:none;">
                <input type="hidden" name="wesub_nonce" value="' . esc_attr(wp_create_nonce($post->ID)) . '">
                <h2><b>' . esc_html__('Subscription Settings', 'we-subscription') . '</b></h2>
                <p class="wesub-form-field">
                    <label><b>' . esc_html__('User will pay every', 'we-subscription') . '</b></label>
                    <span class="wrap">
                        <input type="number" name="wesub-recurring-period" class="wesub-recurring-period short" value="' . esc_html($interval_period) . '">
                        <select name="wesub-recurring-type" class="wesub-recurring-type short">
                            <option value="day" ' . (('day' === $interval_type) ? 'selected' : '') . '>' . esc_html__('Day', 'we-subscription') . '</option>
                            <option value="week"  ' . (('week' === $interval_type) ? 'selected' : '') . '>' . esc_html__('Week', 'we-subscription') . '</option>
                            <option value="month"  ' . (('month' === $interval_type) ? 'selected' : '') . '>' . esc_html__('Month', 'we-subscription') . '</option>
                            <option value="year"  ' . (('year' === $interval_type) ? 'selected' : '') . '>' . esc_html__('Year', 'we-subscription') . '</option>
                        </select>
                    </span>
                </p>
                <p class="wesub-form-field">
                    <label><b>' . esc_html__('Expire after', 'we-subscription') . '</b></label>
                    <input type="number" name="wesub-expiration" class="wesub-expiration short" value="' . esc_html($expire_after) . '">
                    <input type="text" name="wesub-expiration-type" class="wesub-expiration-type short" value="' . esc_html(ucfirst($expire_type)) . '" disabled />
                </p>
                <p class="wesub-form-field">
                    <label><b>' . esc_html__('Sign-up fee', 'we-subscription') . '</b></label>
                    <input type="number" name="wesub-signup-fee" class="wesub-signup-fee short" value="' . esc_html($signup_fee) . '">
                </p>';
			do_action('wesub_subcription_add_more_fields', $post);
			echo '</section>';
		}

		/**
		 * This function used to save the subscription setting.
		 *
		 * @param integer $post_id product id .
		 */
		public function wesub_simple_subscription_settings_save($post_id)
		{

			if (!isset($_POST['post_type']) || 'product' !== $_POST['post_type']) { // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
				return;
			}

			if (!isset($_POST['wesub_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['wesub_nonce'])), $post_id)) {
				return;
			}
			if (!current_user_can('edit_post', $post_id)) {
				return;
			}

			if (isset($_POST['_wes_simple_subscription']) && 'on' === $_POST['_wes_simple_subscription']) {

				$signup_fee = isset($_POST['wesub-signup-fee']) ? sanitize_text_field(wp_unslash($_POST['wesub-signup-fee'])) : null;
				$expire_after = isset($_POST['wesub-expiration']) ? sanitize_text_field(wp_unslash($_POST['wesub-expiration'])) : null;
				$interval_type = isset($_POST['wesub-recurring-type']) ? sanitize_text_field(wp_unslash($_POST['wesub-recurring-type'])) : null;
				$interval_period = isset($_POST['wesub-recurring-period']) ? sanitize_text_field(wp_unslash($_POST['wesub-recurring-period'])) : null;
				$expire_type = isset($_POST['wesub-expiration-type']) ? sanitize_text_field(wp_unslash($_POST['wesub-expiration-type'])) : null;

				$settings_array = array(
					'signup_fee' => $signup_fee,
					'expire_after' => $expire_after,
					'interval_type' => $interval_type,
					'interval_period' => $interval_period,
					'expire_type' => $expire_type,
				);
				update_post_meta($post_id, 'wesub_product_setting', $settings_array);
				update_post_meta($post_id, '_wes_simple_subscription', 'yes');
			} else {
				update_post_meta($post_id, '_wes_simple_subscription', 'no');
			}
			$sanitized_post = map_deep(wp_unslash($_POST), 'sanitize_text_field');
			do_action('wesub_simple_product_setting_save', $sanitized_post);
		}

		/**
		 * Add the checkbox for the variation product
		 *
		 * @param int    $loop refer the variation setting.
		 * @param array  $variation_data .
		 * @param object $variation .
		 */
		public function wesub_variation_subscription_checkbox($loop, $variation_data, $variation)
		{
			$is_checked = get_post_meta($variation->ID, 'wesub_variation_subscription', true);
			$is_checked = ('yes' === $is_checked) ? 'checked' : '';
			?>
			<label class="tips" data-tip="<?php esc_attr_e('Enable Subscription for variation product', 'we-subscription'); ?>">
				<?php esc_html_e('Enable Subscription', 'we-subscription'); ?>
				<input type="checkbox" class="checkbox wesub-variation-subscription"
					name="wesub-variation-subscription[<?php echo esc_attr($loop); ?>]" <?php echo esc_attr($is_checked); ?> />
			</label>
			<?php
			do_action('wesub_variable_product_checkbox', $loop, $variation_data, $variation);
		}

		/**
		 * Add the checkbox for the variation product
		 *
		 * @param int    $loop refer the variation setting.
		 * @param array  $variation_data .
		 * @param object $variation .
		 */
		public function wesub_variation_subscription_settings($loop, $variation_data, $variation)
		{
			$post = $variation;

			$product_settings = get_post_meta($post->ID, 'wesub_product_setting', true);
			$enabled = get_post_meta($post->ID, 'wesub_product_enabled', true);

			$signup_fee = isset($product_settings['signup_fee']) ? $product_settings['signup_fee'] : null;
			$expire_after = isset($product_settings['expire_after']) ? $product_settings['expire_after'] : null;
			$interval_type = isset($product_settings['interval_type']) ? $product_settings['interval_type'] : null;
			$interval_period = isset($product_settings['interval_period']) ? $product_settings['interval_period'] : 1;
			$expire_type = isset($interval_type) ? $interval_type : esc_html__('Day', 'we-subscription');
			// Variation setting goes here.
			echo '<section class="form-row form-row-full wesub-var-product-settings-wrapper" style="display:none;">
                <input type="hidden" name="wesub_nonce[' . esc_attr($loop) . ']" value="' . esc_attr(wp_create_nonce($post->ID)) . '">
                <h2><b>' . esc_html__('Subscription Settings', 'we-subscription') . '</b></h2>
                <p class="form-field">
                    <label><b>' . esc_html__('User will pay every', 'we-subscription') . '</b></label>
                    <input type="number" name="wesub-var-recurring-period[' . esc_attr($loop) . ']" class="wesub-var-recurring-period short" value="' . esc_html($interval_period) . '">
                    <select name="wesub-var-recurring-type[' . esc_attr($loop) . ']" class="wesub-var-recurring-type short">
                        <option value="day" ' . (('day' === $interval_type) ? 'selected' : '') . '>' . esc_html__('Day', 'we-subscription') . '</option>
                        <option value="week"  ' . (('week' === $interval_type) ? 'selected' : '') . '>' . esc_html__('Week', 'we-subscription') . '</option>
                        <option value="month"  ' . (('month' === $interval_type) ? 'selected' : '') . '>' . esc_html__('Month', 'we-subscription') . '</option>
                        <option value="year"  ' . (('year' === $interval_type) ? 'selected' : '') . '>' . esc_html__('Year', 'we-subscription') . '</option>
                    </select>
                </p>
                <p class="form-field">
                    <label><b>' . esc_html__('Expire after', 'we-subscription') . '</b></label>
                    <input type="number" name="wesub-var-expiration[' . esc_attr($loop) . ']" class="wesub-var-expiration short" value="' . esc_html($expire_after) . '">
                    <input type="text" name="wesub-var-expiration-type[' . esc_attr($loop) . ']" class="wesub-var-expiration-type short" value="' . esc_html(ucfirst($expire_type)) . '" disabled />
                </p>
                <p class="form-field">
                    <label><b>' . esc_html__('Sign-up fee', 'we-subscription') . '</b></label>
                    <input type="number" name="wesub-var-signup-fee[' . esc_attr($loop) . ']" class="wesub-var-signup-fee short" value="' . esc_html($signup_fee) . '">
                </p>';
			do_action('wesub_var_subcription_add_more_fields', $$loop, $variation_data, $variation);
			echo '</section>';
		}

		/**
		 * Function to save the variation subscription related data
		 *
		 * @param integer $variation_id .
		 * @param integer $loop refer to variable product setting.
		 */
		public function wesub_variation_subscription_settings_save($variation_id, $loop)
		{
			if (!isset($_POST['wesub_nonce'][$loop]) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['wesub_nonce'][$loop])), $variation_id)) {
				return;
			}

			$parent_id = wp_get_post_parent_id($variation_id);
			if (!current_user_can('edit_post', $parent_id ? $parent_id : $variation_id)) {
				return;
			}

			if (!empty($_POST['wesub-variation-subscription']) && !empty($_POST['wesub-variation-subscription'][$loop])) {
				update_post_meta($variation_id, 'wesub_variation_subscription', 'yes');

				$signup_fee = isset($_POST['wesub-var-signup-fee'][$loop]) ? sanitize_text_field(wp_unslash($_POST['wesub-var-signup-fee'][$loop])) : null;
				$expire_after = isset($_POST['wesub-var-expiration'][$loop]) ? sanitize_text_field(wp_unslash($_POST['wesub-var-expiration'][$loop])) : null;
				$interval_type = isset($_POST['wesub-var-recurring-type'][$loop]) ? sanitize_text_field(wp_unslash($_POST['wesub-var-recurring-type'][$loop])) : null;
				$interval_period = isset($_POST['wesub-var-recurring-period'][$loop]) ? sanitize_text_field(wp_unslash($_POST['wesub-var-recurring-period'][$loop])) : null;
				$expire_type = isset($_POST['wesub-var-expiration-type'][$loop]) ? sanitize_text_field(wp_unslash($_POST['wesub-var-expiration-type'][$loop])) : null;

				$settings_array = array(
					'signup_fee' => $signup_fee,
					'expire_after' => $expire_after,
					'interval_type' => $interval_type,
					'interval_period' => $interval_period,
					'expire_type' => $expire_type,
				);
				update_post_meta($variation_id, 'wesub_product_setting', $settings_array);

			} else {
				update_post_meta($variation_id, 'wesub_variation_subscription', 'no');
			}
			$sanitized_post = map_deep(wp_unslash($_POST), 'sanitize_text_field');
			do_action('wesub_variable_product_setting_save', $sanitized_post, $variation_id, $loop);
		}
	}
}