<?php
/**
 * Override the existing stripe class to handle the subscription compatibility and its renewal
 *
 * @link       https://webeffortless.com/
 * @since      1.0.0
 *
 * @package    we-subscription
 * @subpackage we-subscription/gateways/woocommerce-gateway-stripe
 */

if (!defined('ABSPATH')) {
    exit;
}
if (!class_exists('WESUB_Woocommerce_Gateway_Stripe')) {

    class WESUB_Woocommerce_Gateway_Stripe
    {
        /**
         * The single instance of the class
         */
        private static $instance = null;

        /**
         * Return the instance of Gateway
         *
         * @return WESUB_Woocommerce_Gateway_Stripe
         */
        public static function get_instance()
        {
            return !is_null(self::$instance) ? self::$instance : self::$instance = new self();
        }

        /**
         * Constructor
         */
        public function __construct()
        {
            add_filter('wc_stripe_force_save_source', array($this, 'wesub_stripe_force_save_source'), 10, 2);
            add_filter('wc_stripe_display_save_payment_method_checkbox', array($this, 'display_save_payment_method_checkbox'));
            add_filter('wc_stripe_generate_create_intent_request', [$this, 'add_subscription_information_to_intent'], 10, 4);

            add_action('wesub_do_renewal_payment_stripe', array($this, 'wesub_handle_stripe_renewal_payment'), 10, 2);
            add_action('wesub_do_renewal_payment_stripe_sepa_debit', array($this, 'wesub_handle_stripe_renewal_payment'), 10, 2);

            // Save Stripe customer and source IDs to subscription when payment is processed
            add_action('wc_stripe_save_source', array($this, 'wesub_save_stripe_source_to_subscription'), 10, 2);
            add_action('woocommerce_payment_complete', array($this, 'wesub_update_subscription_payment_meta'), 10, 1);

            // Update Payment method
            add_action('woocommerce_stripe_add_payment_method', [$this, 'handle_add_payment_method_success'], 10, 2);
            add_action('wc_stripe_payment_fields_' . 'stripe', [$this, 'display_update_subs_payment_checkout']);
        }


        public function wesub_stripe_force_save_source($force_save_source, $customer)
        {
            return true;
        }


        public function display_save_payment_method_checkbox($display)
        {
            if (function_exists('wesub_cart_has_subscription') && wesub_cart_has_subscription()) {
                return false;
            }
            // Only render the "Save payment method" checkbox if there are no subscription products in the cart.
            return $display;
        }


        /**
         * Add the necessary information to payment intents for subscriptions to allow Stripe to create
         * mandates for 3DS payments in India. It's ok to apply this across the board; Stripe will
         * take care of handling any authorizations.
         *
         * @param array    $request          The HTTP request that will be sent to Stripe to create the payment intent.
         * @param WC_Order $order            The renewal order.
         * @param object   $prepared_source  The source object.
         */
        public function add_subscription_information_to_intent($request, $order, $prepared_source, $is_setup_intent = false)
        {
            // Just in case the order doesn't contain a subscription we return the base request.
            if (!wesub_order_has_subscription($order->get_id())) {
                return $request;
            }

            // TODO: maybe this isn't necessary since this function should really only be called
            //       when creating the intent? It's called in process_subscription_payment though
            //       so it's probably needed here too?
            // If we've already created a mandate for this order; use that.
            $mandate = $order->get_meta('_stripe_mandate_id', true);
            if (isset($request['confirm']) && filter_var($request['confirm'], FILTER_VALIDATE_BOOLEAN) && !empty($mandate)) {
                $request['mandate'] = $mandate;

                // We already have a mandate -- unset mandate_data and setup_future_usage, if set.
                unset($request['mandate_data']);
                unset($request['setup_future_usage']);

                return $request;
            }

            $subscription_ids = $order->get_meta('wesub_subscription_ids', true);

            $wesub_subscription_id = 0;
            if ($subscription_ids) {
                $wesub_subscription_id = $subscription_ids[0];
            }

            // $subscriptions_for_renewal_order = function_exists( 'wcs_get_subscriptions_for_renewal_order' ) ? wcs_get_subscriptions_for_renewal_order( $order ) : [];

            // Check if mandate already exists.
            if ($wesub_subscription_id) {
                $wesub_subscription = new Wesub_Subscription($wesub_subscription_id);
                $mandate = $this->get_mandate_for_subscription($wesub_subscription, isset($request['payment_method']) ? $request['payment_method'] : '');

                if (!empty($mandate)) {
                    $request['confirm'] = 'true';
                    $request['mandate'] = $mandate;

                    // We already have a mandate -- unset mandate_data and setup_future_usage, if set.
                    unset($request['mandate_data']);
                    unset($request['setup_future_usage']);

                    return $request;
                }
            }
            return $request;
        }
        private function get_mandate_for_subscription($order, $payment_method)
        {
            $renewal_order_ids = $order->get_meta('wesub_renewals', true);
            foreach ($renewal_order_ids as $renewal_order_id) {
                $renewal_order = wc_get_order($renewal_order_id);
                if (!$renewal_order instanceof WC_Order) {
                    continue;
                }

                $mandate = $renewal_order->get_meta('_stripe_mandate_id', true);
                $renewal_order_payment_method = WC_Stripe_Order_Helper::get_instance()->get_stripe_source_id($renewal_order);

                // Return from the most recent renewal order with a valid mandate. Mandate is created against a payment method
                // in Stripe so the payment method should also match to reuse the mandate.
                if (!empty($mandate) && $renewal_order_payment_method === $payment_method) {
                    return $mandate;
                }
            }
            return '';
        }
        /**
         * Handle Stripe renewal payment for subscriptions (both Card and SEPA Debit)
         *
         * @param WC_Order $new_order The renewal order
         * @param int $wesub_subscription_id The subscription ID
         * @return void
         */
        public function wesub_handle_stripe_renewal_payment($new_order, $wesub_subscription_id)
        {
            try {
                $order_id = $new_order->get_id();
                $wesub_subscription = new Wesub_Subscription($wesub_subscription_id);

                // Get the total amount to charge
                $amount = $new_order->get_total();

                // Get the payment method from subscription
                $payment_method = $wesub_subscription->get_meta('_payment_method', true);

                // Determine which gateway to use (stripe or stripe_sepa_debit)
                $gateway_id = 'stripe';
                if ('stripe_sepa_debit' === $payment_method || 'stripe_sepa' === $payment_method) {
                    $gateway_id = 'stripe_sepa';
                }

                // Get the appropriate Stripe gateway instance
                $available_gateways = WC()->payment_gateways->payment_gateways();
                $stripe_gateway = $available_gateways[$gateway_id] ?? $available_gateways['stripe'] ?? null;

                if (!$stripe_gateway) {
                    throw new Exception(__('Stripe payment gateway not found', 'we-subscription'));
                }

                // Get source from subscription
                $stripe_customer_id = $wesub_subscription->get_meta('_stripe_customer_id', true);
                $stripe_source_id = $wesub_subscription->get_meta('_stripe_source_id', true);

                if (empty($stripe_customer_id)) {
                    throw new Exception(__('Stripe customer ID not found for subscription', 'we-subscription'));
                }

                // Prepare source object
                $prepared_source = (object) [
                    'customer' => $stripe_customer_id,
                    'source' => $stripe_source_id,
                    'source_object' => null,
                ];

                // Set the customer and source on the order
                $new_order->update_meta_data('_stripe_customer_id', $stripe_customer_id);
                $new_order->update_meta_data('_stripe_source_id', $stripe_source_id);
                $new_order->update_meta_data('_payment_method', $payment_method);
                $new_order->save();

                WC_Stripe_Logger::log("Info: Begin processing subscription renewal payment for order {$order_id} for the amount of {$amount} using gateway {$gateway_id}");

                $is_authentication_required = false;

                // Check if this is SEPA payment - use charges API
                if ('stripe_sepa' === $gateway_id) {
                    // SEPA uses the charges API (legacy method)
                    if (method_exists($stripe_gateway, 'generate_payment_request')) {
                        $request = $stripe_gateway->generate_payment_request($new_order, $prepared_source);
                        $request['capture'] = 'true';
                        $request['amount'] = WC_Stripe_Helper::get_stripe_amount($amount, $request['currency']);
                        $response = WC_Stripe_API::request($request);
                        $is_authentication_required = false;
                    } else {
                        throw new Exception(__('SEPA payment processing method not available', 'we-subscription'));
                    }
                } else {
                    // Card payments use payment intents
                    if (method_exists($stripe_gateway, 'create_and_confirm_intent_for_off_session')) {
                        $response = $stripe_gateway->create_and_confirm_intent_for_off_session($new_order, $prepared_source, $amount);

                        // Check if authentication is required
                        $is_authentication_required = method_exists($stripe_gateway, 'is_authentication_required_for_payment')
                            ? $stripe_gateway->is_authentication_required_for_payment($response)
                            : false;
                    } else {
                        throw new Exception(__('Payment intent processing method not available', 'we-subscription'));
                    }
                }

                // Handle errors
                if (!empty($response->error) && !$is_authentication_required) {
                    $localized_messages = WC_Stripe_Helper::get_localized_messages();

                    if ('card_error' === $response->error->type) {
                        $localized_message = isset($localized_messages[$response->error->code])
                            ? $localized_messages[$response->error->code]
                            : $response->error->message;
                    } else {
                        $localized_message = isset($localized_messages[$response->error->type])
                            ? $localized_messages[$response->error->type]
                            : $response->error->message;
                    }

                    if (isset($response->error->request_log_url)) {
                        $localized_message .= ' ' . make_clickable($response->error->request_log_url);
                    }

                    $new_order->add_order_note($localized_message);
                    throw new Exception($localized_message);
                }

                // Handle authentication required (only for card payments)
                if ($is_authentication_required) {
                    $error_message = __('This transaction requires authentication.', 'we-subscription');
                    $new_order->add_order_note($error_message);

                    $charge = method_exists($stripe_gateway, 'get_latest_charge_from_intent')
                        ? $stripe_gateway->get_latest_charge_from_intent($response->error->payment_intent)
                        : null;

                    if ($charge && isset($charge->id)) {
                        $new_order->set_transaction_id($charge->id);
                        // Translators: %s is the charge ID
                        $new_order->update_status('failed', sprintf(__('Stripe charge awaiting authentication by user: %s.', 'we-subscription'), $charge->id));
                    } else {
                        $new_order->update_status('failed', $error_message);
                    }
                    $new_order->save();
                    return;
                }

                // Process successful payment
                $latest_charge = method_exists($stripe_gateway, 'get_latest_charge_from_intent')
                    ? $stripe_gateway->get_latest_charge_from_intent($response)
                    : null;

                if (method_exists($stripe_gateway, 'process_response')) {
                    $stripe_gateway->process_response((!empty($latest_charge)) ? $latest_charge : $response, $new_order);
                } else {
                    // Fallback: manually process the response
                    if ($latest_charge && isset($latest_charge->id)) {
                        $new_order->set_transaction_id($latest_charge->id);
                        $new_order->payment_complete($latest_charge->id);
                        // Translators: %s is the charge ID
                        $new_order->add_order_note(sprintf(__('Stripe charge complete (Charge ID: %s)', 'we-subscription'), $latest_charge->id));
                    } elseif (isset($response->id)) {
                        // For SEPA charges that don't go through payment intents
                        $new_order->set_transaction_id($response->id);
                        $new_order->payment_complete($response->id);
                        // Translators: %s is the charge ID
                        $new_order->add_order_note(sprintf(__('Stripe charge complete (Charge ID: %s)', 'we-subscription'), $response->id));
                    }
                }

                WC_Stripe_Logger::log("Info: Successfully processed renewal payment for order {$order_id}");

            } catch (Exception $e) {
                WC_Stripe_Logger::log('Error processing renewal payment: ' . $e->getMessage());
                // Translators: %s is the error message
                $new_order->update_status('failed', sprintf(__('Renewal payment failed: %s', 'we-subscription'), $e->getMessage()));
                $new_order->save();
            }
        }

        /**
         * Save Stripe source information to subscription
         * This is called via the wc_stripe_save_source action
         *
         * @param WC_Order $order The order object
         * @param stdClass $source The Stripe source object
         * @return void
         */
        public function wesub_save_stripe_source_to_subscription($order, $source)
        {
            if (!wesub_order_has_subscription($order->get_id())) {
                return;
            }

            $order_id = $order->get_id();
            $subscription_ids = $order->get_meta('wesub_subscription_ids', true);

            if (empty($subscription_ids) || !is_array($subscription_ids)) {
                return;
            }

            foreach ($subscription_ids as $wesub_subscription_id) {
                $wesub_subscription = new Wesub_Subscription($wesub_subscription_id);

                if (!$wesub_subscription || !$wesub_subscription->get_id()) {
                    continue;
                }

                // Save Stripe customer ID
                if (!empty($source->customer)) {
                    $wesub_subscription->update_meta_data('_stripe_customer_id', $source->customer);
                }

                // Save Stripe source/payment method ID
                if (!empty($source->payment_method)) {
                    $wesub_subscription->update_meta_data('_stripe_source_id', $source->payment_method);
                } elseif (!empty($source->source)) {
                    $wesub_subscription->update_meta_data('_stripe_source_id', $source->source);
                }

                // Save payment method type
                $payment_method = $order->get_payment_method();
                if (!empty($payment_method)) {
                    $wesub_subscription->update_meta_data('_payment_method', $payment_method);
                }

                $wesub_subscription->save();
            }
        }

        /**
         * Update subscription payment meta when payment is complete
         * This is a fallback to ensure payment info is saved
         *
         * @param int $order_id The order ID
         * @return void
         */
        public function wesub_update_subscription_payment_meta($order_id)
        {
            $order = wc_get_order($order_id);

            if (!$order || !wesub_order_has_subscription($order_id)) {
                return;
            }

            // Only process Stripe payments
            $payment_method = $order->get_payment_method();
            if (!in_array($payment_method, array('stripe', 'stripe_sepa', 'stripe_sepa_debit'), true)) {
                return;
            }

            $subscription_ids = $order->get_meta('wesub_subscription_ids', true);

            if (empty($subscription_ids) || !is_array($subscription_ids)) {
                return;
            }

            // Get Stripe customer and source from order
            $stripe_customer_id = $order->get_meta('_stripe_customer_id', true);
            $stripe_source_id = $order->get_meta('_stripe_source_id', true);

            if (empty($stripe_customer_id)) {
                return;
            }

            foreach ($subscription_ids as $wesub_subscription_id) {
                $wesub_subscription = new Wesub_Subscription($wesub_subscription_id);

                if (!$wesub_subscription || !$wesub_subscription->get_id()) {
                    continue;
                }

                // Only update if not already set
                if (empty($wesub_subscription->get_meta('_stripe_customer_id', true))) {
                    $wesub_subscription->update_meta_data('_stripe_customer_id', $stripe_customer_id);
                }

                if (!empty($stripe_source_id) && empty($wesub_subscription->get_meta('_stripe_source_id', true))) {
                    $wesub_subscription->update_meta_data('_stripe_source_id', $stripe_source_id);
                }

                if (empty($wesub_subscription->get_meta('_payment_method', true))) {
                    $wesub_subscription->update_meta_data('_payment_method', $payment_method);
                }

                $wesub_subscription->save();
            }
        }


        /**
         * Displays a checkbox to allow users to update all subs payments with new
         * payment.
         *
         * @since 4.1.11
         */
        public function display_update_subs_payment_checkout()
        {

            if (!isset($_GET['change_payment_method']) || $_GET['change_payment_method'] !== 'true' || !isset($_GET['wesub_subscription_id']) || !apply_filters('wc_stripe_display_update_subs_payment_method_card_checkbox', true)) {
                return;
            }

            if (!is_add_payment_method_page()) {
                return;
            }

            $wesub_subscription_id = isset($_GET['wesub_subscription_id']) ? sanitize_text_field(wp_unslash($_GET['wesub_subscription_id'])) : 0;

            if (function_exists('wesub_get_user_active_subscription_count') && wesub_get_user_active_subscription_count(get_current_user_id())) {
                // Translators: %s is the subscription ID.
                printf(esc_html__('This will update the payment method for the subscription #%s', 'we-subscription'), esc_html($wesub_subscription_id));
            }
        }

        /**
         * Updates all the user's active subscriptions payment method with the new payment method.
         *
         * @since 8.8.0
         *
         * @param int      $wesub_user_id               The user ID.
         * @param stdClass $payment_method_object The newly added payment method object.
         */
        public function handle_add_payment_method_success($wesub_user_id, $payment_method_object)
        {
            // To avoid errors, exit early if there is no WC_Subscriptions_Change_Payment_Gateway class or the payment method object is not complete.
            if (!isset($_GET['change_payment_method']) || $_GET['change_payment_method'] !== 'true' || !isset($_GET['wesub_subscription_id']) || !isset($payment_method_object->id)) {
                return;
            }
            $wesub_subscription_id = isset($_GET['wesub_subscription_id']) ? sanitize_text_field(wp_unslash($_GET['wesub_subscription_id'])) : 0;

            if (!isset($_GET['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'wesub_change_payment_' . $wesub_subscription_id)) {
                return;
            }

            $wesub_subscription = new Wesub_Subscription($wesub_subscription_id);
            $stripe_customer = new WC_Stripe_Customer($wesub_user_id);

            if (!$wesub_subscription || !$wesub_subscription->get_id()) {
                return;
            }

            // Ensure the user owns this subscription.
            if ((int) $wesub_subscription->get_user_id() !== (int) $wesub_user_id) {
                return;
            }

            $wesub_subscription->update_meta_data('_stripe_source_id', $payment_method_object->id);
            $wesub_subscription->update_meta_data('_stripe_customer_id', $stripe_customer->get_id());
            $wesub_subscription->add_order_note(__('Payment method updated successfully.', 'we-subscription'));
            $wesub_subscription->save();
        }
    }
}
