<?php
/**
 * Cancelled Email template
 *
 * @since 1.0.0
 *
 * @package    we-subscription
 * @subpackage we-subscription/email
 */

if (!defined('ABSPATH')) {
	exit;
}

if (!class_exists('WESUB_Renewal_Email')) {
	/**
	 * Cancellation Email template Class
	 *
	 * @since      1.0.0
	 */
	class WESUB_Renewal_Email extends WC_Email
	{
		/**
		 * Subscription
		 *
		 * @var subscription
		 */
		public $wesub_subscription;
		/**
		 * Renewal Order
		 *
		 * @var renewal_order
		 */
		public $renewal_order;

		/**
		 * Create class for email notification.
		 *
		 * @access public
		 */
		public function __construct()
		{

			$this->id = 'wesub_renewal_email';
			$this->title = __('Subscription Renewal Email Notification', 'we-subscription');
			$this->description = __('This email notification send to customer, If the subscription is getting renewed', 'we-subscription');
			$this->template_html = 'wesub-renewal-email-template.php';
			$this->template_plain = 'plain/wesub-renewal-email-template.php';
			$this->template_base = plugin_dir_path(__FILE__) . 'templates/';

			$this->placeholders = array(
				'{site_title}' => $this->get_blogname(),
				'{subscription_id}' => '',
				'{renewal_id}' => '',
			);
			$this->customer_email = true;
			parent::__construct();
		}

		/**
		 * Get email subject.
		 *
		 * @since  1.0.0
		 * @return string
		 */
		public function get_default_subject()
		{
			return __('Susbcription #{subscription_id} Renewal #{renewal_id}', 'we-subscription');
		}

		/**
		 * Get email heading.
		 *
		 * @since  1.0.0
		 * @return string
		 */
		public function get_default_heading()
		{
			return __('Subscription Renewed', 'we-subscription');
		}

		/**
		 * This function is used to trigger for email.
		 *
		 * @since  1.0.0
		 * @param object $renewal_order .
		 * @access public
		 * @return void
		 */
		public function trigger($renewal_order)
		{

			$wesub_subscription_id = $renewal_order->get_meta('wesub_subscription_id');

			$wesub_subscription = new Wesub_Subscription($wesub_subscription_id);

			$this->subscription = $wesub_subscription;
			$this->renewal_order = $renewal_order;

			if (!$this->is_enabled()) {
				return;
			}
			if ($renewal_order) {
				$this->placeholders['{subscription_id}'] = $wesub_subscription->get_id();
				$this->placeholders['{renewal_id}'] = $renewal_order->get_id();
				$billing_email = $renewal_order->get_billing_email();
				$this->send($billing_email, $this->get_subject(), $this->get_content(), $this->get_headers(), $this->get_attachments());
			}
			$this->restore_locale();
		}

		/**
		 * Get_content_html function.
		 *
		 * @access public
		 * @return string
		 */
		public function get_content_html()
		{
			return wc_get_template_html(
				$this->template_html,
				array(
					'subscription' => $this->subscription,
					'renewal_order' => $this->renewal_order,
					'email_heading' => $this->get_heading(),
					'sent_to_admin' => true,
					'plain_text' => false,
					'email' => $this,
				),
				'',
				$this->template_base
			);
		}

		/**
		 * Get_content_plain function.
		 *
		 * @access public
		 * @return string
		 */
		public function get_content_plain()
		{
			return wc_get_template_html(
				$this->template_plain,
				array(
					'subscription' => $this->subscription,
					'renewal_order' => $this->renewal_order,
					'email_heading' => $this->get_heading(),
					'sent_to_admin' => true,
					'plain_text' => true,
					'email' => $this,
				),
				'',
				$this->template_base
			);
		}

		/**
		 * Initialise Settings Form Fields
		 *
		 * @access public
		 * @return void
		 */
		public function init_form_fields()
		{

			$this->form_fields = array(
				'enabled' => array(
					'title' => __('Enable/Disable', 'we-subscription'),
					'type' => 'checkbox',
					'label' => __('Enable this email notification', 'we-subscription'),
					'default' => 'no',
				),
				'subject' => array(
					'title' => __('Subject', 'we-subscription'),
					'type' => 'text',
					'description' => __('Use these shortcodes', 'we-subscription') . ' ' . esc_html(implode(', ', array_keys($this->placeholders))),
					'placeholder' => $this->get_default_subject(),
					'default' => '',
					'desc_tip' => true,
				),
				'heading' => array(
					'title' => __('Email Heading', 'we-subscription'),
					'type' => 'text',
					'description' => __('Use these shortcodes', 'we-subscription') . ' ' . esc_html(implode(', ', array_keys($this->placeholders))),
					'placeholder' => $this->get_default_heading(),
					'default' => '',
					'desc_tip' => true,
				),
				'email_type' => array(
					'title' => __('Email type', 'we-subscription'),
					'type' => 'select',
					'description' => __('Choose which format of email to send.', 'we-subscription'),
					'default' => 'html',
					'class' => 'email_type wc-enhanced-select',
					'options' => $this->get_email_type_options(),
					'desc_tip' => true,
				),
			);
		}
	}
}
