(function() {
    'use strict';

    if (typeof wdTranslatorConfig === 'undefined') {
        return;
    }

    const config = wdTranslatorConfig;
    let currentLang = config.sourceLang;
    let isTranslating = false;
    let originalContent = new Map();
    let translationCache = new Map();
    let widgetElement = null;

    const storage = {
        get: function(key) {
            try {
                return localStorage.getItem('wd_translator_' + key);
            } catch (e) {
                return null;
            }
        },
        set: function(key, value) {
            try {
                localStorage.setItem('wd_translator_' + key, value);
            } catch (e) {}
        },
        remove: function(key) {
            try {
                localStorage.removeItem('wd_translator_' + key);
            } catch (e) {}
        }
    };

    function init() {
        const savedLang = storage.get('lang');
        if (savedLang && config.enabledLanguages.some(l => l.code === savedLang)) {
            currentLang = savedLang;
        }

        renderWidget();
        applyStyles();

        if (currentLang !== config.sourceLang) {
            translatePage(currentLang);
        }

        observeDOM();
    }

    function renderWidget() {
        const container = document.getElementById('wd-translator-widget-container');
        const shortcodes = document.querySelectorAll('.wd-translator-shortcode-widget');

        if (container && config.widgetStyle === 'floating') {
            widgetElement = createFloatingWidget();
            container.appendChild(widgetElement);
        }

        shortcodes.forEach(function(el) {
            const style = el.dataset.style || 'inline';
            let widget;

            if (style === 'flags') {
                widget = createFlagGrid();
            } else {
                widget = createInlineDropdown();
            }

            el.appendChild(widget);
        });
    }

    function createFloatingWidget() {
        const wrapper = document.createElement('div');
        wrapper.className = 'wd-translator-widget wd-translator-floating';
        wrapper.dataset.position = config.widgetPosition;
        wrapper.dataset.radius = config.styles.borderRadius;
        wrapper.dataset.shadow = config.styles.shadow;
        wrapper.dataset.fontSize = config.styles.fontSize;

        const trigger = document.createElement('button');
        trigger.className = 'wd-translator-trigger';
        trigger.setAttribute('aria-expanded', 'false');
        trigger.setAttribute('aria-haspopup', 'true');

        const currentLangData = findLanguage(currentLang);
        trigger.innerHTML = createTriggerContent(currentLangData);

        const dropdown = document.createElement('div');
        dropdown.className = 'wd-translator-dropdown';
        dropdown.setAttribute('role', 'menu');

        config.enabledLanguages.forEach(function(lang) {
            const option = createOption(lang);
            dropdown.appendChild(option);
        });

        const loader = document.createElement('div');
        loader.className = 'wd-translator-loader';
        loader.innerHTML = '<div class="wd-translator-spinner"></div>';

        trigger.addEventListener('click', function(e) {
            e.stopPropagation();
            toggleDropdown(dropdown, trigger);
        });

        document.addEventListener('click', function() {
            closeDropdown(dropdown, trigger);
        });

        wrapper.appendChild(trigger);
        wrapper.appendChild(dropdown);
        wrapper.appendChild(loader);

        return wrapper;
    }

    function createInlineDropdown() {
        const wrapper = document.createElement('div');
        wrapper.className = 'wd-translator-widget wd-translator-inline';
        wrapper.dataset.radius = config.styles.borderRadius;
        wrapper.dataset.shadow = config.styles.shadow;
        wrapper.dataset.fontSize = config.styles.fontSize;

        const trigger = document.createElement('button');
        trigger.className = 'wd-translator-trigger';
        trigger.setAttribute('aria-expanded', 'false');

        const currentLangData = findLanguage(currentLang);
        trigger.innerHTML = createTriggerContent(currentLangData);

        const dropdown = document.createElement('div');
        dropdown.className = 'wd-translator-dropdown';

        config.enabledLanguages.forEach(function(lang) {
            const option = createOption(lang);
            dropdown.appendChild(option);
        });

        trigger.addEventListener('click', function(e) {
            e.stopPropagation();
            toggleDropdown(dropdown, trigger);
        });

        document.addEventListener('click', function() {
            closeDropdown(dropdown, trigger);
        });

        wrapper.appendChild(trigger);
        wrapper.appendChild(dropdown);

        return wrapper;
    }

    function createFlagGrid() {
        const wrapper = document.createElement('div');
        wrapper.className = 'wd-translator-widget wd-translator-flags-grid';
        wrapper.dataset.radius = config.styles.borderRadius;
        wrapper.dataset.fontSize = config.styles.fontSize;

        config.enabledLanguages.forEach(function(lang) {
            const btn = document.createElement('button');
            btn.className = 'wd-translator-flag-btn';
            if (lang.code === currentLang) {
                btn.classList.add('wd-translator-active');
            }

            let content = '';
            if (config.showFlags) {
                content += createFlagElement(lang);
            }
            content += '<span>' + lang.label + '</span>';
            btn.innerHTML = content;

            btn.addEventListener('click', function() {
                selectLanguage(lang.code);
                wrapper.querySelectorAll('.wd-translator-flag-btn').forEach(function(b) {
                    b.classList.remove('wd-translator-active');
                });
                btn.classList.add('wd-translator-active');
            });

            wrapper.appendChild(btn);
        });

        return wrapper;
    }

    function createTriggerContent(lang) {
        let content = '';
        if (config.showFlags) {
            content += createFlagElement(lang);
        }
        content += '<span class="wd-translator-current-label">' + lang.label + '</span>';
        content += '<svg class="wd-translator-chevron" viewBox="0 0 20 20" fill="currentColor"><path fill-rule="evenodd" d="M5.293 7.293a1 1 0 011.414 0L10 10.586l3.293-3.293a1 1 0 111.414 1.414l-4 4a1 1 0 01-1.414 0l-4-4a1 1 0 010-1.414z" clip-rule="evenodd"/></svg>';
        return content;
    }

    function createOption(lang) {
        const option = document.createElement('button');
        option.className = 'wd-translator-option';
        option.setAttribute('role', 'menuitem');
        if (lang.code === currentLang) {
            option.classList.add('wd-translator-active');
        }

        let content = '';
        if (config.showFlags) {
            content += createFlagElement(lang);
        }
        content += '<span>' + lang.label + '</span>';
        option.innerHTML = content;

        option.addEventListener('click', function() {
            selectLanguage(lang.code);
        });

        return option;
    }

    function toggleDropdown(dropdown, trigger) {
        const isOpen = dropdown.classList.contains('wd-translator-open');
        if (isOpen) {
            closeDropdown(dropdown, trigger);
        } else {
            dropdown.classList.add('wd-translator-open');
            trigger.setAttribute('aria-expanded', 'true');
        }
    }

    function closeDropdown(dropdown, trigger) {
        dropdown.classList.remove('wd-translator-open');
        trigger.setAttribute('aria-expanded', 'false');
    }

    function selectLanguage(code) {
        if (code === currentLang) {
            return;
        }

        if (code === config.sourceLang) {
            restoreOriginal();
        } else {
            translatePage(code);
        }
    }

    function translatePage(targetLang) {
        if (isTranslating) {
            return;
        }

        isTranslating = true;
        setLoading(true);

        if (originalContent.size === 0) {
            storeOriginal();
        }

        const textNodes = getTranslatableNodes();
        const texts = [];
        const nodeIndices = [];

        // Always use original text for translation, not current DOM text
        textNodes.forEach(function(node, index) {
            // Get original text if stored, otherwise use current content
            const text = (originalContent.has(index) ? originalContent.get(index) : node.textContent).trim();
            if (text.length > 0 && text.length < 5000) {
                texts.push(text);
                nodeIndices.push(index);
            }
        });

        const uncached = [];
        const uncachedIndices = [];

        texts.forEach(function(text, i) {
            const key = text + '|' + targetLang;
            if (!translationCache.has(key)) {
                uncached.push(text);
                uncachedIndices.push(i);
            }
        });

        if (uncached.length === 0) {
            applyTranslations(textNodes, texts, nodeIndices, targetLang);
            finishTranslation(targetLang);
            return;
        }

        var batches = [];
        for (var i = 0; i < uncached.length; i += 50) {
            batches.push(uncached.slice(i, i + 50));
        }

        var batchIndex = 0;
        var allResults = [];

        function processBatch() {
            if (batchIndex >= batches.length) {
                uncachedIndices.forEach(function(origIndex, i) {
                    var key = texts[origIndex] + '|' + targetLang;
                    translationCache.set(key, allResults[i]);
                });
                applyTranslations(textNodes, texts, nodeIndices, targetLang);
                finishTranslation(targetLang);
                return;
            }

            fetch(config.restUrl + 'translate/batch', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': config.nonce
                },
                body: JSON.stringify({
                    texts: batches[batchIndex],
                    target_lang: targetLang,
                    source_lang: config.sourceLang
                })
            })
            .then(function(response) {
                return response.json();
            })
            .then(function(data) {
                if (data.success && data.translations) {
                    data.translations.forEach(function(t) {
                        allResults.push(t.translated);
                    });
                }
                batchIndex++;
                processBatch();
            })
            .catch(function(error) {
                console.error('Translation failed:', error);
                showToast('Translation failed. Please try again.', 'error');
                finishTranslation(currentLang);
            });
        }

        processBatch();
    }

    function applyTranslations(textNodes, texts, nodeIndices, targetLang) {
        nodeIndices.forEach(function(nodeIndex, i) {
            // Use original text as cache key (texts array already contains original text)
            var key = texts[i] + '|' + targetLang;
            var translated = translationCache.get(key);
            if (translated && textNodes[nodeIndex]) {
                textNodes[nodeIndex].textContent = translated;
            }
        });
    }

    function finishTranslation(lang) {
        currentLang = lang;
        storage.set('lang', lang);
        updateUI();
        setLoading(false);
        isTranslating = false;
        document.documentElement.lang = lang;
    }

    function restoreOriginal() {
        var textNodes = getTranslatableNodes();
        originalContent.forEach(function(text, index) {
            if (textNodes[index]) {
                textNodes[index].textContent = text;
            }
        });

        currentLang = config.sourceLang;
        storage.remove('lang');
        updateUI();
        document.documentElement.lang = config.sourceLang;
    }

    function storeOriginal() {
        var textNodes = getTranslatableNodes();
        textNodes.forEach(function(node, index) {
            originalContent.set(index, node.textContent);
        });
    }

    function getTranslatableNodes() {
        var nodes = [];
        var skipTags = ['SCRIPT', 'STYLE', 'NOSCRIPT', 'IFRAME', 'SVG', 'CODE', 'PRE', 'TEXTAREA', 'INPUT', 'SELECT'];

        var walker = document.createTreeWalker(
            document.body,
            NodeFilter.SHOW_TEXT,
            {
                acceptNode: function(node) {
                    var text = node.textContent.trim();
                    if (!text || text.length === 0) {
                        return NodeFilter.FILTER_REJECT;
                    }

                    var parent = node.parentElement;
                    if (!parent) {
                        return NodeFilter.FILTER_REJECT;
                    }

                    if (skipTags.indexOf(parent.tagName) !== -1) {
                        return NodeFilter.FILTER_REJECT;
                    }

                    if (parent.closest('.wd-translator-widget, .no-translate, [data-no-translate], [translate="no"]')) {
                        return NodeFilter.FILTER_REJECT;
                    }

                    if (parent.isContentEditable) {
                        return NodeFilter.FILTER_REJECT;
                    }

                    return NodeFilter.FILTER_ACCEPT;
                }
            }
        );

        var node;
        while (node = walker.nextNode()) {
            nodes.push(node);
        }

        return nodes;
    }

    function updateUI() {
        var lang = findLanguage(currentLang);

        document.querySelectorAll('.wd-translator-current-label').forEach(function(el) {
            el.textContent = lang.label;
        });

        document.querySelectorAll('.wd-translator-trigger .wd-translator-flag').forEach(function(el) {
            if (lang.customFlagUrl) {
                el.className = 'wd-translator-flag wd-translator-flag-custom';
                el.style.backgroundImage = 'url(\'' + lang.customFlagUrl + '\')';
                el.textContent = '';
            } else {
                el.className = 'wd-translator-flag wd-translator-flag-' + lang.code;
                el.style.backgroundImage = '';
                el.textContent = lang.code.toUpperCase();
            }
        });

        document.querySelectorAll('.wd-translator-option').forEach(function(el) {
            el.classList.remove('wd-translator-active');
        });

        document.querySelectorAll('.wd-translator-flag-btn').forEach(function(el) {
            el.classList.remove('wd-translator-active');
        });
    }

    function findLanguage(code) {
        for (var i = 0; i < config.enabledLanguages.length; i++) {
            if (config.enabledLanguages[i].code === code) {
                return config.enabledLanguages[i];
            }
        }
        return { code: code, label: code };
    }

    function createFlagElement(lang) {
        if (lang.customFlagUrl) {
            return '<span class="wd-translator-flag wd-translator-flag-custom" style="background-image: url(\'' + lang.customFlagUrl + '\')"></span>';
        }
        return '<span class="wd-translator-flag wd-translator-flag-' + lang.code + '">' + lang.code.toUpperCase() + '</span>';
    }

    function setLoading(loading) {
        document.querySelectorAll('.wd-translator-widget').forEach(function(el) {
            if (loading) {
                el.classList.add('wd-translator-loading');
            } else {
                el.classList.remove('wd-translator-loading');
            }
        });
    }

    function showToast(message, type) {
        var toast = document.createElement('div');
        toast.className = 'wd-translator-toast';
        if (type === 'error') {
            toast.classList.add('wd-translator-error');
        }
        toast.textContent = message;
        document.body.appendChild(toast);

        setTimeout(function() {
            toast.remove();
        }, 4000);
    }

    function applyStyles() {
        var root = document.documentElement;
        root.style.setProperty('--wd-translator-bg', config.styles.bgColor);
        root.style.setProperty('--wd-translator-text', config.styles.textColor);
        root.style.setProperty('--wd-translator-border', config.styles.borderColor);
        root.style.setProperty('--wd-translator-hover-bg', config.styles.hoverBgColor);
        root.style.setProperty('--wd-translator-active-bg', config.styles.activeBgColor);
    }

    function observeDOM() {
        if (currentLang === config.sourceLang) {
            return;
        }

        var timeout;
        var observer = new MutationObserver(function(mutations) {
            clearTimeout(timeout);
            timeout = setTimeout(function() {
                translateNewNodes(mutations);
            }, 500);
        });

        observer.observe(document.body, {
            childList: true,
            subtree: true
        });
    }

    function translateNewNodes(mutations) {
        if (currentLang === config.sourceLang || isTranslating) {
            return;
        }

        var newTexts = [];
        var newNodes = [];

        mutations.forEach(function(mutation) {
            mutation.addedNodes.forEach(function(node) {
                if (node.nodeType === Node.ELEMENT_NODE) {
                    var walker = document.createTreeWalker(node, NodeFilter.SHOW_TEXT);
                    var textNode;
                    while (textNode = walker.nextNode()) {
                        var text = textNode.textContent.trim();
                        if (text.length > 0 && text.length < 5000) {
                            var key = text + '|' + currentLang;
                            if (translationCache.has(key)) {
                                textNode.textContent = translationCache.get(key);
                            } else {
                                newTexts.push(text);
                                newNodes.push(textNode);
                            }
                        }
                    }
                }
            });
        });

        if (newTexts.length === 0) {
            return;
        }

        fetch(config.restUrl + 'translate/batch', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-WP-Nonce': config.nonce
            },
            body: JSON.stringify({
                texts: newTexts.slice(0, 50),
                target_lang: currentLang,
                source_lang: config.sourceLang
            })
        })
        .then(function(response) {
            return response.json();
        })
        .then(function(data) {
            if (data.success && data.translations) {
                data.translations.forEach(function(t, i) {
                    if (newNodes[i]) {
                        var key = newTexts[i] + '|' + currentLang;
                        translationCache.set(key, t.translated);
                        newNodes[i].textContent = t.translated;
                    }
                });
            }
        })
        .catch(function() {});
    }

    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
})();
