<?php

namespace WD_TRANSLATOR\Translation;

use WD_TRANSLATOR\Api\OpenAIClient;
use WD_TRANSLATOR\Api\GoogleTranslateClient;
use WD_TRANSLATOR\Cache\CacheManager;

class Translator {

    private $api_client;
    private CacheManager $cache;
    private string $source_lang;
    private string $provider;

    public function __construct(CacheManager $cache) {
        $this->cache = $cache;
        $this->source_lang = get_option('wd_translator_source_language', 'en');
        $this->provider = get_option('wd_translator_translation_provider', 'google');
        $this->api_client = $this->create_api_client();
    }

    private function create_api_client() {
        if ($this->provider === 'openai') {
            return new OpenAIClient();
        }

        return new GoogleTranslateClient();
    }

    public function translate(string $text, string $target_lang, ?string $source_lang = null): TranslationResult {
        $source_lang = $source_lang ?? $this->source_lang;

        if ($source_lang === $target_lang) {
            return new TranslationResult($text, true);
        }

        $cache_key = CacheManager::generate_cache_key($text, $source_lang, $target_lang);
        $cached = $this->cache->get($cache_key);

        if ($cached !== null) {
            $this->cache->increment_hit_count($cache_key);
            return new TranslationResult($cached, true);
        }

        $translated = $this->api_client->translate($text, $source_lang, $target_lang);

        $this->cache->set($cache_key, $translated, [
            'source_lang' => $source_lang,
            'target_lang' => $target_lang,
            'original_text' => $text,
            'word_count' => str_word_count($text),
        ]);

        return new TranslationResult($translated, false);
    }

    public function translate_batch(array $texts, string $target_lang, ?string $source_lang = null): array {
        $source_lang = $source_lang ?? $this->source_lang;
        $results = [];
        $to_translate = [];
        $cache_keys = [];

        foreach ($texts as $index => $text) {
            if ($source_lang === $target_lang) {
                $results[$index] = new TranslationResult($text, true);
                continue;
            }

            $cache_key = CacheManager::generate_cache_key($text, $source_lang, $target_lang);
            $cache_keys[$index] = $cache_key;
            $cached = $this->cache->get($cache_key);

            if ($cached !== null) {
                $results[$index] = new TranslationResult($cached, true);
                $this->cache->increment_hit_count($cache_key);
            } else {
                $to_translate[$index] = $text;
            }
        }

        if (!empty($to_translate)) {
            $batch_results = $this->api_client->translate_batch(
                array_values($to_translate),
                $source_lang,
                $target_lang
            );

            $batch_index = 0;
            foreach ($to_translate as $original_index => $text) {
                $translated = $batch_results[$batch_index] ?? $text;

                $this->cache->set($cache_keys[$original_index], $translated, [
                    'source_lang' => $source_lang,
                    'target_lang' => $target_lang,
                    'original_text' => $text,
                    'word_count' => str_word_count($text),
                ]);

                $results[$original_index] = new TranslationResult($translated, false);
                $batch_index++;
            }
        }

        ksort($results);
        return $results;
    }
}

class TranslationResult {

    public string $text;
    public bool $from_cache;

    public function __construct(string $text, bool $from_cache) {
        $this->text = $text;
        $this->from_cache = $from_cache;
    }

    public function to_array(): array {
        return [
            'translated' => $this->text,
            'from_cache' => $this->from_cache,
        ];
    }
}
