<?php

namespace WD_TRANSLATOR\Frontend;

use WD_TRANSLATOR\Utilities\LanguageList;

class Frontend {

    public function enqueue_assets(): void {
        if (!get_option('wd_translator_enabled', true)) {
            return;
        }

        wp_enqueue_style(
            'wd-translator-widget',
            WD_TRANSLATOR_PLUGIN_URL . 'public/css/widget.css',
            [],
            WD_TRANSLATOR_VERSION
        );

        if (get_option('wd_translator_show_flags', true)) {
            wp_enqueue_style(
                'wd-translator-flags',
                WD_TRANSLATOR_PLUGIN_URL . 'public/css/flags.css',
                [],
                WD_TRANSLATOR_VERSION
            );
        }

        wp_enqueue_script(
            'wd-translator-widget',
            WD_TRANSLATOR_PLUGIN_URL . 'public/js/widget.js',
            [],
            WD_TRANSLATOR_VERSION,
            true
        );

        $display = get_option('wd_translator_language_display', 'native');

        wp_localize_script('wd-translator-widget', 'wdTranslatorConfig', [
            'restUrl' => home_url('/index.php/wp-json/wd-translator/v1/'),
            'nonce' => wp_create_nonce('wp_rest'),
            'sourceLang' => get_option('wd_translator_source_language', 'en'),
            'enabledLanguages' => $this->get_formatted_languages($display),
            'widgetStyle' => get_option('wd_translator_widget_style', 'floating'),
            'widgetPosition' => get_option('wd_translator_widget_position', 'bottom-right'),
            'showFlags' => get_option('wd_translator_show_flags', true),
            'languageDisplay' => $display,
            'styles' => [
                'bgColor' => get_option('wd_translator_bg_color', '#ffffff'),
                'textColor' => get_option('wd_translator_text_color', '#374151'),
                'borderColor' => get_option('wd_translator_border_color', '#e5e7eb'),
                'hoverBgColor' => get_option('wd_translator_hover_bg_color', '#f9fafb'),
                'activeBgColor' => get_option('wd_translator_active_bg_color', '#f3f4f6'),
                'borderRadius' => get_option('wd_translator_border_radius', 'medium'),
                'shadow' => get_option('wd_translator_shadow', 'subtle'),
                'fontSize' => get_option('wd_translator_font_size', 'medium'),
            ],
        ]);

    }

    public function render_widget(): void {
        if (!get_option('wd_translator_enabled', true)) {
            return;
        }

        $style = get_option('wd_translator_widget_style', 'floating');

        if ($style !== 'floating') {
            return;
        }

        echo '<div id="wd-translator-widget-container"></div>';
    }

    public function register_shortcodes(): void {
        add_shortcode('wd_translator', [$this, 'render_shortcode']);
        // Keep old shortcode for backwards compatibility
        add_shortcode('wd_translator_translator', [$this, 'render_shortcode']);
    }

    public function render_shortcode(array $atts = []): string {
        if (!get_option('wd_translator_enabled', true)) {
            return '';
        }

        $atts = shortcode_atts([
            'style' => get_option('wd_translator_widget_style', 'floating'),
        ], $atts);

        $style = sanitize_text_field($atts['style']);
        $id = 'wd-translator-shortcode-' . wp_rand(1000, 9999);

        return sprintf(
            '<div id="%s" class="wd-translator-shortcode-widget" data-style="%s"></div>',
            esc_attr($id),
            esc_attr($style)
        );
    }

    private function get_formatted_languages(string $display): array {
        $languages = LanguageList::get_enabled_languages_with_customizations();
        $formatted = [];

        foreach ($languages as $code => $lang) {
            // Custom label takes priority
            if (!empty($lang['custom_label'])) {
                $label = $lang['custom_label'];
            } else {
                switch ($display) {
                    case 'native':
                        $label = $lang['native'];
                        break;
                    case 'english':
                        $label = $lang['name'];
                        break;
                    case 'code':
                        $label = strtoupper($code);
                        break;
                    default:
                        $label = $lang['native'];
                }
            }

            $formatted[] = [
                'code' => $code,
                'label' => $label,
                'name' => $lang['name'],
                'native' => $lang['native'],
                'rtl' => $lang['rtl'],
                'customFlagUrl' => $lang['custom_flag_url'] ?? '',
            ];
        }

        return $formatted;
    }
}
