<?php

namespace WD_TRANSLATOR\Cache;

if (!defined('ABSPATH')) {
    exit;
}

class DatabaseCache {

    private $wpdb;
    private string $table_name;

    public function __construct() {
        global $wpdb;
        $this->wpdb = $wpdb;
        $this->table_name = $wpdb->prefix . 'wd_translator_translations';
    }

    public function get(string $key): ?string {
        // phpcs:disable WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safely constructed using $wpdb->prefix
        $row = $this->wpdb->get_row(
            $this->wpdb->prepare(
                "SELECT translated_text FROM {$this->table_name}
                 WHERE content_hash = %s
                 AND (expires_at IS NULL OR expires_at > NOW())",
                $key
            )
        );
        // phpcs:enable WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

        if ($row) {
            return $row->translated_text;
        }

        return null;
    }

    public function set(string $key, string $value, array $metadata, int $ttl): bool {
        $expires_at = $ttl > 0 ? gmdate('Y-m-d H:i:s', time() + $ttl) : null;

        $data = [
            'content_hash' => $key,
            'translated_text' => $value,
            'source_lang' => $metadata['source_lang'] ?? 'en',
            'target_lang' => $metadata['target_lang'] ?? '',
            'original_text' => $metadata['original_text'] ?? '',
            'word_count' => $metadata['word_count'] ?? 0,
            'api_tokens_used' => $metadata['tokens_used'] ?? 0,
            'expires_at' => $expires_at,
        ];

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table for translation cache
        $result = $this->wpdb->replace($this->table_name, $data);

        return $result !== false;
    }

    public function delete(string $key): bool {
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table for translation cache
        return $this->wpdb->delete($this->table_name, ['content_hash' => $key]) !== false;
    }

    public function clear_all(): bool {
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.SchemaChange -- Custom table, table name safely constructed
        return $this->wpdb->query("TRUNCATE TABLE {$this->table_name}") !== false;
    }

    public function clear_by_language(string $target_lang): bool {
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table for translation cache
        return $this->wpdb->delete($this->table_name, ['target_lang' => $target_lang]) !== false;
    }

    public function increment_hit_count(string $key): void {
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table, table name safely constructed
        $this->wpdb->query(
            $this->wpdb->prepare(
                "UPDATE {$this->table_name} SET hit_count = hit_count + 1 WHERE content_hash = %s",
                $key
            )
        );
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
    }

    public function get_stats(): array {
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table, table name safely constructed
        $stats = $this->wpdb->get_row(
            "SELECT
                COUNT(*) as total_entries,
                COALESCE(SUM(hit_count), 0) as total_hits,
                COALESCE(SUM(word_count), 0) as total_words,
                COALESCE(SUM(api_tokens_used), 0) as total_tokens,
                COUNT(DISTINCT target_lang) as languages_cached
             FROM {$this->table_name}"
        );
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

        return $stats ? (array) $stats : [
            'total_entries' => 0,
            'total_hits' => 0,
            'total_words' => 0,
            'total_tokens' => 0,
            'languages_cached' => 0,
        ];
    }

    public function prune_expired(): int {
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table, table name safely constructed
        return (int) $this->wpdb->query(
            "DELETE FROM {$this->table_name} WHERE expires_at IS NOT NULL AND expires_at < NOW()"
        );
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
    }
}
