<?php

namespace WD_TRANSLATOR\Cache;

class CacheManager {

    private DatabaseCache $driver;
    private bool $enabled;
    private int $default_ttl;

    public function __construct() {
        $this->enabled = (bool) get_option('wd_translator_cache_enabled', true);
        $this->default_ttl = (int) get_option('wd_translator_cache_ttl', 7) * DAY_IN_SECONDS;
        $this->driver = new DatabaseCache();
    }

    public function get(string $key): ?string {
        if (!$this->enabled) {
            return null;
        }

        return $this->driver->get($key);
    }

    public function set(string $key, string $value, array $metadata = [], ?int $ttl = null): bool {
        if (!$this->enabled) {
            return false;
        }

        $ttl = $ttl ?? $this->default_ttl;
        return $this->driver->set($key, $value, $metadata, $ttl);
    }

    public function delete(string $key): bool {
        return $this->driver->delete($key);
    }

    public function clear_all(): bool {
        return $this->driver->clear_all();
    }

    public function clear_by_language(string $target_lang): bool {
        return $this->driver->clear_by_language($target_lang);
    }

    public function increment_hit_count(string $key): void {
        $this->driver->increment_hit_count($key);
    }

    public function get_stats(): array {
        return $this->driver->get_stats();
    }

    public function prune_expired(): int {
        return $this->driver->prune_expired();
    }

    public static function generate_cache_key(string $text, string $source_lang, string $target_lang): string {
        return hash('sha256', $text . '|' . $source_lang . '|' . $target_lang);
    }
}
