<?php

namespace WD_TRANSLATOR\Api;

use WP_REST_Controller;
use WP_REST_Server;
use WP_REST_Request;
use WP_REST_Response;
use WP_Error;
use WD_TRANSLATOR\Plugin;
use WD_TRANSLATOR\Utilities\LanguageList;

class RestController extends WP_REST_Controller {

    protected $namespace = 'wd-translator/v1';

    public static function init(): void {
        $controller = new self();
        $controller->register_routes();
    }

    public function register_routes(): void {
        $this->register_translation_routes();
        $this->register_settings_routes();
        $this->register_cache_routes();
        $this->register_language_routes();
    }

    private function register_translation_routes(): void {
        register_rest_route($this->namespace, '/translate', [
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'translate_text'],
                'permission_callback' => '__return_true',
                'args' => [
                    'text' => [
                        'required' => true,
                        'type' => 'string',
                        'sanitize_callback' => 'wp_kses_post',
                    ],
                    'target_lang' => [
                        'required' => true,
                        'type' => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                    'source_lang' => [
                        'required' => false,
                        'type' => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                ],
            ],
        ]);

        register_rest_route($this->namespace, '/translate/batch', [
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'translate_batch'],
                'permission_callback' => '__return_true',
                'args' => [
                    'texts' => [
                        'required' => true,
                        'type' => 'array',
                    ],
                    'target_lang' => [
                        'required' => true,
                        'type' => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                    'source_lang' => [
                        'required' => false,
                        'type' => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                ],
            ],
        ]);
    }

    private function register_settings_routes(): void {
        register_rest_route($this->namespace, '/settings', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_public_settings'],
                'permission_callback' => '__return_true',
            ],
        ]);

        register_rest_route($this->namespace, '/admin/settings', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_admin_settings'],
                'permission_callback' => [$this, 'admin_permission_check'],
            ],
            [
                'methods' => WP_REST_Server::EDITABLE,
                'callback' => [$this, 'update_admin_settings'],
                'permission_callback' => [$this, 'admin_permission_check'],
            ],
        ]);

        register_rest_route($this->namespace, '/admin/verify-google-key', [
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'verify_google_key'],
                'permission_callback' => [$this, 'admin_permission_check'],
                'args' => [
                    'api_key' => [
                        'required' => true,
                        'type' => 'string',
                    ],
                ],
            ],
        ]);

        register_rest_route($this->namespace, '/admin/verify-openai-key', [
            [
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => [$this, 'verify_openai_key'],
                'permission_callback' => [$this, 'admin_permission_check'],
                'args' => [
                    'api_key' => [
                        'required' => true,
                        'type' => 'string',
                    ],
                ],
            ],
        ]);
    }

    private function register_cache_routes(): void {
        register_rest_route($this->namespace, '/admin/cache/stats', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_cache_stats'],
                'permission_callback' => [$this, 'admin_permission_check'],
            ],
        ]);

        register_rest_route($this->namespace, '/admin/cache/clear', [
            [
                'methods' => WP_REST_Server::DELETABLE,
                'callback' => [$this, 'clear_cache'],
                'permission_callback' => [$this, 'admin_permission_check'],
                'args' => [
                    'language' => [
                        'required' => false,
                        'type' => 'string',
                    ],
                ],
            ],
        ]);
    }

    private function register_language_routes(): void {
        register_rest_route($this->namespace, '/languages', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_languages'],
                'permission_callback' => '__return_true',
            ],
        ]);

        register_rest_route($this->namespace, '/admin/language-customizations', [
            [
                'methods' => WP_REST_Server::READABLE,
                'callback' => [$this, 'get_language_customizations'],
                'permission_callback' => [$this, 'admin_permission_check'],
            ],
            [
                'methods' => WP_REST_Server::EDITABLE,
                'callback' => [$this, 'update_language_customization'],
                'permission_callback' => [$this, 'admin_permission_check'],
            ],
        ]);
    }

    /**
     * @return WP_REST_Response|WP_Error
     */
    public function translate_text(WP_REST_Request $request) {
        if (!get_option('wd_translator_enabled', true)) {
            return new WP_Error('disabled', __('Translation is disabled.', 'wd-translator'), ['status' => 403]);
        }

        try {
            $translator = Plugin::instance()->get_translator();
            $result = $translator->translate(
                $request->get_param('text'),
                $request->get_param('target_lang'),
                $request->get_param('source_lang')
            );

            return new WP_REST_Response([
                'success' => true,
                'translated' => $result->text,
                'from_cache' => $result->from_cache,
            ], 200);
        } catch (\Exception $e) {
            return new WP_Error('translation_failed', $e->getMessage(), ['status' => 500]);
        }
    }

    /**
     * @return WP_REST_Response|WP_Error
     */
    public function translate_batch(WP_REST_Request $request) {
        if (!get_option('wd_translator_enabled', true)) {
            return new WP_Error('disabled', __('Translation is disabled.', 'wd-translator'), ['status' => 403]);
        }

        $texts = $request->get_param('texts');

        if (!is_array($texts)) {
            return new WP_Error('invalid_input', __('Texts must be an array.', 'wd-translator'), ['status' => 400]);
        }

        if (count($texts) > 100) {
            return new WP_Error('batch_too_large', __('Maximum 100 texts per batch.', 'wd-translator'), ['status' => 400]);
        }

        $texts = array_map('wp_kses_post', $texts);

        try {
            $translator = Plugin::instance()->get_translator();
            $results = $translator->translate_batch(
                $texts,
                $request->get_param('target_lang'),
                $request->get_param('source_lang')
            );

            $translations = array_map(fn($r) => $r->to_array(), $results);

            return new WP_REST_Response([
                'success' => true,
                'translations' => $translations,
            ], 200);
        } catch (\Exception $e) {
            return new WP_Error('translation_failed', $e->getMessage(), ['status' => 500]);
        }
    }

    public function get_public_settings(WP_REST_Request $request): WP_REST_Response {
        $display = get_option('wd_translator_language_display', 'native');

        return new WP_REST_Response([
            'enabled' => get_option('wd_translator_enabled', true),
            'source_language' => get_option('wd_translator_source_language', 'en'),
            'enabled_languages' => $this->format_enabled_languages($display),
            'widget_style' => get_option('wd_translator_widget_style', 'floating'),
            'widget_position' => get_option('wd_translator_widget_position', 'bottom-right'),
            'show_flags' => get_option('wd_translator_show_flags', true),
            'language_display' => $display,
            'styles' => [
                'bg_color' => get_option('wd_translator_bg_color', '#ffffff'),
                'text_color' => get_option('wd_translator_text_color', '#374151'),
                'border_color' => get_option('wd_translator_border_color', '#e5e7eb'),
                'hover_bg_color' => get_option('wd_translator_hover_bg_color', '#f9fafb'),
                'active_bg_color' => get_option('wd_translator_active_bg_color', '#f3f4f6'),
                'border_radius' => get_option('wd_translator_border_radius', 'medium'),
                'shadow' => get_option('wd_translator_shadow', 'subtle'),
                'font_size' => get_option('wd_translator_font_size', 'medium'),
            ],
        ], 200);
    }

    public function get_admin_settings(WP_REST_Request $request): WP_REST_Response {
        return new WP_REST_Response([
            'enabled' => get_option('wd_translator_enabled', true),
            'openai_api_key' => get_option('wd_translator_openai_api_key', ''),
            'openai_model' => get_option('wd_translator_openai_model', 'gpt-4o-mini'),
            'source_language' => get_option('wd_translator_source_language', 'en'),
            'enabled_languages' => get_option('wd_translator_enabled_languages', []),
            'widget_style' => get_option('wd_translator_widget_style', 'floating'),
            'widget_position' => get_option('wd_translator_widget_position', 'bottom-right'),
            'bg_color' => get_option('wd_translator_bg_color', '#ffffff'),
            'text_color' => get_option('wd_translator_text_color', '#374151'),
            'border_color' => get_option('wd_translator_border_color', '#e5e7eb'),
            'hover_bg_color' => get_option('wd_translator_hover_bg_color', '#f9fafb'),
            'active_bg_color' => get_option('wd_translator_active_bg_color', '#f3f4f6'),
            'border_radius' => get_option('wd_translator_border_radius', 'medium'),
            'shadow' => get_option('wd_translator_shadow', 'subtle'),
            'font_size' => get_option('wd_translator_font_size', 'medium'),
            'show_flags' => get_option('wd_translator_show_flags', true),
            'language_display' => get_option('wd_translator_language_display', 'native'),
            'cache_enabled' => get_option('wd_translator_cache_enabled', true),
            'cache_ttl' => get_option('wd_translator_cache_ttl', 7),
        ], 200);
    }

    public function update_admin_settings(WP_REST_Request $request): WP_REST_Response {
        $settings = [
            'wd_translator_enabled' => 'boolean',
            'wd_translator_translation_provider' => 'string',
            'wd_translator_google_api_key' => 'string',
            'wd_translator_openai_api_key' => 'string',
            'wd_translator_openai_model' => 'string',
            'wd_translator_source_language' => 'string',
            'wd_translator_enabled_languages' => 'array',
            'wd_translator_widget_style' => 'string',
            'wd_translator_widget_position' => 'string',
            'wd_translator_bg_color' => 'string',
            'wd_translator_text_color' => 'string',
            'wd_translator_border_color' => 'string',
            'wd_translator_hover_bg_color' => 'string',
            'wd_translator_active_bg_color' => 'string',
            'wd_translator_border_radius' => 'string',
            'wd_translator_shadow' => 'string',
            'wd_translator_font_size' => 'string',
            'wd_translator_show_flags' => 'boolean',
            'wd_translator_language_display' => 'string',
            'wd_translator_cache_enabled' => 'boolean',
            'wd_translator_cache_ttl' => 'integer',
        ];

        $params = $request->get_params();

        foreach ($settings as $key => $type) {
            $param_key = str_replace('wd_translator_', '', $key);
            if (isset($params[$param_key])) {
                $value = $params[$param_key];

                switch ($type) {
                    case 'boolean':
                        $value = (bool) $value;
                        break;
                    case 'integer':
                        $value = (int) $value;
                        break;
                    case 'array':
                        $value = is_array($value) ? array_map('sanitize_text_field', $value) : [];
                        break;
                    default:
                        $value = sanitize_text_field($value);
                }

                update_option($key, $value);
            }
        }

        return new WP_REST_Response(['success' => true], 200);
    }

    public function verify_google_key(WP_REST_Request $request): WP_REST_Response {
        $api_key = $request->get_param('api_key');
        $client = new GoogleTranslateClient($api_key);
        $valid = $client->verify_api_key();

        return new WP_REST_Response([
            'valid' => $valid,
        ], 200);
    }

    public function verify_openai_key(WP_REST_Request $request): WP_REST_Response {
        $api_key = $request->get_param('api_key');
        $client = new OpenAIClient($api_key);
        $valid = $client->verify_api_key();

        return new WP_REST_Response([
            'valid' => $valid,
        ], 200);
    }

    public function get_cache_stats(WP_REST_Request $request): WP_REST_Response {
        $cache = Plugin::instance()->get_cache_manager();
        return new WP_REST_Response($cache->get_stats(), 200);
    }

    public function clear_cache(WP_REST_Request $request): WP_REST_Response {
        $cache = Plugin::instance()->get_cache_manager();
        $language = $request->get_param('language');

        if ($language) {
            $success = $cache->clear_by_language($language);
        } else {
            $success = $cache->clear_all();
        }

        return new WP_REST_Response([
            'success' => $success,
        ], 200);
    }

    public function get_languages(WP_REST_Request $request): WP_REST_Response {
        return new WP_REST_Response(LanguageList::get_all(), 200);
    }

    public function get_language_customizations(WP_REST_Request $request): WP_REST_Response {
        $customizations = get_option('wd_translator_language_customizations', []);
        return new WP_REST_Response(is_array($customizations) ? $customizations : [], 200);
    }

    public function update_language_customization(WP_REST_Request $request): WP_REST_Response {
        $code = sanitize_text_field($request->get_param('language_code') ?? '');
        $custom_label = sanitize_text_field($request->get_param('custom_label') ?? '');
        $custom_flag_id = absint($request->get_param('custom_flag_id') ?? 0);

        if (empty($code)) {
            return new WP_REST_Response(['success' => false, 'message' => 'Language code is required'], 400);
        }

        $customizations = get_option('wd_translator_language_customizations', []);
        if (!is_array($customizations)) {
            $customizations = [];
        }

        $custom_flag_url = $custom_flag_id > 0 ? wp_get_attachment_url($custom_flag_id) : '';

        if (empty($custom_label) && empty($custom_flag_url)) {
            unset($customizations[$code]);
        } else {
            $customizations[$code] = [
                'custom_label' => $custom_label,
                'custom_flag_url' => $custom_flag_url ?: '',
                'custom_flag_id' => $custom_flag_id,
            ];
        }

        update_option('wd_translator_language_customizations', $customizations);
        return new WP_REST_Response(['success' => true], 200);
    }

    public function admin_permission_check(): bool {
        return current_user_can('manage_options');
    }

    private function format_enabled_languages(string $display): array {
        $languages = LanguageList::get_enabled_languages();
        $formatted = [];

        foreach ($languages as $code => $lang) {
            switch ($display) {
                case 'native':
                    $label = $lang['native'];
                    break;
                case 'english':
                    $label = $lang['name'];
                    break;
                case 'code':
                    $label = strtoupper($code);
                    break;
                default:
                    $label = $lang['native'];
            }

            $formatted[] = [
                'code' => $code,
                'label' => $label,
                'name' => $lang['name'],
                'native' => $lang['native'],
                'rtl' => $lang['rtl'],
            ];
        }

        return $formatted;
    }
}
