<?php

namespace WD_TRANSLATOR\Api;

if (!defined('ABSPATH')) {
    exit;
}

use Exception;

class OpenAIClient {

    private string $api_key;
    private string $model;
    private string $api_base = 'https://api.openai.com/v1';
    private int $timeout = 30;

    public function __construct(?string $api_key = null, ?string $model = null) {
        $this->api_key = $api_key ?? get_option('wd_translator_openai_api_key', '');
        $this->model = $model ?? get_option('wd_translator_openai_model', 'gpt-5.2');
    }

    public function translate(string $text, string $source_lang, string $target_lang): string {
        if (empty($this->api_key)) {
            throw new Exception(esc_html__('OpenAI API key is not configured.', 'wd-translator'));
        }

        $system_prompt = $this->build_system_prompt($source_lang, $target_lang);

        $response = $this->chat_completion([
            ['role' => 'system', 'content' => $system_prompt],
            ['role' => 'user', 'content' => $text],
        ]);

        return $response['choices'][0]['message']['content'] ?? $text;
    }

    public function translate_batch(array $texts, string $source_lang, string $target_lang): array {
        if (empty($texts)) {
            return [];
        }

        if (empty($this->api_key)) {
            throw new Exception(esc_html__('OpenAI API key is not configured.', 'wd-translator'));
        }

        $system_prompt = $this->build_batch_system_prompt($source_lang, $target_lang, count($texts));

        $numbered_texts = [];
        foreach ($texts as $index => $text) {
            $numbered_texts[] = '[' . ($index + 1) . '] ' . $text;
        }

        $user_content = implode("\n", $numbered_texts);

        $response = $this->chat_completion([
            ['role' => 'system', 'content' => $system_prompt],
            ['role' => 'user', 'content' => $user_content],
        ]);

        $result = $response['choices'][0]['message']['content'] ?? '';

        return $this->parse_batch_response($result, count($texts));
    }

    public function verify_api_key(): bool {
        if (empty($this->api_key)) {
            return false;
        }

        try {
            $response = $this->make_request('GET', '/models');
            return isset($response['data']);
        } catch (Exception $e) {
            return false;
        }
    }

    private function build_system_prompt(string $source_lang, string $target_lang): string {
        $source_name = \WD_TRANSLATOR\Utilities\LanguageList::get_language_name($source_lang);
        $target_name = \WD_TRANSLATOR\Utilities\LanguageList::get_language_name($target_lang);

        return "You are a professional translator. Translate the following text from {$source_name} to {$target_name}.

Rules:
- Maintain the original meaning and tone
- Preserve any HTML tags, URLs, or code exactly as they appear
- Keep proper nouns, brand names, and technical terms as appropriate
- Provide only the translation, no explanations or alternatives
- If the text is already in the target language, return it unchanged";
    }

    private function build_batch_system_prompt(string $source_lang, string $target_lang, int $count): string {
        $source_name = \WD_TRANSLATOR\Utilities\LanguageList::get_language_name($source_lang);
        $target_name = \WD_TRANSLATOR\Utilities\LanguageList::get_language_name($target_lang);

        return "You are a professional translator. Translate the following {$count} texts from {$source_name} to {$target_name}.

Rules:
- Each text is prefixed with a number in brackets like [1], [2], etc.
- Return translations in the same numbered format
- Maintain the original meaning and tone for each text
- Preserve any HTML tags, URLs, or code exactly as they appear
- Keep proper nouns, brand names, and technical terms as appropriate
- Provide only the translations, no explanations
- Return each translation on its own line with the same number prefix";
    }

    private function parse_batch_response(string $response, int $expected_count): array {
        $results = [];
        $lines = explode("\n", trim($response));

        foreach ($lines as $line) {
            if (preg_match('/^\[(\d+)\]\s*(.+)$/s', trim($line), $matches)) {
                $index = (int) $matches[1] - 1;
                $results[$index] = trim($matches[2]);
            }
        }

        for ($i = 0; $i < $expected_count; $i++) {
            if (!isset($results[$i])) {
                $results[$i] = '';
            }
        }

        ksort($results);
        return array_values($results);
    }

    private function chat_completion(array $messages, array $options = []): array {
        $payload = array_merge([
            'model' => $this->model,
            'messages' => $messages,
            'temperature' => 0.3,
            'max_tokens' => 4096,
        ], $options);

        return $this->make_request('POST', '/chat/completions', $payload);
    }

    private function make_request(string $method, string $endpoint, array $body = []): array {
        $url = $this->api_base . $endpoint;

        $args = [
            'method' => $method,
            'timeout' => $this->timeout,
            'headers' => [
                'Authorization' => 'Bearer ' . $this->api_key,
                'Content-Type' => 'application/json',
            ],
        ];

        if (!empty($body)) {
            $args['body'] = wp_json_encode($body);
        }

        $response = wp_remote_request($url, $args);

        if (is_wp_error($response)) {
            throw new Exception(
                esc_html(
                    sprintf(
                        /* translators: %s: error message from API */
                        __('API request failed: %s', 'wd-translator'),
                        $response->get_error_message()
                    )
                )
            );
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if ($status_code >= 400) {
            $error_message = $data['error']['message'] ?? __('Unknown API error', 'wd-translator');
            throw new Exception(
                esc_html(
                    sprintf(
                        /* translators: %1$d: HTTP status code, %2$s: error message */
                        __('OpenAI API error (%1$d): %2$s', 'wd-translator'),
                        $status_code,
                        $error_message
                    )
                )
            );
        }

        return $data;
    }
}
