<?php

namespace WD_TRANSLATOR\Api;

if (!defined('ABSPATH')) {
    exit;
}

use Exception;

class GoogleTranslateClient {

    private string $api_key;
    private string $api_base = 'https://translation.googleapis.com/language/translate/v2';
    private int $timeout = 10;

    public function __construct(?string $api_key = null) {
        $this->api_key = $api_key ?? get_option('wd_translator_google_api_key', '');
    }

    public function translate(string $text, string $source_lang, string $target_lang): string {
        if (empty($this->api_key)) {
            throw new Exception(esc_html__('Google Translate API key is not configured.', 'wd-translator'));
        }

        $response = $this->make_request([
            'q' => $text,
            'source' => $this->normalize_language_code($source_lang),
            'target' => $this->normalize_language_code($target_lang),
            'format' => 'html',
        ]);

        return $response['data']['translations'][0]['translatedText'] ?? $text;
    }

    public function translate_batch(array $texts, string $source_lang, string $target_lang): array {
        if (empty($texts)) {
            return [];
        }

        if (empty($this->api_key)) {
            throw new Exception(esc_html__('Google Translate API key is not configured.', 'wd-translator'));
        }

        $response = $this->make_request([
            'q' => $texts,
            'source' => $this->normalize_language_code($source_lang),
            'target' => $this->normalize_language_code($target_lang),
            'format' => 'html',
        ]);

        $results = [];
        $translations = $response['data']['translations'] ?? [];

        foreach ($translations as $index => $translation) {
            $results[$index] = $translation['translatedText'] ?? ($texts[$index] ?? '');
        }

        return $results;
    }

    public function verify_api_key(): bool {
        if (empty($this->api_key)) {
            return false;
        }

        try {
            $response = $this->make_request([
                'q' => 'test',
                'source' => 'en',
                'target' => 'es',
            ]);
            return isset($response['data']['translations']);
        } catch (Exception $e) {
            return false;
        }
    }

    public function get_supported_languages(): array {
        $url = 'https://translation.googleapis.com/language/translate/v2/languages';

        $response = wp_remote_get($url . '?key=' . $this->api_key . '&target=en', [
            'timeout' => $this->timeout,
        ]);

        if (is_wp_error($response)) {
            return [];
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        return $body['data']['languages'] ?? [];
    }

    private function make_request(array $params): array {
        $url = $this->api_base . '?key=' . $this->api_key;

        $args = [
            'method' => 'POST',
            'timeout' => $this->timeout,
            'headers' => [
                'Content-Type' => 'application/json',
            ],
            'body' => wp_json_encode($params),
        ];

        $response = wp_remote_request($url, $args);

        if (is_wp_error($response)) {
            throw new Exception(
                esc_html(
                    sprintf(
                        /* translators: %s: error message from API */
                        __('API request failed: %s', 'wd-translator'),
                        $response->get_error_message()
                    )
                )
            );
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if ($status_code >= 400) {
            $error_message = $data['error']['message'] ?? __('Unknown API error', 'wd-translator');
            throw new Exception(
                esc_html(
                    sprintf(
                        /* translators: %1$d: HTTP status code, %2$s: error message */
                        __('Google Translate API error (%1$d): %2$s', 'wd-translator'),
                        $status_code,
                        $error_message
                    )
                )
            );
        }

        return $data;
    }

    private function normalize_language_code(string $code): string {
        $mappings = [
            'zh' => 'zh-CN',
            'zh-CN' => 'zh-CN',
            'zh-TW' => 'zh-TW',
        ];

        return $mappings[$code] ?? $code;
    }
}
