(function($) {
    'use strict';

    const WD_TRANSLATOR_Admin = {
        customizations: {},
        mediaFrame: null,

        init: function() {
            this.initTabs();
            this.initColorPickers();
            this.initLanguageFilter();
            this.initLanguageSelection();
            this.initProviderCards();
            this.initWidgetStyleToggle();
            this.initVerifyKeys();
            this.initClearCache();
            this.initSaveSettings();
            this.loadCacheStats();
            this.initCustomization();
            this.initCssHelp();
        },

        initTabs: function() {
            $('.wd-translator-tab').on('click', function(e) {
                e.preventDefault();
                const tab = $(this).data('tab');

                // Update active tab
                $('.wd-translator-tab').removeClass('active');
                $(this).addClass('active');

                // Show corresponding content
                $('.wd-translator-tab-content').removeClass('active');
                $('.wd-translator-tab-content[data-tab="' + tab + '"]').addClass('active');

                // Hide footer on Help tab (no settings to save)
                $('.wd-translator-settings-footer').toggle(tab !== 'help');
            });
        },

        initColorPickers: function() {
            if ($.fn.wpColorPicker) {
                $('.wd-translator-color-picker').wpColorPicker();
            }
        },

        initLanguageFilter: function() {
            $('#wd-translator-language-filter').on('input', function() {
                const query = $(this).val().toLowerCase();

                $('.wd-translator-language-item').each(function() {
                    const code = $(this).data('code').toLowerCase();
                    const name = $(this).data('name').toLowerCase();
                    const visible = code.includes(query) || name.includes(query);
                    $(this).toggleClass('hidden', !visible);
                });
            });
        },

        initLanguageSelection: function() {
            // Toggle selected class when checkbox changes
            $('.wd-translator-language-item input[type="checkbox"]').on('change', function() {
                $(this).closest('.wd-translator-language-item').toggleClass('selected', this.checked);
                WD_TRANSLATOR_Admin.updateLanguageCount();
            });

            // Select All button
            $('#wd-translator-select-all').on('click', function() {
                $('.wd-translator-language-item:not(.hidden)').each(function() {
                    $(this).addClass('selected');
                    $(this).find('input[type="checkbox"]').prop('checked', true);
                });
                WD_TRANSLATOR_Admin.updateLanguageCount();
            });

            // Select None button
            $('#wd-translator-select-none').on('click', function() {
                $('.wd-translator-language-item:not(.hidden)').each(function() {
                    $(this).removeClass('selected');
                    $(this).find('input[type="checkbox"]').prop('checked', false);
                });
                WD_TRANSLATOR_Admin.updateLanguageCount();
            });
        },

        updateLanguageCount: function() {
            const count = $('.wd-translator-language-item input[type="checkbox"]:checked').length;
            $('#wd-translator-selected-count').text(count);
        },

        initProviderCards: function() {
            $('.wd-translator-provider-card').on('click', function() {
                // Update selected state
                $('.wd-translator-provider-card').removeClass('selected');
                $(this).addClass('selected');

                // Check the radio button
                $(this).find('input[type="radio"]').prop('checked', true);

                // Show/hide API config
                const provider = $(this).find('input[type="radio"]').val();
                $('#wd-translator-google-config').toggleClass('wd-translator-hidden', provider !== 'google');
                $('#wd-translator-openai-config').toggleClass('wd-translator-hidden', provider !== 'openai');
            });
        },

        initWidgetStyleToggle: function() {
            $('#wd_translator_widget_style').on('change', function() {
                const style = $(this).val();
                $('#wd-translator-position-row').toggle(style === 'floating');
            });
        },

        initVerifyKeys: function() {
            // Google Translate API Key verification
            $('#wd-translator-verify-google-key').on('click', function() {
                const $btn = $(this);
                const $status = $('#wd-translator-google-key-status');
                const $badge = $('#wd-translator-google-key-badge');
                const apiKey = $('#wd_translator_google_api_key').val();

                if (!apiKey) {
                    $status.text(wdTranslatorAdmin.strings.invalid).removeClass('success loading').addClass('error');
                    $badge.hide();
                    return;
                }

                $btn.prop('disabled', true);
                $status.text(wdTranslatorAdmin.strings.verifying).removeClass('success error').addClass('loading');
                $badge.hide();

                $.ajax({
                    url: wdTranslatorAdmin.restUrl + 'admin/verify-google-key',
                    method: 'POST',
                    headers: {
                        'X-WP-Nonce': wdTranslatorAdmin.nonce
                    },
                    data: JSON.stringify({ api_key: apiKey }),
                    contentType: 'application/json',
                    success: function(response) {
                        if (response.valid) {
                            $status.text('').removeClass('error loading');
                            $badge.show();
                        } else {
                            $status.text(wdTranslatorAdmin.strings.invalid).removeClass('success loading').addClass('error');
                            $badge.hide();
                        }
                    },
                    error: function() {
                        $status.text(wdTranslatorAdmin.strings.error).removeClass('success loading').addClass('error');
                        $badge.hide();
                    },
                    complete: function() {
                        $btn.prop('disabled', false);
                    }
                });
            });

            // OpenAI API Key verification
            $('#wd-translator-verify-openai-key').on('click', function() {
                const $btn = $(this);
                const $status = $('#wd-translator-openai-key-status');
                const $badge = $('#wd-translator-openai-key-badge');
                const apiKey = $('#wd_translator_openai_api_key').val();

                if (!apiKey) {
                    $status.text(wdTranslatorAdmin.strings.invalid).removeClass('success loading').addClass('error');
                    $badge.hide();
                    return;
                }

                $btn.prop('disabled', true);
                $status.text(wdTranslatorAdmin.strings.verifying).removeClass('success error').addClass('loading');
                $badge.hide();

                $.ajax({
                    url: wdTranslatorAdmin.restUrl + 'admin/verify-openai-key',
                    method: 'POST',
                    headers: {
                        'X-WP-Nonce': wdTranslatorAdmin.nonce
                    },
                    data: JSON.stringify({ api_key: apiKey }),
                    contentType: 'application/json',
                    success: function(response) {
                        if (response.valid) {
                            $status.text('').removeClass('error loading');
                            $badge.show();
                        } else {
                            $status.text(wdTranslatorAdmin.strings.invalid).removeClass('success loading').addClass('error');
                            $badge.hide();
                        }
                    },
                    error: function() {
                        $status.text(wdTranslatorAdmin.strings.error).removeClass('success loading').addClass('error');
                        $badge.hide();
                    },
                    complete: function() {
                        $btn.prop('disabled', false);
                    }
                });
            });
        },

        initClearCache: function() {
            $('#wd-translator-clear-cache').on('click', function() {
                const $btn = $(this);
                const $status = $('#wd-translator-cache-status');

                $btn.prop('disabled', true);
                $status.text(wdTranslatorAdmin.strings.clearing).removeClass('success error').addClass('loading');

                $.ajax({
                    url: wdTranslatorAdmin.restUrl + 'admin/cache/clear',
                    method: 'DELETE',
                    headers: {
                        'X-WP-Nonce': wdTranslatorAdmin.nonce
                    },
                    success: function(response) {
                        if (response.success) {
                            $status.text(wdTranslatorAdmin.strings.cleared).removeClass('error loading').addClass('success');
                            WD_TRANSLATOR_Admin.loadCacheStats();
                        } else {
                            $status.text(wdTranslatorAdmin.strings.error).removeClass('success loading').addClass('error');
                        }
                    },
                    error: function() {
                        $status.text(wdTranslatorAdmin.strings.error).removeClass('success loading').addClass('error');
                    },
                    complete: function() {
                        $btn.prop('disabled', false);
                        setTimeout(function() {
                            $status.text('');
                        }, 3000);
                    }
                });
            });
        },

        initSaveSettings: function() {
            $('#wd-translator-settings-form').on('submit', function(e) {
                e.preventDefault();

                const $form = $(this);
                const $btn = $form.find('button[type="submit"]');
                const $status = $('#wd-translator-save-status');

                const data = {
                    enabled: $('#wd_translator_enabled').is(':checked'),
                    translation_provider: $('input[name="translation_provider"]:checked').val(),
                    google_api_key: $('#wd_translator_google_api_key').val(),
                    openai_api_key: $('#wd_translator_openai_api_key').val(),
                    openai_model: $('#wd_translator_openai_model').val(),
                    source_language: $('#wd_translator_source_language').val(),
                    enabled_languages: $('input[name="enabled_languages[]"]:checked').map(function() {
                        return $(this).val();
                    }).get(),
                    widget_style: $('#wd_translator_widget_style').val(),
                    widget_position: $('#wd_translator_widget_position').val(),
                    bg_color: $('#wd_translator_bg_color').val(),
                    text_color: $('#wd_translator_text_color').val(),
                    border_color: $('#wd_translator_border_color').val(),
                    hover_bg_color: $('#wd_translator_hover_bg_color').val(),
                    active_bg_color: $('#wd_translator_active_bg_color').val(),
                    border_radius: $('#wd_translator_border_radius').val(),
                    shadow: $('#wd_translator_shadow').val(),
                    font_size: $('#wd_translator_font_size').val(),
                    show_flags: $('#wd_translator_show_flags').is(':checked'),
                    language_display: $('#wd_translator_language_display').val(),
                    cache_enabled: $('#wd_translator_cache_enabled').is(':checked'),
                    cache_ttl: parseInt($('#wd_translator_cache_ttl').val(), 10),
                    custom_css: $('#wd_translator_custom_css').val()
                };

                $btn.prop('disabled', true);
                $status.text(wdTranslatorAdmin.strings.saving).removeClass('success error').addClass('loading');

                $.ajax({
                    url: wdTranslatorAdmin.restUrl + 'admin/settings',
                    method: 'POST',
                    headers: {
                        'X-WP-Nonce': wdTranslatorAdmin.nonce
                    },
                    data: JSON.stringify(data),
                    contentType: 'application/json',
                    success: function(response) {
                        if (response.success) {
                            $status.text(wdTranslatorAdmin.strings.saved).removeClass('error loading').addClass('success');
                        } else {
                            $status.text(wdTranslatorAdmin.strings.error).removeClass('success loading').addClass('error');
                        }
                    },
                    error: function() {
                        $status.text(wdTranslatorAdmin.strings.error).removeClass('success loading').addClass('error');
                    },
                    complete: function() {
                        $btn.prop('disabled', false);
                        setTimeout(function() {
                            $status.text('');
                        }, 3000);
                    }
                });
            });
        },

        loadCacheStats: function() {
            $.ajax({
                url: wdTranslatorAdmin.restUrl + 'admin/cache/stats',
                method: 'GET',
                headers: {
                    'X-WP-Nonce': wdTranslatorAdmin.nonce
                },
                success: function(response) {
                    $('#wd-translator-stat-entries').text(response.total_entries || 0);
                    $('#wd-translator-stat-hits').text(response.total_hits || 0);
                    $('#wd-translator-stat-languages').text(response.languages_cached || 0);
                }
            });
        },

        initCustomization: function() {
            var self = this;
            this.loadCustomizations();

            $(document).on('click', '.wd-translator-customize-btn', function(e) {
                e.preventDefault();
                e.stopPropagation();
                self.openCustomizeModal($(this).data('code'), $(this).data('native'));
            });

            $('.wd-translator-modal-close, .wd-translator-modal-cancel, .wd-translator-modal-overlay').on('click', function() {
                self.closeCustomizeModal();
            });

            $('#wd-translator-save-customization').on('click', function() {
                self.saveCustomization();
            });

            $('#wd-translator-select-flag').on('click', function() {
                self.openMediaLibrary();
            });

            $('#wd-translator-remove-flag').on('click', function() {
                $('#wd-translator-custom-flag-id').val(0);
                $('#wd-translator-flag-preview').css('background-image', '').removeClass('has-image');
                $('#wd-translator-remove-flag').hide();
            });
        },

        loadCustomizations: function() {
            var self = this;
            $.ajax({
                url: wdTranslatorAdmin.restUrl + 'admin/language-customizations',
                method: 'GET',
                headers: { 'X-WP-Nonce': wdTranslatorAdmin.nonce },
                success: function(data) {
                    self.customizations = data || {};
                    self.updateCustomizationIndicators();
                }
            });
        },

        updateCustomizationIndicators: function() {
            var self = this;
            $('.wd-translator-language-item').each(function() {
                var code = $(this).data('code');
                var c = self.customizations[code];
                var $flag = $(this).find('.wd-translator-flag');

                $(this).toggleClass('has-customization', !!(c && (c.custom_label || c.custom_flag_url)));

                // Update flag display
                if (c && c.custom_flag_url) {
                    $flag.removeClass('wd-translator-flag-' + code).addClass('wd-translator-flag-custom');
                    $flag.css('background-image', 'url(' + c.custom_flag_url + ')');
                } else {
                    $flag.removeClass('wd-translator-flag-custom').addClass('wd-translator-flag-' + code);
                    $flag.css('background-image', '');
                }
            });
        },

        openCustomizeModal: function(code, native) {
            var c = this.customizations[code] || {};
            $('#wd-translator-modal-title').text(wdTranslatorAdmin.strings.customizeTitle + ': ' + native);
            $('#wd-translator-customize-code').val(code);
            $('#wd-translator-custom-label').val(c.custom_label || '');
            $('#wd-translator-custom-flag-id').val(c.custom_flag_id || 0);

            if (c.custom_flag_url) {
                $('#wd-translator-flag-preview').css('background-image', 'url(' + c.custom_flag_url + ')').addClass('has-image');
                $('#wd-translator-remove-flag').show();
            } else {
                $('#wd-translator-flag-preview').css('background-image', '').removeClass('has-image');
                $('#wd-translator-remove-flag').hide();
            }

            $('#wd-translator-customize-modal').fadeIn(200);
        },

        closeCustomizeModal: function() {
            $('#wd-translator-customize-modal').fadeOut(200);
        },

        saveCustomization: function() {
            var self = this;
            var code = $('#wd-translator-customize-code').val();

            $.ajax({
                url: wdTranslatorAdmin.restUrl + 'admin/language-customizations',
                method: 'POST',
                headers: { 'X-WP-Nonce': wdTranslatorAdmin.nonce },
                data: JSON.stringify({
                    language_code: code,
                    custom_label: $('#wd-translator-custom-label').val(),
                    custom_flag_id: parseInt($('#wd-translator-custom-flag-id').val(), 10) || 0
                }),
                contentType: 'application/json',
                success: function() {
                    self.loadCustomizations();
                    self.closeCustomizeModal();
                },
                error: function() {
                    alert(wdTranslatorAdmin.strings.error);
                }
            });
        },

        openMediaLibrary: function() {
            var self = this;
            if (!this.mediaFrame) {
                this.mediaFrame = wp.media({
                    title: wdTranslatorAdmin.strings.selectFlagImage,
                    button: { text: wdTranslatorAdmin.strings.useThisImage },
                    library: { type: ['image'] },
                    multiple: false
                });
                this.mediaFrame.on('select', function() {
                    var attachment = self.mediaFrame.state().get('selection').first().toJSON();
                    $('#wd-translator-custom-flag-id').val(attachment.id);
                    $('#wd-translator-flag-preview').css('background-image', 'url(' + attachment.url + ')').addClass('has-image');
                    $('#wd-translator-remove-flag').show();
                });
            }
            this.mediaFrame.open();
        },

        initCssHelp: function() {
            $('#wd-translator-toggle-css-help').on('click', function() {
                $('#wd-translator-css-help').slideToggle(200);
            });
        }
    };

    $(document).ready(function() {
        WD_TRANSLATOR_Admin.init();
    });

})(jQuery);
