<?php
/**
 * WCAPF_V4_Migration_Hooks class.
 *
 * @since      4.0.0
 * @package    wc-ajax-product-filter
 * @subpackage wc-ajax-product-filter/includes/migration
 * @author     wptools.io
 */

/**
 * WCAPF_V4_Migration_Hooks class.
 *
 * @since 4.0.0
 */
class WCAPF_V4_Migration_Hooks {

	/**
	 * The constructor.
	 */
	private function __construct() {
	}

	/**
	 * Returns an instance of this class.
	 *
	 * @return WCAPF_V4_Migration_Hooks
	 */
	public static function instance() {
		// Store the instance locally to avoid private static replication
		static $instance = null;

		// Only run these methods if they haven't been run previously
		if ( null === $instance ) {
			$instance = new WCAPF_V4_Migration_Hooks();
			$instance->init_hooks();
		}

		return $instance;
	}

	/**
	 * Hook into actions and filters.
	 */
	private function init_hooks() {
		// Run the v4 migration if required.
		add_action( 'admin_init', array( $this, 'run_v4_migration_from_admin_area' ) );

		add_filter( 'wcapf_admin_js_params', array( $this, 'set_v4_migration_js_params' ) );

		// V4 migration notice.
		add_action( 'admin_notices', array( $this, 'show_v4_migration_notice' ) );
		add_action( 'admin_footer', array( $this, 'dismiss_v4_migration_notice_scripts' ) );
		add_action( 'admin_footer', array( $this, 'dismiss_v4_review_filters_notice_scripts' ) );
		add_action( 'wp_ajax_wcapf_dismiss_v4_migration_notice', array( $this, 'dismiss_v4_migration_notice' ) );
		add_action( 'wp_ajax_wcapf_dismiss_v4_review_filters_notice', array(
			$this,
			'dismiss_v4_review_filters_notice'
		) );

		// Notice to update the pro version.
		add_action( 'admin_notices', array( $this, 'show_pro_version_update_notice' ) );
	}

	/**
	 * Run the v4 migration if required.
	 *
	 * @return void
	 */
	public function run_v4_migration_from_admin_area() {
		WCAPF_V4_Migration()->try_to_run_v4_migration();
	}

	public function set_v4_migration_js_params( $params ) {
		// v4 migration related data.
		$params['show_v4_migration_notice']      = WCAPF_Helper::v4_migration_notice_can_be_shown();
		$params['show_v4_review_filters_notice'] = WCAPF_Helper::v4_review_filters_notice_can_be_shown();
		$params['v4_migrated_form_url']          = $this->get_v4_migrated_form_url();
		$params['v4_migration_doc_url']          = $this->get_v4_migration_doc_url();

		$pro_update_notices = WCAPF_Helper::pro_update_notice_can_be_shown();

		if ( ! empty( $pro_update_notices ) ) {
			// Get the first notice.
			$notice = array_shift( $pro_update_notices );

			$params['pro_update_notice'] = $notice;
		}

		return $params;
	}

	/**
	 * Determines if the v4 migration notice should be shown.
	 *
	 * @since 4.0.0
	 *
	 * @return bool
	 */
	private function v4_migration_notice_can_be_shown() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return false;
		}

		if ( '1' !== get_option( 'wcapf_v4_migration_notice_status' ) ) {
			return false;
		}

		return true;
	}

	/**
	 * The form url that is generated by the v4 migration process.
	 *
	 * @since 4.0.0
	 *
	 * @return string
	 */
	public function get_v4_migrated_form_url() {
		$form_id = get_option( 'wcapf_migrated_filters_form_id' );

		if ( $form_id ) {
			$form_edit_url = WCAPF_Helper::form_edit_url( $form_id );
		} else {
			$form_edit_url = WCAPF_Helper::forms_page_url();
		}

		return $form_edit_url;
	}

	/**
	 * @return string
	 */
	public function get_v4_migration_doc_url() {
		return add_query_arg(
			array(
				'utm_source'   => 'WP+Admin',
				'utm_medium'   => 'inside+plugin',
				'utm_campaign' => 'WCAPF+V4+Migration',
			),
			'https://wptools.io/docs/wc-ajax-product-filter/v3-to-v4-migration/'
		);
	}

	/**
	 * Show the v4 migration notice.
	 *
	 * @since 4.0.0
	 *
	 * @return void
	 */
	public function show_v4_migration_notice() {
		if ( ! WCAPF_Helper::v4_migration_notice_can_be_shown() ) {
			return;
		}

		$form_url          = $this->get_v4_migrated_form_url();
		$migration_doc_url = $this->get_v4_migration_doc_url();
		?>
		<div class="notice notice-info" id="wcapf-v4-migration-notice">
			<p>
				<strong>WCAPF - WooCommerce Ajax Product Filter (v4 Migration Notice)</strong>
			</p>
			<p>
				The WC Ajax Product Filter plugin has been updated to v4 and is now named WCAPF - WooCommerce Ajax
				Product Filter. We have redesigned the admin UI to provide a more intuitive user experience and
				refactored the codebase for improved performance and easier future enhancements. As part of the
				migration process, a form has been automatically created with all the existing filters from your shop.
				We kindly request that you visit the form and review the order of the filters.
			</p>
			<p>
				<a href="<?php echo esc_url( $form_url ); ?>">Review the filters</a>
				|
				<a href="<?php echo esc_url( $migration_doc_url ); ?>" target="_blank">Learn more about migration</a>
				|
				<a href="javascript:void(0)" onclick="removeWCAPFMigrationNotice()">I understand, remove
					the notice</a>
			</p>
		</div>
		<?php
	}

	/**
	 * The js script to dismiss the v4 migration notice.
	 *
	 * @since 4.0.0
	 *
	 * @return void
	 */
	public function dismiss_v4_migration_notice_scripts() {
		if ( ! $this->v4_migration_notice_can_be_shown() ) {
			return;
		}

		$nonce = wp_create_nonce( 'dismiss-v4-migration-notice-nonce' );
		?>
		<!--suppress ES6ConvertVarToLetConst, JSValidateTypes -->
		<script>
			function removeWCAPFMigrationNotice() {
				var $notice = jQuery( '#wcapf-v4-migration-notice' );

				$notice.fadeOut( 300, function() {
					$notice.remove();
				} );

				var data = {
					action: 'wcapf_dismiss_v4_migration_notice',
					nonce: '<?php echo $nonce; ?>',
				};

				jQuery.post( ajaxurl, data, function( response ) {
					console.log( response );
				} );
			}
		</script>
		<?php
	}

	/**
	 * The js script to dismiss the v4 review filters notice.
	 *
	 * @since 4.0.0
	 *
	 * @return void
	 */
	public function dismiss_v4_review_filters_notice_scripts() {
		if ( ! WCAPF_Helper::v4_review_filters_notice_can_be_shown() ) {
			return;
		}

		$nonce = wp_create_nonce( 'dismiss-v4-review-filters-notice-nonce' );
		?>
		<!--suppress ES6ConvertVarToLetConst, JSValidateTypes -->
		<script>
			function removeWCAPFReviewFiltersNotice() {
				var $notice = jQuery( '#wcapf-v4-review-filters-notice' );

				$notice.fadeOut( 300, function() {
					$notice.remove();
				} );

				var data = {
					action: 'wcapf_dismiss_v4_review_filters_notice',
					nonce: '<?php echo $nonce; ?>',
				};

				jQuery.post( ajaxurl, data, function( response ) {
					console.log( response );
				} );
			}
		</script>
		<?php
	}

	/**
	 * Dismiss the v4 migration notice via ajax.
	 *
	 * @since 4.0.0
	 *
	 * @return void
	 */
	public function dismiss_v4_migration_notice() {
		check_ajax_referer( 'dismiss-v4-migration-notice-nonce', 'nonce' );

		update_option( 'wcapf_v4_migration_notice_status', '2' );

		wp_send_json_success( 'v4 migrate notice dismissed' );
	}

	/**
	 * Dismiss the v4 review filters notice via ajax.
	 *
	 * @since 4.0.0
	 *
	 * @return void
	 */
	public function dismiss_v4_review_filters_notice() {
		check_ajax_referer( 'dismiss-v4-review-filters-notice-nonce', 'nonce' );

		update_option( 'wcapf_v4_review_filters_notice_status', '2' );

		wp_send_json_success( 'v4 review filters notice dismissed' );
	}

	/**
	 * Show notice to update the pro version to v2.
	 *
	 * @since 4.0.0
	 *
	 * @return void
	 */
	public function show_pro_version_update_notice() {
		$notices = WCAPF_Helper::pro_update_notice_can_be_shown();

		if ( ! empty( $notices ) ) {
			// Get the first notice.
			$notice = array_shift( $notices );

			echo '<div class="notice notice-error"><p>' . wp_kses_post( $notice ) . '</p></div>';
		}
	}

}

if ( is_admin() ) {
	WCAPF_V4_Migration_Hooks::instance();
}
