<?php
/**
 * The Inventory app connection.
 *
 * @link       https://webtoffee.com
 * @since      1.0.0
 *
 * @package    Wbtfe_Multichannel_Inventory_Sync
 * @subpackage Wbtfe_Multichannel_Inventory_Sync/admin/classes
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Wbtfe_Multichannel_Inventory_Sync_Connector' ) ) {

	/**
	 * This class handles the inventory app connection.
	 *
	 * @package    Wbtfe_Multichannel_Inventory_Sync
	 * @subpackage Wbtfe_Multichannel_Inventory_Sync/admin/classes
	 * @author     WebToffee
	 */
	class Wbtfe_Multichannel_Inventory_Sync_Connector {

		/**
		 *  Inventory app url.
		 *
		 *  @var string
		 */
		private $app_url = 'https://multichannel.webtoffee.com/'; // for production development
		private $disconnect_url = 'https://multichannel-api.webtoffee.com/'; // for production development
		
		/**
		 *  Token db url.
		 *
		 *  @var string
		 */
		private $token_db_url = 'https://feedback.webtoffee.com/wc-api-test/';

		/**
		 *  Nonce action.
		 *
		 *  @var string
		 */
		private $nonce_action = 'connect_wbtfe_multichannel_inventory';

		/**
		 *  Data option name.
		 *
		 *  @var string
		 */
		private $data_option_name = 'wbtfe_multichannel_inventory_connector_data';

		/**
		 *  Key option name.
		 *
		 *  @var string
		 */
		private $key_option_name = 'wbtfe_multichannel_inventory_connector_key';

		/**
		 *  App name.
		 *
		 *  @var string
		 */
		private $app_name = 'WebToffee Multichannel Inventory Sync';

		/**
		 *  WC API permission.
		 *
		 *  @var string
		 */
		private $wc_api_perm = 'read_write';


		/**
		 *  Connection information.
		 *
		 *  @var null|array
		 */
		private $connection_data = null;


		/**
		 *  Connector version.
		 *
		 *  @var string
		 */
		private $version = '1.0.1';

		/**
		 *  Show warning option.
		 *
		 *  @var string
		 */
		private $show_warning_option = 'wbtfe_multichannel_inventory_connector_show_warning';
		private $show_warning = false;

		/**
		 *  Constructer.
		 */
		public function __construct() {
			// Add new admin menu.
			add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
			add_action( 'init', array( $this, 'save_tokens' ) );
			add_action( 'wp_ajax_wbtfe_mis_disconnect', array( $this, 'disconnect_mis' ) );
			add_action('rest_api_init', array( $this, 'register_rest_routes' ) );
			add_action( 'init', array( $this, 'after_auth_redirect' ), 11 );
			add_action( 'init', array( $this, 'delete_connection_data_forcefully' ), 11 );
		}

		/**
		 * Registers REST API routes for the plugin.
		 * To check if the plugin is active.
		 *
		 * The API URL for the ping endpoint is:
		 *   /wp-json/wbtfe-multichannel-inventory-sync/v1/ping
		 */
		public function register_rest_routes() {
			register_rest_route(
				'wbtfe-multichannel-inventory-sync/v1',
				'/ping',
				array(
					'methods'  => 'POST',
					'permission_callback' => '__return_true',
					'callback' => function () {
						$active = is_plugin_active('wbtfe-multichannel-inventory-sync/wbtfe-multichannel-inventory-sync.php');
						return [
							'plugin'   => 'wbtfe-multichannel-inventory-sync',
							'active'   => $active,
							'version'  => WBTFE_MULTICHANNEL_INVENTORY_SYNC_VERSION,
						];
					},
				)
			);

			register_rest_route( 'wc/v3', '/wbtfe-multichannel-inventory-sync/warning-status', array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_warning_status_rest' ),
				'permission_callback' => '__return_true', // WooCommerce handles auth
			) );

			register_rest_route( 'wc/v3', '/wbtfe-multichannel-inventory-sync/clear-warning', array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'clear_warning_rest' ),
				'permission_callback' => '__return_true', // WooCommerce handles auth
			) );
		}

		/**
		 *  Add new admin menu.
		 */
		public function add_admin_menu() {
			/**
			 *  Allow role.
			 *
			 *  @param string $allow_role Allow role.
			 */
			$allowed_role = apply_filters( 'wbtfe_connector_allowed_roles', 'manage_woocommerce' );

			/**
			 *  Add menu page.
			 */
			add_menu_page(
				'Inventory Sync for Amazon',
				'Inventory Sync for Amazon',
				$allowed_role,
				'wbtfe-multichannel-inventory-sync',
				array( $this, 'render_admin_page' ),
				$this->get_menu_icon(),
				56
			);
		}

		private function get_menu_icon() {
			$svg = '<svg width="40" height="40" viewBox="0 0 40 40" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M30.9182 15.089L27.6719 11.9335C27.4675 11.6606 27.1469 11.5 26.8062 11.5H7.32841C6.98776 11.5 6.6671 11.6606 6.46272 11.9335L3.21642 15.089C3.08072 15.2706 3 15.5046 3 15.7393V18.9907C3 20.5918 3.87174 21.9919 5.1642 22.7427V31.6774C5.1642 34.2438 7.24876 36.3316 9.81111 36.3316H24.3235C26.8859 36.3316 28.9704 34.2438 28.9704 31.6774V22.7427C30.2629 21.9919 31.1346 20.5918 31.1346 18.9907V15.7393C31.1346 15.5145 31.0597 15.2784 30.9182 15.089ZM14.9031 34.164V27.6612H19.2315V34.164H14.9031ZM26.8062 31.6774C26.8062 33.0485 25.6925 34.164 24.3235 34.164H21.3957V26.5774C21.3957 25.9788 20.9112 25.4936 20.3136 25.4936H13.821C13.2234 25.4936 12.7389 25.9788 12.7389 26.5774V34.164H9.81111C8.4421 34.164 7.32841 33.0485 7.32841 31.6774V23.326C8.62029 23.326 9.78095 22.7554 10.5747 21.8536C11.3685 22.7554 12.5291 23.326 13.821 23.326C15.1129 23.326 16.2736 22.7554 17.0673 21.8536C17.8611 22.7554 19.0217 23.326 20.3136 23.326C21.6055 23.326 22.7662 22.7554 23.5599 21.8536C24.3537 22.7554 25.5143 23.326 26.8062 23.326V31.6774ZM26.8062 21.1583C25.6129 21.1583 24.642 20.186 24.642 18.9907V16.8231C24.642 16.2246 24.1575 15.7393 23.5599 15.7393C22.9623 15.7393 22.4778 16.2246 22.4778 16.8231V18.9907C22.4778 20.186 21.507 21.1583 20.3136 21.1583C19.1203 21.1583 18.1494 20.186 18.1494 18.9907V16.8231C18.1494 16.2246 17.6649 15.7393 17.0673 15.7393C16.4697 15.7393 15.9852 16.2246 15.9852 16.8231V18.9907C15.9852 20.186 15.0144 21.1583 13.821 21.1583C12.6277 21.1583 11.6568 20.186 11.6568 18.9907V16.8231C11.6568 16.2246 11.1723 15.7393 10.5747 15.7393C9.9771 15.7393 9.49261 16.2246 9.49261 16.8231V18.9907C9.49261 20.186 8.52175 21.1583 7.32841 21.1583C6.13506 21.1583 5.1642 20.186 5.1642 18.9907V16.1006L7.86946 13.6676H26.2652L28.9704 16.1006V18.9907C28.9704 20.186 27.9996 21.1583 26.8062 21.1583Z" fill="#9DA1A7"/>
<path d="M35.8857 3C36.5014 3.00001 37.001 3.49947 37.001 4.11523V11.5547C37.0009 12.1704 36.5014 12.6699 35.8857 12.6699C35.2701 12.6699 34.7715 12.1704 34.7715 11.5547V6.80664L28.5762 13.0029L27 11.4258L33.1943 5.23047H28.4482C27.8326 5.23046 27.333 4.73099 27.333 4.11523C27.333 3.49947 27.8326 3 28.4482 3H35.8857Z" fill="#9DA1A7"/>
</svg>';		
			return 'data:image/svg+xml;base64,' . base64_encode( $svg );
		}

		/**
		 *  Render admin page.
		 */
		public function render_admin_page() {

			$auth_url        = $this->get_auth_url();
			$is_connected    = $this->is_connected();
			$dashboard_url   = $this->get_dashboard_url();
			$connection_data = $this->get_connection_data();
			$connected_email = isset( $connection_data['email'] ) ? $connection_data['email'] : $this->app_name;
			$show_warning 	 = get_option( 'wbtfe_multichannel_inventory_connector_show_warning', false );
			$customer_name  = '';
			$last_sync_date = '';
			
			include_once WBTFE_MULTICHANNEL_INVENTORY_SYNC_PLUGIN_PATH . 'admin/views/admin-page.php';
		}

		/**
		 *  Save tokens and redirect to WC auth page
		 */
		public function save_tokens() {
			if ( isset( $_GET['wbtfe_mis_session_type'] ) && isset( $_GET['_wpnonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), $this->nonce_action ) ) {
				$session_type  = sanitize_text_field( wp_unslash( $_GET['wbtfe_mis_session_type'] ) );
				$website_id    = isset( $_GET['wbtfe_mis_website_id'] ) ? sanitize_text_field( wp_unslash( $_GET['wbtfe_mis_website_id'] ) ) : '';
				$session_id    = isset( $_GET['wbtfe_mis_session_id'] ) ? sanitize_text_field( wp_unslash( $_GET['wbtfe_mis_session_id'] ) ) : '';
				$refresh_token = isset( $_GET['wbtfe_mis_refresh_token'] ) ? sanitize_text_field( wp_unslash( $_GET['wbtfe_mis_refresh_token'] ) ) : '';
				$app_name      = isset( $_GET['wbtfe_mis_app_name'] ) ? sanitize_text_field( wp_unslash( $_GET['wbtfe_mis_app_name'] ) ) : '';
				$expires_at    = isset( $_GET['wbtfe_mis_expires_at'] ) ? sanitize_text_field( wp_unslash( $_GET['wbtfe_mis_expires_at'] ) ) : '';
				$user_id       = isset( $_GET['wbtfe_mis_user_id'] ) ? sanitize_text_field( wp_unslash( $_GET['wbtfe_mis_user_id'] ) ) : '';
				$website_key   = isset( $_GET['wbtfe_mis_website_key'] ) ? sanitize_text_field( wp_unslash( $_GET['wbtfe_mis_website_key'] ) ) : '';
				$email         = isset( $_GET['wbtfe_mis_email'] ) ? sanitize_text_field( wp_unslash( $_GET['wbtfe_mis_email'] ) ) : '';
				$return_url_to_app = isset( $_GET['return_url_to_app'] ) ? sanitize_text_field( wp_unslash( $_GET['return_url_to_app'] ) ) : '';
				$callback_url_to_app = isset( $_GET['callback_url_to_app'] ) ? sanitize_text_field( wp_unslash( $_GET['callback_url_to_app'] ) ) : '';

				if ( 'oauth' === $session_type
					&& $website_id
					&& $session_id
					&& $refresh_token
					&& $app_name
					&& $expires_at
					&& $user_id
					&& $website_key
					&& $email ) {

					// Here we are using nonce key as encryption key.
					$nonce = sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) );

					// Save connection data.
					$this->save_oauth_datas( $website_id, $session_id, $refresh_token, $app_name, $expires_at, $user_id, $nonce, $website_key, $email );

					// Redirect to WC auth page.
					$params       = array(
						'app_name'     => $this->app_name,
						'scope'        => $this->wc_api_perm,
						'user_id'      => $website_id,
						'return_url'   => $return_url_to_app,
						'callback_url' => $callback_url_to_app,
					);
					$query_string = http_build_query( $params );
					wp_safe_redirect( $this->get_wc_auth_url() . '?' . $query_string );
					exit;
				}
			}
		}

		/**
		 *  Get the connect url to the app.
		 *
		 *  @return string Auth url.
		 */
		private function get_auth_url() {
			$nonce = wp_create_nonce( $this->nonce_action );
			return $this->app_url . 'woocommerce/connect?session_type=oauth&platform=woocommerce&_wpnonce=' . $nonce . '&redirect_uri=' . home_url();
		}

		/**
		 *  Get dashboard url.
		 *
		 *  @return string Dashboard url.
		 */
		private function get_dashboard_url() {
			return $this->app_url . 'dashboard/';
		}

		/**
		 *  Get WC auth url.
		 *
		 *  @return string WC auth url.
		 */
		private function get_wc_auth_url() {
			return home_url() . '/wc-auth/v1/authorize';
		}

		/**
		 *  Get disconnect url.
		 *
		 *  @return string Disconnect url.
		 */
		private function get_disconnect_url() {
			return $this->disconnect_url . 'api/woocommerce/disconnect/';
		}

		/**
		 *  Get token storing url.
		 *
		 *  @param string $website_id Website id.
		 *  @param string $access_token Access token.
		 *  @return string Token url.
		 */
		private function get_token_url( $website_id, $access_token ) {
			return $this->token_db_url . 'websites/' . $website_id . '?session_id=' . rawurlencode( $access_token );
		}

		/**
		 *  Get refresh token url.
		 *
		 *  @return string Refresh token url.
		 */
		private function get_refresh_token_url() {
			return $this->token_db_url . 'token/';
		}

		/**
		 *  Encrypt and save oauth tokens.
		 *
		 *  @param string $website_id Website id.
		 *  @param string $session_id Session id.
		 *  @param string $refresh_token Refresh token.
		 *  @param string $app_name App name.
		 *  @param string $expires_at Expires at.
		 *  @param string $user_id User id.
		 *  @param string $nonce Nonce. Here we are using nonce as encryption key.
		 *  @param string $website_key Website key.
		 *  @param string $email Email.
		 */
		private function save_oauth_datas( $website_id, $session_id, $refresh_token, $app_name, $expires_at, $user_id, $nonce, $website_key, $email ) {

			// Save connection data.
			$data_arr = array(
				'refresh_token' => $this->encrypt_data( $refresh_token, $nonce . 'refresh' ),
				'access_token'  => $this->encrypt_data( $session_id, $nonce . 'token' ),
				'expires_at'    => $expires_at,
				'website_id'    => $website_id,
				'user_id'       => $user_id,
				'app_name'      => $app_name,
				'website_key'   => $website_key,
				'email'         => $email,
			);

			update_option( $this->data_option_name, $data_arr );

			// Save key.
			update_option( $this->key_option_name, $nonce );
		}

		/**
		 * Encrypt data securely.
		 *
		 * @param string $data The data to encrypt.
		 * @param string $key  A secure encryption key.
		 * @return string The encrypted data.
		 */
		private function encrypt_data( $data, $key ) {
			// Generate a secure hash of the key to ensure consistent length.
			$key_hash = hash( 'sha256', $key, true );

			// Repeat the key hash to match the length of the data.
			$key_repeated = str_repeat( $key_hash, ceil( strlen( $data ) / strlen( $key_hash ) ) );

			// XOR the data with the repeated key hash.
			$encrypted_data = $data ^ substr( $key_repeated, 0, strlen( $data ) );

			// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode -- Base64 encoding is used to store encrypted binary data as a string in the database.
			return base64_encode( $encrypted_data );
		}

		/**
		 * Decrypt data using base64 and a key.
		 *
		 * @param string $encrypted_data The encrypted data to decrypt.
		 * @param string $key            The encryption key.
		 * @return string The decrypted data.
		 */
		private function decrypt_data( $encrypted_data, $key ) {
			// Generate a secure hash of the key to ensure consistent length.
			$key_hash = hash( 'sha256', $key, true );

			// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode -- Base64 decoding is used to decrypt binary data stored in the database.
			$data = base64_decode( $encrypted_data );

			// Repeat the key hash to match the length of the data.
			$key_repeated = str_repeat( $key_hash, ceil( strlen( $data ) / strlen( $key_hash ) ) );

			// XOR the data with the repeated key hash to decrypt.
			return $data ^ substr( $key_repeated, 0, strlen( $data ) );
		}

		/**
		 *  Get connection data.
		 *
		 *  @return array Connection data.
		 */
		private function get_connection_data() {
			$this->connection_data = is_null( $this->connection_data ) ? get_option( $this->data_option_name, array() ) : $this->connection_data;
			return $this->connection_data;
		}

		/**
		 *  Is connected.
		 */
		private function is_connected() {
			return ! empty( $this->get_connection_data() );
		}

		/**
		 *  Disconnect WebToffee Multichannel Inventory Sync.
		 */
		public function disconnect_mis() {
			if ( isset( $_POST['nonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'wbtfe_mis_disconnect' ) && $this->is_connected() ) {

				$connection_data = $this->get_connection_data();
				$access_key      = $this->get_key( 'token' );
				$access_token    = $this->decrypt_data( $connection_data['access_token'], $access_key );

				$data_arr = array(
					'website_id'  => $connection_data['website_id'],
					'website_key' => $connection_data['website_key'],
					'user_id'     => $connection_data['user_id'],
					'email'       => $connection_data['email'],
					'action'      => 'disconnect',
				);

				$headers = array(
					'Authorization' => 'Bearer ' . $access_token,
					'Content-Type'  => 'application/json',
				);

				$response    = wp_remote_post(
					$this->get_disconnect_url(),
					array(
						'headers'     => $headers,
						'body'        => wp_json_encode( $data_arr ),
						'method'      => 'POST',
						'data_format' => 'body',
					)
				);

				$status_code = wp_remote_retrieve_response_code( $response );

				if ( ! is_wp_error( $response ) && 200 === $status_code ) {
					// Remove the connection data.
					delete_option( $this->data_option_name );
					delete_option( $this->key_option_name );
					update_option( $this->show_warning_option, false );
					wp_send_json_success( array( 'success' => true, 'message' => '1' ) );
				} else {
					wp_send_json_error( array( 'success' => false, 'message' => '2' ) );
				}
			} else {
				wp_send_json_error( array( 'success' => false, 'message' => '3' ) );
			}
		}

		/**
		 *  Is token expired. Important: Call only when connected and token is set.
		 *
		 *  @throws Exception If WebToffee Multichannel Inventory Sync is not connected.
		 *  @return bool True if token is expired, false otherwise.
		 */
		private function is_token_expired() {
			if ( ! $this->is_connected() ) {
				// Throw error.
				throw new Exception( esc_html__( 'WebToffee Multichannel Inventory Sync is not connected.', 'wbtfe-multichannel-inventory-sync' ) );
			}

			$connection_data = $this->get_connection_data();
			return time() > $connection_data['expires_at'];
		}

		/**
		 * Renew token.
		 *
		 * @throws Exception If `WebToffee Multichannel Inventory Sync` App is returned an error.
		 */
		private function renew_token() {
			if ( $this->is_connected() && $this->is_token_expired() ) {
				$connection_data = $this->get_connection_data();
				$refresh_key     = $this->get_key( 'refresh' );
				$refresh_token   = $this->decrypt_data( $connection_data['refresh_token'], $refresh_key );

				// Send a post request to the token url. With refresh token as body application/json.
				$response = wp_remote_post(
					$this->get_refresh_token_url(),
					array(
						'body'    => wp_json_encode( array( 'refresh_token' => $refresh_token ) ),
						'headers' => array( 'Content-Type' => 'application/json' ),
					)
				);

				if ( is_wp_error( $response ) ) {
					// Throw error.
					throw new Exception( esc_html__( 'Failed to renew token.', 'wbtfe-multichannel-inventory-sync' ) );
				}

				$response_body     = json_decode( wp_remote_retrieve_body( $response ), true );
				$new_access_token  = isset( $response_body['access_token'] ) ? $response_body['access_token'] : '';
				$new_refresh_token = isset( $response_body['refresh_token'] ) ? $response_body['refresh_token'] : '';
				$new_expires_at    = isset( $response_body['expires_at'] ) ? $response_body['expires_at'] : '';

				if ( ! $new_access_token || ! $new_refresh_token || ! $new_expires_at ) {
					// Throw error.
					throw new Exception( esc_html__( 'Failed to renew token.', 'wbtfe-multichannel-inventory-sync' ) );
				}

				$token_key                        = $this->get_key( 'token' );
				$connection_data['access_token']  = $this->encrypt_data( $new_access_token, $token_key );
				$connection_data['refresh_token'] = $this->encrypt_data( $new_refresh_token, $refresh_key );
				$connection_data['expires_at']    = $new_expires_at;

				// Update the connection data.
				update_option( $this->data_option_name, $connection_data );
			}
		}

		/**
		 *  Get encryption key.
		 *
		 *  @param string $type Key type.
		 *  @return string Key.
		 */
		private function get_key( $type ) {
			$key = get_option( $this->key_option_name );
			return 'refresh' === $type ? $key . 'refresh' : $key . 'token';
		}

		/**
		 *  After auth redirect.
		 */
		public function after_auth_redirect(){
			if ( ! is_admin() ) {
				return;
			}
			if( 
				isset( $_GET['_app_nonce'] ) &&
				wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_app_nonce'] ) ), $this->nonce_action ) &&
				isset( $_GET['page'], $_GET['tab'], $_GET['success'] ) 
				&& 'wbtfe-multichannel-inventory-sync' === $_GET['page'] 
				&& 'tab2' === $_GET['tab']
				&& isset( $_GET['app-redirect-uri'] ) && !empty( $_GET['app-redirect-uri'] )
			){
				$app_redirect_uri = sanitize_text_field( wp_unslash( $_GET['app-redirect-uri'] ) );
				if( sanitize_text_field( wp_unslash( $_GET['success'] ) ) ){
					update_option($this->show_warning_option, false);
					wp_redirect( $app_redirect_uri );
					exit;
				} else{
					$this->show_warning = true;
					update_option( $this->show_warning_option, true );
					wp_redirect($app_redirect_uri);
					exit;
				}
			}else{
				if( isset( $_GET['page'] ) && 'wbte-multichannel-inventory-sync' === sanitize_text_field( wp_unslash( $_GET['page'] ) ) && get_option( $this->show_warning_option, false ) ){
					$this->show_warning = true;
				}
			}
		}

		/**
		 *  Clear warning option.
		 *  This method can be called to clear the warning state.
		 * 
		 *  @since 1.0.0
		 */
		public function clear_warning() {
			update_option( $this->show_warning_option, false );
			$this->show_warning = false;
		}

		/**
		 *  Get warning status.
		 *  Returns the current warning status.
		 * 
		 *  @since 1.0.0
		 *  @return bool True if warning is active, false otherwise.
		 */
		public function get_warning_status() {
			return get_option( $this->show_warning_option, false );
		}

		/**
		 *  Get warning status via REST API.
		 *
		 *  @return WP_REST_Response Response object.
		 */
		public function get_warning_status_rest() {
			return new WP_REST_Response( array(
				'success' => true,
				'warning' => $this->get_warning_status(),
			), 200 );
		}

		/**
		 *  Clear warning via REST API.
		 *
		 *  @return WP_REST_Response Response object.
		 */
		public function clear_warning_rest() {
			$this->clear_warning();
			return new WP_REST_Response( array(
				'success' => true,
				'message' => 'Warning cleared successfully.',
			), 200 );
		}

		/**
		 * Deletes connection data and key forcefully when explicitly requested via the GET parameter.
		 * Delete this function, when plugin goes to live.
		 *
		 * @since 1.0.0
		 */
		public function delete_connection_data_forcefully() {
			if (
				isset( $_GET['_wpnonce'] ) 
				&& wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'wbtfe_delete_connection_data_forcefully' ) 
				&& isset( $_GET['page'], $_GET['delete_connection_data_forcefully'] )
				&& 'wbtfe-multichannel-inventory-sync' === sanitize_text_field( wp_unslash( $_GET['page'] ) )
				&& 'true' === sanitize_text_field( wp_unslash( $_GET['delete_connection_data_forcefully'] ) ) 
			) {
				// Verify user capability to prevent unauthorized access.
				if ( ! current_user_can( 'manage_options' ) ) {
					wp_die( esc_html__( 'You do not have permission to perform this action.', 'wbtfe-multichannel-inventory-sync' ) );
				}

				// Delete sensitive plugin options.
				delete_option( $this->data_option_name );
				delete_option( $this->key_option_name );
				update_option( $this->show_warning_option, false );

				// Sanitize redirect URL.
				$redirect_url = admin_url( 'admin.php?page=wbtfe-multichannel-inventory-sync' );
				wp_safe_redirect( $redirect_url );
				exit;
			}
		}
	}

	new Wbtfe_Multichannel_Inventory_Sync_Connector();
}
