<?php
/**
 * Plugin Name: Vulnity Security
 * Description: Security monitoring and SIEM integration for WordPress
 * Version: 1.1.9
 * Author: Vulnity
 * License: GPL-2.0-or-later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: vulnity
 * Domain Path: /languages
 */

if (!defined('ABSPATH')) exit;

define('VULNITY_VERSION', '1.1.9');

$vulnity_plugin_dir = plugin_dir_path(__FILE__);

if (!defined('VULNITY_PLUGIN_DIR')) {
    define('VULNITY_PLUGIN_DIR', $vulnity_plugin_dir);
}

if (!function_exists('vulnity_plugin_path')) {
    function vulnity_plugin_path($relative = '') {
        $base = plugin_dir_path(__FILE__);
        if ($relative === '') {
            return $base;
        }

        return rtrim($base, '/\\') . '/' . ltrim($relative, '/\\');
    }
}

if (!function_exists('vulnity_get_log_path')) {
    /**
     * Resolve Vulnity log path inside uploads.
     *
     * @return string
     */
    function vulnity_get_log_path() {
        static $resolved_path = null;

        if (is_string($resolved_path) && $resolved_path !== '') {
            return $resolved_path;
        }

        $candidate_bases = array();

        if (function_exists('wp_upload_dir')) {
            $uploads = wp_upload_dir(null, false);
            if (is_array($uploads) && !empty($uploads['basedir']) && is_string($uploads['basedir'])) {
                $candidate_bases[] = $uploads['basedir'];
            }
        }

        if (defined('WP_CONTENT_DIR')) {
            $candidate_bases[] = rtrim(WP_CONTENT_DIR, '/\\') . '/uploads';
            $candidate_bases[] = rtrim(WP_CONTENT_DIR, '/\\');
        }

        if (defined('ABSPATH')) {
            $candidate_bases[] = rtrim(ABSPATH, '/\\') . '/wp-content/uploads';
            $candidate_bases[] = rtrim(ABSPATH, '/\\') . '/wp-content';
        }

        $candidate_paths = array();
        foreach ($candidate_bases as $base_dir) {
            if (!is_string($base_dir) || $base_dir === '') {
                continue;
            }

            $candidate_paths[] = rtrim($base_dir, '/\\') . '/vulnity-logs/vulnity.log';
        }

        $candidate_paths = array_values(array_unique($candidate_paths));

        foreach ($candidate_paths as $candidate_path) {
            if (vulnity_ensure_log_dir($candidate_path)) {
                $resolved_path = $candidate_path;
                return $resolved_path;
            }
        }

        $resolved_path = isset($candidate_paths[0]) ? $candidate_paths[0] : '';
        return $resolved_path;
    }
}

if (!function_exists('vulnity_ensure_log_dir')) {
    /**
     * Ensure log directory exists and is writable.
     *
     * @param string $path Log file path.
     *
     * @return bool
     */
    function vulnity_ensure_log_dir($path) {
        if (!is_string($path) || $path === '') {
            return false;
        }

        $dir = dirname($path);
        if ($dir === '' || $dir === '.' || $dir === DIRECTORY_SEPARATOR) {
            return false;
        }

        if (!is_dir($dir)) {
            if (!function_exists('wp_mkdir_p') || !wp_mkdir_p($dir)) {
                $filesystem = vulnity_get_wp_filesystem();
                $chmod = defined('FS_CHMOD_DIR') ? FS_CHMOD_DIR : false;
                if (
                    !$filesystem ||
                    !method_exists($filesystem, 'mkdir') ||
                    !$filesystem->mkdir($dir, $chmod)
                ) {
                    return false;
                }
            }
        }

        if (!vulnity_path_is_writable($dir)) {
            return false;
        }

        vulnity_ensure_directory_index($dir);
        vulnity_ensure_log_access_protection($dir);

        return true;
    }
}

if (!function_exists('vulnity_set_log_access_notice')) {
    /**
     * Store a short-lived admin notice about log directory protection.
     *
     * @param string $message Notice message.
     * @param array  $paths   Relevant paths.
     *
     * @return void
     */
    function vulnity_set_log_access_notice($message, $paths = array()) {
        if (!function_exists('set_transient')) {
            return;
        }

        $message = is_string($message) ? trim($message) : '';
        if ($message === '') {
            return;
        }

        $safe_paths = array();
        if (is_array($paths)) {
            foreach ($paths as $path) {
                if (is_string($path) && $path !== '') {
                    $safe_paths[] = $path;
                }
            }
        }

        $expiration = defined('HOUR_IN_SECONDS') ? HOUR_IN_SECONDS : 3600;
        set_transient(
            'vulnity_log_access_notice',
            array(
                'message' => $message,
                'paths'   => array_values(array_unique($safe_paths)),
                'time'    => time(),
            ),
            $expiration
        );
    }
}

if (!function_exists('vulnity_write_file_contents')) {
    /**
     * Write file contents using WordPress filesystem API.
     *
     * @param string $path     File path.
     * @param string $contents Contents.
     *
     * @return bool
     */
    function vulnity_write_file_contents($path, $contents) {
        if (!is_string($path) || $path === '') {
            return false;
        }

        if (!is_string($contents)) {
            $contents = (string) $contents;
        }

        $filesystem = vulnity_get_wp_filesystem();
        if (!$filesystem || !method_exists($filesystem, 'put_contents')) {
            return false;
        }

        $mode = defined('FS_CHMOD_FILE') ? FS_CHMOD_FILE : false;
        return (bool) $filesystem->put_contents($path, $contents, $mode);
    }
}

if (!function_exists('vulnity_ensure_directory_index')) {
    /**
     * Ensure an index.php guard exists in the target directory.
     *
     * @param string $dir Directory path.
     *
     * @return void
     */
    function vulnity_ensure_directory_index($dir) {
        if (!is_string($dir) || $dir === '' || !is_dir($dir)) {
            return;
        }

        if (!vulnity_path_is_writable($dir)) {
            return;
        }

        $index_file = rtrim($dir, '/\\') . '/index.php';
        if (file_exists($index_file)) {
            return;
        }

        vulnity_write_file_contents($index_file, "<?php\n// Silence is golden.\n");
    }
}

if (!function_exists('vulnity_ensure_log_access_protection')) {
    /**
     * Ensure log directory has Apache deny rules and provide Nginx guidance.
     *
     * @param string $dir Log directory.
     *
     * @return void
     */
    function vulnity_ensure_log_access_protection($dir) {
        if (!is_string($dir) || $dir === '' || !is_dir($dir)) {
            return;
        }

        if (!vulnity_path_is_writable($dir)) {
            vulnity_set_log_access_notice(
                'Vulnity could not write log protection rules. Restrict access to the Vulnity log directory in your server configuration.',
                array($dir)
            );
            return;
        }

        $htaccess_path = rtrim($dir, '/\\') . '/.htaccess';
        if (!file_exists($htaccess_path)) {
            $rules = array(
                '# Vulnity Logs',
                '<IfModule mod_authz_core.c>',
                'Require all denied',
                '</IfModule>',
                '<IfModule !mod_authz_core.c>',
                'Deny from all',
                '</IfModule>',
                '<IfModule mod_autoindex.c>',
                'Options -Indexes',
                '</IfModule>',
            );

            $written = vulnity_write_file_contents($htaccess_path, implode("\n", $rules) . "\n");
            if (!$written) {
                vulnity_set_log_access_notice(
                    'Vulnity could not write log protection rules. Restrict access to the Vulnity log directory in your server configuration.',
                    array($dir)
                );
            }
        }

        $server_software = filter_input(INPUT_SERVER, 'SERVER_SOFTWARE', FILTER_UNSAFE_RAW);
        if (is_string($server_software) && function_exists('wp_unslash')) {
            $server_software = wp_unslash($server_software);
        }
        if (is_string($server_software) && function_exists('sanitize_text_field')) {
            $server_software = sanitize_text_field($server_software);
        }
        $server_software = is_string($server_software) ? strtolower($server_software) : '';

        if ($server_software !== '' && strpos($server_software, 'nginx') !== false && !get_option('vulnity_nginx_notice_shown')) {
            $firewall_dir = defined('VULNITY_PLUGIN_DIR')
                ? WP_CONTENT_DIR . '/uploads/vulnity-firewall'
                : '';
            $paths = array($dir);
            if ($firewall_dir !== '') {
                $paths[] = $firewall_dir;
            }
            vulnity_set_log_access_notice(
                'Your server appears to be running Nginx, which ignores .htaccess. Add a deny rule for the Vulnity log and firewall directories.',
                $paths
            );
            update_option('vulnity_nginx_notice_shown', '1', true);
        }
    }
}

if (!function_exists('vulnity_get_wp_filesystem')) {
    /**
     * Resolve WordPress filesystem API instance.
     *
     * @return object|null
     */
    function vulnity_get_wp_filesystem() {
        global $wp_filesystem;

        if (is_object($wp_filesystem)) {
            return $wp_filesystem;
        }

        $file_include = ABSPATH . 'wp-admin/includes/file.php';
        if (file_exists($file_include)) {
            require_once $file_include;
        } else {
            return null;
        }

        $base_include = ABSPATH . 'wp-admin/includes/class-wp-filesystem-base.php';
        if (!class_exists('WP_Filesystem_Base') && file_exists($base_include)) {
            require_once $base_include;
        }

        $direct_include = ABSPATH . 'wp-admin/includes/class-wp-filesystem-direct.php';
        if (!class_exists('WP_Filesystem_Direct') && file_exists($direct_include)) {
            require_once $direct_include;
        }

        if (!function_exists('WP_Filesystem')) {
            if (class_exists('WP_Filesystem_Direct')) {
                return new WP_Filesystem_Direct(false);
            }
            return null;
        }

        if (!WP_Filesystem()) {
            if (class_exists('WP_Filesystem_Direct')) {
                return new WP_Filesystem_Direct(false);
            }
            return null;
        }

        return is_object($wp_filesystem) ? $wp_filesystem : null;
    }
}

if (!function_exists('vulnity_path_is_writable')) {
    /**
     * Determine if a path is writable using WordPress filesystem helpers.
     *
     * @param string $path Path to validate.
     *
     * @return bool
     */
    function vulnity_path_is_writable($path) {
        if (!is_string($path) || $path === '') {
            return false;
        }

        if (function_exists('wp_is_writable')) {
            return wp_is_writable($path);
        }

        return false;
    }
}

if (!function_exists('vulnity_move_file')) {
    /**
     * Move file using WordPress filesystem API.
     *
     * @param string $source      Source file path.
     * @param string $destination Destination file path.
     *
     * @return bool
     */
    function vulnity_move_file($source, $destination) {
        if (
            !is_string($source) || $source === '' ||
            !is_string($destination) || $destination === ''
        ) {
            return false;
        }

        $filesystem = vulnity_get_wp_filesystem();
        if ($filesystem && method_exists($filesystem, 'move')) {
            return (bool) $filesystem->move($source, $destination, true);
        }

        return false;
    }
}

if (!function_exists('vulnity_delete_file')) {
    /**
     * Delete a file using WordPress API.
     *
     * @param string $path File path.
     *
     * @return bool
     */
    function vulnity_delete_file($path) {
        if (!is_string($path) || $path === '') {
            return false;
        }

        if (!file_exists($path)) {
            return true;
        }

        if (function_exists('wp_delete_file')) {
            wp_delete_file($path);
            return !file_exists($path);
        }

        $filesystem = vulnity_get_wp_filesystem();
        if ($filesystem && method_exists($filesystem, 'delete')) {
            return (bool) $filesystem->delete($path, false, 'f');
        }

        return false;
    }
}

if (!function_exists('vulnity_log_line_count_exceeds')) {
    /**
     * Check if line count exceeds a limit without loading full file in memory.
     *
     * @param string $path  File path.
     * @param int    $limit Max lines allowed.
     *
 * @return bool
 */
    function vulnity_log_line_count_exceeds($path, $limit) {
        if (!is_string($path) || $path === '' || !file_exists($path)) {
            return false;
        }

        $limit = max(1, (int) $limit);

        // Use streaming to avoid memory exhaustion with large files
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen -- Required for streaming large files without memory exhaustion
        $fp = fopen($path, 'r');
        if (!$fp) {
            // Conservative fallback: assume limit not exceeded when file is unreadable.
            return false;
        }

        $line_count = 0;
        while (!feof($fp) && $line_count < $limit) {
            if (fgets($fp) !== false) {
                $line_count++;
            }
        }
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- Closing stream opened above
        fclose($fp);

        return $line_count >= $limit;
    }
}

if (!function_exists('vulnity_rotate_log_files')) {
    /**
     * Rotate log files to keep a bounded number of line-limited files.
     *
     * @param string $base_path Base log path.
     * @param int    $max_files Total files including base file.
     * @param int    $max_lines Maximum lines in base file before rotation.
     */
    function vulnity_rotate_log_files($base_path, $max_files, $max_lines) {
        if (!is_string($base_path) || $base_path === '') {
            return;
        }

        $max_files = max(2, (int) $max_files);
        $max_lines = max(1, (int) $max_lines);

        if (!vulnity_log_line_count_exceeds($base_path, $max_lines)) {
            return;
        }

        // Use file locking to prevent race conditions
        $lock_file = $base_path . '.lock';
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen -- Required for flock() atomic locking
        $lock_fp = fopen($lock_file, 'c');

        if (!$lock_fp) {
            vulnity_log('[Vulnity] Failed to create lock file for log rotation');
            return;
        }

        // Try to get exclusive lock with timeout
        $lock_acquired = flock($lock_fp, LOCK_EX | LOCK_NB);

        if (!$lock_acquired) {
            // Another process is rotating, skip
            // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- Closing lock file
            fclose($lock_fp);
            return;
        }

        try {
            // Double-check line count after acquiring lock
            if (!vulnity_log_line_count_exceeds($base_path, $max_lines)) {
                return;
            }

            $max_archive = $max_files - 1;

            for ($i = $max_archive; $i >= 1; $i--) {
                $dst = $base_path . '.' . $i;
                $src = ($i === 1) ? $base_path : ($base_path . '.' . ($i - 1));

                if (file_exists($dst)) {
                    vulnity_delete_file($dst);
                }

                if (file_exists($src)) {
                    vulnity_move_file($src, $dst);
                }
            }
        } finally {
            // Always release lock and cleanup
            flock($lock_fp, LOCK_UN);
            // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- Closing lock file in finally block
            fclose($lock_fp);

            if (file_exists($lock_file)) {
                vulnity_delete_file($lock_file);
            }
        }
    }
}

if (!function_exists('vulnity_log')) {
    /**
     * Write Vulnity logs to plugin-owned rotating files in uploads.
     *
     * - Current file: vulnity.log
     * - Archives: vulnity.log.1, vulnity.log.2
     * - Default limits: 3 files total, 1000 lines per file.
     *
     * @param mixed  $message Message to log.
     * @param string $prefix  Optional prefix for the message.
     */
    function vulnity_log($message, $prefix = '') {
        if (is_array($message) || is_object($message)) {
            $message = wp_json_encode($message);
        }

        $message = (string) $message;
        if ($message === '') {
            return;
        }

        if ($prefix !== '') {
            $message = $prefix . $message;
        }

        // Keep one line per entry for deterministic line-based rotation.
        $message = str_replace(array("\r\n", "\r", "\n"), ' ', $message);

        $log_path = vulnity_get_log_path();
        if (!vulnity_ensure_log_dir($log_path)) {
            return;
        }

        $max_files = (int) apply_filters('vulnity_log_max_files', 3);
        $max_lines = (int) apply_filters('vulnity_log_max_lines', 1000);
        vulnity_rotate_log_files($log_path, $max_files, $max_lines);

        $timestamp = gmdate('Y-m-d H:i:s');
        $entry = '[' . $timestamp . ' UTC] ' . $message . PHP_EOL;

        // Best-effort logging without affecting request flow.
        @file_put_contents($log_path, $entry, FILE_APPEND | LOCK_EX); // phpcs:ignore WordPress.PHP.NoSilencedErrors.Discouraged
    }
}

if (!function_exists('vulnity_get_server_var')) {
    /**
     * Safely retrieve $_SERVER variable using filter_input with fallback.
     *
     * This function follows WordPress Plugin Check standards by using
     * filter_input() instead of direct $_SERVER access.
     *
     * @param string $key    The $_SERVER key to retrieve.
     * @param int    $filter The filter to apply (default: FILTER_SANITIZE_FULL_SPECIAL_CHARS).
     *
     * @return string The sanitized server variable value, or empty string if not found.
     */
    function vulnity_get_server_var($key, $filter = FILTER_SANITIZE_FULL_SPECIAL_CHARS) {
        if (!is_string($key) || $key === '') {
            return '';
        }

        // Use filter_input as primary method (Plugin Check compliant)
        $value = filter_input(INPUT_SERVER, $key, $filter);

        // Fallback for CLI/test environments where filter_input may return null
        if ($value === null || $value === false) {
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotValidated, WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Validation and sanitization handled below
            if (isset($_SERVER[$key])) {
                // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotValidated, WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitization applied immediately below
                $raw_value = $_SERVER[$key];

                if (is_string($raw_value)) {
                    // Apply WordPress sanitization as fallback
                    if (function_exists('wp_unslash')) {
                        $raw_value = wp_unslash($raw_value);
                    }

                    if (function_exists('sanitize_text_field')) {
                        $value = sanitize_text_field($raw_value);
                    } else {
                        $value = $raw_value;
                    }
                } else {
                    $value = '';
                }
            } else {
                $value = '';
            }
        }

        return is_string($value) ? $value : '';
    }
}

if (!defined('VULNITY_BASE_URL')) {
    define('VULNITY_BASE_URL', 'https://euxnoekqasvzwfcbybkg.supabase.co/functions/v1');
}

$vulnity_core_file = vulnity_plugin_path('includes/class-core.php');
$vulnity_anti_collapse_file = vulnity_plugin_path('includes/class-anti-collapse.php');

if (file_exists($vulnity_core_file)) {
    require_once $vulnity_core_file;
} else {
    vulnity_log('[Vulnity] Core file missing at: ' . $vulnity_core_file);
}

if (file_exists($vulnity_anti_collapse_file)) {
    require_once $vulnity_anti_collapse_file;
} else {
    vulnity_log('[Vulnity] Anti-collapse file missing at: ' . $vulnity_anti_collapse_file);
}

// Inicializar sistema anti-colapso automáticamente
add_action('init', function() {
    if (class_exists('Vulnity_Anti_Collapse_System')) {
        Vulnity_Anti_Collapse_System::get_instance();
    }
});

class Vulnity {
    private static $instance = null;
    private $core = null;
    
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));

        // Final cron cleanup at very low priority — other hooks (e.g. alert
        // dispatchers) may re-schedule cron events after deactivate() runs,
        // so we clear them once more after everything else has fired.
        add_action('deactivated_plugin', array($this, 'final_deactivation_cleanup'), 9999);

        // Add cron action for retry queue processing
        add_action('vulnity_process_retry_queue', array($this, 'process_retry_queue'));

        // Add action for triggered inventory sync
        add_action('vulnity_triggered_inventory_sync', array($this, 'handle_triggered_inventory_sync'));

        // Add heartbeat check action
        add_action('vulnity_check_heartbeat', array($this, 'check_heartbeat'));

        // Register custom cron interval
        add_filter('cron_schedules', array($this, 'add_cron_intervals'));

        // Add admin notices for critical failures
        add_action('admin_notices', array($this, 'show_critical_alert_failures'));

        // Schedule retry queue processing if not already scheduled
        if (!wp_next_scheduled('vulnity_process_retry_queue')) {
            wp_schedule_event(time(), 'vulnity_5min', 'vulnity_process_retry_queue');
        }

        // Ensure heartbeat check is scheduled every six hours
        $this->ensure_heartbeat_schedule();
        
        $this->core = Vulnity_Core::get_instance();
    }

    public function activate() {
        if (!get_option('vulnity_config')) {
            update_option('vulnity_config', array(
                'site_id' => '',
                'token' => '',
                'signing_secret' => '',
                'paired_at' => '',
                'status' => 'inactive'
            ));
        }
        
        if (!get_option('vulnity_alerts')) {
            update_option('vulnity_alerts', array());
        }
        
        if (!get_option('vulnity_retry_queue')) {
            update_option('vulnity_retry_queue', array());
        }
        
        if (!get_option('vulnity_failed_alerts')) {
            update_option('vulnity_failed_alerts', array());
        }

        update_option('vulnity_alerts_unread', 0);

        if (!get_option('vulnity_heartbeat_status')) {
            update_option('vulnity_heartbeat_status', array(
                'failures' => 0,
                'last_success' => null
            ));
        }

        if (class_exists('Vulnity_Firewall_Manager')) {
            $firewall = Vulnity_Firewall_Manager::get_instance();
            $firewall->install_firewall();
        }

        // Schedule cron events
        if (!wp_next_scheduled('vulnity_process_retry_queue')) {
            wp_schedule_event(time(), 'vulnity_5min', 'vulnity_process_retry_queue');
        }

        $this->ensure_heartbeat_schedule();

        // Process any pending alerts immediately after activation
        $this->process_unsent_alerts();
    }

    public function deactivate() {
        // Keep pairing and local data on deactivation.
        vulnity_log('[Vulnity] Plugin deactivated - preserving pairing and local data.');

        // Clean up runtime anti-collapse buffers/hooks.
        if (class_exists('Vulnity_Anti_Collapse_System')) {
            $anti_collapse = Vulnity_Anti_Collapse_System::get_instance();
            $anti_collapse->cleanup();
        }

        // Remove plugin-owned hardening marker rules from .htaccess when possible.
        if (class_exists('Vulnity_Static_Security')) {
            Vulnity_Static_Security::get_instance()->cleanup_on_deactivation();
        }

        // Remove runtime firewall integration.
        if (class_exists('Vulnity_Firewall_Manager')) {
            Vulnity_Firewall_Manager::get_instance()->remove_firewall();
        }

        // Always try to remove plugin-owned .htaccess marker blocks on deactivation.
        $this->remove_plugin_htaccess_markers();

        // Remove plugin rewrite rules from WordPress runtime.
        flush_rewrite_rules(false);

        delete_option('vulnity_heartbeat_status');

        // Clear ALL runtime schedules LAST — some cleanup steps above may
        // re-schedule cron events (e.g. alert sending triggers inventory sync),
        // so we clear them after everything else has finished.
        $vulnity_cron_hooks = array(
            'vulnity_sync_inventory',
            'vulnity_sync_inventory_deferred',
            'vulnity_process_retry_queue',
            'vulnity_flush_alert_buffer',
            'vulnity_check_heartbeat',
            'vulnity_sync_mitigation_config',
            'vulnity_cleanup_flood_data',
            'vulnity_process_brute_force_windows',
            'vulnity_triggered_inventory_sync',
        );
        foreach ($vulnity_cron_hooks as $hook) {
            wp_unschedule_hook($hook);
        }
    }

    /**
     * Final sweep after all deactivation hooks have fired.
     * Removes any cron events re-scheduled by late-firing hooks (e.g.
     * plugin-change alerts that trigger inventory syncs) and ensures
     * .htaccess marker blocks are fully removed.
     * Runs on 'deactivated_plugin' at priority 9999 — only for this plugin.
     */
    public function final_deactivation_cleanup($plugin) {
        if ($plugin !== plugin_basename(__FILE__)) {
            return;
        }

        // Final cron sweep — wp_unschedule_hook removes ALL events for a hook
        // regardless of arguments (unlike wp_clear_scheduled_hook).
        $vulnity_cron_hooks = array(
            'vulnity_sync_inventory',
            'vulnity_sync_inventory_deferred',
            'vulnity_process_retry_queue',
            'vulnity_flush_alert_buffer',
            'vulnity_check_heartbeat',
            'vulnity_sync_mitigation_config',
            'vulnity_cleanup_flood_data',
            'vulnity_process_brute_force_windows',
            'vulnity_triggered_inventory_sync',
        );
        foreach ($vulnity_cron_hooks as $hook) {
            wp_unschedule_hook($hook);
        }

        // Final .htaccess sweep — other deactivation hooks may have
        // re-written markers after our initial cleanup pass.
        $this->remove_plugin_htaccess_markers();
    }

    /**
     * Add custom cron intervals
     */
    public function add_cron_intervals($schedules) {
        $schedules['vulnity_every_minute'] = array(
            'interval' => 60, // 1 minute
            'display'  => $this->translate_label('Every Minute (Vulnity)')
        );

        $schedules['vulnity_5min'] = array(
            'interval' => 300, // 5 minutes
            'display'  => $this->translate_label('Every 5 Minutes (Vulnity)')
        );

        $schedules['vulnity_15min'] = array(
            'interval' => 900, // 15 minutes
            'display'  => $this->translate_label('Every 15 Minutes (Vulnity)')
        );

        $schedules['vulnity_6hours'] = array(
            'interval' => 21600, // 6 hours
            'display'  => $this->translate_label('Every 6 Hours (Vulnity)')
        );

        return $schedules;
    }

    /**
     * Avoid early translation loading notices before init.
     *
     * @param string $text Label text.
     *
     * @return string
     */
    private function translate_label($text) {
        if (!is_string($text) || $text === '') {
            return '';
        }

        if (did_action('init') > 0) {
            switch ($text) {
                case 'Every Minute (Vulnity)':
                    return __('Every Minute (Vulnity)', 'vulnity');
                case 'Every 5 Minutes (Vulnity)':
                    return __('Every 5 Minutes (Vulnity)', 'vulnity');
                case 'Every 15 Minutes (Vulnity)':
                    return __('Every 15 Minutes (Vulnity)', 'vulnity');
                case 'Every 6 Hours (Vulnity)':
                    return __('Every 6 Hours (Vulnity)', 'vulnity');
            }
        }

        return $text;
    }

    private function ensure_heartbeat_schedule() {
        $current_schedule = wp_get_schedule('vulnity_check_heartbeat');

        if ($current_schedule && $current_schedule !== 'vulnity_6hours') {
            wp_clear_scheduled_hook('vulnity_check_heartbeat');
        }

        if (!wp_next_scheduled('vulnity_check_heartbeat')) {
            wp_schedule_event(time(), 'vulnity_6hours', 'vulnity_check_heartbeat');
        }
    }

    private function remove_plugin_htaccess_markers() {
        $markers = array(
            'Vulnity Login URL',
            'Vulnity Common Paths',
            'Vulnity Firewall',
        );

        foreach ($markers as $marker) {
            $this->remove_htaccess_marker($marker);
        }
    }

    private function remove_htaccess_marker($marker) {
        if (!is_string($marker) || $marker === '') {
            return;
        }

        if (!function_exists('get_home_path')) {
            $file_include = ABSPATH . 'wp-admin/includes/file.php';
            if (file_exists($file_include)) {
                require_once $file_include;
            }
        }

        $home_path = function_exists('get_home_path') ? get_home_path() : ABSPATH;
        $htaccess_path = trailingslashit($home_path) . '.htaccess';

        if (!file_exists($htaccess_path)) {
            return;
        }

        if (!is_readable($htaccess_path)) {
            vulnity_log('[Vulnity] .htaccess not readable, cannot remove marker: ' . $marker);
            return;
        }

        // Try WP Filesystem first, fall back to native PHP functions.
        $contents = false;
        $use_native = false;

        $filesystem = vulnity_get_wp_filesystem();
        if ($filesystem && method_exists($filesystem, 'get_contents') && method_exists($filesystem, 'put_contents')) {
            $contents = $filesystem->get_contents($htaccess_path);
        }

        if ($contents === false) {
            // Fallback: read with native PHP if WP Filesystem is unavailable.
            $contents = @file_get_contents($htaccess_path);
            $use_native = true;
        }

        if ($contents === false) {
            vulnity_log('[Vulnity] Could not read .htaccess to remove marker: ' . $marker);
            return;
        }

        list($updated, $changed) = $this->strip_marker_blocks_from_contents($contents, $marker);
        if (!$changed) {
            return;
        }

        $written = false;

        if (!$use_native && $filesystem) {
            $chmod = defined('FS_CHMOD_FILE') ? FS_CHMOD_FILE : false;
            $written = $filesystem->put_contents($htaccess_path, $updated, $chmod);
        }

        if (!$written && vulnity_path_is_writable($htaccess_path)) {
            // Fallback: write with native PHP when WP_Filesystem is unavailable.
            $fallback_fs = new WP_Filesystem_Direct(false);
            $chmod = defined('FS_CHMOD_FILE') ? FS_CHMOD_FILE : false;
            $written = $fallback_fs->put_contents($htaccess_path, $updated, $chmod);
        }

        if ($written) {
            vulnity_log('[Vulnity] Removed .htaccess marker: ' . $marker);
        } else {
            vulnity_log('[Vulnity] Failed to write .htaccess after removing marker: ' . $marker);
        }
    }

    private function strip_marker_blocks_from_contents($contents, $marker) {
        if (!is_string($contents) || !is_string($marker) || $marker === '') {
            return array($contents, false);
        }

        $begin_marker = '# BEGIN ' . $marker;
        $end_marker = '# END ' . $marker;
        $changed = false;
        $offset = 0;

        while (($begin_pos = strpos($contents, $begin_marker, $offset)) !== false) {
            $prefix = substr($contents, 0, $begin_pos);
            $line_start = strrpos($prefix, "\n");
            $line_start = ($line_start === false) ? 0 : ($line_start + 1);

            $end_pos = strpos($contents, $end_marker, $begin_pos);
            if ($end_pos === false) {
                break;
            }

            $line_end = strpos($contents, "\n", $end_pos);
            $line_end = ($line_end === false) ? strlen($contents) : ($line_end + 1);

            $contents = substr($contents, 0, $line_start) . substr($contents, $line_end);
            $changed = true;
            $offset = $line_start;
        }

        return array($contents, $changed);
    }
    
    /**
     * Handle triggered inventory sync
     */
    public function handle_triggered_inventory_sync($trigger_type) {
        vulnity_log('[Vulnity] Handling triggered inventory sync from: ' . $trigger_type);
        
        if (class_exists('Vulnity_Inventory_Sync')) {
            $inventory_sync = Vulnity_Inventory_Sync::get_instance();
            $scan_type = 'alert_triggered_' . $trigger_type;
            
            $result = $inventory_sync->perform_sync($scan_type);
            
            if ($result['success']) {
                vulnity_log('[Vulnity] Scheduled inventory sync completed successfully');
            } else {
                vulnity_log('[Vulnity] Scheduled inventory sync failed: ' . $result['error']);
            }
        }
    }
    
    /**
     * Process retry queue via cron
     */
    public function process_retry_queue() {
        vulnity_log('[Vulnity] Processing retry queue...');
        
        // Load the alert base class if needed
        if (!class_exists('Vulnity_Alert_Base')) {
            $alerts_dir = vulnity_plugin_path('includes/alerts/');
            if (file_exists($alerts_dir . 'class-alert-base.php')) {
                require_once $alerts_dir . 'class-alert-base.php';
            }
        }
        
        // Process the queue
        if (class_exists('Vulnity_Alert_Base')) {
            Vulnity_Alert_Base::process_retry_queue();
        }
        
        // Also process any unsent alerts
        $this->process_unsent_alerts();
    }
    
    /**
     * Process unsent alerts
     */
    private function process_unsent_alerts() {
        $alerts = get_option('vulnity_alerts', array());
        $siem = Vulnity_SIEM_Connector::get_instance();
        $processed_count = 0;
        $max_to_process = 10; // Process max 10 alerts at a time to avoid timeout
        
        foreach ($alerts as &$alert) {
            if ($processed_count >= $max_to_process) {
                break;
            }
            
            if (!$alert['sent_to_siem'] && (!isset($alert['retry_count']) || $alert['retry_count'] < 3)) {
                $result = $siem->send_alert($alert);
                
                if ($result['success']) {
                    $alert['sent_to_siem'] = true;
                    $alert['siem_response'] = $result['data'];
                    $alert['sent_at'] = current_time('mysql');
                    vulnity_log('[Vulnity] Unsent alert processed successfully: ' . $alert['id']);
                } else {
                    $alert['retry_count'] = isset($alert['retry_count']) ? $alert['retry_count'] + 1 : 1;
                    $alert['last_retry'] = current_time('mysql');
                    $alert['last_error'] = $result['error'];
                }
                
                $processed_count++;
            }
        }
        
        if ($processed_count > 0) {
            update_option('vulnity_alerts', $alerts);
            vulnity_log('[Vulnity] Processed ' . $processed_count . ' unsent alerts');
        }
    }
    
    /**
     * Send all pending alerts before deactivation
     */
    private function send_all_pending_alerts() {
        $alerts = get_option('vulnity_alerts', array());
        $siem = Vulnity_SIEM_Connector::get_instance();
        $sent_count = 0;

        foreach ($alerts as &$alert) {
            if (!$alert['sent_to_siem']) {
                $result = $siem->send_alert($alert);

                if ($result['success']) {
                    $alert['sent_to_siem'] = true;
                    $alert['sent_at'] = current_time('mysql');
                    $sent_count++;
                }
            }
        }

        if ($sent_count > 0) {
            update_option('vulnity_alerts', $alerts);
            vulnity_log('[Vulnity] Sent ' . $sent_count . ' pending alerts before deactivation');
        }
    }

    public function check_heartbeat() {
        $config = get_option('vulnity_config');

        if (empty($config) || empty($config['site_id']) || empty($config['token']) || (isset($config['status']) && $config['status'] !== 'active')) {
            return;
        }

        $heartbeat_state = get_option('vulnity_heartbeat_status', array('failures' => 0));
        $heartbeat_state['last_checked'] = current_time('mysql');

        $endpoint = VULNITY_BASE_URL . '/heartbeat';

        $theme = function_exists('wp_get_theme') ? wp_get_theme() : null;
        $timezone_string = function_exists('wp_timezone_string') ? wp_timezone_string() : get_option('timezone_string');
        $latency_recorded_at = isset($heartbeat_state['last_latency_recorded_at']) ? $heartbeat_state['last_latency_recorded_at'] : null;

        $system_info = array(
            'site_id'          => $config['site_id'],
            'site_url'         => get_site_url(),
            'home_url'         => home_url(),
            'site_name'        => get_bloginfo('name'),
            'language'         => get_locale(),
            'timezone'         => $timezone_string ?: date_default_timezone_get(),
            'multisite'        => is_multisite(),
            'theme'            => $theme ? $theme->get('Name') : null,
            'theme_version'    => $theme ? $theme->get('Version') : null,
            'php_memory_limit' => ini_get('memory_limit'),
            'server'           => php_sapi_name(),
            'timestamp'        => current_time('c'),
            'latency_recorded_at' => $latency_recorded_at,
        );

        $body_latency = isset($heartbeat_state['last_latency']) ? (int) $heartbeat_state['last_latency'] : null;
        $body = array(
            'status'            => 'online',
            'plugin_version'    => VULNITY_VERSION,
            'wordpress_version' => get_bloginfo('version'),
            'php_version'       => PHP_VERSION,
            'latency_ms'        => $body_latency,
            'system_info'       => $system_info,
        );

        $headers = array(
            'Content-Type' => 'application/json',
            'x-vulnity-token' => $config['token'],
            'x-site-id' => $config['site_id']
        );

        $args = array(
            'method' => 'POST',
            'headers' => $headers,
            'timeout' => 15,
            'sslverify' => true
        );

        $start_time = microtime(true);
        $args['body'] = wp_json_encode($body);
        $response = wp_remote_post($endpoint, $args);
        $latency_ms = round((microtime(true) - $start_time) * 1000);
        $status_code = null;
        $error_message = '';

        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
        } else {
            $status_code = wp_remote_retrieve_response_code($response);

            if ($status_code >= 200 && $status_code < 300) {
                $heartbeat_state['failures'] = 0;
                $heartbeat_state['last_success'] = current_time('mysql');
                $heartbeat_state['last_status'] = $status_code;
                $heartbeat_state['last_latency'] = $latency_ms;
                $heartbeat_state['last_latency_recorded_at'] = current_time('mysql');
                unset($heartbeat_state['last_error']);
                update_option('vulnity_heartbeat_status', $heartbeat_state);
                return;
            }

            $response_body = wp_remote_retrieve_body($response);
            $decoded = json_decode($response_body, true);
            if (is_array($decoded) && isset($decoded['error'])) {
                $error_message = $decoded['error'];
            } else {
                $error_message = sprintf('HTTP %d: %s', $status_code, wp_trim_words($response_body, 40));
            }
        }

        if (!empty($error_message)) {
            $heartbeat_state['last_error'] = $error_message;
        }

        $heartbeat_state['last_status'] = $status_code ?? 'error';
        $heartbeat_state['last_latency'] = $latency_ms;
        $heartbeat_state['last_latency_recorded_at'] = current_time('mysql');
        update_option('vulnity_heartbeat_status', $heartbeat_state);

        vulnity_log('[Vulnity] Heartbeat request failed: ' . ($error_message ?: 'Unknown error'));
    }
    
    /**
     * Show admin notices for critical alert failures
     */
    public function show_critical_alert_failures() {
        $failed_critical = get_transient('vulnity_critical_alert_failed');
        
        if ($failed_critical) {
            ?>
            <div class="notice notice-error is-dismissible">
                <p><strong>Vulnity Security Alert:</strong> <?php echo esc_html($failed_critical['message']); ?></p>
                <p>Alert ID: <?php echo esc_html($failed_critical['alert_id']); ?></p>
                <p><a href="<?php echo esc_url( admin_url( 'options-general.php?page=vulnity&tab=alerts' ) ); ?>">View Alerts</a></p>
            </div>
            <?php
            
            // Delete transient after showing
            delete_transient('vulnity_critical_alert_failed');
        }
        
        // Also check for failed queue
        $failed_alerts = get_option('vulnity_failed_alerts', array());
        $critical_failed = array_filter($failed_alerts, function($item) {
            return $item['alert']['severity'] === 'critical';
        });
        
        // Show anti-collapse panic mode notification if active
        if (class_exists('Vulnity_Anti_Collapse_System')) {
            $anti_collapse = Vulnity_Anti_Collapse_System::get_instance();
            $stats = $anti_collapse->get_stats();
            
            if ($stats['panic_mode']) {
                ?>
                <div class="notice notice-warning is-dismissible">
                    <p><strong>Vulnity Anti-Collapse:</strong> System is in PANIC MODE due to high alert volume (<?php echo esc_html( $stats['current_rate'] ); ?> alerts/min). Alerts are being aggregated to prevent SIEM overload.</p>
                </div>
                <?php
            }
        }

        $firewall_notice = get_transient('vulnity_firewall_storage_unwritable');
        if (is_array($firewall_notice) && !empty($firewall_notice['message'])) {
            $paths = '';
            if (!empty($firewall_notice['paths']) && is_array($firewall_notice['paths'])) {
                $paths = implode(', ', array_map('esc_html', $firewall_notice['paths']));
            }
            ?>
            <div class="notice notice-warning is-dismissible">
                <p><strong>Vulnity Firewall:</strong> <?php echo esc_html($firewall_notice['message']); ?></p>
                <?php if ($paths !== '') : ?>
                    <p><?php echo esc_html('Checked paths: ' . $paths); ?></p>
                <?php endif; ?>
            </div>
            <?php
        }

        $log_notice = get_transient('vulnity_log_access_notice');
        if (is_array($log_notice) && !empty($log_notice['message'])) {
            ?>
            <div class="notice notice-warning is-dismissible">
                <p><strong>Vulnity Logs:</strong> <?php echo esc_html($log_notice['message']); ?></p>
                <?php if (!empty($log_notice['paths']) && is_array($log_notice['paths'])) : ?>
                    <?php foreach ($log_notice['paths'] as $path) : ?>
                        <p><?php echo esc_html('Path: ' . $path); ?></p>
                    <?php endforeach; ?>
                <?php endif; ?>
                <p>See solution documentation: <a href="https://vulnity.gitbook.io/vulnity-docs/instalaciones/nginx-warning" target="_blank" rel="noopener noreferrer">https://vulnity.gitbook.io/vulnity-docs/instalaciones/nginx-warning</a></p>
            </div>
            <?php

            // Delete transient after showing so the notice only appears once
            delete_transient('vulnity_log_access_notice');
        }
    }
}

// Initialize the plugin
Vulnity::get_instance();

// Global function to manually trigger retry queue processing (for debugging)
if (!function_exists('vulnity_process_retry_queue_manual')) {
    function vulnity_process_retry_queue_manual() {
        $vulnity = Vulnity::get_instance();
        $vulnity->process_retry_queue();
        return 'Retry queue processed';
    }
}

// Global function to check anti-collapse status (for debugging)
if (!function_exists('vulnity_check_anti_collapse_status')) {
    function vulnity_check_anti_collapse_status() {
        if (class_exists('Vulnity_Anti_Collapse_System')) {
            $anti_collapse = Vulnity_Anti_Collapse_System::get_instance();
            return $anti_collapse->get_stats();
        }
        return 'Anti-collapse system not loaded';
    }
}
