<?php
/**
 * Vulnity Server Configuration View
 * Shows server-specific configuration instructions for Apache/Nginx
 */

if (!defined('ABSPATH')) {
    exit;
}

// phpcs:disable WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound -- Template variables, not globals
$static_security = Vulnity_Static_Security::get_instance();
$server_type = $static_security->detect_server_type();
$settings = $static_security->get_settings();

$server_name = ucfirst($server_type);
if ($server_type === 'unknown') {
    $server_name = 'Unknown Server';
}

$has_active_features = (
    !empty($settings['rename_login_enabled']) ||
    !empty($settings['protect_common_paths'])
);
// phpcs:enable WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound
?>

<div class="wrap vulnity-server-config">
    <h1><?php echo esc_html__('Server Configuration', 'vulnity'); ?></h1>

    <div class="vulnity-server-detection">
        <h2><?php echo esc_html__('Detected Server', 'vulnity'); ?></h2>
        <p class="vulnity-server-badge vulnity-server-<?php echo esc_attr($server_type); ?>">
            <strong><?php echo esc_html($server_name); ?></strong>
        </p>
    </div>

    <?php if ($server_type === 'apache' || $server_type === 'litespeed'): ?>

        <div class="notice notice-success">
            <p>
                <strong><?php echo esc_html__('Good News!', 'vulnity'); ?></strong>
                <?php echo esc_html__('Your server supports .htaccess files. Vulnity manages these rules automatically. No manual configuration needed.', 'vulnity'); ?>
            </p>
        </div>

        <div class="vulnity-config-section">
            <h3><?php echo esc_html__('Automatic Configuration', 'vulnity'); ?></h3>
            <p><?php echo esc_html__('The following features are configured automatically via .htaccess:', 'vulnity'); ?></p>
            <ul>
                <?php if (!empty($settings['rename_login_enabled'])): ?>
                    <li>✅ <?php echo esc_html__('Custom Login URL', 'vulnity'); ?></li>
                <?php endif; ?>
                <?php if (!empty($settings['protect_common_paths'])): ?>
                    <li>✅ <?php echo esc_html__('Common Paths Protection', 'vulnity'); ?></li>
                <?php endif; ?>
                <li>✅ <?php echo esc_html__('Log Directory Protection', 'vulnity'); ?></li>
            </ul>
        </div>

    <?php elseif ($server_type === 'nginx'): ?>

        <?php if (!$has_active_features): ?>
            <div class="notice notice-info">
                <p><?php echo esc_html__('No active features requiring Nginx configuration. Enable features in the Mitigation tab first.', 'vulnity'); ?></p>
            </div>
        <?php else: ?>
            <div class="notice notice-warning">
                <p>
                    <strong><?php echo esc_html__('Action Required', 'vulnity'); ?></strong>
                    <?php echo esc_html__('Nginx does not support .htaccess files. You need to manually add the configuration below to your nginx.conf or server block.', 'vulnity'); ?>
                </p>
            </div>

            <div class="vulnity-config-section">
                <h3><?php echo esc_html__('Nginx Configuration', 'vulnity'); ?></h3>
                <p><?php echo esc_html__('Add the following to your Nginx server block configuration:', 'vulnity'); ?></p>

                <div class="vulnity-config-wrapper">
                    <pre id="nginx-config-snippet" class="vulnity-config-code"><?php
                        echo esc_html($static_security->get_nginx_config_snippet());
                    ?></pre>
                    <button type="button" class="button button-secondary" onclick="vulnityCopyConfig()">
                        <?php echo esc_html__('Copy Configuration', 'vulnity'); ?>
                    </button>
                </div>

                <div class="vulnity-instructions">
                    <h4><?php echo esc_html__('Installation Steps', 'vulnity'); ?></h4>
                    <ol>
                        <li><?php echo esc_html__('Copy the configuration above', 'vulnity'); ?></li>
                        <li><?php echo esc_html__('Open your Nginx configuration file (usually /etc/nginx/sites-available/your-site)', 'vulnity'); ?></li>
                        <li><?php echo esc_html__('Paste the configuration inside your server { } block', 'vulnity'); ?></li>
                        <li><?php echo esc_html__('Test the configuration: sudo nginx -t', 'vulnity'); ?></li>
                        <li><?php echo esc_html__('Reload Nginx: sudo systemctl reload nginx', 'vulnity'); ?></li>
                    </ol>
                </div>

                <div class="vulnity-php-fallback">
                    <h4><?php echo esc_html__('Alternative: PHP-Only Protection', 'vulnity'); ?></h4>
                    <p><?php echo esc_html__('Vulnity also provides basic PHP-level protection that works without server configuration. However, server-level protection is more secure and performant.', 'vulnity'); ?></p>
                </div>
            </div>
        <?php endif; ?>

    <?php else: ?>

        <div class="notice notice-warning">
            <p>
                <strong><?php echo esc_html__('Unknown Server Type', 'vulnity'); ?></strong>
                <?php echo esc_html__('Vulnity could not detect your web server. Please configure manually based on your server type.', 'vulnity'); ?>
            </p>
        </div>

        <div class="vulnity-config-section">
            <h3><?php echo esc_html__('Manual Configuration Required', 'vulnity'); ?></h3>
            <p><?php echo esc_html__('If you are using:', 'vulnity'); ?></p>
            <ul>
                <li><strong>Apache/LiteSpeed:</strong> <?php echo esc_html__('Configuration should work automatically via .htaccess', 'vulnity'); ?></li>
                <li><strong>Nginx:</strong> <?php echo esc_html__('See the Nginx configuration section above', 'vulnity'); ?></li>
                <li><strong>Other:</strong> <?php echo esc_html__('Consult your server documentation for implementing URL rewrites and path protection', 'vulnity'); ?></li>
            </ul>
        </div>

    <?php endif; ?>

    <div class="vulnity-help-section">
        <h3><?php echo esc_html__('Need Help?', 'vulnity'); ?></h3>
        <p>
            <?php
            printf(
                /* translators: %s: Link to documentation */
                esc_html__('If you need assistance with server configuration, please visit our %s or contact your hosting provider.', 'vulnity'),
                '<a href="https://vulnity.com/docs/server-configuration" target="_blank">' . esc_html__('documentation', 'vulnity') . '</a>'
            );
            ?>
        </p>
    </div>
</div>

<style>
.vulnity-server-config {
    max-width: 1000px;
}

.vulnity-server-badge {
    display: inline-block;
    padding: 10px 20px;
    background: #f0f0f1;
    border-radius: 4px;
    font-size: 16px;
}

.vulnity-server-badge.vulnity-server-apache,
.vulnity-server-badge.vulnity-server-litespeed {
    background: #d4edda;
    color: #155724;
    border: 1px solid #c3e6cb;
}

.vulnity-server-badge.vulnity-server-nginx {
    background: #fff3cd;
    color: #856404;
    border: 1px solid #ffeaa7;
}

.vulnity-server-badge.vulnity-server-unknown {
    background: #f8d7da;
    color: #721c24;
    border: 1px solid #f5c6cb;
}

.vulnity-config-section {
    background: #fff;
    border: 1px solid #ccd0d4;
    border-radius: 4px;
    padding: 20px;
    margin: 20px 0;
}

.vulnity-config-section h3 {
    margin-top: 0;
}

.vulnity-config-section ul {
    list-style: none;
    padding-left: 0;
}

.vulnity-config-section ul li {
    padding: 5px 0;
    font-size: 14px;
}

.vulnity-config-wrapper {
    position: relative;
    margin: 15px 0;
}

.vulnity-config-code {
    background: #f6f7f7;
    border: 1px solid #dcdcde;
    border-radius: 4px;
    padding: 15px;
    font-family: 'Courier New', monospace;
    font-size: 13px;
    line-height: 1.6;
    overflow-x: auto;
    white-space: pre;
}

.vulnity-config-wrapper .button {
    margin-top: 10px;
}

.vulnity-instructions {
    background: #f0f6fc;
    border-left: 4px solid #0073aa;
    padding: 15px;
    margin: 20px 0;
}

.vulnity-instructions h4 {
    margin-top: 0;
}

.vulnity-instructions ol {
    margin-bottom: 0;
}

.vulnity-instructions ol li {
    padding: 5px 0;
}

.vulnity-php-fallback {
    background: #fffbf0;
    border-left: 4px solid #f0b429;
    padding: 15px;
    margin: 20px 0;
}

.vulnity-php-fallback h4 {
    margin-top: 0;
}

.vulnity-help-section {
    margin-top: 30px;
    padding-top: 20px;
    border-top: 1px solid #dcdcde;
}
</style>

<script>
function vulnityCopyConfig() {
    const snippet = document.getElementById('nginx-config-snippet');
    const textArea = document.createElement('textarea');
    textArea.value = snippet.textContent;
    document.body.appendChild(textArea);
    textArea.select();

    try {
        document.execCommand('copy');
        const button = event.target;
        const originalText = button.textContent;
        button.textContent = '<?php echo esc_js(__('Copied!', 'vulnity')); ?>';
        button.style.background = '#00a32a';
        button.style.color = '#fff';

        setTimeout(function() {
            button.textContent = originalText;
            button.style.background = '';
            button.style.color = '';
        }, 2000);
    } catch (err) {
        alert('<?php echo esc_js(__('Failed to copy. Please copy manually.', 'vulnity')); ?>');
    }

    document.body.removeChild(textArea);
}
</script>
