<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class Vulnity_Security_Monitor {
    
    private static $instance = null;
    private $core = null;
    private $suspicious_activity_threshold = 10;
    private $critical_events_count = 0;
    
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_action('vulnity_critical_event_detected', array($this, 'handle_critical_event'), 10, 2);
        add_action('vulnity_security_breach_detected', array($this, 'handle_security_breach'), 10, 2);
        add_filter('vulnity_unpair_reason', array($this, 'determine_unpair_reason'), 10, 2);
    }
    
    /**
     * Handle critical security events that might require immediate action
     */
    public function handle_critical_event($event_type, $event_data) {
        // Security audit trail logging
        vulnity_log('[Vulnity Security Monitor] Critical event detected: ' . $event_type);
        
        $this->critical_events_count++;
        update_option('vulnity_critical_events_count', $this->critical_events_count);
        
        // If too many critical events, consider the site compromised
        if ($this->critical_events_count >= $this->suspicious_activity_threshold) {
            $this->initiate_security_lockdown('excessive_critical_events', $event_data);
        }
        
        // Specific critical events that require immediate action
        switch ($event_type) {
            case 'malware_detected':
            case 'backdoor_detected':
            case 'webshell_detected':
                $this->initiate_security_lockdown('malware_detected', $event_data);
                break;
                
            case 'mass_admin_creation':
                if (isset($event_data['count']) && $event_data['count'] > 3) {
                    $this->initiate_security_lockdown('mass_admin_creation', $event_data);
                }
                break;
                
            case 'unauthorized_core_modification':
                $this->initiate_security_lockdown('core_files_modified', $event_data);
                break;
        }
    }
    
    /**
     * Handle confirmed security breaches
     */
    public function handle_security_breach($breach_type, $breach_data) {
        // Security audit trail logging
        vulnity_log('[Vulnity Security Monitor] SECURITY BREACH DETECTED: ' . $breach_type);
        
        // Send critical alert to SIEM
        $this->send_breach_alert($breach_type, $breach_data);
        
        // Determine if auto-unpair is needed based on breach severity
        $auto_unpair_breaches = array(
            'ransomware_detected',
            'cryptominer_detected',
            'command_execution',
            'privilege_escalation',
            'mass_data_exfiltration'
        );
        
        if (in_array($breach_type, $auto_unpair_breaches)) {
            $this->initiate_emergency_unpair($breach_type, $breach_data);
        }
    }
    
    /**
     * Initiate security lockdown procedures
     */
    private function initiate_security_lockdown($reason, $data) {
        // Security audit trail logging
        vulnity_log('[Vulnity Security Monitor] Initiating security lockdown - Reason: ' . $reason);
        
        // Notify SIEM of lockdown
        $lockdown_data = array(
            'action' => 'security_lockdown',
            'reason' => $reason,
            'timestamp' => current_time('c'),
            'data' => $data,
            'site_url' => get_site_url()
        );
        
        $siem = Vulnity_SIEM_Connector::get_instance();
        $siem->send_alert(array(
            'type' => 'security_lockdown',
            'severity' => 'critical',
            'title' => 'Security Lockdown Initiated',
            'message' => 'Security lockdown initiated due to: ' . $reason,
            'details' => $lockdown_data,
            'timestamp' => current_time('mysql')
        ));
        
        // Set site to lockdown mode
        update_option('vulnity_lockdown_mode', array(
            'active' => true,
            'reason' => $reason,
            'initiated_at' => current_time('mysql'),
            'data' => $data
        ));
    }
    
    /**
     * Emergency response in case of severe compromise
     */
    private function initiate_emergency_unpair($breach_type, $breach_data) {
        // Security audit trail logging
        vulnity_log('[Vulnity Security Monitor] EMERGENCY RESPONSE REQUIRED - Site compromised: ' . $breach_type);

        // Send final breach report so the SIEM is aware of the situation
        $this->send_final_breach_report($breach_type, $breach_data);

        // Determine recommended action without disconnecting automatically
        $unpair_reason = $this->determine_unpair_reason($breach_type, $breach_data);

        update_option('vulnity_compromised', array(
            'status' => true,
            'breach_type' => $breach_type,
            'timestamp' => current_time('mysql'),
            'requires_manual_action' => true,
            'recommended_unpair_reason' => $unpair_reason
        ));

        do_action('vulnity_emergency_manual_action_required', array(
            'breach_type' => $breach_type,
            'breach_data' => $breach_data,
            'recommended_unpair_reason' => $unpair_reason
        ));

        $config = get_option('vulnity_config');
        if (is_array($config)) {
            $config['status'] = 'active';
            update_option('vulnity_config', $config);
        }

        // Security audit trail logging
        vulnity_log('[Vulnity Security Monitor] Automatic unpair prevented - manual intervention required');
    }
    
    /**
     * Send breach alert to SIEM
     */
    private function send_breach_alert($breach_type, $breach_data) {
        $siem = Vulnity_SIEM_Connector::get_instance();

        $alert_data = array(
            'type' => 'security_breach',
            'severity' => 'critical',
            'title' => 'Security Breach Detected: ' . ucwords(str_replace('_', ' ', $breach_type)),
            'message' => 'A security breach has been detected on the WordPress site',
            'details' => array(
                'breach_type' => $breach_type,
                'breach_data' => $breach_data,
                'site_url' => get_site_url(),
                'wordpress_version' => get_bloginfo('version'),
                'php_version' => PHP_VERSION,
                'timestamp' => current_time('c'),
                'active_plugins' => get_option('active_plugins', array()),
                'current_theme' => $this->get_safe_theme_name()
            ),
            'timestamp' => current_time('mysql')
        );

        $alert = $this->store_local_breach_alert($alert_data);
        $result = $siem->send_alert($alert);

        if ($result['success']) {
            $this->mark_breach_alert_sent($alert['id'], $result);
            $this->remove_from_retry_queue($alert['id']);
        } else {
            $this->update_breach_retry_info($alert['id'], 1, $result);
        }

        return $result;
    }

    /**
     * Store critical breach alerts locally so they appear in the alert panel and retry queue.
     */
    private function store_local_breach_alert($alert_data) {
        $alert_id = 'alert_' . time() . '_' . wp_generate_password(8, false);

        $alert = array_merge(array(
            'id' => $alert_id,
            'type' => 'security_breach',
            'severity' => 'critical',
            'title' => 'Security Breach Detected',
            'message' => 'A security breach has been detected on the WordPress site',
            'details' => array(),
            'timestamp' => current_time('mysql'),
            'timestamp_utc' => current_time('c', true),
            'status' => 'new',
            'sent_to_siem' => false,
            'retry_count' => 0,
            'last_retry' => null,
            'wordpress_version' => get_bloginfo('version'),
            'plugin_version' => VULNITY_VERSION,
            'site_url' => get_site_url(),
            'site_name' => get_bloginfo('name')
        ), $alert_data);

        $alerts = get_option('vulnity_alerts', array());
        array_unshift($alerts, $alert);
        $alerts = array_slice($alerts, 0, 100);
        update_option('vulnity_alerts', $alerts);

        $unread = get_option('vulnity_alerts_unread', 0);
        update_option('vulnity_alerts_unread', $unread + 1);

        if (in_array($alert['severity'], array('critical', 'high'), true)) {
            $this->add_to_retry_queue($alert);
        }

        do_action('vulnity_alert_created', $alert['id'], $alert);

        return $alert;
    }

    /**
     * Update stored alert after a successful send.
     */
    private function mark_breach_alert_sent($alert_id, $result) {
        $alerts = get_option('vulnity_alerts', array());
        foreach ($alerts as &$stored_alert) {
            if ($stored_alert['id'] === $alert_id) {
                $stored_alert['sent_to_siem'] = true;
                $stored_alert['siem_response'] = isset($result['data']) ? $result['data'] : array();
                $stored_alert['sent_at'] = current_time('mysql');
                break;
            }
        }

        update_option('vulnity_alerts', $alerts);
    }

    /**
     * Track failed attempts so retries can be processed.
     */
    private function update_breach_retry_info($alert_id, $retry_count, $result) {
        $alerts = get_option('vulnity_alerts', array());

        foreach ($alerts as &$stored_alert) {
            if ($stored_alert['id'] === $alert_id) {
                $stored_alert['retry_count'] = $retry_count;
                $stored_alert['last_retry'] = current_time('mysql');
                $stored_alert['last_error'] = isset($result['error']) ? $result['error'] : '';
                break;
            }
        }

        update_option('vulnity_alerts', $alerts);
    }

    /**
     * Add critical breach alerts to the existing retry queue structure.
     */
    private function add_to_retry_queue($alert) {
        $queue = get_option('vulnity_retry_queue', array());

        $exists = false;
        foreach ($queue as $queued_alert) {
            if ($queued_alert['id'] === $alert['id']) {
                $exists = true;
                break;
            }
        }

        if ($exists) {
            return;
        }

        $queue[] = array(
            'id' => $alert['id'],
            'added_at' => current_time('mysql'),
            'priority' => $this->get_severity_priority($alert['severity'])
        );

        usort($queue, function($a, $b) {
            return $a['priority'] - $b['priority'];
        });

        $queue = array_slice($queue, 0, 50);

        update_option('vulnity_retry_queue', $queue);
    }

    private function remove_from_retry_queue($alert_id) {
        $queue = get_option('vulnity_retry_queue', array());
        $queue = array_filter($queue, function($item) use ($alert_id) {
            return $item['id'] !== $alert_id;
        });
        update_option('vulnity_retry_queue', array_values($queue));
    }

    private function get_severity_priority($severity) {
        $priorities = array(
            'critical' => 1,
            'high' => 2,
            'medium' => 3,
            'low' => 4,
            'info' => 5
        );

        return isset($priorities[$severity]) ? $priorities[$severity] : 5;
    }
    
    /**
     * Send final breach report during emergency response
     */
    private function send_final_breach_report($breach_type, $breach_data) {
        $siem = Vulnity_SIEM_Connector::get_instance();
        
        // Collect all available forensic data
        $forensic_data = array(
            'breach_type' => $breach_type,
            'breach_data' => $breach_data,
            'recent_alerts' => get_option('vulnity_alerts', array()),
            'critical_events_count' => get_option('vulnity_critical_events_count', 0),
            'last_known_good_state' => get_option('vulnity_previous_state', array()),
            'active_sessions' => $this->get_active_sessions(),
            'recent_file_changes' => $this->get_recent_file_changes(),
            'timestamp' => current_time('c')
        );
        
        $report = array(
            'type' => 'final_breach_report',
            'severity' => 'critical',
            'title' => 'Final Security Breach Report - Emergency Response',
            'message' => 'Emergency response triggered due to confirmed compromise: ' . $breach_type,
            'details' => $forensic_data,
            'timestamp' => current_time('mysql')
        );
        
        return $siem->send_alert($report);
    }
    
    /**
     * Determine appropriate unpair reason based on context
     */
    public function determine_unpair_reason($context, $data = array()) {
        $reason_map = array(
            'ransomware_detected' => 'ransomware',
            'cryptominer_detected' => 'cryptominer',
            'malware_detected' => 'malware',
            'backdoor_detected' => 'backdoor',
            'webshell_detected' => 'webshell',
            'command_execution' => 'remote_execution',
            'privilege_escalation' => 'privilege_escalation',
            'mass_data_exfiltration' => 'data_breach',
            'mass_admin_creation' => 'unauthorized_access',
            'core_files_modified' => 'integrity_breach',
            'excessive_critical_events' => 'compromised',
            'manual' => 'manual',
            'deactivation' => 'deactivation'
        );
        
        return isset($reason_map[$context]) ? $reason_map[$context] : 'security_breach';
    }
    
    /**
     * Get active user sessions for forensic data
     */
    private function get_active_sessions() {
        global $wpdb;
        
        $sessions = array();
        
        // Get logged in users
        $logged_in_users = get_transient('logged_in_users');
        if ($logged_in_users) {
            $sessions['logged_in_users'] = $logged_in_users;
        }

        // Get recent authentication cookies
        $sessions['auth_cookies_count'] = $this->count_auth_cookies();

        return $sessions;
    }

    /**
     * Count authentication cookies without iterating over the full cookie stack.
     */
    private function count_auth_cookies() {
        $cookie_names = array();

        if (defined('LOGGED_IN_COOKIE')) {
            $cookie_names[] = LOGGED_IN_COOKIE;
        }

        if (defined('SECURE_AUTH_COOKIE')) {
            $cookie_names[] = SECURE_AUTH_COOKIE;
        }

        if (defined('AUTH_COOKIE')) {
            $cookie_names[] = AUTH_COOKIE;
        }

        if (defined('COOKIEHASH')) {
            $cookie_names[] = 'wordpress_logged_in_' . COOKIEHASH;
            $cookie_names[] = 'wordpress_sec_' . COOKIEHASH;
        }

        $cookie_names = array_filter(array_unique($cookie_names));
        $count = 0;

        foreach ($cookie_names as $name) {
            $cookie_value = filter_input(INPUT_COOKIE, $name, FILTER_SANITIZE_FULL_SPECIAL_CHARS);

            if (null === $cookie_value && isset($_COOKIE[$name])) {
                $cookie_value = sanitize_text_field(wp_unslash($_COOKIE[$name]));
            }

            if (!empty($cookie_value)) {
                $count++;
            }
        }

        return $count;
    }
    
    /**
     * Get recent file changes for forensic data
     */
    private function get_recent_file_changes() {
        $changes = array();

        if (!function_exists('get_home_path')) {
            $file_include = path_join(ABSPATH, 'wp-admin/includes/file.php');
            if (file_exists($file_include)) {
                require_once $file_include;
            }
        }

        if (!function_exists('get_home_path')) {
            return $changes;
        }

        // Check for recently modified core files
        $home_path = trailingslashit(get_home_path());

        $core_files = array(
            $home_path . 'wp-config.php',
            $home_path . 'index.php',
            $home_path . '.htaccess'
        );
        
        foreach ($core_files as $file) {
            if (file_exists($file)) {
                $changes[basename($file)] = array(
                    'modified' => wp_date('Y-m-d H:i:s', filemtime($file)),
                    'size' => filesize($file)
                );
            }
        }
        
        return $changes;
    }
    
    /**
     * Check if site is in lockdown mode
     */
    public function is_lockdown_mode() {
        $lockdown = get_option('vulnity_lockdown_mode', array());
        return isset($lockdown['active']) && $lockdown['active'] === true;
    }
    
    /**
     * Get lockdown status
     */
    public function get_lockdown_status() {
        return get_option('vulnity_lockdown_mode', array(
            'active' => false,
            'reason' => null,
            'initiated_at' => null
        ));
    }
    
    /**
     * Clear lockdown mode (admin action)
     */
    public function clear_lockdown() {
        delete_option('vulnity_lockdown_mode');
        delete_option('vulnity_critical_events_count');
        // Security audit trail logging
        vulnity_log('[Vulnity Security Monitor] Lockdown mode cleared');
    }

    /**
     * Get theme name safely without causing fatal error
     *
     * @return string Theme name or 'Unknown' if theme cannot be retrieved
     */
    private function get_safe_theme_name() {
        if (!function_exists('wp_get_theme')) {
            return 'Unknown';
        }

        $theme = wp_get_theme();

        if (!$theme || !is_object($theme)) {
            return 'Unknown';
        }

        $name = $theme->get('Name');
        return is_string($name) && $name !== '' ? $name : 'Unknown';
    }
}
