<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class Vulnity_Alert_Manager {
    
    private static $instance = null;
    private $alert_classes = array();
    
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        $this->load_alert_classes();
        $this->initialize_alerts();
    }
    
    private function load_alert_classes() {
        $alerts_dir = vulnity_plugin_path('includes/alerts/');
        
        if (file_exists($alerts_dir . 'class-alert-base.php')) {
            require_once $alerts_dir . 'class-alert-base.php';
        }
        
        $alert_files = array(
            'class-brute-force-alert.php',
            'class-php-upload-alert.php',
            'class-admin-created-alert.php',
            'class-permission-change-alert.php',
            'class-file-editor-alert.php',
            'class-plugin-change-alert.php',
            'class-theme-change-alert.php',
            'class-core-update-alert.php',
            'class-suspicious-query-alert.php',
            'class-scanner-detection-alert.php'
        );
        
        foreach ($alert_files as $file) {
            if (file_exists($alerts_dir . $file)) {
                require_once $alerts_dir . $file;
            }
        }
    }
    
    private function initialize_alerts() {
        $alert_classes = array(
            'Vulnity_Brute_Force_Alert',
            'Vulnity_PHP_Upload_Alert',
            'Vulnity_Admin_Created_Alert',
            'Vulnity_Permission_Change_Alert',
            'Vulnity_File_Editor_Alert',
            'Vulnity_Plugin_Change_Alert',
            'Vulnity_Theme_Change_Alert',
            'Vulnity_Core_Update_Alert',
            'Vulnity_Suspicious_Query_Alert',
            'Vulnity_Scanner_Detection_Alert'
        );
        
        foreach ($alert_classes as $class) {
            if (class_exists($class)) {
                $this->alert_classes[] = new $class();
            }
        }
    }
    
    public function get_alerts($limit = 50, $filter_type = 'all') {
        $alerts = get_option('vulnity_alerts', array());
        
        if ($filter_type !== 'all') {
            $alerts = array_filter($alerts, function($alert) use ($filter_type) {
                return $alert['type'] === $filter_type;
            });
        }
        
        return array_slice($alerts, 0, $limit);
    }
    
    public function mark_alert_as_viewed($alert_id) {
        $alerts = get_option('vulnity_alerts', array());
        
        foreach ($alerts as &$alert) {
            if ($alert['id'] === $alert_id && $alert['status'] === 'new') {
                $alert['status'] = 'viewed';
                
                $unread = get_option('vulnity_alerts_unread', 0);
                if ($unread > 0) {
                    update_option('vulnity_alerts_unread', $unread - 1);
                }
                break;
            }
        }
        
        update_option('vulnity_alerts', $alerts);
    }
    
    public function mark_all_as_viewed() {
        $alerts = get_option('vulnity_alerts', array());
        
        foreach ($alerts as &$alert) {
            if ($alert['status'] === 'new') {
                $alert['status'] = 'viewed';
            }
        }
        
        update_option('vulnity_alerts', $alerts);
        update_option('vulnity_alerts_unread', 0);
    }
    
    public function clear_all_alerts() {
        delete_option('vulnity_alerts');
        update_option('vulnity_alerts_unread', 0);
    }
    
    public function get_statistics() {
        $alerts = get_option('vulnity_alerts', array());
        $cutoff = current_time('timestamp') - DAY_IN_SECONDS;

        $alerts_24h = array_filter($alerts, function($alert) use ($cutoff) {
            if (empty($alert['timestamp'])) {
                return false;
            }

            $alert_timestamp = strtotime($alert['timestamp']);

            if ($alert_timestamp === false) {
                return false;
            }

            return $alert_timestamp > $cutoff;
        });
        
        $type_counts = array();
        $severity_counts = array(
            'critical' => 0,
            'high' => 0,
            'medium' => 0,
            'low' => 0
        );
        
        foreach ($alerts_24h as $alert) {
            if (!isset($type_counts[$alert['type']])) {
                $type_counts[$alert['type']] = 0;
            }
            $type_counts[$alert['type']]++;
            
            if (isset($severity_counts[$alert['severity']])) {
                $severity_counts[$alert['severity']]++;
            }
        }
        
        return array(
            'total_alerts' => count($alerts),
            'alerts_24h' => count($alerts_24h),
            'unread_alerts' => get_option('vulnity_alerts_unread', 0),
            'type_counts' => $type_counts,
            'severity_counts' => $severity_counts
        );
    }
}
