<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class Vulnity_Plugin_Change_Alert extends Vulnity_Alert_Base {
    
    public function __construct() {
        $this->alert_type = 'plugin_change';
        parent::__construct();
    }
    
    protected function register_hooks() {
        add_action('activated_plugin', array($this, 'on_plugin_activated'), 10, 2);
        add_action('deactivated_plugin', array($this, 'on_plugin_deactivated'), 10, 2);
        add_action('upgrader_process_complete', array($this, 'on_plugin_updated'), 10, 2);
        add_action('deleted_plugin', array($this, 'on_plugin_deleted'), 10, 2);
        
        // Hook adicional para capturar desactivaciones antes de que se procesen
        add_action('deactivate_plugin', array($this, 'on_plugin_about_to_deactivate'), 5, 2);
    }
    
    public function on_plugin_activated($plugin, $network_wide) {
        $this->evaluate(array(
            'action' => 'activated',
            'plugin' => $plugin,
            'network_wide' => $network_wide
        ));
    }
    
    public function on_plugin_deactivated($plugin, $network_wide) {
        $this->evaluate(array(
            'action' => 'deactivated',
            'plugin' => $plugin,
            'network_wide' => $network_wide
        ));
    }
    
    // Nuevo método para capturar desactivaciones antes de que ocurran
    public function on_plugin_about_to_deactivate($plugin, $network_deactivating) {
        // Solo procesar si no es el plugin Vulnity mismo
        if (strpos($plugin, 'vulnity') === false) {
            // Guardar información del plugin antes de que se desactive
            $plugin_data = get_plugin_data(WP_PLUGIN_DIR . '/' . $plugin);
            
            // Crear la alerta inmediatamente
            $this->evaluate(array(
                'action' => 'deactivated',
                'plugin' => $plugin,
                'network_wide' => $network_deactivating,
                'plugin_data' => $plugin_data // Pasar los datos del plugin directamente
            ));
        }
    }
    
    public function on_plugin_updated($upgrader, $hook_extra) {
        if ($hook_extra['type'] !== 'plugin') {
            return;
        }
        
        if ($hook_extra['action'] === 'install') {
            $this->evaluate(array(
                'action' => 'installed',
                'plugins' => isset($hook_extra['plugins']) ? $hook_extra['plugins'] : array()
            ));
        } elseif ($hook_extra['action'] === 'update') {
            $this->evaluate(array(
                'action' => 'updated',
                'plugins' => isset($hook_extra['plugins']) ? $hook_extra['plugins'] : array()
            ));
        }
    }
    
    public function on_plugin_deleted($plugin_file, $deleted) {
        if ($deleted) {
            $this->evaluate(array(
                'action' => 'deleted',
                'plugin' => $plugin_file
            ));
        }
    }
    
    protected function evaluate($data) {
        $current_user_info = $this->get_current_user_info();
        
        $severity = 'medium';
        $title = '';
        $message = '';
        $plugin_info = array();
        
        // Verificar si ya tenemos los datos del plugin (desde on_plugin_about_to_deactivate)
        if (isset($data['plugin_data']) && !empty($data['plugin_data'])) {
            $plugin_data = $data['plugin_data'];
        } elseif (isset($data['plugin']) && !is_array($data['plugin']) && file_exists(WP_PLUGIN_DIR . '/' . $data['plugin'])) {
            $plugin_data = get_plugin_data(WP_PLUGIN_DIR . '/' . $data['plugin']);
        } else {
            $plugin_data = array();
        }
        
        if (!empty($plugin_data)) {
            $plugin_info = array(
                'name' => $plugin_data['Name'],
                'version' => $plugin_data['Version'],
                'author' => $plugin_data['Author'],
                'file' => isset($data['plugin']) ? $data['plugin'] : ''
            );
        }
        
        switch ($data['action']) {
            case 'activated':
                $severity = 'medium';
                $title = 'Plugin Activated';
                $message = sprintf(
                    'Plugin "%s" was activated by user "%s" from IP %s',
                    $plugin_info['name'],
                    $current_user_info['user_login'],
                    $current_user_info['user_ip']
                );
                break;
                
            case 'deactivated':
                $severity = 'low';
                $title = 'Plugin Deactivated';
                $message = sprintf(
                    'Plugin "%s" was deactivated by user "%s" from IP %s',
                    !empty($plugin_info['name']) ? $plugin_info['name'] : 'Unknown',
                    $current_user_info['user_login'],
                    $current_user_info['user_ip']
                );
                break;
                
            case 'installed':
                $severity = 'high';
                $title = 'New Plugin Installed';
                $message = sprintf(
                    'New plugin was installed by user "%s" from IP %s',
                    $current_user_info['user_login'],
                    $current_user_info['user_ip']
                );
                break;
                
            case 'updated':
                $severity = 'low';
                $title = 'Plugin Updated';
                $message = sprintf(
                    'Plugin was updated by user "%s" from IP %s',
                    $current_user_info['user_login'],
                    $current_user_info['user_ip']
                );
                break;
                
            case 'deleted':
                $severity = 'high';
                $title = 'Plugin Deleted';
                $message = sprintf(
                    'Plugin "%s" was deleted by user "%s" from IP %s',
                    !empty($plugin_info['name']) ? $plugin_info['name'] : 'Unknown',
                    $current_user_info['user_login'],
                    $current_user_info['user_ip']
                );
                break;
        }
        
        if (empty($title)) {
            return;
        }
        
        $this->create_alert(array(
            'severity' => $severity,
            'title' => $title,
            'message' => $message,
            'details' => array(
                'action' => $data['action'],
                'plugin_info' => $plugin_info,
                'network_wide' => isset($data['network_wide']) ? $data['network_wide'] : false,
                'user_id' => $current_user_info['user_id'],
                'user_login' => $current_user_info['user_login'],
                'user_email' => $current_user_info['user_email'],
                'user_ip' => $current_user_info['user_ip'],
                'timestamp' => current_time('mysql')
            )
        ));
    }
}
