<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class Vulnity_PHP_Upload_Alert extends Vulnity_Alert_Base {
    
    private $processed_files = array(); // Cache para evitar duplicados
    
    public function __construct() {
        $this->alert_type = 'php_file_uploaded';
        parent::__construct();
    }
    
    protected function register_hooks() {
        add_action('add_attachment', array($this, 'check_uploaded_file'), 10, 1);
        add_filter('wp_handle_upload', array($this, 'check_file_upload'), 10, 2);
        add_filter('wp_handle_sideload', array($this, 'check_file_upload'), 10, 2);
    }
    
    public function check_uploaded_file($attachment_id) {
        $file = get_attached_file($attachment_id);
        
        if (!$file) {
            return;
        }
        
        $this->evaluate(array(
            'file_path' => $file,
            'attachment_id' => $attachment_id,
            'source' => 'media_library'
        ));
    }
    
    public function check_file_upload($file, $type) {
        if (!empty($file['file'])) {
            $this->evaluate(array(
                'file_path' => $file['file'],
                'source' => 'direct_upload',
                'type' => $file['type'] ?? 'unknown'
            ));
        }
        return $file;
    }
    
    protected function evaluate($data) {
        $file_path = $data['file_path'];
        $filename = basename($file_path);
        $extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        
        $suspicious_extensions = array(
            'php','php3','php4','php5','php6','php7','php8','phtml','phar','pht','phps',
            'asp','aspx','jsp','jspx','do','action','cgi','pl',
            'sh','bash','py','rb','ps1',
            'exe','dll','com','bat','msi','scr','vbs','jse','wsf',
            'js','inc','class','shtml','shtm',
            'bak','old','backup','tmp','swp','~','test',
            'zip','tar','gz','tgz','tar.gz','rar','7z','jar','war',
            'bz2','tbz','tbz2','xz','lzma','lz','lz4','zst','zstd',
            'iso','img','dmg','cab','apk','aar','ar','cpio','ace','lha','rz','pak',
            'rpm','deb','pem','key','crt','csr','bin'
        );
        
        if (!in_array($extension, $suspicious_extensions)) {
            return;
        }
        
        $upload_dir = wp_upload_dir();
        $is_in_uploads = strpos($file_path, $upload_dir['basedir']) === 0;
        
        if (!$is_in_uploads) {
            return;
        }
        
        // =====================================================
        // NUEVA LÓGICA DE DEDUPLICACIÓN
        // =====================================================
        
        // Crear identificador único basado en la ruta del archivo
        $file_hash = md5($file_path);
        
        // Si ya procesamos este archivo en los últimos 30 segundos, omitir
        $current_time = time();
        if (isset($this->processed_files[$file_hash])) {
            $time_diff = $current_time - $this->processed_files[$file_hash];
            if ($time_diff < 30) { // 30 segundos de ventana de deduplicación
                vulnity_log('[Vulnity] Skipping duplicate PHP file detection: ' . $filename . ' (processed ' . $time_diff . 's ago)');
                return;
            }
        }
        
        // Marcar archivo como procesado
        $this->processed_files[$file_hash] = $current_time;
        
        // Limpiar caché antiguo (mantener solo archivos de los últimos 5 minutos)
        $this->clean_processed_files_cache();
        
        // =====================================================
        // CONTINUAR CON LÓGICA NORMAL
        // =====================================================
        
        $file_content = @file_get_contents($file_path);
        $is_php_code = false;
        
        if ($file_content !== false) {
            $php_indicators = array('<?php', '<?=', '<script language="php"', '<?', '<%');
            foreach ($php_indicators as $indicator) {
                if (stripos($file_content, $indicator) !== false) {
                    $is_php_code = true;
                    break;
                }
            }
        }
        
        $user_info = $this->get_current_user_info();
        
        $alert_data = array(
            'severity' => 'critical',
            'title' => 'PHP File Uploaded to Uploads Directory',
            'message' => sprintf(
                'A PHP file "%s" was uploaded to the uploads directory by user "%s" (ID: %d)',
                $filename,
                $user_info['user_login'],
                $user_info['user_id']
            ),
            'details' => array(
                'filename' => $filename,
                'file_path' => $file_path,
                'file_size' => filesize($file_path),
                'upload_time' => current_time('mysql'),
                'user_id' => $user_info['user_id'],
                'user_login' => $user_info['user_login'],
                'user_email' => $user_info['user_email'],
                'user_ip' => $user_info['user_ip'],
                'contains_php_code' => $is_php_code,
                'attachment_id' => isset($data['attachment_id']) ? $data['attachment_id'] : null,
                'source' => $data['source'],
                'deduplicated' => true // Marcar que se aplicó deduplicación
            )
        );
        
        $this->create_alert($alert_data);
        
        vulnity_log('[Vulnity] PHP file upload alert created (deduplicated): ' . $filename);
    }
    
    /**
     * Limpiar caché de archivos procesados
     */
    private function clean_processed_files_cache() {
        $current_time = time();
        $cache_expiry = 300; // 5 minutos
        
        foreach ($this->processed_files as $hash => $timestamp) {
            if (($current_time - $timestamp) > $cache_expiry) {
                unset($this->processed_files[$hash]);
            }
        }
    }
}
