<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class Vulnity_Core_Update_Alert extends Vulnity_Alert_Base {
    
    public function __construct() {
        $this->alert_type = 'core_updated';
        parent::__construct();
    }
    
    protected function register_hooks() {
        add_action('_core_updated_successfully', array($this, 'on_core_updated'));
        add_action('automatic_updates_complete', array($this, 'on_auto_update_complete'));
    }
    
    public function on_core_updated($wp_version) {
        $this->evaluate(array(
            'action' => 'manual_update',
            'new_version' => $wp_version,
            'old_version' => get_bloginfo('version')
        ));
    }
    
    public function on_auto_update_complete($update_results) {
        if (isset($update_results['core'])) {
            $this->evaluate(array(
                'action' => 'auto_update',
                'update_results' => $update_results['core']
            ));
        }
    }
    
    /**
     * Compare version numbers to detect downgrades
     */
    private function is_downgrade($old_version, $new_version) {
        // Use WordPress version_compare function
        // Returns -1 if old < new (upgrade)
        // Returns 0 if old == new (same)
        // Returns 1 if old > new (downgrade)
        return version_compare($old_version, $new_version) > 0;
    }
    
    protected function evaluate($data) {
        $current_user_info = $this->get_current_user_info();
        
        // Default to info severity for normal updates
        $severity = 'info';
        $title = '';
        $message = '';
        
        // Check if this is a downgrade (HIGH severity)
        if (isset($data['new_version']) && isset($data['old_version'])) {
            if ($this->is_downgrade($data['old_version'], $data['new_version'])) {
                $severity = 'high';
                $title = 'WordPress Core DOWNGRADED - Suspicious Activity';
                $message = sprintf(
                    'WARNING: WordPress was DOWNGRADED from version %s to %s by user "%s" from IP %s',
                    $data['old_version'],
                    $data['new_version'],
                    $current_user_info['user_login'],
                    $current_user_info['user_ip']
                );
            }
        }
        
        if ($severity === 'info') {
            if ($data['action'] === 'manual_update') {
                $severity = 'info';  
                $title = 'WordPress Core Updated';
                $message = sprintf(
                    'WordPress was manually updated to version %s by user "%s" from IP %s',
                    $data['new_version'],
                    $current_user_info['user_login'],
                    $current_user_info['user_ip']
                );
            } elseif ($data['action'] === 'auto_update') {
                $severity = 'info';  
                $title = 'WordPress Auto-Updated';
                $message = 'WordPress was automatically updated';
                
                if (isset($data['update_results']) && is_array($data['update_results'])) {
                    if (isset($data['update_results']['item']) && isset($data['update_results']['item']->version)) {
                        $message .= sprintf(' to version %s', $data['update_results']['item']->version);
                    }
                }
            }
        }
        
        if (empty($title)) {
            return;
        }
        
        $details = array(
            'action' => $data['action'],
            'update_data' => $data,
            'wordpress_version' => get_bloginfo('version'),
            'php_version' => PHP_VERSION,
            'user_id' => $current_user_info['user_id'],
            'user_login' => $current_user_info['user_login'],
            'user_ip' => $current_user_info['user_ip'],
            'timestamp' => current_time('mysql')
        );
        
        if ($severity === 'high' && isset($data['old_version']) && isset($data['new_version'])) {
            $details['is_downgrade'] = true;
            $details['version_change'] = sprintf('%s → %s', $data['old_version'], $data['new_version']);
            $details['risk_assessment'] = 'High risk - WordPress downgrades are unusual and potentially dangerous';
        }
        
        $this->create_alert(array(
            'severity' => $severity,
            'title' => $title,
            'message' => $message,
            'details' => $details
        ));
    }
}
