<?php
// This file can be executed via auto_prepend_file before WordPress loads.
if (!defined('ABSPATH')) {
    $vulnity_firewall_script_filename = filter_input(INPUT_SERVER, 'SCRIPT_FILENAME', FILTER_UNSAFE_RAW);
    if (basename((string) $vulnity_firewall_script_filename) === 'bootstrap.php') {
        exit;
    }

    $vulnity_doc_root_input = filter_input(INPUT_SERVER, 'DOCUMENT_ROOT', FILTER_UNSAFE_RAW);
    $vulnity_doc_root = is_string($vulnity_doc_root_input) ? $vulnity_doc_root_input : '';
    if (function_exists('wp_unslash')) {
        $vulnity_doc_root = wp_unslash($vulnity_doc_root);
    }
    $vulnity_doc_root = preg_replace('/[^A-Za-z0-9:\\/._\\-\\\\]/', '', (string) $vulnity_doc_root);
    $vulnity_doc_root = rtrim($vulnity_doc_root, '/\\');
    if ($vulnity_doc_root !== '') {
        $vulnity_doc_root .= '/';
        if (file_exists($vulnity_doc_root . 'wp-load.php')) {
            define('ABSPATH', $vulnity_doc_root);
        } elseif (file_exists($vulnity_doc_root . 'wp/wp-load.php')) {
            define('ABSPATH', $vulnity_doc_root . 'wp/');
        }
    }

    if (!defined('ABSPATH') && is_string($vulnity_firewall_script_filename) && $vulnity_firewall_script_filename !== '') {
        $vulnity_script_dir = dirname($vulnity_firewall_script_filename);
        for ($vulnity_i = 0; $vulnity_i < 6; $vulnity_i++) {
            if (file_exists($vulnity_script_dir . '/wp-load.php')) {
                define('ABSPATH', rtrim($vulnity_script_dir, '/\\') . '/');
                break;
            }
            $vulnity_parent = dirname($vulnity_script_dir);
            if ($vulnity_parent === $vulnity_script_dir) {
                break;
            }
            $vulnity_script_dir = $vulnity_parent;
        }
    }

    if (!defined('ABSPATH')) {
        $vulnity_candidate_abspath = dirname(__DIR__, 4);
        $vulnity_candidate_abspath = rtrim((string) $vulnity_candidate_abspath, '/\\') . '/';
        if (is_dir($vulnity_candidate_abspath . 'wp-includes')) {
            define('ABSPATH', $vulnity_candidate_abspath);
        }
    }
}

if (!defined('ABSPATH')) {
    // Fail-safe: do nothing if WordPress context cannot be resolved.
    exit;
}

if (!function_exists('vulnity_firewall_sanitize_text')) {
    function vulnity_firewall_sanitize_text($value) {
        if (function_exists('wp_unslash')) {
            $value = wp_unslash($value);
        }

        if (function_exists('sanitize_text_field')) {
            return sanitize_text_field($value);
        }

        $value = (string) $value;

        return preg_replace('/[^A-Za-z0-9:\.\-_, ]/', '', $value);
    }
}

if (!function_exists('vulnity_firewall_trailingslashit')) {
    function vulnity_firewall_trailingslashit($path) {
        return rtrim((string) $path, "/\\") . '/';
    }
}

if (!function_exists('vulnity_firewall_esc_html')) {
    function vulnity_firewall_esc_html($value) {
        if (function_exists('esc_html')) {
            return esc_html($value);
        }

        return htmlspecialchars((string) $value, ENT_QUOTES, 'UTF-8');
    }
}

if (!function_exists('vulnity_firewall_esc_url')) {
    function vulnity_firewall_esc_url($value) {
        if (function_exists('esc_url')) {
            return esc_url($value);
        }

        $sanitized = filter_var((string) $value, FILTER_SANITIZE_URL);
        return is_string($sanitized) ? $sanitized : '';
    }
}

if (!function_exists('vulnity_firewall_wp_kses')) {
    function vulnity_firewall_wp_kses($value, $allowed = array()) {
        if (function_exists('wp_kses')) {
            return wp_kses($value, $allowed);
        }

        return $value;
    }
}

if (!function_exists('vulnity_firewall_plugins_url')) {
    function vulnity_firewall_plugins_url($relative_path, $file) {
        if (function_exists('plugins_url')) {
            return plugins_url($relative_path, $file);
        }

        $relative_path = ltrim((string) $relative_path, '/\\');
        $file_dir = dirname($file);
        $target = realpath($file_dir . DIRECTORY_SEPARATOR . $relative_path);

        if ($target === false) {
            return '';
        }

        $doc_root_input = filter_input(INPUT_SERVER, 'DOCUMENT_ROOT', FILTER_UNSAFE_RAW);
        $doc_root_raw = is_string($doc_root_input) ? $doc_root_input : '';
        if (function_exists('wp_unslash')) {
            $doc_root_raw = wp_unslash($doc_root_raw);
        }
        $doc_root = rtrim(vulnity_firewall_sanitize_text($doc_root_raw), '/\\');
        if ($doc_root === '') {
            return '';
        }

        $doc_root_norm = str_replace('\\', '/', $doc_root);
        $target_norm = str_replace('\\', '/', $target);

        if (strpos($target_norm, $doc_root_norm) !== 0) {
            return '';
        }

        $path = '/' . ltrim(substr($target_norm, strlen($doc_root_norm)), '/');

        $host_input = filter_input(INPUT_SERVER, 'HTTP_HOST', FILTER_UNSAFE_RAW);
        $host_raw = is_string($host_input) ? $host_input : '';
        if (function_exists('wp_unslash')) {
            $host_raw = wp_unslash($host_raw);
        }
        $host = vulnity_firewall_sanitize_text($host_raw);
        if ($host === '') {
            return $path;
        }

        $https_value = filter_input(INPUT_SERVER, 'HTTPS', FILTER_UNSAFE_RAW);
        $scheme = (!empty($https_value) && $https_value !== 'off') ? 'https' : 'http';
        return $scheme . '://' . $host . $path;
    }
}

if (!function_exists('vulnity_firewall_get_uploads_dir')) {
    function vulnity_firewall_get_uploads_dir() {
        if (function_exists('wp_upload_dir')) {
            $uploads = wp_upload_dir(null, false);
            if (is_array($uploads) && !empty($uploads['basedir'])) {
                return vulnity_firewall_trailingslashit($uploads['basedir']);
            }
        }

        if (defined('WP_CONTENT_DIR')) {
            return vulnity_firewall_trailingslashit(WP_CONTENT_DIR) . 'uploads/';
        }

        $content_dir = dirname(__DIR__, 3);
        return vulnity_firewall_trailingslashit($content_dir) . 'uploads/';
    }
}

if (!function_exists('vulnity_firewall_get_content_dir')) {
    function vulnity_firewall_get_content_dir() {
        if (defined('WP_CONTENT_DIR')) {
            return vulnity_firewall_trailingslashit(WP_CONTENT_DIR);
        }

        $content_dir = dirname(__DIR__, 3);
        return vulnity_firewall_trailingslashit($content_dir);
    }
}

if (!function_exists('vulnity_validate_style_path')) {
    function vulnity_validate_style_path($style_path) {
        $real_path = realpath($style_path);

        if (false === $real_path) {
            return false;
        }

        $plugin_base = defined('VULNITY_PLUGIN_DIR') ? VULNITY_PLUGIN_DIR : vulnity_firewall_trailingslashit(dirname(__DIR__));

        if (strpos($real_path, vulnity_firewall_trailingslashit($plugin_base)) !== 0) {
            return false;
        }

        return $real_path;
    }
}

if (!function_exists('vulnity_firewall_get_style_markup')) {
    function vulnity_firewall_get_style_markup($style_url, $style_path = '') {
        $handle       = 'vulnity-firewall-style';
        $markup       = '';
        $safe_url     = vulnity_firewall_esc_url($style_url);

        if ($safe_url !== '' && function_exists('wp_register_style') && function_exists('wp_enqueue_style')) {
            $version   = defined('VULNITY_VERSION') ? VULNITY_VERSION : null;

            wp_register_style($handle, $safe_url, array(), $version);
            wp_enqueue_style($handle);

            if (function_exists('wp_print_styles')) {
                ob_start();
                wp_print_styles($handle);
                $markup = ob_get_clean();
            }
        }

        if ($markup !== '' || $style_path === '') {
            return $markup;
        }

        $validated_path = vulnity_validate_style_path($style_path);
        if ($validated_path === false || !is_readable($validated_path)) {
            return $markup;
        }

        $css = file_get_contents($validated_path);
        if ($css === false) {
            return $markup;
        }

        $css = trim($css);
        if ($css === '') {
            return $markup;
        }

        return '<style>' . htmlspecialchars($css, ENT_QUOTES, 'UTF-8') . '</style>';
    }
}

if (!function_exists('vulnity_firewall_render_block_page')) {
    function vulnity_firewall_render_block_page() {
        if (defined('VULNITY_FIREWALL_BYPASS')) {
            return;
        }

        if (PHP_SAPI === 'cli') {
            return;
        }

        $uploads_dir = vulnity_firewall_get_uploads_dir();
        $content_dir = vulnity_firewall_get_content_dir();

        $candidate_files = array(
            vulnity_firewall_trailingslashit($uploads_dir) . 'vulnity-firewall/blocks.php',
            vulnity_firewall_trailingslashit($content_dir) . 'vulnity-firewall/blocks.php',
        );

        $data_file = '';

        foreach ($candidate_files as $candidate_file) {
            if (file_exists($candidate_file)) {
                $data_file = $candidate_file;
                break;
            }
        }

        if ($data_file === '') {
            return;
        }

        $firewall_data = @include $data_file;
        if (!is_array($firewall_data)) {
            return;
        }

        $remote_addr     = '';
        $remote_addr_raw = filter_input( INPUT_SERVER, 'REMOTE_ADDR', FILTER_SANITIZE_FULL_SPECIAL_CHARS );

        if ( null !== $remote_addr_raw ) {
            if ( function_exists( 'wp_unslash' ) ) {
                $remote_addr_raw = wp_unslash( $remote_addr_raw );
            }

            if ( is_array( $remote_addr_raw ) ) {
                $remote_addr_raw = reset( $remote_addr_raw );
            }

            if ( function_exists( 'sanitize_text_field' ) ) {
                $remote_addr = sanitize_text_field( $remote_addr_raw );
            } else {
                $remote_addr = vulnity_firewall_sanitize_text( $remote_addr_raw );
            }
        }

        $client_ip = filter_var( $remote_addr, FILTER_VALIDATE_IP ) ? $remote_addr : 'unknown';

        $trusted_proxies = array();
        if (isset($firewall_data['trusted_proxies']) && is_array($firewall_data['trusted_proxies'])) {
            $trusted_proxies = $firewall_data['trusted_proxies'];
        }

        if ($client_ip !== 'unknown' && in_array($client_ip, $trusted_proxies, true)) {
            $headers = array(
                'HTTP_CF_CONNECTING_IP',
                'HTTP_X_FORWARDED_FOR',
                'HTTP_X_FORWARDED',
                'HTTP_X_CLUSTER_CLIENT_IP',
                'HTTP_FORWARDED_FOR',
                'HTTP_FORWARDED'
            );

            foreach ( $headers as $header ) {
                $header_value     = '';
                $header_raw_value = filter_input( INPUT_SERVER, $header, FILTER_SANITIZE_FULL_SPECIAL_CHARS );

                if ( null !== $header_raw_value ) {
                    if ( function_exists( 'wp_unslash' ) ) {
                        $header_raw_value = wp_unslash( $header_raw_value );
                    }

                    if ( is_array( $header_raw_value ) ) {
                        $header_raw_value = reset( $header_raw_value );
                    }

                    if ( function_exists( 'sanitize_text_field' ) ) {
                        $header_value = sanitize_text_field( $header_raw_value );
                    } else {
                        $header_value = vulnity_firewall_sanitize_text( $header_raw_value );
                    }
                }

                if ( '' === $header_value ) {
                    continue;
                }

                if (strpos($header_value, ',') !== false) {
                    $parts = explode(',', $header_value);
                    $header_value = trim($parts[0]);
                }

                if (filter_var($header_value, FILTER_VALIDATE_IP)) {
                    $client_ip = $header_value;
                    break;
                }
            }
        }

        if ($client_ip === 'unknown') {
            return;
        }

        $whitelist = isset($firewall_data['whitelist']) && is_array($firewall_data['whitelist']) ? $firewall_data['whitelist'] : array();
        if (in_array($client_ip, $whitelist, true)) {
            return;
        }

        $blocked_ips = isset($firewall_data['blocked_ips']) && is_array($firewall_data['blocked_ips']) ? $firewall_data['blocked_ips'] : array();
        if (!isset($blocked_ips[$client_ip])) {
            return;
        }

        $block = $blocked_ips[$client_ip];
        $expires_at = isset($block['expires_at']) ? (int) $block['expires_at'] : 0;

        if ($expires_at !== 0 && $expires_at <= time()) {
            return;
        }

        $reason = isset($block['reason']) ? $block['reason'] : 'Security block';
        $blocked_at = isset($block['blocked_at']) ? $block['blocked_at'] : '';
        $display_expires = isset($block['display_expires']) ? $block['display_expires'] : null;

        if (!headers_sent()) {
            header('HTTP/1.1 403 Forbidden');
            header('X-Blocked-By: Vulnity Firewall');
            header('Cache-Control: no-cache, no-store, must-revalidate');
        }

        $reason_escaped = vulnity_firewall_esc_html($reason);
        $ip_escaped = vulnity_firewall_esc_html($client_ip);
        $blocked_at_escaped = vulnity_firewall_esc_html($blocked_at);

        $expires_message = '';
        if (!empty($display_expires) || !empty($block['permanent'])) {
            $expires_message = '<div class="d"><div class="l">Status</div><div class="v">Blocked</div></div>';
        }

        $logo_url = vulnity_firewall_plugins_url('../assets/Logo_Vulnity.png', __FILE__);
        $style_url = vulnity_firewall_plugins_url('../assets/block-page.css', __FILE__);
        $style_path = dirname(__DIR__) . '/assets/block-page.css';
        $logo_url_escaped = vulnity_firewall_esc_url($logo_url);
        $style_markup = vulnity_firewall_get_style_markup($style_url, $style_path);
        $expires_message_escaped = vulnity_firewall_wp_kses(
            $expires_message,
            array(
                'div' => array(
                    'class' => true,
                ),
            )
        );

        $reference = substr(md5($client_ip . $blocked_at . $reason), 0, 8);
        $reference_escaped = vulnity_firewall_esc_html($reference);

        $html_template = implode('', array(
            '<html>',
            '<head>',
            '<meta charset="UTF-8">',
            '<meta name="viewport" content="width=device-width,initial-scale=1">',
            '<title>Access Denied</title>',
            '%1$s',
            '</head>',
            '<body>',
            '<div class="c">',
            '    <div class="logo"><img src="%2$s" alt="Vulnity Security Logo"></div>',
            '    <h1>Access Denied</h1>',
            '    <div class="st">Your access has been blocked by Vulnity Security</div>',
            '    <div class="b">',
            '        <div class="d"><div class="l">Your IP Address</div><div class="v">%3$s</div></div>',
            '        <div class="d"><div class="l">Block Reason</div><div class="v">%4$s</div></div>',
            '        <div class="d"><div class="l">Blocked Since</div><div class="v">%5$s</div></div>',
            '        %6$s',
            '    </div>',
            '    <div class="f"><strong>Protected by Vulnity Security</strong><br>Reference: %7$s</div>',
            '</div>',
            '</body>',
            '</html>'
        ));

        $final_output = sprintf(
            $html_template,
            $style_markup,
            $logo_url_escaped,
            $ip_escaped,
            $reason_escaped,
            $blocked_at_escaped,
            $expires_message_escaped,
            $reference_escaped
        );

        $safe_final_output = vulnity_firewall_wp_kses(
            $final_output,
            array(
                'html'   => array(),
                'head'   => array(),
                'meta'   => array(
                    'charset' => true,
                    'name'    => true,
                    'content' => true,
                ),
                'title'  => array(),
                'link'   => array(
                    'rel'  => true,
                    'href' => true,
                ),
                'body'   => array(),
                'div'    => array(
                    'class' => true,
                ),
                'img'    => array(
                    'src' => true,
                    'alt' => true,
                ),
                'h1'     => array(),
                'strong' => array(),
                'br'     => array(),
            )
        );
        echo '<!DOCTYPE html>';
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Sanitized above with vulnity_firewall_wp_kses (wp_kses wrapper with bootstrap-safe fallback).
        echo $safe_final_output;

        exit;
    }
}

vulnity_firewall_render_block_page();
