<?php
declare(strict_types=1);
if (!defined('ABSPATH'))
{
	exit;
}

if (!class_exists('VulnDexBeacon')):
	final class VulnDexBeacon
	{
		const OPT_SETTINGS = 'vulndex-beacon_settings';
		const OPT_NODE_ID  = 'vulndex-beacon_node_id';
		const OPT_STATUS   = 'vulndex-beacon_last_status';

		const CRON_HOOK    = 'vulndex-beacon_cron_hook';
		const SCHED_KEY    = 'vulndex_two_hours';
		const NONCE_MANUAL = 'vulndex-beacon_manual_send';
		const API_ENDPOINT = 'https://api.vulndex.at/beacon/wordpress';

		private static $instance = null;

		public static function instance()
		{
			if (self::$instance === null)
			{
				self::$instance = new self();
			}
			return self::$instance;
		}

		private function __construct()
		{
			add_action('admin_menu', [$this, 'add_settings_page']);
			add_action('admin_init', [$this, 'register_settings']);

			add_filter('cron_schedules', [$this, 'register_two_hour_schedule']);
			add_action(self::CRON_HOOK, [$this, 'send_report']);
			add_action('admin_notices', [$this, 'display_sync_notice']);


			add_action('activated_plugin', [$this, 'send_report_debounced'], 10, 2);
			add_action('deactivated_plugin', [$this, 'send_report_debounced'], 10, 2);
			add_action('after_switch_theme', [$this, 'send_report_debounced']);
			add_action('switch_theme', [$this, 'send_report_debounced']);
			add_action('upgrader_process_complete', [$this, 'on_upgrader_done'], 10, 2);

			add_action('update_option_' . self::OPT_SETTINGS, [$this, 'on_settings_updated'], 10, 3);
		}

		public static function on_activation()
		{


			if (!get_option(self::OPT_NODE_ID))
			{
				$node = self::generate_node_id();
				add_option(self::OPT_NODE_ID, $node, '', false);
			}
			if (!get_option(self::OPT_SETTINGS))
			{
				add_option(self::OPT_SETTINGS, ['api_key' => ''], '', false);
			}
			if (!get_option(self::OPT_STATUS))
			{
				add_option(self::OPT_STATUS, ['ok' => null, 'http_code' => null, 'time' => null, 'error' => null, 'error_code' => null], '', false);
			}
			if (!wp_next_scheduled(self::CRON_HOOK))
			{
				wp_schedule_event(time() + 120, self::SCHED_KEY, self::CRON_HOOK);
			}
		}

		public static function on_deactivation()
		{
			$ts = wp_next_scheduled(self::CRON_HOOK);
			if ($ts)
			{
				wp_unschedule_event($ts, self::CRON_HOOK);
			}

			$ts_burst = wp_next_scheduled('vulndex-beacon_single_burst');
			if ($ts_burst)
			{
				wp_unschedule_event($ts_burst, 'vulndex-beacon_single_burst');
			}
		}

		public static function on_uninstall()
		{
			delete_option(self::OPT_SETTINGS);
			delete_option(self::OPT_STATUS);
			delete_option(self::OPT_NODE_ID);

		}

		public function register_two_hour_schedule($schedules)
		{
			$schedules[self::SCHED_KEY] = ['interval' => 2 * HOUR_IN_SECONDS, 'display' => __('Every two hours (VulnDex Beacon)', 'vulndex-beacon')];
			return $schedules;
		}

		public function add_settings_page()
		{
			add_options_page('VulnDex Beacon', 'VulnDex Beacon', 'manage_options', 'vulndex-beacon', [$this, 'render_settings_page']);
		}


		public function register_settings()
		{
			register_setting(self::OPT_SETTINGS, self::OPT_SETTINGS, [
				'type'              => 'array',
				'sanitize_callback' => [$this, 'sanitize_settings'],
				'default'           => ['api_key' => ''],
			]);
			add_settings_section('vulndex-beacon_main', __('Reporting Settings', 'vulndex-beacon'), function ()
			{


			},                   'vulndex-beacon');
			add_settings_field('endpoint_url', __('API endpoint', 'vulndex-beacon'), [$this, 'field_api_endpoint'], 'vulndex-beacon', 'vulndex-beacon_main');
			add_settings_field('api_key', __('API key (Bearer)', 'vulndex-beacon'), [$this, 'field_api_key'], 'vulndex-beacon', 'vulndex-beacon_main');
			add_settings_field('node_id', __('Node identifier', 'vulndex-beacon'), [$this, 'field_node_id'], 'vulndex-beacon', 'vulndex-beacon_main');
			add_settings_field('last_status', __('Last API status', 'vulndex-beacon'), [$this, 'field_last_status'], 'vulndex-beacon', 'vulndex-beacon_main');

		}

		public function sanitize_settings($input)
		{
			return ['api_key' => isset($input['api_key']) ? trim(wp_unslash($input['api_key'])) : ''];
		}

		public function field_api_key()
		{
			$opts     = get_option(self::OPT_SETTINGS, []);
			$docs_url = 'https://vulndex.at/docs/10-vulndex-docs/127-beacon';
			printf(
				'<input type="password" name="%1$s[api_key]" value="%2$s" class="regular-text" autocomplete="off" placeholder="eyJhb..." /> ' .
				'<p class="description">' . esc_html__('To retrieve an API key contact administrator or review the', 'vulndex-beacon') . ' ' .
				'<a href="%3$s" target="_blank" rel="noopener noreferrer">' . esc_html__('VulnDex Docs', 'vulndex-beacon') . '</a>.' .
				'</p>',
				esc_attr(self::OPT_SETTINGS),
				esc_attr($opts['api_key'] ?? ''),
				esc_url($docs_url)
			);
		}

		public function field_node_id()
		{
			$node = get_option(self::OPT_NODE_ID, '');
			printf('<input type="text" value="%s" class="regular-text" readonly /> <p class="description">%s</p>',
				   esc_attr($node),
				   esc_html__('This value uniquely identifies your site and prevents duplicate reports. To connect another WordPress site, ask your administrator to add a new Beacon integration.', 'vulndex-beacon'));
		}

		public function display_sync_notice()
		{
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- No nonce verification needed for displaying admin notice
			if (!isset($_GET['page']) || $_GET['page'] !== 'vulndex-beacon')
			{
				return;
			}




			if (!current_user_can('manage_options'))
			{
				return;
			}

			// Check if sync is pending
			$sync_pending = get_transient('vulndex-beacon_sync_pending');
			$next_burst   = wp_next_scheduled('vulndex-beacon_single_burst');

			if ($sync_pending && $next_burst)
			{
				echo '<div class="notice notice-info is-dismissible">';
				echo '<p>';
				echo '<span style="color: #0073aa; font-weight: bold;">🔄 ' . esc_html__('VulnDex Beacon synchronization in progress...', 'vulndex-beacon') . '</span><br>';
				echo '<small style="color: #666;">' . esc_html__('Your data will be synchronized with VulnDex shortly.', 'vulndex-beacon') . '</small>';
				echo '</p>';
				echo '</div>';
			}
		}

		public function field_last_status()
		{


			$st = get_option(self::OPT_STATUS, []);

			$ok        = $st['ok'];
			$http_code = $st['http_code'] ?? null;
			// Nicht mit __() übersetzen - ist ein Timestamp von der Datenbank
			$time  = !empty($st['time']) ? $st['time'] : '-';
			$error = isset($st['error']) && $st['error'] !== null ? esc_html($st['error']) : '';
			$ecode = isset($st['error_code']) && $st['error_code'] !== null ? (int)($st['error_code']) : null;

			if ($ok === null)
			{
				echo '<span class="notice notice-info" style="display:inline-block;padding:6px 10px;">' . esc_html__('No request sent so far.', 'vulndex-beacon') . '</span>';
				return;
			}

			if ($ok)
			{
				printf('<span class="notice notice-success" style="display:inline-block;padding:6px 10px;">%s (HTTP %d) • %s</span>', esc_html__('Success', 'vulndex-beacon'), (int)($http_code), esc_html($time));
			}
			else
			{
				$msg = esc_html__('Error', 'vulndex-beacon');
				if ($error)
				{
					$msg .= ' – ' . $error;
				}
				if ($ecode !== null)
				{
					$msg .= ' (Code ' . $ecode . ')';
				}
				if ($http_code !== null)
				{
					$msg .= ' • HTTP ' . (int)$http_code;
				}
				$msg .= ' • ' . $time;
				printf('<span class="notice notice-error" style="display:inline-block;padding:6px 10px;">%s</span>', esc_html($msg));
			}
		}

		public function field_api_endpoint()
		{
			//echo json_encode($this->build_payload());

			echo esc_html(self::API_ENDPOINT);
			return;

		}

		public function render_settings_page()
		{
			if (!current_user_can('manage_options'))
			{
				return;
			}

			if (isset($_POST['vulndex_manual']))
			{

				check_admin_referer(self::NONCE_MANUAL, self::NONCE_MANUAL);


				$result = $this->send_report(true);
				if (is_wp_error($result))
				{
					printf('<div class="notice notice-error"><p>%s</p></div>', esc_html($result->get_error_message()));
				}
				else
				{
					echo '<div class="notice notice-success"><p>' . esc_html__('Information successfully sent to the VulnDex API.', 'vulndex-beacon') . '</p></div>';
				}

			}

			echo '<div class="wrap">';
			echo '<h1>VulnDex Beacon</h1>';
			echo '<form method="post" action="options.php">';
			settings_fields(self::OPT_SETTINGS);
			do_settings_sections('vulndex-beacon');
			submit_button(__('Save', 'vulndex-beacon'));
			echo '</form>';

			echo '<hr/>';
			echo '<form method="post">';
			wp_nonce_field(self::NONCE_MANUAL, self::NONCE_MANUAL);
			echo '<input type="hidden" name="vulndex_manual" value="1" />';
			submit_button(__('Send data manually', 'vulndex-beacon'), 'secondary');
			echo '</form>';


			echo '</div>';
		}

		public function build_payload()
		{
			global $wp_version;
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
			require_once ABSPATH . 'wp-includes/version.php';

			$node     = get_option(self::OPT_NODE_ID, '');
			$site_url = site_url();
			$home_url = home_url();

			$all_plugins = function_exists('get_plugins') ? get_plugins() : [];


			$active_map = array_flip((array)get_option('active_plugins', []));
			$plugins    = [];
			foreach ($all_plugins as $slug => $data)
			{
				$plugins[] = ['path' => $slug, 'slug' => $data['TextDomain'], 'name' => $data['Name'] ?? '', 'version' => $data['Version'] ?? '', 'active' => isset($active_map[$slug])];
			}

			$themes       = [];
			$active_theme = wp_get_theme();
			foreach (wp_get_themes() as $stylesheet => $theme)
			{
				$themes[] = ['stylesheet' => $stylesheet, 'name' => $theme->get('Name'), 'version' => $theme->get('Version'), 'active' => $active_theme && ($active_theme->get_stylesheet() === $stylesheet)];
			}

			$os_info = [];
			if (function_exists('php_uname'))
			{
				try
				{
					$os_info = [
						'name'    => php_uname('a'),
						'version' => php_uname('r')
					];
				}
				catch (Throwable $e)
				{

					$os_info = [
						'name'    => null,
						'version' => null,

					];
				}
			}
			else
			{
				$os_info = [
					'name'    => null,
					'version' => null
				];
			}


			return [
				'node_id'           => $node,
				'wordpress_version' => $wp_version,
				'php_version'       => PHP_VERSION,
				'os'                => $os_info,
				'site_url'          => $site_url,
				'home_url'          => $home_url,
				'version'           => 1,
				'inventory'         => ['plugins' => $plugins, 'themes' => $themes],
			];
		}

		public function send_report($manual = false)
		{
			$opts    = get_option(self::OPT_SETTINGS, []);
			$api_key = $opts['api_key'] ?? '';

			if (empty($api_key))
			{
				$this->save_status(false, null, __('API key is missing. Please configure it in settings.', 'vulndex-beacon'), null);
				return new \WP_Error('vulndex-beacon_missing_config', __('API key is missing. Please configure it in settings.', 'vulndex-beacon'));
			}

			$args = [
				'timeout' => 20,
				'headers' => [
					'Authorization' => 'Bearer ' . $api_key,
					'Content-Type'  => 'application/json',
					'Accept'        => 'application/json',
				],
				'body'    => wp_json_encode($this->build_payload()),
			];

			$response = wp_remote_post(self::API_ENDPOINT, $args);

			if (is_wp_error($response))
			{
				$this->save_status(false, null, $response->get_error_message(), null);
				return $response;
			}

			$code = wp_remote_retrieve_response_code($response);
			$body = wp_remote_retrieve_body($response);

			if ($code >= 200 && $code < 300)
			{
				$this->save_status(true, $code, null, null);
				return true;
			}

			$err   = null;
			$ecode = null;
			$json  = json_decode($body, true);
			if (is_array($json))
			{
				$err   = isset($json['error']) ? (string)$json['error'] : null;
				$ecode = isset($json['code']) ? (int)($json['code']) : null;
			}
			if (!$err)
			{
				$err = substr((string)$body, 0, 500);
			}
			$this->save_status(false, $code, $err, $ecode);

			/* translators: %1$d is the HTTP status code, %2$s is the API response body */
			return new \WP_Error('vulndex-beacon_http_error', sprintf(__('HTTP %1$d error during sending. Response: %2$s', 'vulndex-beacon'), esc_html($code), esc_html($body)));
		}

		private function save_status($ok, $http_code = null, $error = null, $error_code = null)
		{
			$status = [
				'ok'         => (bool)$ok,
				'http_code'  => $http_code !== null ? (int)$http_code : null,
				'time'       => gmdate('Y-m-d H:i:s') . ' UTC',
				'error'      => $error !== null ? (string)$error : null,
				'error_code' => $error_code !== null ? (int)$error_code : null,
			];
			update_option(self::OPT_STATUS, $status, false);

		}

		public function send_report_debounced()
		{
			$ts = wp_next_scheduled('vulndex-beacon_single_burst');
			if ($ts)
			{
				return;
			}
			wp_schedule_single_event(time() + 30, 'vulndex-beacon_single_burst');


		}

		public function on_upgrader_done($upgrader, $hook_extra)
		{
			$this->send_report_debounced();
		}

		public function on_settings_updated($old_value, $value, $option)
		{
			$old = is_array($old_value) ? ($old_value['api_key'] ?? '') : '';
			$new = is_array($value) ? ($value['api_key'] ?? '') : '';


			if ($new !== '' && $new !== $old)
			{
				set_transient('vulndex-beacon_sync_pending', true, 60);
				wp_schedule_single_event(time() + 5, 'vulndex-beacon_single_burst');
			}
		}


		private static function generate_node_id()
		{
			if (function_exists('wp_generate_uuid4'))
			{
				try
				{
					return wp_generate_uuid4();
				}
				catch (Throwable $e)
				{
					// Fallback
				}
			}


			if (function_exists('random_bytes'))
			{
				try
				{
					return bin2hex(random_bytes(16));
				}
				catch (Throwable $e)
				{
					// Fallback
				}
			}


			return uniqid('vulndex_', true);
		}
	}
endif;