<?php
if (!defined('ABSPATH')) {
    exit;
}

class Voxfor_Summary_Generator {
    
    private static function get_system_prompt() {
        $prompt = 'You are a CONCISE and ACCURATE summarizer. Create a clear, faithful summary.' . "\n\n";
        $prompt .= 'STRICT RULES:' . "\n";
        $prompt .= '1. ONLY use information EXPLICITLY in the source text' . "\n";
        $prompt .= '2. NEVER add external knowledge or interpretation' . "\n";
        $prompt .= '3. Optimal length: 60-90 words OR 5-7 bullet points' . "\n";
        $prompt .= '4. NO markdown formatting (no **, #, ##, -, etc) - plain text only' . "\n";
        $prompt .= '5. Match language: Hebrew text → Hebrew summary, English text → English summary' . "\n";
        $prompt .= '6. Every statement must be traceable to the source' . "\n\n";
        $prompt .= 'FORMAT (choose ONE):' . "\n";
        $prompt .= 'A) Single paragraph: 60-90 words, plain text' . "\n";
        $prompt .= 'B) 5-7 bullet points, each one clear sentence' . "\n\n";
        $prompt .= 'EXAMPLES:' . "\n\n";
        $prompt .= 'Good (English, 75 words):' . "\n";
        $prompt .= 'The article explores AI applications in healthcare including diagnostic tools, treatment recommendations, and patient monitoring. It discusses how machine learning models can detect cancers as effectively as radiologists according to recent studies. Privacy concerns remain a major challenge as healthcare organizations must protect patient data while leveraging AI capabilities. The article mentions that regulatory frameworks are being developed to ensure safe implementation, and emphasizes that AI integration must prioritize patient safety above all else.' . "\n\n";
        $prompt .= 'Good (Hebrew, 70 words):' . "\n";
        $prompt .= 'המאמר עוסק ביישומי בינה מלאכותית ברפואה כולל כלי אבחון, המלצות טיפול וניטור מטופלים. המאמר דן כיצד מודלים של למידת מכונה יכולים לזהות סרטן ביעילות כמו רדיולוגים לפי מחקרים עדכניים. חששות פרטיות נותרים אתגר מרכזי כאשר ארגוני בריאות חייבים להגן על נתוני מטופלים תוך שימוש ביכולות AI. המאמר מזכיר כי מסגרות רגולטוריות מפותחות להבטיח יישום בטוח, ומדגיש שאינטגרציה של AI חייבת לתעדף בטיחות מטופלים מעל הכל.' . "\n\n";
        $prompt .= 'Good (7 points):' . "\n";
        $prompt .= '• The article examines AI applications in healthcare diagnostics and treatment' . "\n";
        $prompt .= '• Machine learning models detect cancers as effectively as experienced radiologists' . "\n";
        $prompt .= '• Current AI uses include medical imaging, drug discovery, and patient monitoring' . "\n";
        $prompt .= '• Privacy concerns remain paramount as organizations must protect patient information' . "\n";
        $prompt .= '• Data security challenges exist despite the promise of AI technology' . "\n";
        $prompt .= '• Regulatory frameworks are being developed for safe AI implementation' . "\n";
        $prompt .= '• The article emphasizes that AI integration must prioritize patient safety' . "\n\n";
        $prompt .= 'BAD (too short):' . "\n";
        $prompt .= 'AI in healthcare. Privacy concerns. Regulations needed.' . "\n";
        $prompt .= '[This is too brief]' . "\n\n";
        $prompt .= 'BAD (too formatted):' . "\n";
        $prompt .= '# Summary' . "\n";
        $prompt .= '**Introduction:** The article provides...' . "\n";
        $prompt .= '[Has markdown formatting]' . "\n\n";
        $prompt .= 'Now create a clear summary (60-90 words OR 5-7 points):';

        return $prompt;
    }
    
    public static function generate($post_id, $force_regenerate = false) {
        $existing_summary = Voxfor_Database_Handler::get_summary($post_id);
        
        if (!$force_regenerate && $existing_summary) {
            $post = get_post($post_id);
            $content_hash = self::calculate_content_hash($post->post_content, $post->post_title);
            
            if ($existing_summary['content_hash'] === $content_hash) {
                return [
                    'success' => true,
                    'summary' => $existing_summary['summary'],
                    'cached' => true,
                    'generated_at' => $existing_summary['generated_at']
                ];
            }
        }
        
        $content = Voxfor_Content_Extractor::extract_post_content($post_id);
        
        if (empty($content)) {
            return [
                'success' => false,
                'error' => __('Unable to extract content from post.', 'voxfor-ai-content-summary')
            ];
        }
        
        $validation = Voxfor_Content_Extractor::validate_content($content);
        
        if (!$validation['valid']) {
            return [
                'success' => false,
                'error' => $validation['message']
            ];
        }
        
        $settings = get_option('voxfor_ai_summary_settings', []);
        $provider = $settings['provider'] ?? 'claude';
        $api_key = '';
        
        if ($provider === 'claude') {
            $api_key = $settings['claude_api_key'] ?? '';
        } else {
            $api_key = $settings['chatgpt_api_key'] ?? '';
        }
        
        if (empty($api_key)) {
            return [
                'success' => false,
                'error' => __('API key not configured. Please configure it in the plugin settings.', 'voxfor-ai-content-summary')
            ];
        }
        
        $ai_provider = new Voxfor_AI_Provider($provider, $api_key);
        $system_prompt = self::get_system_prompt();
        
        $result = $ai_provider->generate_summary($content, $system_prompt);
        
        if (!$result['success']) {
            return $result;
        }
        
        $post = get_post($post_id);
        $content_hash = self::calculate_content_hash($post->post_content, $post->post_title);
        
        Voxfor_Database_Handler::save_summary(
            $post_id,
            $result['summary'],
            $content,
            $content_hash,
            $provider
        );
        
        return [
            'success' => true,
            'summary' => $result['summary'],
            'cached' => false,
            'generated_at' => current_time('mysql')
        ];
    }
    
    private static function calculate_content_hash($content, $title) {
        return md5($title . '|' . $content);
    }
    
    public static function get_summary_display($post_id) {
        $summary_data = Voxfor_Database_Handler::get_summary($post_id);
        
        if (!$summary_data) {
            return null;
        }
        
        return [
            'summary' => $summary_data['summary'],
            'generated_at' => $summary_data['generated_at'],
            'provider' => $summary_data['provider_used'],
            'can_compare' => !empty($summary_data['original_content'])
        ];
    }
}
