<?php
if (!defined('ABSPATH')) {
    exit;
}

class Voxfor_Frontend_Display {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_filter('the_content', [$this, 'add_summary_button_to_content']);
        add_shortcode('voxfor_summary', [$this, 'summary_shortcode']);
        add_action('wp_enqueue_scripts', [$this, 'enqueue_frontend_assets']);
        add_action('wp_ajax_voxfor_get_summary_frontend', [$this, 'ajax_get_summary']);
        add_action('wp_ajax_nopriv_voxfor_get_summary_frontend', [$this, 'ajax_get_summary']);
    }
    
    public function enqueue_frontend_assets() {
        if (!is_singular()) {
            return;
        }
        
        $settings = get_option('voxfor_ai_summary_settings', []);
        $allowed_post_types = isset($settings['display_on_post_types']) ? $settings['display_on_post_types'] : ['post'];
        $current_post_type = get_post_type();
        
        if (!in_array($current_post_type, $allowed_post_types)) {
            return;
        }
        
        wp_enqueue_style(
            'voxfor-ai-content-summary-frontend',
            VOXFOR_AI_SUMMARY_PLUGIN_URL . 'css/frontend.css',
            [],
            VOXFOR_AI_SUMMARY_VERSION
        );
        
        wp_enqueue_script(
            'voxfor-ai-content-summary-frontend',
            VOXFOR_AI_SUMMARY_PLUGIN_URL . 'js/frontend.js',
            ['jquery'],
            VOXFOR_AI_SUMMARY_VERSION,
            true
        );
        
        wp_localize_script('voxfor-ai-content-summary-frontend', 'voxforSummary', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'postId' => get_the_ID(),
            'nonce' => wp_create_nonce('voxfor_summary_nonce'),
            'strings' => [
                'loading' => __('Generating summary...', 'voxfor-ai-content-summary'),
                'error' => __('Failed to generate summary. Please try again.', 'voxfor-ai-content-summary')
            ]
        ]);
    }
    
    private function should_display_button($post_id = null) {
        if (!$post_id) {
            $post_id = get_the_ID();
        }
        
        if (!$post_id) {
            return false;
        }
        
        $settings = get_option('voxfor_ai_summary_settings', []);
        
        $button_position = isset($settings['button_position']) ? $settings['button_position'] : 'before_content';
        if ($button_position === 'shortcode_only') {
            return false;
        }
        
        $post_type = get_post_type($post_id);
        $allowed_types = isset($settings['display_on_post_types']) ? $settings['display_on_post_types'] : ['post'];
        
        if (!in_array($post_type, $allowed_types)) {
            return false;
        }
        
        $auto_display = isset($settings['auto_display']) ? $settings['auto_display'] : 'manual';
        $per_post_setting = get_post_meta($post_id, '_voxfor_display_summary', true);
        
        if ($auto_display === 'auto') {
            return $per_post_setting !== 'no';
        } else {
            return $per_post_setting === 'yes';
        }
    }
    
    public function add_summary_button_to_content($content) {
        if (!is_singular() || !is_main_query()) {
            return $content;
        }
        
        global $post;
        if (!$post) {
            return $content;
        }
        
        if (!$this->should_display_button($post->ID)) {
            return $content;
        }
        
        $settings = get_option('voxfor_ai_summary_settings', []);
        $button_position = isset($settings['button_position']) ? $settings['button_position'] : 'before_content';
        
        $button_html = $this->render_summary_button($post->ID);
        
        if ($button_position === 'after_content') {
            return $content . $button_html;
        }
        
        return $button_html . $content;
    }
    
    private function render_summary_button($post_id) {
        ob_start();
        ?>
        <div class="voxfor-summary-wrapper" id="voxfor-summary-wrapper-<?php echo esc_attr($post_id); ?>">
            <button type="button" class="voxfor-summary-button" id="voxfor-summary-btn-<?php echo esc_attr($post_id); ?>" data-post-id="<?php echo esc_attr($post_id); ?>">
                <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"></path>
                    <polyline points="14 2 14 8 20 8"></polyline>
                    <line x1="16" y1="13" x2="8" y2="13"></line>
                    <line x1="16" y1="17" x2="8" y2="17"></line>
                </svg>
                <span class="voxfor-button-text"><?php esc_html_e('AI Summary', 'voxfor-ai-content-summary'); ?></span>
                <svg class="voxfor-spinner" style="display: none;" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <circle cx="12" cy="12" r="10"></circle>
                </svg>
            </button>
            
            <div class="voxfor-summary-container" id="voxfor-summary-container-<?php echo esc_attr($post_id); ?>" style="display: none;">
            </div>
        </div>
        <?php
        return ob_get_clean();
    }
    
    private function render_summary($summary_data) {
        $summary = esc_html($summary_data['summary']);
        $generated_at = date_i18n(get_option('date_format'), strtotime($summary_data['generated_at']));
        $provider = ucfirst($summary_data['provider_used']);
        
        ob_start();
        ?>
        <div class="voxfor-summary-box" itemscope itemtype="https://schema.org/Article">
            <div class="voxfor-summary-header">
                <h3 class="voxfor-summary-title">
                    <svg class="voxfor-summary-icon" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"></path>
                        <polyline points="14 2 14 8 20 8"></polyline>
                        <line x1="16" y1="13" x2="8" y2="13"></line>
                        <line x1="16" y1="17" x2="8" y2="17"></line>
                        <polyline points="10 9 9 9 8 9"></polyline>
                    </svg>
                    <?php esc_html_e('Content Summary', 'voxfor-ai-content-summary'); ?>
                </h3>
                <div class="voxfor-summary-badge">
                    <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <circle cx="12" cy="12" r="10"></circle>
                        <path d="M12 16v-4"></path>
                        <path d="M12 8h.01"></path>
                    </svg>
                    <?php esc_html_e('AI-Generated', 'voxfor-ai-content-summary'); ?>
                </div>
            </div>
            
            <div class="voxfor-summary-content" itemprop="abstract">
                <?php echo nl2br(esc_html(trim($summary))); ?>
            </div>
            
            <div class="voxfor-summary-footer">
                <div class="voxfor-summary-meta">
                    <span class="voxfor-summary-meta-item">
                        <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <circle cx="12" cy="12" r="10"></circle>
                            <polyline points="12 6 12 12 16 14"></polyline>
                        </svg>
                        <?php
                        // translators: %s: Date when the summary was generated.
                        printf(esc_html__('Generated: %s', 'voxfor-ai-content-summary'), esc_html($generated_at));
                        ?>
                    </span>
                    <span class="voxfor-summary-meta-item">
                        <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z"></path>
                        </svg>
                        <?php echo esc_html($provider); ?>
                    </span>
                </div>
                <div class="voxfor-summary-disclaimer">
                    <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M10.29 3.86L1.82 18a2 2 0 0 0 1.71 3h16.94a2 2 0 0 0 1.71-3L13.71 3.86a2 2 0 0 0-3.42 0z"></path>
                        <line x1="12" y1="9" x2="12" y2="13"></line>
                        <line x1="12" y1="17" x2="12.01" y2="17"></line>
                    </svg>
                    <?php esc_html_e('This summary is generated from the article content only, with no external information added.', 'voxfor-ai-content-summary'); ?>
                </div>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }
    
    public function ajax_get_summary() {
        check_ajax_referer('voxfor_summary_nonce', 'nonce');
        
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        
        if (!$post_id) {
            wp_send_json_error(['message' => __('Invalid post ID.', 'voxfor-ai-content-summary')]);
        }
        
        if (!get_post($post_id)) {
            wp_send_json_error(['message' => __('Post not found.', 'voxfor-ai-content-summary')]);
        }
        
        if (get_post_status($post_id) !== 'publish') {
            wp_send_json_error(['message' => __('Post is not published.', 'voxfor-ai-content-summary')]);
        }
        
        $summary_data = Voxfor_Database_Handler::get_summary($post_id);
        
        if ($summary_data && !empty($summary_data['summary'])) {
            wp_send_json_success([
                'html' => $this->render_summary($summary_data),
                'cached' => true
            ]);
        }
        
        $this->check_rate_limit($post_id);
        
        if (!$this->acquire_generation_lock($post_id)) {
            wp_send_json_error([
                'message' => __('Summary is being generated. Please wait a moment.', 'voxfor-ai-content-summary')
            ]);
        }
        
        $result = Voxfor_Summary_Generator::generate($post_id, false);
        
        $this->release_generation_lock($post_id);
        
        if ($result['success']) {
            $summary_data = Voxfor_Database_Handler::get_summary($post_id);
            wp_send_json_success([
                'html' => $this->render_summary($summary_data),
                'cached' => false
            ]);
        } else {
            wp_send_json_error(['message' => $result['error']]);
        }
    }
    
    private function check_rate_limit($post_id) {
        $ip = isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'])) : '';
        $transient_key = 'voxfor_rate_limit_' . md5($ip . $post_id);
        $attempts = get_transient($transient_key);
        
        if ($attempts && $attempts >= 15) {
            wp_send_json_error([
                'message' => __('Too many requests. Please wait a few minutes.', 'voxfor-ai-content-summary')
            ]);
        }
        
        $new_attempts = $attempts ? $attempts + 1 : 1;
        set_transient($transient_key, $new_attempts, 5 * MINUTE_IN_SECONDS);
    }
    
    private function acquire_generation_lock($post_id) {
        $lock_key = 'voxfor_generating_' . $post_id;
        
        if (get_transient($lock_key)) {
            return false;
        }
        
        set_transient($lock_key, time(), 60);
        return true;
    }
    
    private function release_generation_lock($post_id) {
        $lock_key = 'voxfor_generating_' . $post_id;
        delete_transient($lock_key);
    }
    
    public function summary_shortcode($atts) {
        $atts = shortcode_atts([
            'post_id' => get_the_ID()
        ], $atts);
        
        $post_id = intval($atts['post_id']);
        if (!$post_id) {
            return '';
        }
        
        return $this->render_summary_button($post_id);
    }
}
