<?php
if (!defined('ABSPATH')) {
    exit;
}

class Voxfor_Database_Handler {
    
    const TABLE_NAME = 'voxfor_ai_summaries';
    
    public static function create_tables() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . self::TABLE_NAME;
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            post_id bigint(20) UNSIGNED NOT NULL,
            summary longtext NOT NULL,
            original_content longtext NOT NULL,
            content_hash varchar(32) NOT NULL,
            provider_used varchar(20) NOT NULL,
            generated_at datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP NOT NULL,
            PRIMARY KEY (id),
            UNIQUE KEY post_id (post_id),
            KEY content_hash (content_hash)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    public static function save_summary($post_id, $summary, $original_content, $content_hash, $provider) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . self::TABLE_NAME;
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table requires direct query; table name safely constructed.
        $existing = $wpdb->get_var( $wpdb->prepare( "SELECT id FROM $table_name WHERE post_id = %d", $post_id ) );
        
        $data = [
            'post_id' => $post_id,
            'summary' => $summary,
            'original_content' => $original_content,
            'content_hash' => $content_hash,
            'provider_used' => $provider
        ];
        
        if ($existing) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table requires direct query; cache not applicable for write operations.
            $result = $wpdb->update(
                $table_name,
                $data,
                ['post_id' => $post_id],
                ['%d', '%s', '%s', '%s', '%s'],
                ['%d']
            );
        } else {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query.
            $result = $wpdb->insert(
                $table_name,
                $data,
                ['%d', '%s', '%s', '%s', '%s']
            );
        }
        
        return $result !== false;
    }
    
    public static function get_summary($post_id) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . self::TABLE_NAME;
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table requires direct query; table name safely constructed.
        $result = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM $table_name WHERE post_id = %d", $post_id ), ARRAY_A );
        
        return $result;
    }
    
    public static function delete_summary($post_id) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . self::TABLE_NAME;
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table requires direct query; cache not applicable for delete operations.
        return $wpdb->delete(
            $table_name,
            ['post_id' => $post_id],
            ['%d']
        );
    }
    
    public static function get_all_summaries($limit = 100, $offset = 0) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . self::TABLE_NAME;
        
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table requires direct query; table name safely constructed.
        $results = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT s.*, p.post_title FROM $table_name s LEFT JOIN {$wpdb->posts} p ON s.post_id = p.ID ORDER BY s.updated_at DESC LIMIT %d OFFSET %d",
                $limit,
                $offset
            ),
            ARRAY_A
        );
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        
        return $results;
    }
    
    public static function get_summary_count() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . self::TABLE_NAME;
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table requires direct query; table name safely constructed.
        return (int) $wpdb->get_var( "SELECT COUNT(*) FROM $table_name" );
    }
}
