<?php
if (!defined('ABSPATH')) {
    exit;
}

class Voxfor_AI_Provider {
    
    const PROVIDER_CLAUDE = 'claude';
    const PROVIDER_CHATGPT = 'chatgpt';
    
    private $api_key;
    private $provider;
    
    public function __construct($provider, $api_key) {
        $this->provider = $provider;
        $this->api_key = $api_key;
    }
    
    public function generate_summary($content, $system_prompt) {
        if (empty($this->api_key)) {
            return [
                'success' => false,
                'error' => __('API key not configured.', 'voxfor-ai-content-summary')
            ];
        }
        
        switch ($this->provider) {
            case self::PROVIDER_CLAUDE:
                return $this->call_claude_api($content, $system_prompt);
            
            case self::PROVIDER_CHATGPT:
                return $this->call_chatgpt_api($content, $system_prompt);
            
            default:
                return [
                    'success' => false,
                    'error' => __('Invalid provider specified.', 'voxfor-ai-content-summary')
                ];
        }
    }
    
    private function call_claude_api($content, $system_prompt) {
        $url = 'https://api.anthropic.com/v1/messages';
        
        $data = [
            'model' => 'claude-haiku-4-5-20251001',
            'max_tokens' => 2000,
            'system' => $system_prompt,
            'messages' => [
                [
                    'role' => 'user',
                    'content' => $content
                ]
            ]
        ];
        
        $args = [
            'timeout' => 60,
            'headers' => [
                'Content-Type' => 'application/json',
                'x-api-key' => $this->api_key,
                'anthropic-version' => '2023-06-01'
            ],
            'body' => wp_json_encode($data)
        ];
        
        $response = wp_remote_post($url, $args);
        
        if (is_wp_error($response)) {
            return [
                'success' => false,
                'error' => $response->get_error_message()
            ];
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $result = json_decode($body, true);
        
        if ($status_code !== 200) {
            $error_message = isset($result['error']['message']) 
                ? $result['error']['message'] 
                : __('Unknown API error occurred.', 'voxfor-ai-content-summary');
            
            return [
                'success' => false,
                'error' => $error_message
            ];
        }
        
        if (isset($result['content'][0]['text'])) {
            return [
                'success' => true,
                'summary' => trim($result['content'][0]['text'])
            ];
        }
        
        return [
            'success' => false,
            'error' => __('Invalid API response format.', 'voxfor-ai-content-summary')
        ];
    }
    
    private function call_chatgpt_api($content, $system_prompt) {
        $url = 'https://api.openai.com/v1/chat/completions';
        
        $data = [
            'model' => 'gpt-5.2-chat-latest',
            'messages' => [
                [
                    'role' => 'system',
                    'content' => $system_prompt
                ],
                [
                    'role' => 'user',
                    'content' => $content
                ]
            ],
            'max_completion_tokens' => 500
        ];
        
        $args = [
            'timeout' => 60,
            'headers' => [
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $this->api_key
            ],
            'body' => wp_json_encode($data)
        ];
        
        $response = wp_remote_post($url, $args);
        
        if (is_wp_error($response)) {
            return [
                'success' => false,
                'error' => $response->get_error_message()
            ];
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $result = json_decode($body, true);
        
        if ($status_code !== 200) {
            $error_message = isset($result['error']['message']) 
                ? $result['error']['message'] 
                : __('Unknown API error occurred.', 'voxfor-ai-content-summary');
            
            return [
                'success' => false,
                'error' => $error_message
            ];
        }
        
        if (isset($result['choices'][0]['message']['content'])) {
            return [
                'success' => true,
                'summary' => trim($result['choices'][0]['message']['content'])
            ];
        }
        
        return [
            'success' => false,
            'error' => __('Invalid API response format.', 'voxfor-ai-content-summary')
        ];
    }
    
    public static function get_available_providers() {
        return [
            self::PROVIDER_CLAUDE => __('Claude Haiku 4.5 (claude-haiku-4-5-20251001)', 'voxfor-ai-content-summary'),
            self::PROVIDER_CHATGPT => __('ChatGPT 5.2 (gpt-5.2-chat-latest)', 'voxfor-ai-content-summary')
        ];
    }
    
    public static function validate_api_key($provider, $api_key) {
        if (empty($api_key)) {
            return false;
        }
        
        switch ($provider) {
            case self::PROVIDER_CLAUDE:
                return strpos($api_key, 'sk-ant-') === 0;
            
            case self::PROVIDER_CHATGPT:
                return strpos($api_key, 'sk-') === 0;
            
            default:
                return false;
        }
    }
}
