<?php
if (!defined('ABSPATH')) {
    exit;
}

class Voxfor_Meta_Box {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_action('add_meta_boxes', [$this, 'add_meta_box']);
        add_action('save_post', [$this, 'save_meta_box']);
        add_action('wp_ajax_voxfor_generate_summary', [$this, 'ajax_generate_summary']);
        add_action('wp_ajax_voxfor_delete_summary', [$this, 'ajax_delete_summary']);
    }
    
    public function add_meta_box() {
        $settings = get_option('voxfor_ai_summary_settings', []);
        $post_types = isset($settings['display_on_post_types']) ? $settings['display_on_post_types'] : ['post', 'page'];
        
        foreach ($post_types as $post_type) {
            add_meta_box(
                'voxfor_ai_summary',
                __('AI Content Summary', 'voxfor-ai-content-summary'),
                [$this, 'render_meta_box'],
                $post_type,
                'side',
                'default'
            );
        }
    }
    
    public function render_meta_box($post) {
        wp_nonce_field('voxfor_ai_summary_meta_box', 'voxfor_ai_summary_nonce');
        
        $summary_data = Voxfor_Summary_Generator::get_summary_display($post->ID);
        $settings = get_option('voxfor_ai_summary_settings', []);
        $auto_display = isset($settings['auto_display']) ? $settings['auto_display'] : 'manual';
        $display_summary = get_post_meta($post->ID, '_voxfor_display_summary', true);
        
        if (empty($display_summary)) {
            $display_summary = ($auto_display === 'auto') ? 'yes' : 'no';
        }
        
        ?>
        <div class="voxfor-meta-box-wrapper">
            <div class="voxfor-display-control" style="margin-bottom: 15px; padding: 10px; background: #f9f9f9; border: 1px solid #ddd; border-radius: 4px;">
                <label style="display: flex; align-items: center; cursor: pointer;">
                    <input type="checkbox" 
                           name="voxfor_display_summary" 
                           value="yes" 
                           <?php checked($display_summary, 'yes'); ?>
                           style="margin: 0 8px 0 0;">
                    <strong><?php esc_html_e('Display AI Summary Button', 'voxfor-ai-content-summary'); ?></strong>
                </label>
                <p style="margin: 8px 0 0 0; font-size: 12px; color: #666;">
                    <?php 
                    if ($auto_display === 'auto') {
                        esc_html_e('Uncheck to hide the button on this post.', 'voxfor-ai-content-summary');
                    } else {
                        esc_html_e('Check to show the button on this post.', 'voxfor-ai-content-summary');
                    }
                    ?>
                </p>
                <p style="margin: 8px 0 0 0; font-size: 11px; color: #999;">
                    <?php esc_html_e('Or use shortcode: [voxfor_summary]', 'voxfor-ai-content-summary'); ?>
                </p>
            </div>
            <?php if ($summary_data): ?>
                <div class="voxfor-summary-display">
                    <div class="voxfor-summary-meta">
                        <p style="margin: 0 0 10px 0; font-size: 12px; color: #666;">
                            <strong><?php esc_html_e('Generated:', 'voxfor-ai-content-summary'); ?></strong>
                            <?php echo esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($summary_data['generated_at']))); ?>
                        </p>
                        <p style="margin: 0 0 10px 0; font-size: 12px; color: #666;">
                            <strong><?php esc_html_e('Provider:', 'voxfor-ai-content-summary'); ?></strong>
                            <?php echo esc_html(ucfirst($summary_data['provider'])); ?>
                        </p>
                    </div>
                    
                    <div class="voxfor-summary-content" style="padding: 15px; background: #f6f7f7; border-left: 4px solid #2271b1; margin: 10px 0;">
                        <p style="margin: 0; line-height: 1.6; white-space: pre-wrap;"><?php echo esc_html($summary_data['summary']); ?></p>
                    </div>
                    
                    <div class="voxfor-summary-actions" style="margin-top: 15px;">
                        <button type="button" class="button button-primary voxfor-regenerate-summary" data-post-id="<?php echo esc_attr($post->ID); ?>">
                            <?php esc_html_e('Regenerate Summary', 'voxfor-ai-content-summary'); ?>
                        </button>
                        <button type="button" class="button voxfor-delete-summary" data-post-id="<?php echo esc_attr($post->ID); ?>" style="margin-left: 5px;">
                            <?php esc_html_e('Delete Summary', 'voxfor-ai-content-summary'); ?>
                        </button>
                        <?php if ($summary_data['can_compare']): ?>
                            <button type="button" class="button voxfor-compare-summary" data-post-id="<?php echo esc_attr($post->ID); ?>" style="margin-top: 10px; width: 100%;">
                                <?php esc_html_e('Compare with Original', 'voxfor-ai-content-summary'); ?>
                            </button>
                        <?php endif; ?>
                    </div>
                    
                    <div class="voxfor-comparison-view" style="display: none; margin-top: 15px; padding: 15px; background: #fff; border: 1px solid #ddd;">
                        <h4 style="margin-top: 0;"><?php esc_html_e('Original Content', 'voxfor-ai-content-summary'); ?></h4>
                        <div class="voxfor-original-content" style="max-height: 300px; overflow-y: auto; padding: 10px; background: #f9f9f9; font-size: 12px; line-height: 1.6;"></div>
                        <button type="button" class="button voxfor-hide-comparison" style="margin-top: 10px;">
                            <?php esc_html_e('Hide', 'voxfor-ai-content-summary'); ?>
                        </button>
                    </div>
                </div>
            <?php else: ?>
                <div class="voxfor-no-summary">
                    <p><?php esc_html_e('No summary generated yet.', 'voxfor-ai-content-summary'); ?></p>
                    <p style="font-size: 12px; color: #666; line-height: 1.6;">
                        <?php esc_html_e('The AI will create an accurate, faithful summary based solely on the actual content of this post.', 'voxfor-ai-content-summary'); ?>
                    </p>
                    <button type="button" class="button button-primary voxfor-generate-summary" data-post-id="<?php echo esc_attr($post->ID); ?>">
                        <?php esc_html_e('Generate Summary', 'voxfor-ai-content-summary'); ?>
                    </button>
                </div>
            <?php endif; ?>
            
            <div class="voxfor-loading" style="display: none; text-align: center; padding: 20px;">
                <span class="spinner is-active" style="float: none; margin: 0;"></span>
                <p style="margin-top: 10px;"><?php esc_html_e('Generating summary...', 'voxfor-ai-content-summary'); ?></p>
            </div>
            
            <div class="voxfor-error" style="display: none; padding: 10px; background: #f8d7da; border-left: 4px solid #dc3545; margin-top: 10px;">
                <p style="margin: 0; color: #721c24;"></p>
            </div>
            
            <div class="voxfor-info-box" style="margin-top: 20px; padding: 10px; background: #e7f3ff; border-left: 4px solid #0073aa; font-size: 12px;">
                <p style="margin: 0; line-height: 1.6;">
                    <strong><?php esc_html_e('Accuracy Guarantee:', 'voxfor-ai-content-summary'); ?></strong>
                    <?php esc_html_e('Every sentence in the summary is traceable to your original content. No external knowledge or fabrication.', 'voxfor-ai-content-summary'); ?>
                </p>
            </div>
        </div>
        <?php
    }
    
    public function ajax_generate_summary() {
        check_ajax_referer('voxfor_ai_summary_nonce', 'nonce');
        
        if (!current_user_can('edit_posts')) {
            wp_send_json_error(['message' => __('Insufficient permissions.', 'voxfor-ai-content-summary')]);
        }
        
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        $force_regenerate = isset($_POST['regenerate']) && $_POST['regenerate'] === 'true';
        
        if (!$post_id) {
            wp_send_json_error(['message' => __('Invalid post ID.', 'voxfor-ai-content-summary')]);
        }
        
        $result = Voxfor_Summary_Generator::generate($post_id, $force_regenerate);
        
        if ($result['success']) {
            $summary_data = Voxfor_Database_Handler::get_summary($post_id);
            
            wp_send_json_success([
                'summary' => $result['summary'],
                'generated_at' => date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($result['generated_at'])),
                'provider' => isset($summary_data['provider_used']) ? ucfirst($summary_data['provider_used']) : '',
                'original_content' => isset($summary_data['original_content']) ? $summary_data['original_content'] : '',
                'cached' => $result['cached'] ?? false
            ]);
        } else {
            wp_send_json_error(['message' => $result['error']]);
        }
    }
    
    public function ajax_delete_summary() {
        check_ajax_referer('voxfor_ai_summary_nonce', 'nonce');
        
        if (!current_user_can('edit_posts')) {
            wp_send_json_error(['message' => __('Insufficient permissions.', 'voxfor-ai-content-summary')]);
        }
        
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        
        if (!$post_id) {
            wp_send_json_error(['message' => __('Invalid post ID.', 'voxfor-ai-content-summary')]);
        }
        
        $deleted = Voxfor_Database_Handler::delete_summary($post_id);
        
        if ($deleted) {
            wp_send_json_success(['message' => __('Summary deleted successfully.', 'voxfor-ai-content-summary')]);
        } else {
            wp_send_json_error(['message' => __('Failed to delete summary.', 'voxfor-ai-content-summary')]);
        }
    }
    
    public function save_meta_box($post_id) {
        if (!isset($_POST['voxfor_ai_summary_nonce'])) {
            return;
        }
        
        $nonce = sanitize_text_field(wp_unslash($_POST['voxfor_ai_summary_nonce']));
        if (!wp_verify_nonce($nonce, 'voxfor_ai_summary_meta_box')) {
            return;
        }
        
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }
        
        if (isset($_POST['voxfor_display_summary']) && $_POST['voxfor_display_summary'] === 'yes') {
            update_post_meta($post_id, '_voxfor_display_summary', 'yes');
        } else {
            update_post_meta($post_id, '_voxfor_display_summary', 'no');
        }
    }
}
