<?php
if (!defined('ABSPATH')) {
    exit;
}

class Voxfor_Admin_Settings {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_action('admin_menu', [$this, 'add_settings_page']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
    }
    
    public function add_settings_page() {
        add_menu_page(
            __('Voxfor AI Summary', 'voxfor-ai-content-summary'),
            __('Voxfor AI', 'voxfor-ai-content-summary'),
            'manage_options',
            'voxfor-ai-content-summary',
            [$this, 'render_settings_page'],
            'dashicons-analytics',
            30
        );
        
        add_submenu_page(
            'voxfor-ai-content-summary',
            __('Settings', 'voxfor-ai-content-summary'),
            __('Settings', 'voxfor-ai-content-summary'),
            'manage_options',
            'voxfor-ai-content-summary',
            [$this, 'render_settings_page']
        );
    }
    
    public function register_settings() {
        register_setting('voxfor_ai_summary_settings', 'voxfor_ai_summary_settings', [
            'type'              => 'array',
            'description'       => 'Voxfor AI Content Summary plugin settings',
            'sanitize_callback' => [$this, 'sanitize_settings'],
            'default'           => [
                'provider'              => 'claude',
                'claude_api_key'        => '',
                'chatgpt_api_key'       => '',
                'display_on_post_types' => ['post'],
                'auto_display'          => 'manual',
                'button_position'       => 'before_content',
            ],
        ]);
        
        add_settings_section(
            'voxfor_ai_summary_provider',
            __('AI Provider Configuration', 'voxfor-ai-content-summary'),
            [$this, 'render_provider_section'],
            'voxfor-ai-content-summary'
        );
        
        add_settings_field(
            'provider',
            __('AI Provider', 'voxfor-ai-content-summary'),
            [$this, 'render_provider_field'],
            'voxfor-ai-content-summary',
            'voxfor_ai_summary_provider'
        );
        
        add_settings_field(
            'claude_api_key',
            __('Claude API Key', 'voxfor-ai-content-summary'),
            [$this, 'render_claude_api_key_field'],
            'voxfor-ai-content-summary',
            'voxfor_ai_summary_provider'
        );
        
        add_settings_field(
            'chatgpt_api_key',
            __('ChatGPT API Key', 'voxfor-ai-content-summary'),
            [$this, 'render_chatgpt_api_key_field'],
            'voxfor-ai-content-summary',
            'voxfor_ai_summary_provider'
        );
        
        add_settings_section(
            'voxfor_ai_summary_display',
            __('Display Settings', 'voxfor-ai-content-summary'),
            [$this, 'render_display_section'],
            'voxfor-ai-content-summary'
        );
        
        add_settings_field(
            'display_on_post_types',
            __('Display Button On', 'voxfor-ai-content-summary'),
            [$this, 'render_post_types_field'],
            'voxfor-ai-content-summary',
            'voxfor_ai_summary_display'
        );
        
        add_settings_field(
            'auto_display',
            __('Auto Display', 'voxfor-ai-content-summary'),
            [$this, 'render_auto_display_field'],
            'voxfor-ai-content-summary',
            'voxfor_ai_summary_display'
        );
        
        add_settings_field(
            'button_position',
            __('Button Position', 'voxfor-ai-content-summary'),
            [$this, 'render_button_position_field'],
            'voxfor-ai-content-summary',
            'voxfor_ai_summary_display'
        );
    }
    
    public function sanitize_settings($input) {
        $output = [];
        
        // Whitelist validation for provider
        $output['provider'] = isset($input['provider']) && in_array($input['provider'], ['claude', 'chatgpt'], true) 
            ? $input['provider'] 
            : 'claude';
        
        // API keys: trim only, no sanitization to preserve special characters
        $output['claude_api_key'] = isset($input['claude_api_key']) 
            ? trim($input['claude_api_key']) 
            : '';
        
        $output['chatgpt_api_key'] = isset($input['chatgpt_api_key']) 
            ? trim($input['chatgpt_api_key']) 
            : '';
        
        // Validate post types: sanitize keys and whitelist against actual registered post types
        if (isset($input['display_on_post_types']) && is_array($input['display_on_post_types'])) {
            $sanitized_post_types = array_map('sanitize_key', $input['display_on_post_types']);
            $registered_post_types = get_post_types(['public' => true], 'names');
            $output['display_on_post_types'] = array_values(array_intersect($sanitized_post_types, $registered_post_types));
            
            // Fallback to 'post' if no valid post types remain
            if (empty($output['display_on_post_types'])) {
                $output['display_on_post_types'] = ['post'];
            }
        } else {
            $output['display_on_post_types'] = ['post'];
        }
        
        // Whitelist validation for auto_display
        $output['auto_display'] = isset($input['auto_display']) && in_array($input['auto_display'], ['auto', 'manual'], true)
            ? $input['auto_display']
            : 'manual';
        
        // Whitelist validation for button_position
        $output['button_position'] = isset($input['button_position']) && in_array($input['button_position'], ['before_content', 'after_content', 'shortcode_only'], true)
            ? $input['button_position']
            : 'before_content';
        
        return $output;
    }
    
    public function render_provider_section() {
        echo '<p>' . esc_html__('Configure your AI provider and API keys. Both providers operate under the same strict accuracy standards.', 'voxfor-ai-content-summary') . '</p>';
        
        echo '<div class="notice notice-warning inline" style="margin: 15px 0;">';
        echo '<p><strong>' . esc_html__('⚠️ External Service Disclosure', 'voxfor-ai-content-summary') . '</strong></p>';
        echo '<p>' . esc_html__('This plugin sends post content to external AI services to generate summaries:', 'voxfor-ai-content-summary') . '</p>';
        echo '<ul style="list-style: disc; margin-left: 20px;">';
        $allowed_html = array(
            'code' => array(),
            'a'    => array(
                'href'   => array(),
                'target' => array(),
            ),
        );
        $claude_api_text = sprintf(
            // translators: %1$s: API endpoint URL, %2$s: Privacy policy link.
            esc_html__('API endpoint: %1$s | Privacy: %2$s', 'voxfor-ai-content-summary'),
            '<code>https://api.anthropic.com/v1/messages</code>',
            '<a href="https://www.anthropic.com/privacy" target="_blank">anthropic.com/privacy</a>'
        );
        echo '<li><strong>Claude (Anthropic):</strong> ' . wp_kses($claude_api_text, $allowed_html) . '</li>';
        $chatgpt_api_text = sprintf(
            // translators: %1$s: API endpoint URL, %2$s: Privacy policy link.
            esc_html__('API endpoint: %1$s | Privacy: %2$s', 'voxfor-ai-content-summary'),
            '<code>https://api.openai.com/v1/chat/completions</code>',
            '<a href="https://openai.com/privacy" target="_blank">openai.com/privacy</a>'
        );
        echo '<li><strong>ChatGPT (OpenAI):</strong> ' . wp_kses($chatgpt_api_text, $allowed_html) . '</li>';
        echo '</ul>';
        echo '<p><strong>' . esc_html__('Data sent:', 'voxfor-ai-content-summary') . '</strong> ' . esc_html__('Post title and content only (no personal data, no user information, no site credentials).', 'voxfor-ai-content-summary') . '</p>';
        echo '<p><strong>' . esc_html__('When:', 'voxfor-ai-content-summary') . '</strong> ' . esc_html__('Only when a visitor clicks the summary button (not automatic).', 'voxfor-ai-content-summary') . '</p>';
        echo '<p><strong>' . esc_html__('Security:', 'voxfor-ai-content-summary') . '</strong> ' . esc_html__('Rate limiting (15 requests/5min), nonce verification, caching to minimize API calls.', 'voxfor-ai-content-summary') . '</p>';
        echo '<p style="margin-top: 10px;">' . esc_html__('By using this plugin, you acknowledge sending content to these services and accept responsibility for compliance with their terms and privacy policies.', 'voxfor-ai-content-summary') . '</p>';
        echo '</div>';
        
        echo '<div class="notice notice-info inline"><p><strong>' . esc_html__('Accuracy Standard:', 'voxfor-ai-content-summary') . '</strong> ' . esc_html__('The AI only summarizes information that actually appears in your content. No external knowledge, interpretation, or fabrication.', 'voxfor-ai-content-summary') . '</p></div>';
    }
    
    public function render_provider_field() {
        $settings = get_option('voxfor_ai_summary_settings', []);
        $provider = $settings['provider'] ?? 'claude';
        
        $providers = Voxfor_AI_Provider::get_available_providers();
        
        echo '<select name="voxfor_ai_summary_settings[provider]" id="voxfor_provider">';
        foreach ($providers as $key => $label) {
            printf(
                '<option value="%s" %s>%s</option>',
                esc_attr($key),
                selected($provider, $key, false),
                esc_html($label)
            );
        }
        echo '</select>';
        echo '<p class="description">' . esc_html__('Select which AI provider to use for generating summaries.', 'voxfor-ai-content-summary') . '</p>';
    }
    
    public function render_claude_api_key_field() {
        $settings = get_option('voxfor_ai_summary_settings', []);
        $api_key = $settings['claude_api_key'] ?? '';
        
        printf(
            '<input type="password" name="voxfor_ai_summary_settings[claude_api_key]" id="voxfor_claude_api_key" value="%s" class="regular-text" />',
            esc_attr($api_key)
        );
        echo '<p class="description">' . esc_html__('Enter your Claude API key (starts with sk-ant-). Model: claude-haiku-4-5-20251001', 'voxfor-ai-content-summary') . '</p>';
    }
    
    public function render_chatgpt_api_key_field() {
        $settings = get_option('voxfor_ai_summary_settings', []);
        $api_key = $settings['chatgpt_api_key'] ?? '';
        
        printf(
            '<input type="password" name="voxfor_ai_summary_settings[chatgpt_api_key]" id="voxfor_chatgpt_api_key" value="%s" class="regular-text" />',
            esc_attr($api_key)
        );
        echo '<p class="description">' . esc_html__('Enter your ChatGPT API key (starts with sk-). Model: gpt-5.2-chat-latest', 'voxfor-ai-content-summary') . '</p>';
    }
    
    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- This is a standard WordPress Settings API pattern; nonce is verified by settings_fields().
        if (isset($_GET['settings-updated'])) {
            add_settings_error(
                'voxfor_ai_summary_messages',
                'voxfor_ai_summary_message',
                __('Settings saved successfully.', 'voxfor-ai-content-summary'),
                'success'
            );
        }
        
        settings_errors('voxfor_ai_summary_messages');
        
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <div class="voxfor-ai-content-summary-header" style="background: white; padding: 20px; margin: 20px 0; border-left: 4px solid #2271b1;">
                <h2><?php esc_html_e('Voxfor AI Content Summary', 'voxfor-ai-content-summary'); ?></h2>
                <p style="font-size: 14px; line-height: 1.6;">
                    <?php esc_html_e('This plugin generates accurate, faithful summaries of your post content using AI.', 'voxfor-ai-content-summary'); ?>
                </p>
                <p style="font-size: 14px; line-height: 1.6;">
                    <strong><?php esc_html_e('Accuracy-First Approach:', 'voxfor-ai-content-summary'); ?></strong>
                    <?php esc_html_e('Every summary is 100% based on the actual content. No external knowledge, no interpretation, no fabrication.', 'voxfor-ai-content-summary'); ?>
                </p>
            </div>
            
            <form action="options.php" method="post">
                <?php
                settings_fields('voxfor_ai_summary_settings');
                do_settings_sections('voxfor-ai-content-summary');
                submit_button(__('Save Settings', 'voxfor-ai-content-summary'));
                ?>
            </form>
            
            <div class="voxfor-ai-content-summary-info" style="background: #f0f6fc; padding: 20px; margin: 20px 0; border-left: 4px solid #0073aa;">
                <h3><?php esc_html_e('How It Works', 'voxfor-ai-content-summary'); ?></h3>
                <ol style="line-height: 1.8;">
                    <li><?php esc_html_e('Edit any post and find the "AI Content Summary" meta box', 'voxfor-ai-content-summary'); ?></li>
                    <li><?php esc_html_e('Click "Generate Summary" to create an accurate summary', 'voxfor-ai-content-summary'); ?></li>
                    <li><?php esc_html_e('The summary is saved and can be compared with the original content', 'voxfor-ai-content-summary'); ?></li>
                    <li><?php esc_html_e('Summaries are automatically regenerated if content changes', 'voxfor-ai-content-summary'); ?></li>
                </ol>
                
                <h3><?php esc_html_e('What Gets Summarized', 'voxfor-ai-content-summary'); ?></h3>
                <ul style="line-height: 1.8;">
                    <li>✓ <?php esc_html_e('Post title', 'voxfor-ai-content-summary'); ?></li>
                    <li>✓ <?php esc_html_e('Text paragraphs', 'voxfor-ai-content-summary'); ?></li>
                    <li>✓ <?php esc_html_e('Headings (H2, H3, etc.)', 'voxfor-ai-content-summary'); ?></li>
                    <li>✓ <?php esc_html_e('Lists (bullets and numbered)', 'voxfor-ai-content-summary'); ?></li>
                    <li>✓ <?php esc_html_e('Quotes in the text', 'voxfor-ai-content-summary'); ?></li>
                    <li>✓ <?php esc_html_e('Facts, dates, names (if present)', 'voxfor-ai-content-summary'); ?></li>
                </ul>
                
                <h3><?php esc_html_e('What Is Excluded', 'voxfor-ai-content-summary'); ?></h3>
                <ul style="line-height: 1.8;">
                    <li>✗ <?php esc_html_e('HTML/CSS/JavaScript', 'voxfor-ai-content-summary'); ?></li>
                    <li>✗ <?php esc_html_e('Buttons and forms', 'voxfor-ai-content-summary'); ?></li>
                    <li>✗ <?php esc_html_e('Dynamic content', 'voxfor-ai-content-summary'); ?></li>
                    <li>✗ <?php esc_html_e('External knowledge or interpretation', 'voxfor-ai-content-summary'); ?></li>
                </ul>
            </div>
            
            <div class="voxfor-ai-content-summary-stats" style="background: white; padding: 20px; margin: 20px 0;">
                <h3><?php esc_html_e('Statistics', 'voxfor-ai-content-summary'); ?></h3>
                <?php
                $count = Voxfor_Database_Handler::get_summary_count();
                $summary_text = sprintf(
                    // translators: %d: Number of summaries generated.
                    esc_html__('Total summaries generated: %d', 'voxfor-ai-content-summary'),
                    absint($count)
                );
                echo '<p>' . esc_html($summary_text) . '</p>';
                ?>
            </div>
        </div>
        <?php
    }
    
    public function enqueue_admin_assets($hook) {
        // phpcs:disable WordPress.Security.NonceVerification.Recommended -- Only checking page slug for asset enqueuing, no data processing.
        $is_settings_page = (
            $hook === 'settings_page_voxfor-ai-content-summary' || 
            $hook === 'toplevel_page_voxfor-ai-content-summary' ||
            (isset($_GET['page']) && $_GET['page'] === 'voxfor-ai-content-summary')
        );
        // phpcs:enable WordPress.Security.NonceVerification.Recommended
        
        $is_post_page = ($hook === 'post.php' || $hook === 'post-new.php');
        
        if (!$is_settings_page && !$is_post_page) {
            return;
        }
        
        wp_enqueue_style(
            'voxfor-ai-content-summary-admin',
            VOXFOR_AI_SUMMARY_PLUGIN_URL . 'css/admin.css',
            [],
            VOXFOR_AI_SUMMARY_VERSION
        );
        
        wp_enqueue_script(
            'voxfor-ai-content-summary-admin',
            VOXFOR_AI_SUMMARY_PLUGIN_URL . 'js/admin.js',
            ['jquery'],
            VOXFOR_AI_SUMMARY_VERSION,
            true
        );
        
        wp_localize_script('voxfor-ai-content-summary-admin', 'voxforAiSummary', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('voxfor_ai_summary_nonce'),
            'strings' => [
                'generating' => __('Generating summary...', 'voxfor-ai-content-summary'),
                'success' => __('Summary generated successfully!', 'voxfor-ai-content-summary'),
                'error' => __('Error generating summary. Please try again.', 'voxfor-ai-content-summary'),
                'confirm_regenerate' => __('Are you sure you want to regenerate this summary?', 'voxfor-ai-content-summary')
            ]
        ]);
    }
    
    public function render_display_section() {
        echo '<p>' . esc_html__('Control where and how the AI Summary button appears on your site.', 'voxfor-ai-content-summary') . '</p>';
    }
    
    public function render_post_types_field() {
        $settings = get_option('voxfor_ai_summary_settings', []);
        $selected = isset($settings['display_on_post_types']) ? $settings['display_on_post_types'] : ['post'];
        
        $post_types = get_post_types(['public' => true], 'objects');
        
        echo '<fieldset>';
        foreach ($post_types as $post_type) {
            if (in_array($post_type->name, ['attachment'])) continue;
            
            $checked = in_array($post_type->name, $selected) ? 'checked' : '';
            printf(
                '<label><input type="checkbox" name="voxfor_ai_summary_settings[display_on_post_types][]" value="%s" %s /> %s</label><br>',
                esc_attr($post_type->name),
                esc_attr($checked),
                esc_html($post_type->label)
            );
        }
        echo '</fieldset>';
        echo '<p class="description">' . esc_html__('Select which post types should display the AI Summary button.', 'voxfor-ai-content-summary') . '</p>';
    }
    
    public function render_auto_display_field() {
        $settings = get_option('voxfor_ai_summary_settings', []);
        $auto_display = isset($settings['auto_display']) ? $settings['auto_display'] : 'manual';
        ?>
        <fieldset>
            <label>
                <input type="radio" name="voxfor_ai_summary_settings[auto_display]" value="auto" <?php checked($auto_display, 'auto'); ?> />
                <?php esc_html_e('Automatic - Show button on all posts (can be disabled per post)', 'voxfor-ai-content-summary'); ?>
            </label><br>
            <label>
                <input type="radio" name="voxfor_ai_summary_settings[auto_display]" value="manual" <?php checked($auto_display, 'manual'); ?> />
                <?php esc_html_e('Manual - Only show when enabled per post (or use shortcode)', 'voxfor-ai-content-summary'); ?>
            </label>
        </fieldset>
        <p class="description">
            <?php esc_html_e('Automatic: Button appears by default (you can disable it per post in the meta box).', 'voxfor-ai-content-summary'); ?><br>
            <?php esc_html_e('Manual: Button only appears when you enable it per post, or use [voxfor_summary] shortcode.', 'voxfor-ai-content-summary'); ?>
        </p>
        <?php
    }
    
    public function render_button_position_field() {
        $settings = get_option('voxfor_ai_summary_settings', []);
        $position = isset($settings['button_position']) ? $settings['button_position'] : 'before_content';
        ?>
        <select name="voxfor_ai_summary_settings[button_position]">
            <option value="before_content" <?php selected($position, 'before_content'); ?>>
                <?php esc_html_e('Before Content', 'voxfor-ai-content-summary'); ?>
            </option>
            <option value="after_content" <?php selected($position, 'after_content'); ?>>
                <?php esc_html_e('After Content', 'voxfor-ai-content-summary'); ?>
            </option>
            <option value="shortcode_only" <?php selected($position, 'shortcode_only'); ?>>
                <?php esc_html_e('Shortcode Only [voxfor_summary]', 'voxfor-ai-content-summary'); ?>
            </option>
        </select>
        <p class="description">
            <?php esc_html_e('Choose where the button appears automatically. Use "Shortcode Only" for complete manual control.', 'voxfor-ai-content-summary'); ?>
        </p>
        <?php
    }
}
