<?php
/**
 * Video management for Voomo Blog to Video.
 *
 * @package WP_Blog_To_Video
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles publish hooks, REST helpers, and admin columns related to videos.
 */
class WPBTV_Video_Manager {

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_filter( 'manage_post_posts_columns', array( $this, 'add_video_column' ) );
		add_action( 'manage_post_posts_custom_column', array( $this, 'render_video_column' ), 10, 2 );
		add_action( 'transition_post_status', array( $this, 'handle_post_publish' ), 10, 3 );
	}

	/**
	 * Add custom column to Posts list table.
	 *
	 * @param array $columns Existing columns.
	 *
	 * @return array
	 */
	public function add_video_column( $columns ) {
		$columns['wpbtv_video'] = __( 'Voomo', 'voomo-blog-to-video' );
		return $columns;
	}

	/**
	 * Render custom column content.
	 *
	 * @param string $column  Column key.
	 * @param int    $post_id Post ID.
	 *
	 * @return void
	 */
	public function render_video_column( $column, $post_id ) {
		if ( 'wpbtv_video' !== $column ) {
			return;
		}

		$video_url     = get_post_meta( $post_id, 'wpbtv_video_url', true );
		$attachment_id = get_post_meta( $post_id, 'wpbtv_video_attachment_id', true );
		$file_url      = $attachment_id ? wp_get_attachment_url( $attachment_id ) : '';
		$requested     = get_post_meta( $post_id, 'wpbtv_generation_requested', true );
		$error_message = get_post_meta( $post_id, 'wpbtv_generation_error', true );

		if ( $error_message || empty( $requested ) ) {
			echo '&mdash;';
			return;
		}

		if ( empty( $video_url ) || empty( $file_url ) ) {
			echo esc_html__( 'Generating video…', 'voomo-blog-to-video' );
			return;
		}

		printf(
			'<a class="button button-small" href="%1$s" download>%2$s</a>',
			esc_url( $file_url ),
			esc_html__( 'Download Video', 'voomo-blog-to-video' )
		);
	}

	/**
	 * Handle publish events to trigger content creation.
	 *
	 * @param string  $new_status New post status.
	 * @param string  $old_status Old post status.
	 * @param WP_Post $post       Post object.
	 *
	 * @return void
	 */
	public function handle_post_publish( $new_status, $old_status, $post ) {
		if ( 'publish' !== $new_status || 'publish' === $old_status || 'post' !== $post->post_type ) {
			return;
		}

		$post_id  = $post->ID;
		$title    = get_the_title( $post_id );
		$post_url = get_permalink( $post_id );
		$org_id   = get_option( 'wpbtv_organization_id', '' );
		$api_key  = get_option( 'wpbtv_api_key', '' );

		if ( empty( $org_id ) || empty( $api_key ) ) {
			return;
		}

		$api_url = sprintf(
			'%s/organisation/%s/plugin/create-content',
			rtrim( WPBTV_API_BASE, '/' ),
			rawurlencode( $org_id )
		);

		update_post_meta( $post_id, 'wpbtv_generation_requested', 1 );

		$payload = array(
			'post_id'         => (string) $post_id,
			'post_title'      => $title,
			'post_url'        => $post_url,
			'api_key'         => $api_key,
		);

		delete_post_meta( $post_id, 'wpbtv_generation_error' );

		$response = wp_remote_post(
			$api_url,
			array(
				'headers' => array(
					'Content-Type' => 'application/json',
				),
				'body'    => wp_json_encode( $payload ),
				'timeout' => 30,
			)
		);

		if ( is_wp_error( $response ) ) {
			$error_message = $response->get_error_message();
			$this->handle_generation_error( $post_id, $error_message );
			return;
		}

		$response_code = wp_remote_retrieve_response_code( $response );
		$response_body = wp_remote_retrieve_body( $response );

		if ( $response_code >= 400 ) {
			$data    = json_decode( $response_body, true );
			$message = isset( $data['message'] ) ? $data['message'] : '';
			$type    = 'error';

			if ( isset( $data['code'] ) && 'UPGRADE_REQUIRED' === $data['code'] ) {
				$message = sprintf(
					/* translators: %1$s opening anchor tag, %2$s closing anchor tag. */
					__( 'Please upgrade your plan to create a video from this post. Visit %1$sapp.voomo.ai%2$s to upgrade.', 'voomo-blog-to-video' ),
					'<a href="https://app.voomo.ai" target="_blank" rel="noopener noreferrer">',
					'</a>'
				);
				$type = 'warning';
			}

			if ( empty( $message ) ) {
				$message = sprintf(
					/* translators: %d HTTP status code. */
					__( 'Video generation failed with status code %d.', 'voomo-blog-to-video' ),
					$response_code
				);
			}

			$this->handle_generation_error( $post_id, $message, $type );

			return;
		}
	}

	/**
	 * Save a remotely generated video into the media library and post meta.
	 *
	 * @param int    $post_id   Post ID.
	 * @param string $video_url Remote video URL.
	 *
	 * @return array|WP_Error Array containing attachment_id and stored_url or WP_Error.
	 */
	public function save_remote_video( $post_id, $video_url ) {
		$attachment_id = $this->sideload_video_to_media( $post_id, $video_url );

		if ( is_wp_error( $attachment_id ) ) {
			return $attachment_id;
		}

		$stored_url = wp_get_attachment_url( $attachment_id );

		update_post_meta( $post_id, 'wpbtv_video_source_url', $video_url );
		update_post_meta( $post_id, 'wpbtv_video_attachment_id', $attachment_id );
		update_post_meta( $post_id, 'wpbtv_video_url', $stored_url ? $stored_url : $video_url );
		delete_post_meta( $post_id, 'wpbtv_generation_error' );

		if ( get_option( WPBTV_Admin::OPTION_AUTO_EMBED, 1 ) ) {
			$this->maybe_embed_video_in_post( $post_id, $attachment_id );
		}

		return array(
			'attachment_id' => $attachment_id,
			'video_url'     => $stored_url ? $stored_url : $video_url,
		);
	}

	/**
	 * Download the generated video and attach it to the media library.
	 *
	 * @param int    $post_id   Post ID.
	 * @param string $video_url Remote video URL.
	 *
	 * @return int|WP_Error Attachment ID on success, WP_Error on failure.
	 */
	public function sideload_video_to_media( $post_id, $video_url ) {
		require_once ABSPATH . 'wp-admin/includes/file.php';
		require_once ABSPATH . 'wp-admin/includes/media.php';
		require_once ABSPATH . 'wp-admin/includes/image.php';

		$temp_file = download_url( $video_url );

		if ( is_wp_error( $temp_file ) ) {
			return $temp_file;
		}

		$extension = pathinfo( wp_parse_url( $video_url, PHP_URL_PATH ), PATHINFO_EXTENSION );
		$extension = $extension ? $extension : 'mp4';
		$file_name = sprintf( 'voomo-video-%d.%s', $post_id, $extension );

		$file_array             = array();
		$file_array['name']     = $file_name;
		$file_array['tmp_name'] = $temp_file;

		$attachment_id = media_handle_sideload( $file_array, $post_id );

		if ( is_wp_error( $attachment_id ) ) {
			wp_delete_file( $temp_file );
			return $attachment_id;
		}

		return $attachment_id;
	}

	/**
	 * Embed the downloaded video into the post content if not already embedded.
	 *
	 * @param int $post_id       Post ID.
	 * @param int $attachment_id Attachment ID.
	 *
	 * @return void
	 */
	private function maybe_embed_video_in_post( $post_id, $attachment_id ) {
		$post = get_post( $post_id );

		if ( ! $post || 'post' !== $post->post_type ) {
			return;
		}

		$content = $post->post_content;

		if ( false !== strpos( $content, 'wpbtv-video-embed' ) || false !== strpos( $content, '[video ' ) ) {
			return;
		}

		$video_shortcode = sprintf(
			"\n\n[video src=\"%s\"]\n\n",
			esc_url( wp_get_attachment_url( $attachment_id ) )
		);

		$updated_content = $content . $video_shortcode;

		wp_update_post(
			array(
				'ID'           => $post_id,
				'post_content' => $updated_content,
			)
		);
	}

	/**
	 * Queue an admin notice for the current user.
	 *
	 * @param string $message Notice message content.
	 * @param string $type    Notice type (info, warning, error, success).
	 *
	 * @return void
	 */
	private function queue_admin_notice( $message, $type = 'info' ) {
		$key = WPBTV_Admin::get_notice_storage_key();

		if ( empty( $key ) ) {
			return;
		}

		set_transient(
			$key,
			array(
				'message' => $message,
				'type'    => $type,
			),
			5 * MINUTE_IN_SECONDS
		);
	}

	/**
	 * Handle errors returned from the video generation API.
	 *
	 * @param int    $post_id Post ID.
	 * @param string $message Error message to display.
	 * @param string $type    Notice type.
	 *
	 * @return void
	 */
	private function handle_generation_error( $post_id, $message, $type = 'error' ) {
		update_post_meta( $post_id, 'wpbtv_generation_error', $message );
		delete_post_meta( $post_id, 'wpbtv_generation_requested' );
		$this->queue_admin_notice( $message, $type );
	}
}

