<?php
/**
 * REST routes for Voomo Blog to Video.
 *
 * @package WP_Blog_To_Video
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Registers REST API endpoints used by Voomo.
 */
class WPBTV_REST_Controller {

	/**
	 * Video manager instance.
	 *
	 * @var WPBTV_Video_Manager
	 */
	private $video_manager;

	/**
	 * Constructor.
	 *
	 * @param WPBTV_Video_Manager $video_manager Video manager dependency.
	 */
	public function __construct( WPBTV_Video_Manager $video_manager ) {
		$this->video_manager = $video_manager;
		add_action( 'rest_api_init', array( $this, 'register_rest_routes' ) );
	}

	/**
	 * Register REST API routes.
	 *
	 * @return void
	 */
	public function register_rest_routes() {
		register_rest_route(
			'wpbtv/v1',
			'/video',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'handle_video_webhook' ),
				'permission_callback' => array( $this, 'verify_webhook_permission' ),
				'args'                => array(
					'post_id'   => array(
						'required'          => true,
						'type'              => 'integer',
						'sanitize_callback' => 'absint',
						'validate_callback' => function( $param ) {
							return is_numeric( $param ) && $param > 0;
						},
					),
					'video_url' => array(
						'required'          => true,
						'type'              => 'string',
						'sanitize_callback' => 'esc_url_raw',
						'validate_callback' => function( $param ) {
							return ! empty( $param ) && filter_var( $param, FILTER_VALIDATE_URL );
						},
					),
				),
			)
		);
	}

	/**
	 * Verify webhook permission by checking API key.
	 *
	 * @param WP_REST_Request $request Request object.
	 *
	 * @return bool|WP_Error
	 */
	public function verify_webhook_permission( $request ) {
		$api_key_header = $request->get_header( 'X-API-Key' );
		$saved_api_key  = get_option( 'wpbtv_api_key', '' );

		if ( empty( $saved_api_key ) ) {
			return new WP_Error(
				'no_api_key_configured',
				__( 'API key not configured.', 'voomo-blog-to-video' ),
				array( 'status' => 500 )
			);
		}

		if ( empty( $api_key_header ) || $api_key_header !== $saved_api_key ) {
			return new WP_Error(
				'invalid_api_key',
				__( 'Invalid API key.', 'voomo-blog-to-video' ),
				array( 'status' => 401 )
			);
		}

		return true;
	}

	/**
	 * Handle video webhook callback.
	 *
	 * @param WP_REST_Request $request Request object.
	 *
	 * @return WP_REST_Response|WP_Error
	 */
	public function handle_video_webhook( $request ) {
		$post_id   = $request->get_param( 'post_id' );
		$video_url = $request->get_param( 'video_url' );

		$post = get_post( $post_id );
		if ( ! $post ) {
			return new WP_Error(
				'post_not_found',
				__( 'Post not found.', 'voomo-blog-to-video' ),
				array( 'status' => 404 )
			);
		}

		$result = $this->video_manager->save_remote_video( $post_id, $video_url );

		if ( is_wp_error( $result ) ) {
			return new WP_Error(
				'video_download_failed',
				$result->get_error_message(),
				array( 'status' => 500 )
			);
		}

		return new WP_REST_Response(
			array(
				'success'       => true,
				'message'       => __( 'Video saved to media library.', 'voomo-blog-to-video' ),
				'post_id'       => $post_id,
				'attachment_id' => $result['attachment_id'],
				'video_url'     => $result['video_url'],
			),
			200
		);
	}
}

