<?php
/**
 * Admin-facing functionality for Voomo Blog to Video.
 *
 * @package WP_Blog_To_Video
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles the WordPress admin UI for the plugin.
 */
class WPBTV_Admin {

	/**
	 * Option name for auto embed toggle.
	 */
	const OPTION_AUTO_EMBED = 'wpbtv_auto_embed';

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_action( 'admin_menu', array( $this, 'register_admin_menu' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
		add_action( 'admin_init', array( $this, 'handle_settings_form' ) );
		add_action( 'admin_notices', array( $this, 'render_generation_notice' ) );
	}

	/**
	 * Return the storage key used for admin notices.
	 *
	 * @return string
	 */
	public static function get_notice_storage_key() {
		return 'wpbtv_notice_global';
	}

	/**
	 * Register the plugin admin menu.
	 *
	 * @return void
	 */
	public function register_admin_menu() {
		add_menu_page(
			__( 'Voomo', 'voomo-blog-to-video' ),
			__( 'Voomo', 'voomo-blog-to-video' ),
			'manage_options',
			'voomo-blog-to-video',
			array( $this, 'render_admin_page' ),
			WPBTV_PLUGIN_URL . 'assets/icon-20x20.png',
			30
		);
	}

	/**
	 * Render the settings page.
	 *
	 * @return void
	 */
	public function render_admin_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'voomo-blog-to-video' ) );
		}

		$organization_id  = get_option( 'wpbtv_organization_id', '' );
		$api_key          = get_option( 'wpbtv_api_key', '' );
		$auto_embed       = (bool) get_option( self::OPTION_AUTO_EMBED, true );
		$workspace_base   = WPBTV_APP_BASE;
		$workspace_url    = ! empty( $organization_id ) ? sprintf( '%s/dashboard/organisation/%s/settings?page=workspace', rtrim( $workspace_base, '/' ), rawurlencode( $organization_id ) ) : '';

		settings_errors( 'wpbtv_settings' );
		?>
		<div class="wrap">
			<h1><?php esc_html_e( 'Voomo', 'voomo-blog-to-video' ); ?></h1>
			<p><?php esc_html_e( 'Following settings are required to enable the plugin.', 'voomo-blog-to-video' ); ?></p>
		<p>
			<?php
			printf(
				/* translators: %1$s: Opening anchor tag, %2$s: Closing anchor tag */
				esc_html__( 'You will find these settings in your Voomo account. In case you have not created an account yet, %1$sclick here to signup%2$s.', 'voomo-blog-to-video' ),
				'<a href="https://app.voomo.ai/signup" target="_blank" rel="noopener noreferrer">',
				'</a>'
			);
			?>
		</p>
			<?php if ( $workspace_url ) : ?>
				<p>
					<a href="<?php echo esc_url( $workspace_url ); ?>" target="_blank" rel="noopener noreferrer">
						<?php esc_html_e( 'Open your Voomo workspace settings to change how videos are generated.', 'voomo-blog-to-video' ); ?>
					</a>
				</p>
			<?php else : ?>
				<p><?php esc_html_e( 'Need to change how videos are generated? Log into app.voomo.ai, open your workspace, and visit the workspace settings page.', 'voomo-blog-to-video' ); ?></p>
			<?php endif; ?>
			<form method="post" action="">
				<?php wp_nonce_field( 'wpbtv_settings_action', 'wpbtv_settings_nonce' ); ?>
				<table class="form-table" role="presentation">
					<tbody>
						<tr>
							<th scope="row">
								<label for="wpbtv_organization_id"><?php esc_html_e( 'Organization ID', 'voomo-blog-to-video' ); ?></label>
							</th>
							<td>
								<input type="text" id="wpbtv_organization_id" name="wpbtv_organization_id" value="<?php echo esc_attr( $organization_id ); ?>" class="regular-text" />
							</td>
						</tr>
						<tr>
							<th scope="row">
								<label for="wpbtv_api_key"><?php esc_html_e( 'API Key', 'voomo-blog-to-video' ); ?></label>
							</th>
							<td>
								<input type="password" id="wpbtv_api_key" name="wpbtv_api_key" value="<?php echo esc_attr( $api_key ); ?>" class="regular-text" />
							</td>
						</tr>
						<tr>
							<th scope="row">
								<label for="wpbtv_auto_embed"><?php esc_html_e( 'Embed video automatically', 'voomo-blog-to-video' ); ?></label>
							</th>
							<td>
								<label for="wpbtv_auto_embed">
									<input type="checkbox" id="wpbtv_auto_embed" name="wpbtv_auto_embed" value="1" <?php checked( $auto_embed ); ?> />
									<?php esc_html_e( 'Insert the downloaded video into post content automatically.', 'voomo-blog-to-video' ); ?>
								</label>
							</td>
						</tr>
					</tbody>
				</table>
				<?php submit_button( __( 'Save Changes', 'voomo-blog-to-video' ), 'primary', 'wpbtv_settings_submit' ); ?>
			</form>
		</div>
		<?php
	}

	/**
	 * Enqueue admin assets.
	 *
	 * @param string $hook Current admin page hook.
	 *
	 * @return void
	 */
	public function enqueue_admin_assets( $hook ) {
		if ( 'toplevel_page_voomo-blog-to-video' !== $hook ) {
			return;
		}

		wp_enqueue_style(
			'voomo-blog-to-video-admin',
			WPBTV_PLUGIN_URL . 'public/css/admin.css',
			array(),
			WPBTV_VERSION
		);
	}

	/**
	 * Output queued admin notices from background processes.
	 *
	 * @return void
	 */
	public function render_generation_notice() {
		$key = self::get_notice_storage_key();

		if ( empty( $key ) ) {
			return;
		}

		$notice = get_transient( $key );

		if ( empty( $notice ) || empty( $notice['message'] ) ) {
			return;
		}

		$type = isset( $notice['type'] ) ? $notice['type'] : 'info';

		printf(
			'<div class="notice notice-%1$s is-dismissible"><p>%2$s</p></div>',
			esc_attr( $type ),
			wp_kses_post( $notice['message'] )
		);
	}

	/**
	 * Handle settings form submission.
	 *
	 * @return void
	 */
	public function handle_settings_form() {
		if ( ! isset( $_POST['wpbtv_settings_submit'] ) ) {
			return;
		}

		if ( ! isset( $_POST['wpbtv_settings_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['wpbtv_settings_nonce'] ) ), 'wpbtv_settings_action' ) ) {
			wp_die( esc_html__( 'Security check failed.', 'voomo-blog-to-video' ) );
		}

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'voomo-blog-to-video' ) );
		}

		$organization_id = isset( $_POST['wpbtv_organization_id'] ) ? sanitize_text_field( wp_unslash( $_POST['wpbtv_organization_id'] ) ) : '';
		$api_key         = isset( $_POST['wpbtv_api_key'] ) ? sanitize_text_field( wp_unslash( $_POST['wpbtv_api_key'] ) ) : '';
		$auto_embed      = isset( $_POST['wpbtv_auto_embed'] );

		if ( empty( $organization_id ) || empty( $api_key ) ) {
			add_settings_error(
				'wpbtv_settings',
				'wpbtv_settings_missing_fields',
				__( 'Organization ID and API Key are required.', 'voomo-blog-to-video' ),
				'error'
			);
			return;
		}

		$verification_result = $this->verify_api_credentials( $organization_id, $api_key );

		if ( $verification_result['success'] ) {
			update_option( 'wpbtv_organization_id', $organization_id );
			update_option( 'wpbtv_api_key', $api_key );
			update_option( self::OPTION_AUTO_EMBED, $auto_embed ? 1 : 0 );

			add_settings_error(
				'wpbtv_settings',
				'wpbtv_settings_success',
				__( 'Settings saved successfully!', 'voomo-blog-to-video' ),
				'success'
			);
		} else {
			$error_message = isset( $verification_result['message'] ) ? $verification_result['message'] : __( 'API verification failed. Please check your credentials.', 'voomo-blog-to-video' );
			add_settings_error(
				'wpbtv_settings',
				'wpbtv_settings_error',
				$error_message,
				'error'
			);
		}
	}

	/**
	 * Verify API credentials with the remote service.
	 *
	 * @param string $organization_id Organization ID.
	 * @param string $api_key         API key.
	 *
	 * @return array
	 */
	private function verify_api_credentials( $organization_id, $api_key ) {
		$api_url = sprintf(
			'%s/organisation/%s/plugin/verify',
			rtrim( WPBTV_API_BASE, '/' ),
			rawurlencode( $organization_id )
		);

		$args = array(
			'method'  => 'POST',
			'timeout' => 30,
			'headers' => array(
				'Content-Type' => 'application/json',
			),
			'body'    => wp_json_encode(
				array(
					'api_key' => $api_key,
				)
			),
		);

		$response = wp_remote_request( $api_url, $args );

		if ( is_wp_error( $response ) ) {
			return array(
				'success' => false,
				'message' => $response->get_error_message(),
			);
		}

		$response_code = wp_remote_retrieve_response_code( $response );
		$response_body = wp_remote_retrieve_body( $response );
		$data          = json_decode( $response_body, true );
		$message       = isset( $data['message'] ) ? $data['message'] : '';

		if ( 200 === $response_code && isset( $data['success'] ) && true === $data['success'] ) {
			return array(
				'success' => true,
				'message' => $message ? $message : __( 'Credentials verified successfully.', 'voomo-blog-to-video' ),
			);
		}

		return array(
			'success' => false,
			'message' => $message ? $message : sprintf(
				/* translators: %d: HTTP response code */
				__( 'API verification failed with status code: %d', 'voomo-blog-to-video' ),
				$response_code
			),
		);
	}
}

