document.addEventListener('DOMContentLoaded', function () {

    try {
        const calendarDays = document.getElementById('calendarDays');
        const monthYear = document.getElementById('monthYear');
        const prevMonthButton = document.getElementById('prevMonth');
        const nextMonthButton = document.getElementById('nextMonth');
        const timeBlocks = document.querySelectorAll('.time-block a');
        const bookNowBtn = document.querySelector('.btn-book-now');
        const popup = document.getElementById('bookingPopup');
        const vooduxBookingCalendar = document.getElementById('voodux-booking-calendar');
        const closeBtn = document.querySelector('.popup .close');
        const errorMessage = document.getElementById('errorMessage');
        const bookingFormFieldsMessage = document.getElementById('bookingFormFieldsMessage');
        const bookingDateInput = document.getElementById('bookingDate');
        const bookingTimeInput = document.getElementById('bookingTime');
        const bookingsPerHourDiv = document.querySelector('.book-per-hour');
        let currentMonth = new Date().getMonth();
        let currentYear = new Date().getFullYear();
        let selectedDate = null;
        let selectedTime = null;
        let selectedPrice = null;

        // settings
        const bookings = bookingData.bookings;        
        const bookingPerHour = bookPerHour;
        const firstDayOfWeek = firstDayOfWeekChoice;
        const scrollToCalendarValue = scrollToCalendar;
        const bookingFromTimesValues = bookingFromTimes;
        const bookingUntillTimesValues = bookingUntillTimes;

        // Global setting: scroll calendar in view.
        if (scrollToCalendarValue == "yes") vooduxBookingCalendar.scrollIntoView({ behavior: 'smooth' });

        const availabilityPrices = firstDayOfWeek === "sunday" ? [
            availabilitySunday,
            availabilityMonday,
            availabilityTuesday,
            availabilityWednesday,
            availabilityThursday,
            availabilityFriday,
            availabilitySaterday
        ] : [
            availabilityMonday,
            availabilityTuesday,
            availabilityWednesday,
            availabilityThursday,
            availabilityFriday,
            availabilitySaterday,
            availabilitySunday
        ];

        const unavailableDateRanges = [
            [unavailableFromDate1, unavailableUntillDate1],
            [unavailableFromDate2, unavailableUntillDate2],
            [unavailableFromDate3, unavailableUntillDate3],
            [unavailableFromDate4, unavailableUntillDate4],
            [unavailableFromDate5, unavailableUntillDate5]
        ];        

        /**
         * Checks if a given date falls within any of the unavailable date ranges.
         *
         * This function loops through the defined unavailable date ranges and checks if the given date 
         * falls within any of these ranges. If the date is unavailable, it returns `true`, otherwise `false`.
         *
         * @param {Date} date - The date to check for availability.
         * @returns {boolean} - Returns `true` if the date is within an unavailable range, `false` otherwise.
         */
        function isDateUnavailable(date) {

            try {
                return unavailableDateRanges.some(range => {
                    const [start, end] = range.map(d => new Date(d).setHours(0, 0, 0, 0)); // Normalize time to start of the day
                    const checkDate = date.setHours(0, 0, 0, 0);
                    return checkDate >= start && checkDate <= end;
                });
            } catch (error) {
                console.error('Error checking date availability:', error);
                return false; // Default to not unavailable in case of error
            }

        }

        /**
         * Renders a calendar for the given month and year, marking days as unavailable, past, or selectable.
         *
         * This function generates the calendar grid for the specified month and year. It marks days as 
         * unavailable based on booking availability, past days, and days that are fully booked. It also 
         * handles day selection, updating the booking date and showing time slots available for the selected date.
         *
         * @param {number} month - The month to render (0-11, where 0 is January and 11 is December).
         * @param {number} year - The year to render.
         */
        function renderCalendar(month, year) {

            try {
                calendarDays.innerHTML = '';
                const firstDay = new Date(year, month, 1).getDay();
                const lastDay = new Date(year, month + 1, 0).getDate();
                const today = new Date();
                const todayUTC = new Date(Date.UTC(today.getFullYear(), today.getMonth(), today.getDate()));
                const tomorrow = new Date(todayUTC);
                tomorrow.setDate(todayUTC.getDate() + 1);
                tomorrow.setHours(0, 0, 0, 0); // Set time to start of the day

                const firstDayOffset = (firstDayOfWeek === 'monday') ? ((firstDay === 0) ? 6 : firstDay - 1) : firstDay;

                for (let i = 0; i < firstDayOffset; i++) {
                    const day = document.createElement('div');
                    day.className = 'calendar-day empty';
                    calendarDays.appendChild(day);
                }

                for (let i = 1; i <= lastDay; i++) {
                    const day = document.createElement('div');
                    day.className = 'calendar-day';
                    const date = new Date(year, month, i);                    

                    // Determine the first day of the week.
                    const rawDayOfWeek = date.getDay();
                    const dayOfWeek = firstDayOfWeek != "sunday" ? (rawDayOfWeek + 6) % 7 : rawDayOfWeek;
                    day.textContent = i;
                    day.setAttribute('data-date', `${year}-${month + 1}-${i}`);

                    if (availabilityPrices[dayOfWeek] === "" || isDateUnavailable(date) || (bookingPerHour !== 'yes' && dayHasBooking(date)) || isFullyBooked(date, bookingFromTimesValues[dayOfWeek], bookingUntillTimesValues[dayOfWeek]) ) {
                        // Mark the day as unavailable
                        day.classList.add('unavailable-day');                        
                    } else if (date < tomorrow) {
                        day.classList.add('past-day');
                    } else {
                        day.addEventListener('click', function () {
                            try {
                                const bookingConfirmation = document.querySelector('.booking-confirmation');
                                if (bookingConfirmation) { bookingConfirmation.style.display = 'none'; }

                                document.querySelectorAll('.calendar-day').forEach(d => d.classList.remove('selected'));
                                if (!day.classList.contains('past-day')) {
                                    day.classList.add('selected');
                                    selectedDate = day.getAttribute('data-date');
                                    selectedTime = null;
                                    bookingDateInput.value = selectedDate;
                                    bookingTimeInput.value = '';
                                    errorMessage.style.display = 'none';
                                    bookingFormFieldsMessage.innerHTML = '';

                                    document.querySelectorAll('.time-block').forEach(el => el.style.display = 'none');
                                    timeBlocks.forEach(b => b.parentElement.classList.remove('selected'));

                                    const selectedDateObj = new Date(selectedDate);                                    
                                    const dayOfWeekLetters = selectedDateObj.toLocaleDateString("en-GB", { weekday: 'short' }).toLowerCase();

                                    document.querySelectorAll('.select-date').forEach(el => el.style.display = 'none');
                                    document.querySelectorAll(`.time-picker .${dayOfWeekLetters}`).forEach(el => el.style.display = 'flex');
                                    document.querySelector('.total-price-container').style.display = 'flex';
                                    let formattedPrice = parseFloat(availabilityPrices[dayOfWeek]).toLocaleString("en-GB", { minimumFractionDigits: 2, maximumFractionDigits: 2 });
                                    selectedPrice = formattedPrice;
                                    document.querySelector('.total-price-amount').textContent = selectedPrice;

                                    disableBookedTimes(selectedDate);
                                }
                            } catch (error) {
                                console.error('Error during day selection:', error);
                            }
                        });
                    }

                    if (date.getDate() === todayUTC.getDate() && date.getMonth() === todayUTC.getMonth() && date.getFullYear() === todayUTC.getFullYear()) {
                        day.classList.add('today');
                    }

                    calendarDays.appendChild(day);
                }

                monthYear.textContent = `${monthNames[month]} ${year}`;
            } catch (error) {
                console.error('Error rendering calendar:', error);
            }

        }

        /**
         * Checks if there is at least one booking on the given date.
         *
         * This function takes a date and checks the list of bookings to see if there is at least one
         * booking that matches the provided date. The dates are compared in the 'YYYY-MM-DD' format.
         *
         * @param {Date} date - The target date to check for bookings.
         * @returns {boolean} - Returns true if there is at least one booking on the given date, otherwise false.
         */
        function dayHasBooking(date) {
            try {
                // Format the date to YYYY-MM-DD in WordPress local time
                const dateStr = wp.date.dateI18n('Y-m-d', date); 
        
                return bookings.some(booking => {
                    const bookingDate = new Date(booking.date);
                    const bookingDateStr = wp.date.dateI18n('Y-m-d', bookingDate);
                            
                    return bookingDateStr === dateStr;
                });
            } catch (error) {
                console.error('Error checking if date has booking:', error);
                return false; // Default to not booked in case of error
            }
        }        

        /**
         * Checks if all booking times between a specified start and end time are fully booked on a given date.
         *
         * This function takes a date, a start time, and an end time, and verifies whether all available time slots
         * between the start and end times are booked for that day. It compares the date and time with the existing bookings
         * and returns true if all times are booked, otherwise false.
         *
         * @param {Date} date - The target date to check for bookings.
         * @param {number} startTime - The start hour (in 24-hour format) for the time range.
         * @param {number} endTime - The end hour (in 24-hour format) for the time range.
         * @returns {boolean} - Returns true if all time slots between the start and end times are booked, otherwise false.
         */
        function isFullyBooked(date, startTime, endTime) {

            try {        
                // Convert the input date to a string format for comparison (YYYY-MM-DD)
                const year = date.getFullYear();
                const month = String(date.getMonth() + 1).padStart(2, '0'); // Months are zero-indexed, so we add 1
                const day = String(date.getDate()).padStart(2, '0');
                const targetDate = `${year}-${month}-${day}`;

                // Filter bookings to get only the ones that match the target date
                const bookingsForDate = bookings.filter(booking => {
                    try {
                        // Manually split and normalize the booking date string (assuming booking.date is in 'YYYY-M-D' format)
                        const [year, month, day] = booking.date.split('-').map(num => num.padStart(2, '0')); // Normalize month/day
                        const bookingDate = `${year}-${month}-${day}`;                        

                        // Return true if the booking date matches the target date and the booking hour is within the required range                        
                        return bookingDate === targetDate;
                    } catch (error) {
                        console.error("Error processing booking time:", booking, error);
                        return false;
                    }
                });                

                const countHours = Math.abs(startTime - endTime) + 1;                
                return countHours <= bookingsForDate.length;
            } catch (error) {
                console.error("An error occurred while checking bookings:", error);
                return false; // Default to false if an error occurs
            }

        }

        /**
         * Disables time blocks that are already booked for the selected date.
         *
         * This function takes a selected date, filters out the bookings that match the selected date,
         * and disables time blocks that are booked by adding a 'booked' class to their parent elements.
         * If a time block is not booked, it ensures the 'booked' class is removed.
         *
         * @param {string} selectedDate - The date for which to check and disable booked time slots.
         */
        function disableBookedTimes(selectedDate) {

            try {
                const bookedTimes = bookings.filter(booking => booking.date === selectedDate).map(booking => booking.time);

                timeBlocks.forEach(block => {
                    const time = block.getAttribute('data-value');
                    if (bookedTimes.includes(time)) {
                        block.parentElement.classList.add('booked');
                        block.parentElement.classList.remove('selected');
                    } else {
                        block.parentElement.classList.remove('booked');
                    }
                });
            } catch (error) {
                console.error('Error disabling booked times:', error);
            }
        }

        const monthNames = [bookingTranslations.January, bookingTranslations.February, bookingTranslations.March, bookingTranslations.April, bookingTranslations.May, bookingTranslations.June, bookingTranslations.July, bookingTranslations.August, bookingTranslations.September, bookingTranslations.October, bookingTranslations.November, bookingTranslations.December];

        prevMonthButton.addEventListener('click', function () {
            try {
                currentMonth--;
                if (currentMonth < 0) {
                    currentMonth = 11;
                    currentYear--;
                }
                renderCalendar(currentMonth, currentYear);
            } catch (error) {
                console.error('Error navigating to previous month:', error);
            }
        });

        nextMonthButton.addEventListener('click', function () {
            try {
                currentMonth++;
                if (currentMonth > 11) {
                    currentMonth = 0;
                    currentYear++;
                }
                renderCalendar(currentMonth, currentYear);
            } catch (error) {
                console.error('Error navigating to next month:', error);
            }
        });

        renderCalendar(currentMonth, currentYear);

        timeBlocks.forEach(block => {
            block.addEventListener('click', function (event) {
                try {
                    event.preventDefault();
                    const bookingConfirmation = document.querySelector('.booking-confirmation');
                    if (bookingConfirmation) { bookingConfirmation.style.display = 'none'; }
                    selectedTime = block.getAttribute('data-value');
                    bookingTimeInput.value = selectedTime;
                    timeBlocks.forEach(b => b.parentElement.classList.remove('selected'));
                    block.parentElement.classList.add('selected');
                    errorMessage.style.display = 'none';
                    bookingFormFieldsMessage.innerHTML = '';
                } catch (error) {
                    console.error('Error during time block selection:', error);
                }
            });
        });

        bookNowBtn.addEventListener('click', function (event) {
            try {
                event.preventDefault();
                if ((selectedDate && selectedTime) || bookingPerHour != 'yes') {

                    const time = bookingPerHour != 'yes' ? 'Whole day' : `${new Date(selectedTime).toLocaleTimeString(bookingTranslations.Language_code, {
                        hour: '2-digit',
                        minute: '2-digit',
                        hour12: bookingTranslations.hour12,
                        timeZone: 'UTC'
                    })}`;
                    
                    const dateString = selectedDate;
                    const date = new Date(dateString);

                    // Convert to "29 September 2024"
                    const formattedDate = date.toLocaleDateString(bookingTranslations.Language_code, {
                        day: 'numeric',
                        month: 'long',
                        year: 'numeric'
                    });                    

                    document.getElementById('bookingDetails').innerHTML = `<b>${bookingTranslations.Date}:</b> ${formattedDate}<br> <b>${bookingTranslations.Time}:</b> ${time}<br> <b>${bookingTranslations.Price}:</b> € ${selectedPrice}`;
                    popup.style.display = 'block';
                } else {
                    errorMessage.textContent = bookingTranslations.ErrorMessageDateTime;
                    errorMessage.style.display = 'contents';
                    bookingFormFieldsMessage.innerHTML = '';
                    const bookingConfirmation = document.querySelector('.booking-confirmation');
                    if (bookingConfirmation) { bookingConfirmation.style.display = 'none'; }
                }
            } catch (error) {
                console.error('Error during booking process:', error);
            }
        });

        closeBtn.addEventListener('click', function () {
            try {
                popup.style.display = 'none';
            } catch (error) {
                console.error('Error closing popup:', error);
            }
        });

        window.addEventListener('click', function (event) {
            try {
                if (event.target === popup) {
                    popup.style.display = 'none';
                }
            } catch (error) {
                console.error('Error handling window click event:', error);
            }
        });

        // Check the value of bookPerHour and display/hide the book-per-hour div
        try {
            if (bookingPerHour == 'yes') {
                bookingsPerHourDiv.style.display = 'block';
            } else {
                bookingsPerHourDiv.style.display = 'none';
            }
        } catch (error) {
            console.error('Error handling book-per-hour visibility:', error);
        }

    } catch (error) {

        console.error('Error initializing DOM content:', error);

    }
});

function disableSubmitButton() {

    document.getElementById('submitButton').disabled = true;

}