<?php

/**
 * Responsible for validating the booking form fields.
 * 
 *
 * @link       http://example.com
 * @since      1.0.0
 *
 * @package    Voodux_Bookings
 * @subpackage Voodux_Bookings/includes
 */

/**
 * Responsible for validating the booking form fields.
 * 
 *
 * @since      1.0.0
 * @package    Voodux_Bookings
 * @subpackage Voodux_Bookings/includes
 * @author     Paul Visser
 */

defined('ABSPATH') || exit(); // Exit if accessed directly.

if (!class_exists('vdx_bks_voodux_booking_form_validator')) {

    /**
     * Booking Product Custom Block class.
     */
    class vdx_bks_voodux_booking_form_validator {

        /**
		 * Validate the booking form fields.
		 *
		 * This function ensures that the necessary booking details are provided based on the booking type.
		 * If the booking is per hour, both the date and time are required. If the booking is for a full day,
		 * only the date is required. The email field is always required.
		 *
		 * @param string $booking_per_hour Indicates whether the booking is per hour ('yes') or for a full day (any other value).
		 * @param array  $booking_fields Booking form fields.
		 * @param string $booking_date The date of the booking.
		 * @param string $booking_time The time of the booking (required if booking per hour).
		 * @return array Returns araay with first element true/false (form validation) and second element the error message.
		 */
		function vdx_bks_validate_booking_fields($booking_per_hour, $booking_fields, $booking_date, $booking_time) {

			$results = [true, ''];  // Default: No errors, valid.

			// Checks if the date and time have valid formats.
			if (!$this->vdx_bks_is_valid_date($booking_date)) {
				$results = [false, esc_html(__("The&nbsp;<i>booking date</i>&nbsp;is not in the right format.", "voodux-bookings"))];
			} else if (!$this->vdx_bks_is_valid_dateTime($booking_time) && $booking_per_hour == "yes") {
				$results = [false, esc_html(__("The&nbsp;<i>booking time</i>&nbsp;is not in the right format.", "voodux-bookings"))];
			}        
		
			// Checks if any booking field's value is empty or if the email is invalid.
			foreach ($booking_fields as $field) {
				if (empty($field['value'])) {
					$results = [false, esc_html("Please provide your&nbsp;<i>" . $field['name'] . "</i>&nbsp;it is required to proceed.", "voodux-bookings")];
					break;
				} else if ($field['name'] === 'email address' && !filter_var($field['value'], FILTER_VALIDATE_EMAIL)) {
					$results = [false, esc_html("The&nbsp;<i>" . $field['name'] . "</i>&nbsp;you entered is not valid.", "voodux-bookings")];
					break;
				}
			}
		
			// Check booking-specific requirements if no errors have been found so far.
			if ($results[0]) {
				if ($booking_per_hour === "yes") {
					if (empty($booking_date) || empty($booking_time)) {
						$results = [false, esc_html(__("Both the &nbsp;<i>booking date and time</i>&nbsp;must be specified.", "voodux-bookings"))];
					} else {
						$orders = $this->vdx_bks_vdx_bks_get_orders_by_date_and_time($booking_date, $booking_time);
						if ( !empty($orders) ) {
							$booking_date_formatted = gmdate("j F Y", strtotime($booking_date));
							$booking_time_formatted = gmdate("H:i", strtotime($booking_time));                        
														
							$results = [false, ""];
						}
					}
				} else {
					if (empty($booking_date)) {
						$results = [false, esc_html(__("The &nbsp;<i>booking date</i>&nbsp;is not been specified.", "voodux-bookings"))];
					} else {
						$orders = $this->vdx_bks_get_orders_by_date($booking_date);
						if ( !empty($orders) ) {
							$results = [false, esc_html(__("The selected&nbsp;<i>booking date</i>&nbsp;is taken, please choose another date.", "voodux-bookings"))];
						}
					}
				}
			}
		
			return $results;

		}
		
		/**
		 * Validates if the given date string is in a valid format (Y-n-j).
		 *
		 * This function handles dates where the month and day can be single-digit values.
		 * It ensures the day is within the valid range for the given month and year.
		 *
		 * @param string $date The date string in 'Y-n-j' format (e.g., '2024-8-27').
		 * @return bool Returns true if the date is valid and properly formatted, otherwise returns false.
		 */
		function vdx_bks_is_valid_date($date) {
			
			// Create a DateTime object from the date string using a loose format ('!Y-n-j')
			$d = DateTime::createFromFormat('!Y-n-j', $date);

			// Check if the DateTime object is created
			if ($d === false) {
				return false;
			}

			// Validate that the normalized date matches the original input
			$normalizedDate = $d->format('Y-n-j');

			// Create another DateTime object to check for invalid days
			$checkDate = DateTime::createFromFormat('Y-n-j', $normalizedDate);
			if ($checkDate === false || $checkDate->format('Y-n-j') !== $normalizedDate) {
				return false;
			}

			// Ensure that the date is valid and check for any edge cases
			$dateParts = explode('-', $date);
			$year = (int)$dateParts[0];
			$month = (int)$dateParts[1];
			$day = (int)$dateParts[2];

			// Validate month and day ranges
			if ($month < 1 || $month > 12) {
				return false;
			}

			// Get the maximum number of days in the month
			$maxDays = cal_days_in_month(CAL_GREGORIAN, $month, $year);
			if ($day < 1 || $day > $maxDays) {
				return false;
			}

			return true;
		}

		/**
		 * Validates if the given datetime string is in a valid ISO 8601 format.
		 *
		 * @param string $datetime The ISO 8601 datetime string.
		 * @return bool Returns true if the datetime is valid and properly formatted, otherwise returns false.
		 */
		function vdx_bks_is_valid_dateTime($datetime) {
			
			// Create a DateTime object from the datetime string
			$d = DateTime::createFromFormat(DateTime::ISO8601, $datetime);
			
			// If DateTime creation failed, return false
			if ($d === false) {
				return false;
			}
		
			// Check if the formatted DateTime matches the original input
			// Convert the DateTime object to the format with +0000 time zone offset
			$normalizedDatetime = $d->format('Y-m-d\TH:i:sO');
			return $normalizedDatetime === $datetime;
		}

		/**
		 * Retrieve all WooCommerce orders with a specific booking date and time.
		 *
		 * This function fetches all orders that match a given `booking_date` and `booking_time`.
		 * The query is performed using meta fields stored in the order data.
		 *
		 * @param string $booking_date The booking date to filter by (format: 'YYYY-MM-DD').
		 * @param string $booking_time The booking time to filter by (format: 'YYYY-MM-DDTHH:MM:SS+0000').
		 * @return array An array of WC_Order objects that match the criteria.
		 */
		function vdx_bks_vdx_bks_get_orders_by_date_and_time($booking_date, $booking_time) {

			$args = [
				'status' => 'pending',
				'limit' => -1,
				'meta_query' => [
					'relation' => 'AND',
					[
						'key' => 'booking_date',
						'value' => $booking_date,
						'compare' => '='
					],
					[
						'key' => 'booking_time',
						'value' => $booking_time,
						'compare' => '='
					]
				]
			];

			$query = new WC_Order_Query($args);
			return $query->get_orders();

		}

		/**
		 * Retrieve all WooCommerce orders with a specific booking date, ignoring booking time.
		 *
		 * This function fetches all orders that match a given `booking_date`. It does not take
		 * into account the `booking_time` field.
		 *
		 * @param string $booking_date The booking date to filter by (format: 'YYYY-MM-DD').
		 * @return array An array of WC_Order objects that match the criteria.
		 */
		function vdx_bks_get_orders_by_date($booking_date) {

			$args = [
				'status' => 'pending',
				'limit' => -1,
				'meta_query' => [
					[
						'key' => 'booking_date',
						'value' => $booking_date,
						'compare' => '='
					]
				]
			];

			$query = new WC_Order_Query($args);
			return $query->get_orders();

		}
    }

}

new vdx_bks_voodux_booking_form_validator();
