<?php

/**
 * Responsible for creating a custom WordPress block that displays the booking calendar.
 * 
 * Extends WooCommerce functionality by creating a custom WordPress block 
 * that displays the booking calendar.
 *
 * @link       http://example.com
 * @since      1.0.0
 *
 * @package    Voodux_Bookings
 * @subpackage Voodux_Bookings/includes
 */

/**
 * Responsible for creating a custom WordPress block that displays the booking calendar.
 * 
 * Extends WooCommerce functionality by creating a custom WordPress block 
 * that displays the booking calendar.
 *
 * @since      1.0.0
 * @package    Voodux_Bookings
 * @subpackage Voodux_Bookings/includes
 * @author     Paul Visser
 */

defined('ABSPATH') || exit(); // Exit if accessed directly.

if (!class_exists('vdx_bks_Booking_Custom_Block')) {

    /**
     * Booking Product Custom Block class.
     */
    class vdx_bks_Booking_Custom_Block {

        /**
         * Constructor.
         */
        public function __construct() {

            add_action('enqueue_block_editor_assets', array($this, 'vdx_bks_enqueue_calendar_php_block_assets'));
            add_action('enqueue_block_assets', array($this, 'vdx_bks_enqueue_calendar_block_styles'));
            add_action('init', array($this, 'vdx_bks_register_bookings_calendar_block'));
            add_action('wp_enqueue_scripts', array($this, 'vdx_bks_enqueue_calendar_block_scripts_for_booking_product'));
            add_action('wp_enqueue_scripts', array($this, 'vdx_bks_enqueue_calendar_block_styles_for_booking_product'));
            add_action('wp_enqueue_scripts', array($this, 'vdx_bks_enqueue_bookings_to_calendar_js'));
            add_action('wp', array($this, 'vdx_bks_set_global_product_for_booking'));
            add_action('woocommerce_admin_order_data_after_billing_address', array($this, 'vdx_bks_display_custom_order_fields') );            
            add_filter('is_protected_meta', array($this, 'vdx_bks_hide_woocommercer_order_custom_fields'), 10, 2);            

        }

        /**
         * Enqueues the block editor JavaScript file and localizes the PHP template URL.
         */
        public function vdx_bks_enqueue_calendar_php_block_assets() {

            // Enqueue the JS file for the block editor
            wp_enqueue_script(
                'booking-calendar-block-creation-script',
                plugins_url('js/booking-calendar-creation.js', __FILE__),
                array('wp-blocks', 'wp-element'),
                filemtime(plugin_dir_path(__FILE__) . 'js/booking-calendar-creation.js'),
                true
            );

            // Enqueue the CSS file for the block editor
            wp_enqueue_style(
                'booking-calendar-block-editor-style',
                plugins_url('css/booking-calendar-block.css', __FILE__),
                array(),
                filemtime(plugin_dir_path(__FILE__) . 'css/booking-calendar-block.css'),
                true
            );

            // Add the Glutenborg bloc keditor preview of the booking calendar
            wp_localize_script('booking-calendar-block-creation-script', 'calendarBlockContent', array(
                'calendar_block_file_url' => plugins_url('templates/admin-booking-calendar-content.html', __FILE__)
            ));

        }

        /**
         * Enqueues the CSS file for both frontend and backend. Is used at the "edit page" pages.
         */
        public function vdx_bks_enqueue_calendar_block_styles() {

            wp_enqueue_style(
                'booking-calendar-block-style',
                plugins_url('css/booking-calendar-block.css', __FILE__),
                array(),
                filemtime(plugin_dir_path(__FILE__) . 'css/booking-calendarblock.css')
            );

        }

        /**
         * Registers the Voodux Bookings custom block with its editor script and render callback.
         */
        public function vdx_bks_register_bookings_calendar_block() {

            register_block_type('voodux-bookings/booking-calendar-block', array(
                'editor_script' => 'booking-calendar-block-creation-script',
                'render_callback' => array($this, 'vdx_bks_render_bookings_custom_block'),
            ));

        }

        /**
         * Renders the Voodux Bookings custom block by loading its PHP content from a template file.
         *
         * @return string The PHP content of the custom block.
         */
        public function vdx_bks_render_bookings_custom_block() {
            
            ob_start();
            include plugin_dir_path(__FILE__) . 'templates/block-booking-calendar-content.php';
            return ob_get_clean();

        }

        /**
         * Enqueues the custom block JavaScript file only for booking products on the product page.
         */
        public function vdx_bks_enqueue_calendar_block_scripts_for_booking_product() {
            
            global $product;

            if ($product && $product->get_type() === 'booking_product') {
                wp_enqueue_script(
                    'booking-calendar-block-script',
                    plugin_dir_url(__FILE__) . 'js/booking-calendar-block.js',
                    array(),
                    filemtime(plugin_dir_path(__FILE__) . 'js/booking-calendar-block.js'),
                    true // Load script in footer
                );

                // Localize the script with translatable strings                
                wp_localize_script('booking-calendar-block-script', 'bookingTranslations', array(                    
                    'January' => esc_html(__('January', "voodux-bookings")),
                    'February' => esc_html(__('February', "voodux-bookings")),
                    'March' => esc_html(__('March', "voodux-bookings")),
                    'April' => esc_html(__('April', "voodux-bookings")),
                    'May' => esc_html(__('May', "voodux-bookings")),
                    'June' => esc_html(__('June', "voodux-bookings")),
                    'July' => esc_html(__('July', "voodux-bookings")),
                    'August' => esc_html(__('August', "voodux-bookings")),
                    'September' => esc_html(__('September', "voodux-bookings")),
                    'October' => esc_html(__('October', "voodux-bookings")),
                    'November' => esc_html(__('November', "voodux-bookings")),
                    'December' => esc_html(__('December', "voodux-bookings")),
                    'ErrorMessageDateTime' => esc_html(__('Please select both a date and a time.', "voodux-bookings")),
                    'Date' => esc_html(__('Date', "voodux-bookings")),
                    'Time' => esc_html(__('Time', "voodux-bookings")),
                    'Price' => esc_html(__('Price', "voodux-bookings")),
                    'Language_code' => str_replace('_', '-', get_locale()),
                    'hour12' => get_option('vdx_bks_time_format') == '24' ? false : true
                ));

            }            

        }

        /**
         * Enqueues the custom block CSS file only for booking products on the product page.
         */
        public function vdx_bks_enqueue_calendar_block_styles_for_booking_product() {

            if (is_product()) {
                global $product;

                if ($product && $product->get_type() === 'booking_product') {
                    wp_enqueue_style(
                        'booking-calendar-block-style',
                        plugin_dir_url(__FILE__) . 'css/booking-calendar-block.css',
                        array(), // No dependencies
                        filemtime(plugin_dir_path(__FILE__) . 'css/booking-calendar-block.css')
                    );
                }
            }

        }

        /**
         * Enqueue booking data to the calendar JavaScript.
         */
        function vdx_bks_enqueue_bookings_to_calendar_js() {            

            $post_id = get_the_ID();

            // Set week day availability according to if a price is set.            
            wp_localize_script('booking-calendar-block-script', 'availabilityMonday', get_post_meta($post_id, 'mon_price', true));            
            wp_localize_script('booking-calendar-block-script', 'availabilityTuesday', get_post_meta($post_id, 'tue_price', true));                     
            wp_localize_script('booking-calendar-block-script', 'availabilityWednesday', get_post_meta($post_id, 'wed_price', true));            
            wp_localize_script('booking-calendar-block-script', 'availabilityThursday', get_post_meta($post_id, 'thu_price', true));            
            wp_localize_script('booking-calendar-block-script', 'availabilityFriday', get_post_meta($post_id, 'fri_price', true));         
            wp_localize_script('booking-calendar-block-script', 'availabilitySaterday', get_post_meta($post_id, 'sat_price', true));            
            wp_localize_script('booking-calendar-block-script', 'availabilitySunday', get_post_meta($post_id, 'sun_price', true));

            // Set the not avaiable date ranges.            
            wp_localize_script('booking-calendar-block-script', 'unavailableFromDate1', get_post_meta($post_id, 'unavailable_from_date_1', true));
            wp_localize_script('booking-calendar-block-script', 'unavailableUntillDate1', get_post_meta($post_id, 'unavailable_untill_date_1', true));
            wp_localize_script('booking-calendar-block-script', 'unavailableFromDate2', get_post_meta($post_id, 'unavailable_from_date_2', true));
            wp_localize_script('booking-calendar-block-script', 'unavailableUntillDate2', get_post_meta($post_id, 'unavailable_untill_date_2', true));
            wp_localize_script('booking-calendar-block-script', 'unavailableFromDate3', get_post_meta($post_id, 'unavailable_from_date_3', true));
            wp_localize_script('booking-calendar-block-script', 'unavailableUntillDate3', get_post_meta($post_id, 'unavailable_untill_date_3', true));
            wp_localize_script('booking-calendar-block-script', 'unavailableFromDate4', get_post_meta($post_id, 'unavailable_from_date_4', true));
            wp_localize_script('booking-calendar-block-script', 'unavailableUntillDate4', get_post_meta($post_id, 'unavailable_untill_date_4', true));
            wp_localize_script('booking-calendar-block-script', 'unavailableFromDate5', get_post_meta($post_id, 'unavailable_from_date_5', true));
            wp_localize_script('booking-calendar-block-script', 'unavailableUntillDate5', get_post_meta($post_id, 'unavailable_untill_date_5', true));

            // Set the booking per hour or whole day value
            wp_localize_script('booking-calendar-block-script', 'bookPerHour', get_post_meta($post_id, 'book_per_hour', true));

            // Scroll to calendar on webpage.
            wp_localize_script('booking-calendar-block-script', 'scrollToCalendar', get_option('vdx_bks_scroll_to_calendar'));

            // Collect the booking from ans untill times for each day
            $from_times = [
                0 => ! $this->vdx_bks_is_empty_or_null( get_post_meta( $post_id, 'mon_from_time', true ) ) ? get_post_meta( $post_id, 'mon_from_time', true ) : '',
                1 => ! $this->vdx_bks_is_empty_or_null( get_post_meta( $post_id, 'tue_from_time', true ) ) ? get_post_meta( $post_id, 'tue_from_time', true ) : '',
                2 => ! $this->vdx_bks_is_empty_or_null( get_post_meta( $post_id, 'wed_from_time', true ) ) ? get_post_meta( $post_id, 'wed_from_time', true ) : '',
                3 => ! $this->vdx_bks_is_empty_or_null( get_post_meta( $post_id, 'thu_from_time', true ) ) ? get_post_meta( $post_id, 'thu_from_time', true ) : '',
                4 => ! $this->vdx_bks_is_empty_or_null( get_post_meta( $post_id, 'fri_from_time', true ) ) ? get_post_meta( $post_id, 'fri_from_time', true ) : '',
                5 => ! $this->vdx_bks_is_empty_or_null( get_post_meta( $post_id, 'sat_from_time', true ) ) ? get_post_meta( $post_id, 'sat_from_time', true ) : '',
                6 => ! $this->vdx_bks_is_empty_or_null( get_post_meta( $post_id, 'sun_from_time', true ) ) ? get_post_meta( $post_id, 'sun_from_time', true ) : ''
            ];

            $untill_times = [
                0 => ! $this->vdx_bks_is_empty_or_null( get_post_meta( $post_id, 'mon_untill_time', true ) ) ? get_post_meta( $post_id, 'mon_untill_time', true ) : '',
                1 => ! $this->vdx_bks_is_empty_or_null( get_post_meta( $post_id, 'tue_untill_time', true ) ) ? get_post_meta( $post_id, 'tue_untill_time', true ) : '',
                2 => ! $this->vdx_bks_is_empty_or_null( get_post_meta( $post_id, 'wed_untill_time', true ) ) ? get_post_meta( $post_id, 'wed_untill_time', true ) : '',
                3 => ! $this->vdx_bks_is_empty_or_null( get_post_meta( $post_id, 'thu_untill_time', true ) ) ? get_post_meta( $post_id, 'thu_untill_time', true ) : '',
                4 => ! $this->vdx_bks_is_empty_or_null( get_post_meta( $post_id, 'fri_untill_time', true ) ) ? get_post_meta( $post_id, 'fri_untill_time', true ) : '',
                5 => ! $this->vdx_bks_is_empty_or_null( get_post_meta( $post_id, 'sat_untill_time', true ) ) ? get_post_meta( $post_id, 'sat_untill_time', true ) : '',
                6 => ! $this->vdx_bks_is_empty_or_null( get_post_meta( $post_id, 'sun_untill_time', true ) ) ? get_post_meta( $post_id, 'sun_untill_time', true ) : ''
            ];
            wp_localize_script('booking-calendar-block-script', 'bookingFromTimes', $from_times);
            wp_localize_script('booking-calendar-block-script', 'bookingUntillTimes', $untill_times);  
            
            // Set bookings for this product            
            wp_localize_script('booking-calendar-block-script', 'bookingData', array('bookings' => $this->vdx_bks_get_existing_bookings()));
             
            // Set the first day of the week
            wp_localize_script('booking-calendar-block-script', 'firstDayOfWeekChoice', get_option('vdx_bks_first_day_of_week'));

            // Add the WordPress date library
            wp_enqueue_script('wp-date');
            
        }

        /**
         * Checks if the given value is either null or empty.
         *
         * A value is considered empty if it is:
         * - An empty string ("")
         * - The integer 0 or the string "0"
         * - The boolean false
         * - An empty array
         * - The special value NULL
         * 
         * @param mixed $value The value to check.
         * 
         * @return bool Returns true if the value is null or empty, false otherwise.
         */
        function vdx_bks_is_empty_or_null($value) {

            // Check if the value is either null or empty
            return is_null($value) || empty($value);

        }

        /**
         * Retrieve existing bookings from pending orders.
         *
         * This function fetches all WooCommerce orders with a status of 'pending' that have
         * both 'booking_date' and 'booking_time' meta fields. It then collects the booking
         * dates and times from these orders and returns them as an array.
         *
         * @return array $bookings Array of bookings, each with 'date' and 'time' keys.
         */
        function vdx_bks_get_existing_bookings() {

            // Get page admin setting booking_per_hour (if the booking can be made per hour or is for the whole day).
            $booking_per_hour = get_post_meta( get_the_ID(), 'book_per_hour', true);

            // Fetch orders with 'pending' status that have booking_date and booking_time meta fields
            $args = array(
                'status' => 'pending', // Fetch only pending payment orders
                'limit' => -1, // To get all pending orders
                'meta_query' => array(                    
                    array(
                        'key' => 'booking_date',
                        'compare' => 'EXISTS',
                    )
                ),
            );
        
            $orders = wc_get_orders($args);
            $bookings = array();
            
            foreach ($orders as $order) {        
                // Get order meta data
                $order_date = $order->get_meta('booking_date', true);
                $order_time = $order->get_meta('booking_time', true);
        
                // Check if both date and time are available
                if (!empty($order_date) && ( !empty($order_time) || $booking_per_hour != "yes") ) {
                    $booking = array(
                        'date' => $order_date,
                        'time' => $order_time
                    );
                    $bookings[] = $booking;
                }
            }
        
            return $bookings;

        }

        /**
         * Sets the global $product variable for use on single product pages.
         *
         * This function is necessary to ensure that the $product variable is properly
         * set for booking products, allowing conditional checks and operations based
         * on the product type.
         */
        public function vdx_bks_set_global_product_for_booking() {

            if (is_product()) {
                global $post, $product;
                $product = wc_get_product($post->ID);
            }

        }

        /**
         * Display custom order fields on the order edit page
         */
        function vdx_bks_display_custom_order_fields( $order ) {            

            $order_meta_data = $order->get_meta_data();
            $order_date = '';
            $order_time = '';

            foreach ($order_meta_data as $meta) {
                if ($meta->get_data()['key'] === 'booking_date') {
                    $order_date = $meta->get_data()['value'];
                }
                if ($meta->get_data()['key'] === 'booking_time') {
                    $order_time = $meta->get_data()['value'];
                }
            }
            
            // Check if either order_date or order_time is empty or null.
            if (empty($order_date) || empty($order_time)) {
                return;
            }

            $date = new DateTime($order_date);
            $time = new DateTime($order_time);
        
            // Format order date and time according to Wordpress date and time format settings.
            $formatted_booking_date = $date->format(get_option('date_format'));                                
            $formatted_booking_time = $time->format(get_option('time_format'));

            echo wp_kses_post('<p><strong>'. __('Booking date', "voodux-bookings").':</p></strong>' . $formatted_booking_date, "voodux-bookings");
            echo wp_kses_post('<p><strong>'. __('Booking time', "voodux-bookings").':</p></strong>' . $formatted_booking_time, "voodux-bookings");

        }

        /**
         * Hide Woocommer order detail page custom fields.
         */
        public function vdx_bks_hide_woocommercer_order_custom_fields($protected, $meta_key) {
            return true;

        }

    }

}

new vdx_bks_Booking_Custom_Block();