<?php
/**
 * Redirection Manager
 */
if (!defined('ABSPATH')) exit;

class VonSEOWP_Redirects {

    private $redirects = array();
    private $log_404 = false;

    public function __construct() {
        $options = get_option('vonseowp_settings', array());
        
        // Parse Redirects (newline separated string: /old -> /new)
        if (!empty($options['redirects_list'])) {
            $lines = explode("\n", $options['redirects_list']);
            foreach ($lines as $line) {
                $parts = explode('->', $line);
                if (count($parts) === 2) {
                    $this->redirects[trim($parts[0])] = trim($parts[1]);
                }
            }
        }

        $this->log_404 = !empty($options['enable_404_log']);

        add_action('template_redirect', array($this, 'handle_redirects'), 1);
        add_action('template_redirect', array($this, 'log_404_error'), 99);
    }

    public function handle_redirects() {
        if (empty($this->redirects)) return;

        $current_uri = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';
        $path = wp_parse_url($current_uri, PHP_URL_PATH);

        // Simple Match
        if (isset($this->redirects[$path])) {
            wp_safe_redirect($this->redirects[$path], 301);
            exit;
        }

        // Check for trailing slash variations
        $untrail = untrailingslashit($path);
        if (isset($this->redirects[$untrail])) {
            wp_safe_redirect($this->redirects[$untrail], 301);
            exit;
        }

        $trail = trailingslashit($path);
        if (isset($this->redirects[$trail])) {
             wp_safe_redirect($this->redirects[$trail], 301);
             exit;
        }
    }

    public function log_404_error() {
        if (!is_404() || !$this->log_404) return;

        $log = get_option('vonseowp_404_log', array());
        $current_uri = isset($_SERVER['REQUEST_URI']) ? esc_url_raw(wp_unslash($_SERVER['REQUEST_URI'])) : '';
        
        // Add new entry
        if (!isset($log[$current_uri])) {
            $log[$current_uri] = array(
                'count' => 1,
                'last_hit' => current_time('mysql')
            );
        } else {
            $log[$current_uri]['count']++;
            $log[$current_uri]['last_hit'] = current_time('mysql');
        }

        // Limit log size
        if (count($log) > 50) {
            // Sort by time ASC and remove first
            uasort($log, function($a, $b) {
                return strtotime($a['last_hit']) - strtotime($b['last_hit']);
            });
            array_shift($log);
        }

        update_option('vonseowp_404_log', $log);
    }
}
