/*!
 * voNavPilot Lite – Voice Navigation & Accessibility
 * https://vonavpilot.com
 *
 * MIT License
 * 
 * Copyright (c) 2025 Hamid Mosahebi
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
(function() {
    "use strict";
    const vonavpilot_botName = "voNavPilot";
    const vonavpilot_color = '#00aaff';
    const vonavpilot_noSleep = new window.NoSleep(); // Prevent device from sleeping while voNavPilot is active
    const vonavpilot_currentDomain = window.location.hostname; // Get the current domain name
    const recognition = window.SpeechRecognition || window.webkitSpeechRecognition; // Use native speech recognition (cross-browser support)
    const vonavpilot_DEBUG = {
        main: false, // Enable logs for voNavPilot's main function
        promptMatch: false,  // Enable logs for prompt matching (Fuzzysort + Levenshtein)
        tts: false, // Enable logs for Text-to-Speech (TTS) actions
        recognition: false, // Enable logs for window Speech Recognition actions
        form: false,  // Enable logs for form-filling logic
        readPage: false, // Enable logs for page reading and analysis
        commandMap: false, // Enable logs for content analysis and link generation
        readingIntro: false,  // Enable logs for reading Intro Text
        links : false // Enable logs for internal and external links detection
    };
    const vonavpilot_language = vonavpilotData.lang; // Language setting from voNavPilot config
    // Static microphone icon (no animation)
    const vonavpilot_micSVG = "<svg viewBox='0 0 24 24' fill='rgba(86, 81, 81, 0.5)' xmlns='http://www.w3.org/2000/svg'><path d='M12 14c1.66 0 3-1.34 3-3V5a3 3 0 0 0-6 0v6c0 1.66 1.34 3 3 3zm5-3c0 2.76-2.24 5-5 5s-5-2.24-5-5H5c0 3.53 2.61 6.43 6 6.92V21h2v-3.08c3.39-.49 6-3.39 6-6.92h-2z'/></svg>";
    // Animated microphone icon with pulsing effect
    const vonavpilot_micSVGAnimated = "<svg id='mic-svg' viewBox='0 0 24 24' fill='"+vonavpilot_contrastVariant(vonavpilot_color)+"' xmlns='http://www.w3.org/2000/svg'><style>@keyframes pulseFade{0%,100%{opacity:0.3;}50%{opacity:1;}}#mic-svg path{animation:pulseFade 2s infinite ease-in-out;}</style><path d='M12 14c1.66 0 3-1.34 3-3V5a3 3 0 0 0-6 0v6c0 1.66 1.34 3 3 3zm5-3c0 2.76-2.24 5-5 5s-5-2.24-5-5H5c0 3.53 2.61 6.43 6 6.92V21h2v-3.08c3.39-.49 6-3.39 6-6.92h-2z'/></svg>";
    // Animated 3 points icon with light effect
    const vonavpilot_waitPointsAnimated = "<svg viewBox='0 0 40 20' width='24' height='12'><circle fill='"+vonavpilot_contrastVariant(vonavpilot_color)+"' cx='10' cy='10' r='4'><animate attributeName='opacity' values='0.3;1;0.3' dur='1s' repeatCount='indefinite'/></circle><circle fill='"+vonavpilot_contrastVariant(vonavpilot_color)+"' cx='20' cy='10' r='4'><animate attributeName='opacity' values='0.3;1;0.3' dur='1s' begin='0.2s' repeatCount='indefinite'/></circle><circle fill='"+vonavpilot_contrastVariant(vonavpilot_color)+"' cx='30' cy='10' r='4'><animate attributeName='opacity' values='0.3;1;0.3' dur='1s' begin='0.4s' repeatCount='indefinite'/></circle></svg>";
    const vonavpilot_fuzzysortThreshold = -100; // Fuzzysort threshold for tighter matching (commands)
    const vonavpilot_fuzzysortThresholdForLinks = -1000; // Fuzzysort threshold for looser matching (links)
    const vonavpilot_ttsDelayTime = 1000; // Delay (ms) before TTS starts, to avoid overlapping speech
    const vonavpilot_vonavpilotDelayTime = 300; // General-purpose voNavPilot delay (ms)
    const vonavpilot_readTextPauseTime = 200; // Pause (ms) between sentences during reading
    const vonavpilot_maxFontSizePercent = 300; // Maximum zoom percentage
    const vonavpilot_minFontSizePercent = 5; // Minimum zoom percentage
    const vonavpilot_fontSizePercentStep = 5; // Step size for zoom in/out (%)
    const vonavpilot_defaultFontSizePercent = 100; // Default font zoom level (%)
    const vonavpilot_contentAnalyseOutput = document.createElement("div"); // Placeholder for raw website content analysis
    const vonavpilot_synth = window.speechSynthesis; // Browser speech synthesis engine
    const vonavpilot_utteranceRate = 1; // TTS speaking rate from config
    let vonavpilot_status = "init"; // Initial voNavPilot vonavpilot_status ("init" or "start")
    let vonavpilot_commandDetectionInProgress = false; // Indicates if voice command detection is currently active
    let vonavpilot_voNavPilotStopped = false; // Indicates if voNavPilot is manually stopped
    let vonavpilot_voNavPilotRecognition; // Recognition instance for main command processing
    let vonavpilot_readVoNavPilotIntroRecognition; // Recognition instance for intro vonavpilot_messages
    let vonavpilot_fillFormRecognition; // Recognition instance for form filling
    let vonavpilot_readFillFormInstrunctionsRecognition; // Recognition for reading form instructions
    let vonavpilot_readElementIds = new Set(); // Tracks elements already read aloud
    let vonavpilot_cancelPageReadingRequested = false; // Flag to interrupt page reading
    let vonavpilot_internalLinksMap = []; // Mapped internal site links
    let vonavpilot_internalLinksNames = []; // Names (text) of internal links
    let vonavpilot_externalLinksMap = []; // Mapped external links
    let vonavpilot_externalLinksNames = []; // Names (text) of external links
    let vonavpilot_totalLinksMap; // Combined internal + external links (used in mapping process)
    let vonavpilot_homeIS = false; // Whether the current page is considered "home"
    let vonavpilot_vonavPilotSpeak = false; // Flag to indicate if voNavPilot is speaking
    let formSpeak = false; // Flag to indicate if form instructions should be spoken
    let formSpelling = false; // Flag to enable spelling mode in forms
    let vonavpilot_activeFormField = null; // Currently active form field
    let vonavpilot_isListening = false; // Indicates if voice recognition is actively listening
    let form; // Currently selected form element
    let formFields; // Fields within the selected form
    let vonavpilot_fieldCont = 0; // Index or counter for form fields
    let vonavpilot_curUserSaid; // Stores the latest user voice input
    let vonavpilot_stopwords; // Common vonavpilot_stopwords to filter out
    let vonavpilot_spellingMap = null; // Map of words used in spelling mode
    let vonavpilot_languageMap = null; // Map switch Languages commands
    let vonavpilot_controlCommandsMap = null; // Command map for general control commands
    let formControlCommandsMap = null; // Command map specific to form control
    let vonavpilot_audioCtx; // Web Audio API context for sound feedback
    let vonavpilot_beepBuffer; // Buffer for playing beep sounds
    let vonavpilot_fontSizePercent = vonavpilot_defaultFontSizePercent; // Current font zoom level
    let vonavpilot_disabledStylesheets = []; // Stylesheets disabled for Simple View
    let vonavpilot_disabledStyleTags = []; // <style> tags disabled for Simple View
    let vonavpilot_disabledInlineStyles = []; // Inline styles disabled for Simple View
    let vonavpilot_linkCommandMap; // Map for voice navigation of links
    let vonavpilot_currentIndex = 0; // Current index in link navigation or form filling
    let vnpStarted = false;
    const vonavpilot_cookieBannerIds = [ // Common cookie banner IDs to detect and handle
        "BorlabsCookieBox",
        "cmplz-cookie-banner",
        "cookie-notice",
        "gdpr-cookie-consent",
        "cookiebot-banner",
        "wpcc-banner",
        "cookie-law-info-bar",
        "real-cookie-banner",
        "hibou-cookie-consent",
        "accept-cookies-bar",
        "iubenda-cookie-banner"
    ];
    const vonavpilot_domElements = { // DOM elements used by voNavPilot
        button: document.createElement("button"),
        InfoBox: document.createElement("div"),
        SpellingMap: document.createElement("div")
    };
    let vonavpilot_messages; // Dynamic labels/vonavpilot_messages used by voNavPilot (based on vonavpilot_language or state)
    let vonavpilot_storedVnp; 
    const vonavpilot_settings = {
        session: false,                // If instance of vonavpilot is already running
        intro: false,                  // If Intro has been read or not
        deactive: false,               // If instance of vonavpilot is deactivated
        grayScale: false,              // Grayscale enabled or not
        highContrast: false,           // High Contrast enabled or not
        zoom: vonavpilot_defaultFontSizePercent,  // Zoom level (default 100%)
        plainView: false,              // Plain View enabled or not (default false)
        images: true                   // Page Images enabled or not
    };

    if(sessionStorage.getItem('vnp') === null){
        sessionStorage.setItem('vnp', JSON.stringify(vonavpilot_settings));
    }

    function isUnsupportedBrowser() {
       return /iPad|iPhone|iPod/.test(navigator.userAgent) && !window.MSStream;
    }

    document.addEventListener("DOMContentLoaded", async () => {
        if (!recognition || isUnsupportedBrowser()) {
            await vonavpilot_loadMessages();
            // Fallback if Browser doesn"t support SpeechRecognition
            if (sessionStorage.getItem("speechRecognitionSupport") === null) {
                vonavpilot_domElements.InfoBox.classList.add("vonavpilot-info");
                vonavpilot_domElements.InfoBox.tabIndex = 0;
                vonavpilot_domElements.InfoBox.style.right = "20px";
                vonavpilot_domElements.InfoBox.style.bottom = "20px";
                const icon = document.createElementNS("http://www.w3.org/2000/svg", "svg"); // create error icon 
                icon.setAttribute("xmlns", "http://www.w3.org/2000/svg");
                icon.setAttribute("width", "40");
                icon.setAttribute("height", "40");
                icon.setAttribute("viewBox", "0 0 24 24");
                icon.setAttribute("fill", "none");
                icon.innerHTML = "<path d='M4 4l16 16M20 4l-16 16' stroke='red'/>";
                vonavpilot_domElements.InfoBox.appendChild(icon);
                const text = document.createElement("span");
                text.innerHTML = vonavpilot_messages.noSpeechRecognitionSupport[vonavpilot_language];
                vonavpilot_domElements.InfoBox.appendChild(text);
                document.body.appendChild(vonavpilot_domElements.InfoBox);
                setTimeout(() => {
                    vonavpilot_domElements.InfoBox.focus();
                    ["click", "keydown"].forEach(eventType => {
                        vonavpilot_domElements.InfoBox.addEventListener(eventType, () => {
                            vonavpilot_domElements.InfoBox.remove();
                            sessionStorage.setItem("speechRecognitionSupport", JSON.stringify(false));
                        });
                    });
                }, vonavpilot_vonavpilotDelayTime);
            } else {
                // User already closed No-Speech-Recognition Info-Box, no need for more Action.
            }
        } else {
            await vonavpilot_loadJsons();
            vonavpilot_getSettings();
            vonavpilot_loadBeep(); // Load mp3 file for later playing (sound effect)
            vonavpilot_initVoNavPilotRecognition(); // Initialize main voice recognition for navigation
            setTimeout(()=>{ // wait for loading dynamic elements
                let pageContent = document.documentElement.outerHTML; // Get the entire HTML content of the page
                pageContent = vonavpilot_cleanContent(pageContent); // Clean up the content (remove unwanted elements)
                vonavpilot_contentAnalyseOutput.classList.add("vonavpilot-content-analyse-output"); // Add a class for styling
                document.body.appendChild(vonavpilot_contentAnalyseOutput); // Append the output container to the body
                vonavpilot_contentAnalyseOutput.innerHTML = pageContent; // Set cleaned content into the output container
                vonavpilot_createContentCommandsMap(pageContent); // Create a map of content commands based on cleaned content
            }, vonavpilot_vonavpilotDelayTime);
        }
    });

    async function vonavpilot_loadMessages() {
        vonavpilot_messages = await vonavpilot_loadJson('vonavpilotLabelsMap.json', {}, data => data);
        // Now do the recursive replacement of [BOT-NAME]
        (function replace(obj, name) {
            for (const key in obj) {
                const val = obj[key];
                obj[key] = (typeof val === 'string') ?
                    val.replace(/\[BOT-NAME\]/g, name) :
                    (val && typeof val === 'object') ? (replace(val, name), val) : val;
            }
        })(vonavpilot_messages, vonavpilot_botName);
    }
    
    // Function to load JSON from server or cached server-side
    async function vonavpilot_loadJson(file, fallback = {}, transform = data => data) {
        try {
            // Add nonce to the URL
            const url = `${vonavpilotData.ajaxUrl}?action=vonavpilot_get&file=${file}&_wpnonce=${vonavpilotData.ajaxNonce}`;
            const response = await fetch(url, { credentials: 'same-origin' });
            const result = await response.json();
            if (result.success) {
                return transform(result.data);
            }
            throw new Error(result.data || 'Failed to load JSON');
        } catch (error) {
            if (vonavpilot_DEBUG.main) console.error(`Failed to load ${file}:`, error);
            return fallback;
        }
    }

    // Function to load all JSON files at once with specific handling for each one
    async function vonavpilot_loadJsons() {
        // Define the JSON files and their respective transformations
        const jsonFiles = [
            { file: 'vonavpilotLabelsMap.json', transform: data => data }, // Normal handling for labelsMap
            { file: 'vonavpilotControlCommandsMap.json', transform: data => data }, // Normal handling for vonavpilot_controlCommandsMap
            { file: 'vonavpilotLinkCommandsMap.json', transform: data => data }, // Normal handling for linkCommandsMap
            { file: 'vonavpilotStopWords.json', transform: data => new Set(data.stopWords) }, // Special handling for stopWords
            { file: 'vonavpilotLanMap.json', transform: data => data.lanMap }, // Special handling for lanMap
            { file: 'vonavpilotFormCommandsMap.json', transform: data => data }, // Normal handling for formCommandsMap
            { file: 'vonavpilotSpellingMap.json', transform: data => data }, // Special handling for vonavpilot_spellingMap
        ];
        // Load JSON files in parallel, applying the transformation for each
        const jsonPromises = jsonFiles.map(({ file, transform }) => {
            return vonavpilot_loadJson(file, {}, transform); // Use the transform function for each file
        });
        // Wait for all the JSON files to be loaded
        const results = await Promise.all(jsonPromises);
        // Assign results to the respective variables
        vonavpilot_messages = results[0];            // Loaded labelsMap.json
        vonavpilot_controlCommandsMap = results[1];  // Loaded vonavpilot_controlCommandsMap.json
        vonavpilot_linkCommandMap = results[2];      // Loaded linkCommandsMap.json
        vonavpilot_stopwords = results[3];           // Loaded stopWords.json (transformed to Set)
        vonavpilot_languageMap = results[4];         // Loaded lanMap.json (transformed to lanMap)
        formControlCommandsMap = results[5]; // Loaded formCommandsMap.json
        vonavpilot_spellingMap = results[6];         // Loaded vonavpilot_spellingMap.json (transformed to Set)
    }

    function vonavpilot_contrastVariant(c){
        c = c.replace(/^#/, "");
        let R = parseInt(c.substr(0, 2), 16),
            G = parseInt(c.substr(2, 2), 16),
            B = parseInt(c.substr(4, 2), 16);
        const brighten = 200; // adjust to make it more or less "Brighter"
        return "#" + [R, G, B]
            .map(v => Math.min(255, v + brighten).toString(16).padStart(2, "0"))
            .join("");
    }

    function vonavpilot_getSettings() {
        vonavpilot_storedVnp = sessionStorage.getItem('vnp');
        if (vonavpilot_storedVnp) {
            vonavpilot_storedVnp = JSON.parse(vonavpilot_storedVnp);  // Parse the JSON string back into an object
            vonavpilot_loadSettings(vonavpilot_storedVnp);
        }
        return null;  // Return null if no data is found in sessionStorage
    }

    function vonavpilot_loadSettings(settings) {
        if (settings.grayScale) vonavpilot_enableGrayscale();
        if (settings.highContrast) vonavpilot_enableHighContrast();
        if (settings.plainView) vonavpilot_enablePlainView();
        if (settings.images === false) setTimeout(()=>{vonavpilot_hideImages();}, vonavpilot_vonavpilotDelayTime);
        if (settings.zoom !== vonavpilot_defaultFontSizePercent){
            vonavpilot_fontSizePercent = settings.zoom;
            vonavpilot_applyPageFontSize();
        }
    }

    function vonavpilot_saveSettings(settings) {
        // Utility function to check for undefined or null, otherwise return the provided value
        const getValue = (value, defaultValue) => (value === undefined || value === null) ? defaultValue : value;
        const vonavpilot_settings = {
            session: getValue(settings.session, false),                 // If instance of vonavpilot is already running (default false)
            intro: getValue(settings.intro, false),                     // If Intro has been read or not (default false)
            deactive: getValue(settings.deactive, false),               // If instance of vonavpilot is deactivated (default false)
            grayScale: getValue(settings.grayScale, false),             // Grayscale enabled or not (default false)
            highContrast: getValue(settings.highContrast, false),       // High Contrast enabled or not (default false)
            zoom: getValue(settings.zoom, vonavpilot_defaultFontSizePercent),      // Zoom level (default 100%)
            plainView: getValue(settings.plainView, false),             // Plain View enabled or not (default false)
            images: getValue(settings.images, true),                    // Page Images enabled or not (default true)
        };
        sessionStorage.setItem('vnp', JSON.stringify(vonavpilot_settings));    // Save settings to sessionStorage
    }

    function vonavpilot_resetSettings() {
        const defaultSettings = {
            session: false,
            intro: false,
            deactive: false,
            grayScale: false,
            highContrast: false,
            zoom: vonavpilot_defaultFontSizePercent,
            plainView: false,
            images: true,
        };
        vonavpilot_saveSettings(defaultSettings);
    }

    async function vonavpilot_createContentCommandsMap(content) {
        const alreayExists = new Set();
        let linkIgnoreList = ["wpml"];
        let openLinkCommandWords;
        openLinkCommandWords = vonavpilot_linkCommandMap.openLinkCommandWords || {};
        vonavpilot_totalLinksMap = [...content.matchAll(/<a\s+[^>]*href="([^"]+)"[^>]*>([\s\S]*?)<\/a>/gi)]
            .map(([, link, innerHTML]) => {
                try {
                    let name;
                    let visibleText = innerHTML
                        .replace(/<[^>]+>/g, ' ')       // remove tags
                        .replace(/&nbsp;/g, ' ')        // replace non-breaking spaces
                        .replace(/[^a-zA-Z0-9\u00C0-\u017F\s\-_]/g, '') // keep letters, numbers, accented chars, space, -, _
                        .replace(/ä/g, 'ae').replace(/Ä/g, 'Ae')
                        .replace(/ö/g, 'oe').replace(/Ö/g, 'Oe')
                        .replace(/ü/g, 'ue').replace(/Ü/g, 'Ue')
                        .replace(/ß/g, 'ss')
                        .replace(/\s+/g, ' ')
                        .trim();
                    if (link.trim() === "#") return null;
                    if (linkIgnoreList.some(ignore => link.includes(ignore))) {
                        if (vonavpilot_DEBUG.main) console.log('ignored links: ', link);
                        return null;
                    }
                    // Get name from URL last segment
                    const raw = decodeURIComponent(
                        link.startsWith('#')
                            ? link.slice(1)
                            : link.split('/').filter(Boolean).pop() || ''
                    );
                    name = raw.split('?')[0].split('#')[0]
                        .replace(/[^a-zA-Z0-9\s\-_\.]/g, '')
                        .toLowerCase()
                        .trim()
                        .replace(/[-_]/g, ' ')
                        .replace(/^[a-zA-Z]{2}\./, '')
                        .replace(/\.[^.]*$/, '')
                        .replace(/\s+/g, ' ');

                    if (!name && !visibleText) return null;
                    const normalize = str => str
                        .toLowerCase()
                        .trim()
                        .replace(/\s+/g, ' ');
                    const nameNorm = normalize(name);
                    const textNorm = normalize(visibleText);
                    // A helper to add name variants
                    const addNameVariants = linkName => {
                        const nameNoSpace = linkName.replace(/\s+/g, '');
                        const key1 = `${linkName}|||${link}`;
                        const key2 = `${nameNoSpace}|||${link}`;
                        if (!alreayExists.has(key1)) {
                            alreayExists.add(key1);
                            vonavpilot_initLinksMap(linkName, link);
                        }
                        // Add "open" variants
                        let openCommands = openLinkCommandWords[vonavpilot_language];
                        if (!Array.isArray(openCommands)) {
                            openCommands = openLinkCommandWords.en || [];
                        }
                        openCommands.forEach(command => {
                            const prepend = `${command} ${linkName}`;
                            const append = `${linkName} ${command}`;
                            const keyPrepend = `${prepend}|||${link}`;
                            const keyAppend = `${append}|||${link}`;
                            if (!alreayExists.has(keyPrepend)) {
                                alreayExists.add(keyPrepend);
                                vonavpilot_initLinksMap(prepend, link);
                            }
                            if (!alreayExists.has(keyAppend)) {
                                alreayExists.add(keyAppend);
                                vonavpilot_initLinksMap(append, link);
                            }
                        });
                        // Add no-space variants
                        if (nameNoSpace !== linkName && !alreayExists.has(key2)) {
                            alreayExists.add(key2);
                            openCommands.forEach(command => {
                                const prepend = `${command} ${nameNoSpace}`;
                                const append = `${nameNoSpace} ${command}`;
                                const keyPrepend = `${prepend}|||${link}`;
                                const keyAppend = `${append}|||${link}`;
                                if (!alreayExists.has(keyPrepend)) {
                                    alreayExists.add(keyPrepend);
                                    vonavpilot_initLinksMap(prepend, link);
                                }
                                if (!alreayExists.has(keyAppend)) {
                                    alreayExists.add(keyAppend);
                                    vonavpilot_initLinksMap(append, link);
                                }
                            });
                        }
                    };
                    // Always add URL-derived name
                    if (nameNorm) addNameVariants(nameNorm);
                    // If visible text exists and is different from URL segment, add it too
                    if (textNorm && textNorm !== nameNorm) {
                        addNameVariants(textNorm);
                    }
                    return { name: nameNorm, link };
                } catch (e) {
                    if (vonavpilot_DEBUG.main) console.warn("Error processing link:", link, e);
                    return null;
                }
            })
            .filter(Boolean);
        if (vonavpilot_DEBUG.links) console.log("vonavpilot_internalLinksMap: ", vonavpilot_internalLinksMap);
        if (vonavpilot_DEBUG.links) console.log("vonavpilot_externalLinksMap: ", vonavpilot_externalLinksMap);
        vonavpilot_internalLinksMap.forEach(function(link) {
            vonavpilot_internalLinksNames.push(link.name.toLowerCase());
        });
        vonavpilot_externalLinksMap.forEach(function(link) {
            vonavpilot_externalLinksNames.push(link.name.toLowerCase());
        });
        vonavpilot_addDomElements();
    }

    async function vonavpilot_loadBeep() {
        const AudioContextClass = window.AudioContext || window.webkitAudioContext;
        vonavpilot_audioCtx = new AudioContextClass();
        const response = await fetch(vonavpilotData.beepUrl);
        const arrayBuffer = await response.arrayBuffer();
        vonavpilot_beepBuffer = await vonavpilot_audioCtx.decodeAudioData(arrayBuffer);
    }

    async function vonavpilot_setBotName(){
        (function replace(obj, name) {
            for (const key in obj) {
                const val = obj[key];
                obj[key] = (typeof val === 'string') ?
                    val.replace(/\[BOT-NAME\]/g, name) :
                    (val && typeof val === 'object') ? (replace(val, name), val) : val;
            }
        })(vonavpilot_messages, vonavpilot_botName);
    }

    async function vonavpilot_addDomElements() {
        let infoMessage = "";
        vonavpilot_setBotName();// Load Labels Map and set vonavpilot_botName
        if (vonavpilot_storedVnp.intro === false) {
            infoMessage = vonavpilot_messages.intro[vonavpilot_language];
        } else {
            infoMessage = vonavpilot_messages.stop[vonavpilot_language];
        }
        vonavpilot_domElements.InfoBox.classList.add("vonavpilot-info");
        vonavpilot_domElements.InfoBox.tabIndex = 0;
        vonavpilot_domElements.InfoBox.innerHTML = "<span class='vonavpilot-info-text'>" + infoMessage + "</span>";
        document.body.appendChild(vonavpilot_domElements.InfoBox);
        vonavpilot_domElements.InfoBox.classList.add("hidden");
        vonavpilot_domElements.button.classList.add('vonavpilot-btn');
        vonavpilot_domElements.button.title = vonavpilot_botName + "-btn";
        vonavpilot_setWidgetDynamicDesign();
        document.body.appendChild(vonavpilot_domElements.button);
        if (vonavpilot_storedVnp.deactive === false) {
            vonavpilot_speakBlock("icon");
            vonavpilot_domElements.InfoBox.classList.remove("hidden");
        } else {
            vonavpilot_speakDeactive();
        }
        setTimeout(() => {
            vonavpilot_domElements.InfoBox.focus();
            if (vonavpilot_storedVnp.intro === true && vonavpilot_storedVnp.session === true && vonavpilot_storedVnp.deactive === false) {
                vonavpilot_status = "vonavpilot_start";
                vonavpilot_detectActivatedVoNavPilot();
            }
        }, vonavpilot_vonavpilotDelayTime);
        vonavpilot_unbindEventListeners();
        vonavpilot_bindEventListeners();
    }

    function vonavpilot_unbindEventListeners() {
        document.querySelectorAll(".vonavpilot-btn").forEach(function(btn) {
            btn.removeEventListener("click", vonavpilot_handleVonavPilotButtonClick);
        });
        document.querySelectorAll(".vonavpilot-info-text").forEach(function(infoText) {
            infoText.removeEventListener("click", vonavpilot_handleVonavPilotInfoTextClick);
        });
        document.removeEventListener("keydown", vonavpilot_handleKeyDown);
    }

    function vonavpilot_bindEventListeners() {
        document.querySelectorAll(".vonavpilot-btn").forEach(function(btn) {
            btn.addEventListener("click", vonavpilot_handleVonavPilotButtonClick);
        });
        document.querySelectorAll(".vonavpilot-info-text").forEach(function(infoText) {
            infoText.addEventListener("click", vonavpilot_handleVonavPilotInfoTextClick);
        });
        document.addEventListener("keydown", vonavpilot_handleKeyDown, {
            passive: false
        });
    }

    function vonavpilot_handleVonavPilotButtonClick() {
        if (vonavpilot_storedVnp.deactive === false) {
            vonavpilot_storedVnp.deactive = true;
            vonavpilot_saveSettings(vonavpilot_storedVnp);
            vonavpilot_deactivateBot();
        } else {
            vonavpilot_storedVnp.deactive = false;
            vonavpilot_saveSettings(vonavpilot_storedVnp);
            vonavpilot_activateBot();
        }
    }

    function vonavpilot_handleVonavPilotInfoTextClick() {
        switch (vonavpilot_status) {
            case "init":
                vonavpilot_status = "vonavpilot_start";
                if (vonavpilot_audioCtx?.state === "suspended") {
                    vonavpilot_audioCtx.resume();
                }
                vonavpilot_start();
                break;
            case "vonavpilot_start":
                if (vonavpilot_DEBUG.main) console.log("already started...");
                if (vonavpilot_cancelPageReadingRequested) {
                    if (speechSynthesis.speaking) {
                        speechSynthesis.cancel(); // stop TTS
                    }
                    vonavpilot_cancelPageReading();
                }
                break;
        }
    }

    function vonavpilot_deactivateBot() {
        vonavpilot_domElements.InfoBox.classList.add("hidden");
        vonavpilot_resetSettings();
        vonavpilot_storedVnp.deactive = true;
        vonavpilot_saveSettings(vonavpilot_storedVnp);
        vonavpilot_speakDeactive();
        setTimeout(() => {
            vonavpilot_synth.cancel();
            if (speechSynthesis.speaking) {
                speechSynthesis.cancel();
            }
            if (vonavpilot_voNavPilotRecognition) vonavpilot_killVoNavPilotRecognition();
            if (vonavpilot_readVoNavPilotIntroRecognition) vonavpilot_killReadVoNavPilotIntroRecognition();
            if (vonavpilot_fillFormRecognition) vonavpilot_killFillFormRecognition();
            if (vonavpilot_readFillFormInstrunctionsRecognition) killReadFillFormInstrunctionsRecognition();
        }, vonavpilot_ttsDelayTime);
    }

    function vonavpilot_activateBot() {
        vonavpilot_domElements.InfoBox.classList.remove("hidden");
        vonavpilot_curUserSaid = undefined; //Hide Command in Widget InfoBox, otherwise set to transcript
        vonavpilot_status = "init";
        if(vonavpilot_storedVnp.intro === false){
            vonavpilot_domElements.InfoBox.classList.add("vonavpilot-info");
            vonavpilot_domElements.InfoBox.tabIndex = 0;
            vonavpilot_domElements.InfoBox.innerHTML = "<span class='vonavpilot-info-text'>" + vonavpilot_messages.intro[vonavpilot_language] + "</span>";
        } else {
            vonavpilot_handleKeyDown(null);                
        }
        const btns = document.querySelectorAll(".vonavpilot-btn");
        btns.forEach(b => b.disabled = true);
            setTimeout(() => {
                document.querySelectorAll(".vonavpilot-info-text").forEach(function(infoText) {
                infoText.addEventListener("click", vonavpilot_handleVonavPilotInfoTextClick);
            });
            btns.forEach(b => b.disabled = false);
        }, vonavpilot_ttsDelayTime);
    }

    function vonavpilot_handleKeyDown(event) {
        if(event){
            const key = event.code || event.key;
            // Prevent scrolling on spacebar
            if (key === "Space" || event.key === " ") {
                event.preventDefault();
            }
        }
        // Unlock audio context if suspended (mobile-safe)
        if (vonavpilot_audioCtx?.state === "suspended") {
            vonavpilot_audioCtx.resume();
        }
        // Handle voNavPilot state
        if (vonavpilot_storedVnp.deactive === false) {
            switch (vonavpilot_status) {
                case "init":
                    vonavpilot_status = "vonavpilot_start";
                    vonavpilot_start(); // start voNavPilot
                    break;
                case "vonavpilot_start":
                    vnpStarted = true;
                    if (vonavpilot_DEBUG.main) console.log("already started...");
                    if(!vnpStarted){
                        const utterance = vonavpilot_createUtterance(vonavpilot_sanitizeMessageText(vonavpilot_messages.startShort[vonavpilot_language]));
                        vonavpilot_synth.speak(utterance);
                        vonavpilot_updateInfoBoxText(vonavpilot_messages.startShort[vonavpilot_language]);
                    }
                    if (vonavpilot_cancelPageReadingRequested) {
                        if (speechSynthesis.speaking) {
                            speechSynthesis.cancel(); // stop TTS
                        }
                        vonavpilot_cancelPageReading();
                    }
                    break;
            }
        }
    }

    function vonavpilot_getLangCode(lang) {
        const map = {
            en: "en-US",
            de: "de-DE",
            fr: "fr-FR",
            es: "es-ES",
            it: "it-IT"
        };
        return map[lang] || "en-US"; // fallback
    }

    function vonavpilot_initVoNavPilotRecognition() {
        vonavpilot_voNavPilotRecognition = new recognition();
        vonavpilot_voNavPilotRecognition.lang = vonavpilot_getLangCode(vonavpilot_language);
        vonavpilot_voNavPilotRecognition.interimResults = false;
        vonavpilot_voNavPilotRecognition.continuous = false;
        if(vonavpilot_DEBUG.recognition)console.log('vonavpilot_initVoNavPilotRecognition - lang:', vonavpilot_getLangCode(vonavpilot_language));
    }

    function vonavpilot_killVoNavPilotRecognition() {
        vonavpilot_voNavPilotRecognition.stop();
        vonavpilot_voNavPilotRecognition.onresult = null;
        vonavpilot_voNavPilotRecognition.onerror = null;
        vonavpilot_voNavPilotRecognition.onend = null;
        if(vonavpilot_DEBUG.recognition)console.log('vonavpilot_killVoNavPilotRecognition');
    }

    function vonavpilot_initReadVoNavPilotIntroRecognition() {
        vonavpilot_readVoNavPilotIntroRecognition = new recognition();
        vonavpilot_readVoNavPilotIntroRecognition.lang = vonavpilot_getLangCode(vonavpilot_language);
        vonavpilot_readVoNavPilotIntroRecognition.interimResults = false;
        vonavpilot_readVoNavPilotIntroRecognition.continuous = false;
        if(vonavpilot_DEBUG.recognition)console.log('vonavpilot_initReadVoNavPilotIntroRecognition - lang:', vonavpilot_getLangCode(vonavpilot_language));
    }

    function vonavpilot_killReadVoNavPilotIntroRecognition() {
        vonavpilot_readVoNavPilotIntroRecognition.stop();
        vonavpilot_readVoNavPilotIntroRecognition.onresult = null;
        vonavpilot_readVoNavPilotIntroRecognition.onerror = null;
        vonavpilot_readVoNavPilotIntroRecognition.onend = null;
        if(vonavpilot_DEBUG.recognition)console.log('vonavpilot_killReadVoNavPilotIntroRecognition');
    }

    function vonavpilot_initFillFormRecognition() {
        vonavpilot_fillFormRecognition = new recognition();
        vonavpilot_fillFormRecognition.lang = vonavpilot_getLangCode(vonavpilot_language);
        vonavpilot_fillFormRecognition.interimResults = false;
        vonavpilot_fillFormRecognition.continuous = false;
        if(vonavpilot_DEBUG.recognition)console.log('vonavpilot_initFillFormRecognition - lang:', vonavpilot_getLangCode(vonavpilot_language));
    }

    function vonavpilot_killFillFormRecognition() {
        vonavpilot_fillFormRecognition.stop();
        vonavpilot_fillFormRecognition.onresult = null;
        vonavpilot_fillFormRecognition.onerror = null;
        vonavpilot_fillFormRecognition.onend = null;
        if(vonavpilot_DEBUG.recognition)console.log('vonavpilot_killFillFormRecognition');
    }

    function initReadFillFormInstrunctionsRecognition() {
        vonavpilot_readFillFormInstrunctionsRecognition = new recognition();
        vonavpilot_readFillFormInstrunctionsRecognition.lang = vonavpilot_getLangCode(vonavpilot_language);
        vonavpilot_readFillFormInstrunctionsRecognition.interimResults = false;
        vonavpilot_readFillFormInstrunctionsRecognition.continuous = false;
        if(vonavpilot_DEBUG.recognition)console.log('initReadFillFormInstrunctionsRecognition - lang:', vonavpilot_getLangCode(vonavpilot_language));
    }

    function killReadFillFormInstrunctionsRecognition() {
        vonavpilot_readFillFormInstrunctionsRecognition.stop();
        vonavpilot_readFillFormInstrunctionsRecognition.onresult = null;
        vonavpilot_readFillFormInstrunctionsRecognition.onerror = null;
        vonavpilot_readFillFormInstrunctionsRecognition.onend = null;
        if(vonavpilot_DEBUG.recognition)console.log('killReadFillFormInstrunctionsRecognition');
    }

    function vonavpilot_cleanContent(pageContent) {
        const parser = new DOMParser();
        const doc = parser.parseFromString(pageContent, "text/html");
        // Remove SVG elements, <style>, <script>, <meta> and <link> tags
        ["svg", "style", "script", "meta", "link"].forEach(sel => 
            doc.querySelectorAll(sel).forEach(el => el.remove())
        );
        // Remove unwanted elements like <font>, <center>, etc.
        const nonSemanticTags = ["font", "center"];
        nonSemanticTags.forEach(tag => {
            const elements = doc.querySelectorAll(tag);
            elements.forEach(el => el.remove());
        });
        // Remove elements with `style="display: none"` or `visibility: hidden`
        const hiddenElements = doc.querySelectorAll("[style*='display: none'], [style*='visibility: hidden']");
        hiddenElements.forEach(el => el.remove());
        // Remove `javascript:` links from href
        const javascriptLinks = doc.querySelectorAll("a[href^='javascript:']");
        javascriptLinks.forEach(link => link.remove());
        // Remove inline style attributes
        const allElements = doc.querySelectorAll("*");
        allElements.forEach(element => {
            // Remove inline styles
            element.removeAttribute("style");
            // Keep only specific attributes
            Array.from(element.attributes).forEach(attr => {
                if (!["alt", "target", "srcset", "src", "href", "class"].includes(attr.name)) {
                    element.removeAttribute(attr.name);
                }
            });
            // Remove empty elements or those with no meaningful text content
            if (element.childNodes.length === 0 || !/\S/.test(element.textContent)) {
                element.remove();
            }
            // Clean up text content: remove non-breaking spaces and normalize other whitespaces
            if (element.hasChildNodes()) {
                element.childNodes.forEach(child => {
                    if (child.nodeType === Node.TEXT_NODE) {
                        child.textContent = child.textContent.replace(/\u00A0/g, ""); // Remove non-breaking spaces
                        child.textContent = child.textContent.replace(/\s+/g, " ").trim(); // Normalize whitespace
                    }
                });
            }
        });
        // Remove all comments from the document
        const comments = doc.createTreeWalker(doc, NodeFilter.SHOW_COMMENT);
        let comment;
        while ((comment = comments.nextNode())) {
            // Remove WPML and other specific comments
            if (comment.nodeValue.includes("googleoff") || comment.nodeValue.includes("wpml.org")) {
                comment.parentNode.removeChild(comment);
            }
        }
        // Remove <strong> tags but keep their content
        doc.querySelectorAll("strong").forEach(el => {
            const spaceBefore = document.createTextNode("\u00A0");
            const spaceAfter = document.createTextNode("\u00A0");
            el.replaceWith(spaceBefore, ...el.childNodes, spaceAfter);
        });
        // Remove Wordpress Internal Language Links(#en, #de,...)
        const langCodes = ["en", "de", "fr", "es"];
        doc.querySelectorAll("a").forEach(link => {
            const text = link.textContent.trim().toLowerCase();
            if (langCodes.includes(text)) {
                link.remove();
            }
        });
        // Output cleaned HTML for vonavpilot_DEBUGging or review
        return doc.documentElement.outerHTML;
    }

    function vonavpilot_initLinksMap(name, link) {
        const origin = window.location.origin;
        const trimmedLink = link.trim();
        const ignoredAnchors = new Set(["#wp--skip-link--target"]);
        const lowerName = name.toLowerCase();
        const isWpAdminLink = lowerName.includes("wp admin") || lowerName.includes("wpadmin");
        if (
            (trimmedLink.startsWith("http://") || trimmedLink.startsWith("https://")) &&
            trimmedLink !== origin &&
            !isWpAdminLink
        ) {
            vonavpilot_externalLinksMap.push({ name, link: trimmedLink });
        } else if (
            trimmedLink.startsWith("#") &&
            !ignoredAnchors.has(trimmedLink) &&
            trimmedLink.length > 1
        ) {
            vonavpilot_internalLinksMap.push({ name, link: trimmedLink });
        }
    }

    function vonavpilot_updateInfoBoxText(result) {
         // Replace any vonavpilot_stop words in vonavpilot_curUserSaid with "..."
        if (vonavpilot_curUserSaid) {
            vonavpilot_stopwords.forEach(word => {
                const regex = new RegExp(`\\b${word}\\b`, 'gi'); // Word boundary to match whole words only, case-insensitive
                vonavpilot_curUserSaid = vonavpilot_curUserSaid.replace(regex, '...');
            });
        }
        let textToDisplay = vonavpilot_curUserSaid === undefined ? result : `${vonavpilot_messages.detectedCommand[vonavpilot_language]} ${vonavpilot_curUserSaid}<br>${result.replace(/_/g, " ")}`;
        document.querySelector(".vonavpilot-info-text").innerHTML = textToDisplay;
    }

    function vonavpilot_speakBlock(type) {
        if (type === "icon") {
            vonavpilot_domElements.button.innerHTML = vonavpilot_micSVG;
        } else {
            vonavpilot_domElements.button.innerHTML = vonavpilot_waitPointsAnimated;
        }
        vonavpilot_domElements.button.classList.remove("vonavpilot-speak-free");
        vonavpilot_domElements.button.classList.remove("vonavpilot-speak-deactive");
        vonavpilot_domElements.button.classList.add("vonavpilot-speak-block");
    }

    function vonavpilot_speakFree() {
        vonavpilot_playBeep();
        vonavpilot_domElements.button.innerHTML = vonavpilot_micSVGAnimated;
        vonavpilot_domElements.button.classList.remove("vonavpilot-speak-block");
        vonavpilot_domElements.button.classList.remove("vonavpilot-speak-deactive");
        vonavpilot_domElements.button.classList.add("vonavpilot-speak-free");
    }

    function vonavpilot_speakDeactive() {
        vonavpilot_domElements.button.innerHTML = vonavpilot_micSVG;
        vonavpilot_domElements.button.classList.remove("vonavpilot-speak-block");
        vonavpilot_domElements.button.classList.remove("vonavpilot-speak-free");
        vonavpilot_domElements.button.classList.add("vonavpilot-speak-deactive");
    }

    function vonavpilot_setWidgetDynamicDesign() {
        const widgetButton = vonavpilot_domElements.button;
        const widgetInfoBox = vonavpilot_domElements.InfoBox;
        widgetButton.style.backgroundColor = vonavpilot_color;
        widgetButton.style.bottom = "20px";
        widgetButton.style.right = "20px";
        widgetInfoBox.style.bottom = "50px";
        widgetInfoBox.style.right = "10px";
    }

    function vonavpilot_playBeep() {
        if (vonavpilot_audioCtx && vonavpilot_beepBuffer) {
            if (vonavpilot_audioCtx.state === "suspended") {
                vonavpilot_updateInfoBoxText("<span class='vonavpilot-fail-sound'>🔇" + vonavpilot_messages.audioSuspendedMessage[vonavpilot_language]+ "</span>");
                vonavpilot_audioCtx.resume().then(() => {
                    vonavpilot_startBeep();
                }).catch(() => {
                   vonavpilot_updateInfoBoxText("<span class='vonavpilot-fail-sound'>🔇" + vonavpilot_messages.audioSuspendedMessage[vonavpilot_language]+ "</span>");
                });
            } else {
                vonavpilot_startBeep();
            }
        }
    }

    function vonavpilot_startBeep() {
        const source = vonavpilot_audioCtx.createBufferSource();
        source.buffer = vonavpilot_beepBuffer;
        source.connect(vonavpilot_audioCtx.destination);
        source.start(0);
        //remove Sound Error Message if Exist
        setTimeout(() => {
            const el = document.querySelector(".vonavpilot-fail-sound"); 
            if (el){
               // el.innerHTML = "";
            }
        }, vonavpilot_vonavpilotDelayTime);
    }

    function vonavpilot_isSpeechSynthesisSupported() {
        return "speechSynthesis" in window && typeof SpeechSynthesisUtterance === "function";
    }

    async function vonavpilot_initSpellingTable() {
        vonavpilot_domElements.SpellingMap.classList.add("vonavpilot-contact-form-help");
        vonavpilot_domElements.SpellingMap.tabIndex = 0;
        const template = `
            <div class="vonavpilot-spelling-table-container">
                <div class="vonavpilot-spelling-table">
                    <span class="spelling-letter-head">${vonavpilot_messages.spellingTableHead[vonavpilot_language]}</span>
                    <div class="letters-container"></div>
                </div>
                <div class="vonavpilot-spelling-panel" style="background-color:${vonavpilot_botName}"><div class="vonavpilot-spelling-panel-text">${vonavpilot_messages.formSpellingTitle[vonavpilot_language]}</text></div>
            </div>
        `;
        vonavpilot_domElements.SpellingMap.innerHTML = template;
        document.body.appendChild(vonavpilot_domElements.SpellingMap);
        vonavpilot_renderSpellingTable();
        vonavpilot_domElements.SpellingMap.querySelector(".vonavpilot-spelling-panel").removeEventListener("click", vonavpilot_handleClick);
        vonavpilot_domElements.SpellingMap.querySelector(".vonavpilot-spelling-panel").addEventListener("click", vonavpilot_handleClick);
    }

    function vonavpilot_handleClick() { 
        vonavpilot_domElements.SpellingMap.classList.toggle("slide-out");
    }

    function vonavpilot_renderSpellingTable() {
        const container = vonavpilot_domElements.SpellingMap.querySelector(".letters-container");
        container.innerHTML = "";
        // Render letters first (A-Z, @, ., -)
        for (const letter in vonavpilot_spellingMap) {
            if (["0", "1", "2", "3", "4", "5", "6", "7", "8", "9", "space"].includes(letter)) {
                continue; // Skip numbers and space
            }
            if (vonavpilot_spellingMap[letter][vonavpilot_language]) {
                const words = vonavpilot_spellingMap[letter][vonavpilot_language];
                const firstWord = words[0] || "-";
                const span = document.createElement("span");
                span.className = "letter-row";
                span.innerHTML = `<span class="letter">${letter.toUpperCase()}</span><span class="word">| ${firstWord}</span>`;
                container.appendChild(span);
            }
        }
        // Render numbers last (0-9)
        for (let i = 0; i <= 9; i++) {
            const number = i.toString();
            if (vonavpilot_spellingMap[number][vonavpilot_language]) {
                const words = vonavpilot_spellingMap[number][vonavpilot_language];
                const firstWord = words[0] || "-";
                const span = document.createElement("span");
                span.className = "letter-row";
                span.innerHTML = `<span class="letter">${number}</span><span class="word">| ${firstWord}</span>`;
                container.appendChild(span);
            }
        }
        // Render space symbol last
        if (vonavpilot_spellingMap.space[vonavpilot_language]) {
            const words = vonavpilot_spellingMap.space[vonavpilot_language];
            const firstWord = words[0] || "-";
            const span = document.createElement("span");
            span.className = "letter-row";
            span.innerHTML = `<span class="letter">SPACE</span><span class="word">| ${firstWord}</span>`;
            container.appendChild(span);
        }
    }

    function vonavpilot_createUtterance(text) {
        const utterance = new SpeechSynthesisUtterance(text);
        utterance.lang = vonavpilot_language;
        utterance.rate = vonavpilot_utteranceRate;
        utterance.volume = 0.7;
        return utterance;
    }

    function vonavpilot_cancelPageReading() {
        vonavpilot_resetReadTextVisible();
        vonavpilot_readElementIds.clear();
        vonavpilot_cancelPageReadingRequested = true;
        document.querySelectorAll(".vonavpilot-read-highlight").forEach(el =>
            el.classList.remove("vonavpilot-read-highlight")
        );
        vonavpilot_updateInfoBoxText(vonavpilot_messages.pageReadCanceled[vonavpilot_language]);
        vonavpilot_domElements.InfoBox.focus();
        if (vonavpilot_synth && vonavpilot_isSpeechSynthesisSupported) {
            const utterance = vonavpilot_createUtterance(vonavpilot_messages.pageReadCanceled[vonavpilot_language]);
            vonavpilot_speakBlock("icon");
            vonavpilot_synth.speak(utterance);
            utterance.onend = function() {
                vonavpilot_vonavPilotSpeak = false;
                if (vonavpilot_DEBUG.tts) console.log("vonavpilot_cancelPageReading - vonavpilot_stop speaking page text");
                if (vonavpilot_voNavPilotRecognition) {
                    vonavpilot_killVoNavPilotRecognition();
                }
                setTimeout(() => {
                    vonavpilot_domElements.InfoBox.focus();
                    const utterance = vonavpilot_createUtterance(vonavpilot_sanitizeMessageText(vonavpilot_messages.startShort[vonavpilot_language]));
                    vonavpilot_synth.speak(utterance);
                    utterance.onend = function() {
                        vonavpilot_initVoNavPilotRecognition();
                        setTimeout(() => {
                            vonavpilot_updateInfoBoxText(vonavpilot_messages.startShort[vonavpilot_language]);
                            vonavpilot_startRecognition(); // start voNavPilot recognition after intro text finished
                        }, vonavpilot_ttsDelayTime - 300);
                    };
                }, vonavpilot_ttsDelayTime - 700);
            };
        }
    }

    function vonavpilot_start() {
        if (vonavpilot_DEBUG.main) console.log('start');
        if (vonavpilot_noSleep && document.visibilityState === "visible") {
            vonavpilot_noSleep.enable();
        }
        vonavpilot_synth.cancel();
        vonavpilot_speakBlock('text');
        let startMessage;
        if (vonavpilot_storedVnp.session === false) {
            vonavpilot_storedVnp.session = true;
            vonavpilot_saveSettings(vonavpilot_storedVnp);
            if (vonavpilot_DEBUG.main) console.log('start without session, read intro');
            vonavpilot_initReadVoNavPilotIntroRecognition();
            startMessage = '<span class="intro-message">' + vonavpilot_messages.startSkip[vonavpilot_language] + '<br><i>' + vonavpilot_messages.startCommands[vonavpilot_language] + '</i></span>';
            setTimeout(() => {
                if (vonavpilot_readVoNavPilotIntroRecognition && vonavpilot_storedVnp.deactive === false) {
                    vonavpilot_readVoNavPilotIntroRecognition.start();
                    vonavpilot_speakFree();
                    vonavpilot_readVoNavPilotIntroRecognitionTriggers();
                }
            }, vonavpilot_dynamicLanguageDelay(vonavpilot_language, startMessage)); //enable hearing after a dynamic delay du to vonavpilot_language to skip the instruction...
        } else {
            if (vonavpilot_DEBUG.main) console.log('start with session');
            startMessage = vonavpilot_messages.start[vonavpilot_language];
        }
        vonavpilot_updateInfoBoxText(startMessage.replace(/,/g, ''));
        if (vonavpilot_synth && vonavpilot_isSpeechSynthesisSupported) {
            vonavpilot_currentIndex = 0;
            // start speaking the first sentence
            startIntroSpeakNext(sanitizeIntroText(startMessage));
        }
    }

    function vonavpilot_sanitizeMessageText(text) {
        return text.replace(/©/g, "").replace(/<\/?strong>/gi, "");
    }

    function sanitizeIntroText(text){
        const sanitizedMessage = text
            .replace(/<\/?strong>/gi, '')
            .replace(/<\/?h4>/gi, '')
            .replace(/<\/?i>/gi, '')
            .replace(/©/g, '')
            .trim();
        const rawSentences = sanitizedMessage
            .split(/<br\s*\/?>|[,:.]/i)
            .map(s => s.trim())
            .filter(Boolean);
        const sentences = rawSentences.map(s =>
            s.replace(/<\/?[^>]+(>|$)/g, '').trim()
        );
        return sentences;
    }

    function startIntroSpeakNext(sentences) {
        if (vonavpilot_currentIndex >= sentences.length) return; // No more sentences to speak
        const currentSentence = sentences[vonavpilot_currentIndex].trim();
        if (currentSentence === "") {
            vonavpilot_currentIndex++;
            startIntroSpeakNext(sentences); // Skip empty sentences
            return;
        }
        if (vonavpilot_DEBUG.readingIntro) console.log('currentSentence:', currentSentence);
        const utterance = vonavpilot_createUtterance(currentSentence);
        utterance.rate = vonavpilot_utteranceRate;
        vonavpilot_vonavPilotSpeak = true;
        utterance.onend = function() {
            vonavpilot_currentIndex++;
            if (vonavpilot_currentIndex < sentences.length) {
                setTimeout(() => startIntroSpeakNext(sentences), vonavpilot_readTextPauseTime);
            } else {
                if (vonavpilot_voNavPilotRecognition) {
                    vonavpilot_updateInfoBoxText(vonavpilot_messages.startShort[vonavpilot_language]);
                    if (vonavpilot_readVoNavPilotIntroRecognition) {
                        vonavpilot_killReadVoNavPilotIntroRecognition();
                    }
                    vonavpilot_initVoNavPilotRecognition();
                    setTimeout(() => {
                        vonavpilot_vonavPilotSpeak = false;
                        vonavpilot_startRecognition(); // start voNavPilot recognition after intro text/short text finished
                    }, vonavpilot_ttsDelayTime*2);
                }
            }
        };
        // Speak the current sentence
        vonavpilot_synth.speak(utterance);
    }

    function vonavpilot_stop() {
        if (vonavpilot_noSleep) {
            vonavpilot_noSleep.disable();
        }
        if (vonavpilot_voNavPilotStopped) return;
        vonavpilot_voNavPilotStopped = true;
        vonavpilot_killVoNavPilotRecognition();
        vonavpilot_isListening = false;
        vonavpilot_status = "init";
        setTimeout(() => {
            if (vonavpilot_isSpeechSynthesisSupported) {
                const utterance = vonavpilot_createUtterance(vonavpilot_sanitizeMessageText(vonavpilot_messages.stop[vonavpilot_language]));
                vonavpilot_synth.speak(utterance);
                utterance.onend = function() {
                    vonavpilot_domElements.InfoBox.focus();
                };
            } else {
                vonavpilot_domElements.InfoBox.focus();
            }
            vonavpilot_curUserSaid = undefined; //Hide Command in Widget InfoBox, otherwise set to transcript
            vonavpilot_updateInfoBoxText(vonavpilot_messages.stop[vonavpilot_language]);
            vonavpilot_voNavPilotStopped = false;
            vonavpilot_speakBlock("icon");
        }, vonavpilot_ttsDelayTime);
    }

    function vonavpilot_startRecognition() {
        vonavpilot_voNavPilotRecognitionTriggers();
        vonavpilot_speakFree();
        try {
            vonavpilot_voNavPilotRecognition.start();
        } catch (error) {
            if (vonavpilot_DEBUG.recognition) console.error("Speech recognition start failed:", error);
            vonavpilot_stop();
        }
        vonavpilot_cancelPageReadingRequested = false;
        vonavpilot_isListening = true;
        vonavpilot_storedVnp.intro = true;
        vonavpilot_storedVnp.session = true;
        vonavpilot_saveSettings(vonavpilot_storedVnp);
    }
    
    function vonavpilot_spellOutAcronyms(word) {
        const knownAcronyms = ["FAQ", "CSS", "HTML", "API", "JSON", "HTTP", "URL", "JS", "XML", "TCP", "UDP", "SQL", "PHP", "SVG", "DOM", "RAM", "ROM", "SSL", "AWS", "SQL", "IDE", "URL", "FTP", "SaaS", "PaaS", "IaaS", "UI", "UX", "JWT", "OAuth", "REST", "SOAP", "CSS3", "ES6"];
        // Convert the word to uppercase to check against known acronyms
        let upperWord = word.toUpperCase();
        // Check if the word is in the list of known acronyms and has 2 or more characters
        if (knownAcronyms.includes(upperWord)) {
            // Split the acronym and join with spaces (e.g., "F A Q")
            return upperWord.split('').join(' ');
        }
        return word;  // Return the word as is if it's not in the acronym list
    }

    function vonavpilot_detectActivatedVoNavPilot() {
        let infoText;
        if (vonavpilot_DEBUG.main) console.log("vonavpilot_detectActivatedVoNavPilot: voNavPilot is already activated!");
        vonavpilot_speakBlock("text");
        vonavpilot_isListening = true;
        if (vonavpilot_noSleep && document.visibilityState === "visible") {
            vonavpilot_noSleep.enable();
        }
        if (vonavpilot_DEBUG.main) console.log("vonavpilot_detectActivatedVoNavPilot: wait few seconds for the page to load before voice recognition activates automatic.");
        let curPath = location.pathname.split("/").filter(Boolean).pop();
        if (curPath !== undefined) {
            curPath = curPath.replace(vonavpilot_language, "");
            if(curPath !== ""){
                infoText = vonavpilot_messages.pageLoaded[vonavpilot_language].replace("{curPath}", vonavpilot_spellOutAcronyms(curPath));
            } else {
               infoText = vonavpilot_messages.homeLoaded[vonavpilot_language]; 
            }
            
        } else {
            infoText = vonavpilot_messages.homeLoaded[vonavpilot_language];
        }
        if (typeof infoText !== "string") {
            infoText = infoText[vonavpilot_language];
        }
        vonavpilot_updateInfoBoxText(vonavpilot_messages.start[vonavpilot_language]);
        const utterance = vonavpilot_createUtterance(infoText);
        vonavpilot_synth.speak(utterance);
        setTimeout(() => {
            try {
                vonavpilot_speakFree();
                vonavpilot_voNavPilotRecognition.start();
                vonavpilot_voNavPilotRecognitionTriggers();
            } catch (error) {
                if (vonavpilot_DEBUG.main) console.error("Speech recognition start failed:", error);
                vonavpilot_stop();
            }
        }, vonavpilot_ttsDelayTime * 2);
    }

    async function vonavpilot_readVoNavPilotIntroRecognitionTriggers() {
        vonavpilot_readVoNavPilotIntroRecognition.onresult = async (event) => {
            const transcript = event.results[0][0].transcript.toLowerCase().replace(/[.\s]+$/g, "").trim();
            if (vonavpilot_DEBUG.recognition) console.log("vonavpilot_readVoNavPilotIntroRecognitionTriggers - transcript: ", transcript);
            if (Object.values(vonavpilot_controlCommandsMap.skipInstruction).flat().some(w => transcript.toLowerCase().includes(w.toLowerCase()))) {
                // skip reading Intro
                if (vonavpilot_readVoNavPilotIntroRecognition) {
                    vonavpilot_killReadVoNavPilotIntroRecognition();
                }
                if (speechSynthesis.speaking) {
                    speechSynthesis.cancel(); // stop TTS immediately
                }
                vonavpilot_curUserSaid = undefined; //Hide Command in Widget InfoBox, otherwise set to transcript
                vonavpilot_updateInfoBoxText(vonavpilot_messages.startShort[vonavpilot_language]);
                const utterance = vonavpilot_createUtterance(vonavpilot_sanitizeMessageText(vonavpilot_messages.startShort[vonavpilot_language]));
                if (vonavpilot_DEBUG.tts) console.log("start speaking page text");
                vonavpilot_speakBlock("icon");
                vonavpilot_synth.speak(utterance);
                utterance.onend = function() {
                    setTimeout(() => {
                        vonavpilot_vonavPilotSpeak = false;
                        if (vonavpilot_DEBUG.tts) console.log("stop speaking page text");
                        vonavpilot_startRecognition(); // start voNavPilot recognition after speaking page text finished
                    }, vonavpilot_ttsDelayTime - 300);
                };
            } else {
                if (vonavpilot_DEBUG.tts) console.log("Ignoring command during reading. Say 'cancel reading page' to exit reading.");
            }
        };
        vonavpilot_readVoNavPilotIntroRecognition.onend = function() {
            vonavpilot_readVoNavPilotIntroRecognition.start();
        };

        vonavpilot_readVoNavPilotIntroRecognition.onerror = () => {
            setTimeout(() => {
                //vonavpilot_domElements.InfoBox.style.display = "none";
            }, 5000);
        };
    }

    function vonavpilot_voNavPilotRecognitionTriggers() {
        vonavpilot_voNavPilotRecognition.onresult = async (event) => {
            if (vonavpilot_commandDetectionInProgress) return;
            const transcript = event.results[0][0].transcript.toLowerCase().replace(/[.\s]+$/g, "").trim();
            if (vonavpilot_DEBUG.recognition) console.log("vonavpilot_voNavPilotRecognitionTriggers-vonavpilot_vonavPilotSpeak: ", vonavpilot_vonavPilotSpeak);
            if (vonavpilot_vonavPilotSpeak) {
                let matchedCommand = await vonavpilot_getCommandResponse(transcript, "CONTROL-COMMANDS-MAP");
                // During reading, only allow stop reading command
                if (matchedCommand === "cancelPageRead") {
                    // stop the reading
                    if (speechSynthesis.speaking) {
                        speechSynthesis.cancel(); // stop TTS immediately
                    }
                    vonavpilot_cancelPageReading();
                } else {
                    if (vonavpilot_DEBUG.tts) console.log("Ignoring command during reading. Say 'cancel reading page' to exit reading.");
                }
                return; // stop processing other commands
            }
            let debounceTimeout = setTimeout(async () => {
                let matchedCommand = await vonavpilot_getCommandResponse(transcript, "CONTROL-COMMANDS-MAP");
                if (vonavpilot_actionMap[matchedCommand]) {
                    vonavpilot_executeAction(matchedCommand, transcript);
                    if (matchedCommand === "backToPreviousPage" && vonavpilot_homeIS) {
                        matchedCommand = vonavpilot_messages.backNotPossible[vonavpilot_language];
                        setTimeout(() => {
                            if (vonavpilot_isSpeechSynthesisSupported) {
                                const utterance = vonavpilot_createUtterance(matchedCommand);
                                vonavpilot_vonavPilotSpeak = true;
                                if (vonavpilot_DEBUG.recognition) console.log("vonavpilot_voNavPilotRecognitionTriggers debounceTimeout - start speaking page text");
                                vonavpilot_synth.speak(utterance);
                                utterance.onend = function() {
                                    setTimeout(() => {
                                        vonavpilot_vonavPilotSpeak = false;
                                        if (vonavpilot_DEBUG.recognition) console.log("vonavpilot_voNavPilotRecognitionTriggers - stop speaking page text");
                                    }, vonavpilot_ttsDelayTime);
                                };
                            }
                        }, vonavpilot_ttsDelayTime - 700);
                    }
                    vonavpilot_updateInfoBoxText(matchedCommand + "<span class='vonavpilot-success-symbol'>✓</span>");
                } else {
                    vonavpilot_actionMap.UNKOWN();
                    vonavpilot_updateInfoBoxText(matchedCommand + "<span class='vonavpilot-fail-symbol'>✗</span>");
                }
                vonavpilot_domElements.InfoBox.style.display = "block";
                setTimeout(() => {
                    vonavpilot_speakFree();
                }, vonavpilot_ttsDelayTime - 700);
                debounceTimeout = null;
            }, vonavpilot_vonavpilotDelayTime);
        };

        vonavpilot_voNavPilotRecognition.onend = function() {
            if (vonavpilot_isListening) {
                try {
                    setTimeout(()=>{vonavpilot_voNavPilotRecognition.start();}, vonavpilot_vonavpilotDelayTime);
                } catch (error) {
                    if (vonavpilot_DEBUG.recognition) console.error("Speech recognition start failed:", error);
                    vonavpilot_stop();
                }
            } else {
                vonavpilot_stop();
            }
        };

        vonavpilot_voNavPilotRecognition.onerror = () => {
            setTimeout(() => {
                //vonavpilot_domElements.InfoBox.style.display = "none";
            }, 5000);
        };
    }

    async function vonavpilot_executeAction(command, transcript) {
        vonavpilot_synth.cancel();
        vonavpilot_commandDetectionInProgress = true;
        if (vonavpilot_actionMap[command]) {
            await vonavpilot_actionMap[command](transcript);
        }
        setTimeout(() => {
            vonavpilot_commandDetectionInProgress = false;
        }, 2000);
    }

    async function vonavpilot_applyPageFontSize() {
        // Set root font size (affects rem/em fonts)
        document.documentElement.style.fontSize = vonavpilot_fontSizePercent + "%";
        const zoomFactor = vonavpilot_fontSizePercent / 100;
        // Select all visible elements that contain text (common text tags + form controls)
        // You can expand or customize this list as needed
        const textElements = Array.from(document.querySelectorAll(
            "p, span, a, li, dt, dd, h1, h2, h3, h4, h5, h6, button, input, select, textarea, label, div"
        ));
        // Filter elements that have text content and visible (not display:none or visibility:hidden)
        const visibleTextElements = textElements.filter(el => {
            const style = getComputedStyle(el);
            return el.textContent.trim().length > 0 &&
                style.display !== "none" &&
                style.visibility !== "hidden" &&
                el.offsetParent !== null; // visible in layout
        });
        visibleTextElements.forEach(el => {
            // For fallback: apply zoom only if font-size is fixed px (not relative)
            const computedFontSize = getComputedStyle(el).fontSize;
            if (computedFontSize && computedFontSize.endsWith("px")) {
                el.style.zoom = zoomFactor.toString();
            } else {
                // Reset zoom for elements already scaling well
                el.style.zoom = "";
            }
        });
        // Optional: small delay for style update
        await new Promise(resolve => setTimeout(resolve, 50));
    }

    function vonavpilot_removeNoiseWords(input, vonavpilot_stopwords) {
        const inputWords = input.split(/\s+/).filter(w => !vonavpilot_stopwords.has(w.toLowerCase())); // Ensure case insensitivity
        return inputWords.join(" "); // Return the words joined together as a string
    }

    function vonavpilot_hasMeaningfulCommonWords(input, phrase, vonavpilot_stopwords) {
        const inputWords = input.split(/\s+/).filter(w => !vonavpilot_stopwords.has(w));
        const phraseWords = phrase.split(/\s+/).filter(w => !vonavpilot_stopwords.has(w));
        const common = inputWords.filter(word => phraseWords.includes(word));
        return common.length > 0;
    }

    async function vonavpilot_getCommandResponse(prompt, type) {
        const userSaidMatch = prompt.match(/"(.+?)"/);
        let userSaid = userSaidMatch ? userSaidMatch[1].toLowerCase() : prompt.toLowerCase();
        vonavpilot_curUserSaid = userSaid;
        // Remove the word "page" in many languages (case-insensitive), then trim extra spaces
        userSaid = userSaid.replace(
        /\b(seite|page|pagina|pag[eè]|página|pagine|páginas|seiten|paginas|strona|strony|paginae|страница|страницы|σελίδα|sayfa|صفحة|页面|頁面|ページ|쪽)\b/gi,"").trim();
        if (userSaid.endsWith(".")) {
            userSaid = userSaid.slice(0, -1); //remove point if added to end of userSaid
        }
        userSaid = vonavpilot_removeNoiseWords(userSaid, vonavpilot_stopwords);
        if (vonavpilot_DEBUG.commandMap) console.log("userSaid:", userSaid);
        if (vonavpilot_DEBUG.commandMap) console.log("type:", type);
        if (vonavpilot_DEBUG.commandMap) console.log("vonavpilot_spellingMap: ", vonavpilot_spellingMap);
        if (vonavpilot_DEBUG.commandMap) console.log("vonavpilot_controlCommandsMap: ", vonavpilot_controlCommandsMap);
        let controlCommandMapRaw;
        (function replace(obj, name) {
            for (const key in obj) {
                const val = obj[key];
                obj[key] = (typeof val === "string") ?
                    val.replace(/\[BOT-NAME\]/g, name) : (val && typeof val === "object") ? (replace(val, name), val) : val;
            }
        })(vonavpilot_controlCommandsMap, vonavpilot_botName);
        
        if (type === "SPELLING-MAP") {
            controlCommandMapRaw = vonavpilot_spellingMap;
        } else {
            controlCommandMapRaw = vonavpilot_controlCommandsMap;
        }
        const controlCommandMap = {};
        for (const command in controlCommandMapRaw) {
            const entry = controlCommandMapRaw[command];
            if (Array.isArray(entry)) {
                controlCommandMap[command] = entry;
            } else if (typeof entry === "object" && entry !== null) {
                // Merge all languages arrays into one big array for this command
                controlCommandMap[command] = Object.values(entry)
                    .filter(Array.isArray)
                    .flat();
            } else {
                controlCommandMap[command] = [];
            }
        }
        if (type !== "SPELLING-MAP") {
            controlCommandMap.openInternalUrl = [...new Set(vonavpilot_internalLinksNames.map(s => s.trim().replace(/\s+/g, " ")))].map(phrase => {
                const words = phrase.split(/\s+/);
                const halfLen = Math.floor(words.length / 2);
                const firstHalf = words.slice(0, halfLen).join(" ");
                const secondHalf = words.slice(halfLen).join(" ");
                return firstHalf === secondHalf ? firstHalf : phrase;
            });
            controlCommandMap.openUrl = [...new Set(vonavpilot_externalLinksNames.map(s => s.trim().replace(/\s+/g, " ")))].map(phrase => {
                const words = phrase.split(/\s+/);
                const halfLen = Math.floor(words.length / 2);
                const firstHalf = words.slice(0, halfLen).join(" ");
                const secondHalf = words.slice(halfLen).join(" ");
                return firstHalf === secondHalf ? firstHalf : phrase;
            });
        }
        if (vonavpilot_DEBUG.promptMatch) console.log("controlCommandMap: ", controlCommandMap);
        let bestMatchCommand = null;
        let highestScore = -Infinity;
        // 1. Fuzzysort on full input
        for (const [command, phrases] of Object.entries(controlCommandMap)) {
            const results = fuzzysort.go(userSaid, phrases, {
                threshold: vonavpilot_fuzzysortThreshold
            });
            if (vonavpilot_DEBUG.commandMap) console.log("userSaid:", userSaid, "phrases: ", phrases, "Fuzzy results:", results);
            if (
                results.length > 0 &&
                results[0].score > vonavpilot_fuzzysortThreshold &&
                results[0].score > highestScore
            ) {
                // Check meaningful word overlap to avoid false positives
                if (vonavpilot_hasMeaningfulCommonWords(userSaid, results[0].target, vonavpilot_stopwords)) {
                    bestMatchCommand = command;
                    highestScore = results[0].score;
                    if (vonavpilot_DEBUG.promptMatch) console.log("Fuzzy BestMatched:", bestMatchCommand, "HighestScore:", highestScore);
                } else {
                    if (vonavpilot_DEBUG.promptMatch) console.log(`Discarded fuzzy match on "${command}" due to no meaningful word overlap.`);
                }
            } else if (results.length > 0 && vonavpilot_DEBUG.promptMatch) {
                if (vonavpilot_DEBUG.promptMatch) console.log(`Fuzzysort Discarded "${command}" with too low score:`, results[0].score);
            }
        }

        // 2. Levenshtein fallback on full phrase
        if (!bestMatchCommand) {
            let bestScore = Infinity;
            const cleanedUserSaid = userSaid.toLowerCase().trim();
            for (const [command, phrases] of Object.entries(controlCommandMap)) {
                for (const phrase of phrases) {
                    const cleanedPhrase = phrase.toLowerCase().trim();
                    const distance = Levenshtein.get(cleanedUserSaid, cleanedPhrase);
                    const normalizedDistance = distance / Math.max(cleanedUserSaid.length, cleanedPhrase.length);
                    if (vonavpilot_DEBUG.promptMatch) console.log(`Levenshtein Match "${cleanedUserSaid}" with "${cleanedPhrase}" — distance: ${distance}, normalized: ${normalizedDistance}`);
                    //const dynamicThreshold = Math.max(0.45, 1 / cleanedPhrase.length);
                    const dynamicThreshold = Math.min(0.35, 1 / Math.max(6, cleanedPhrase.length));
                    if (normalizedDistance < bestScore && normalizedDistance <= dynamicThreshold) {
                        // Also check meaningful word overlap here
                        if (vonavpilot_hasMeaningfulCommonWords(cleanedUserSaid, cleanedPhrase, vonavpilot_stopwords)) {
                            bestMatchCommand = command;
                            bestScore = normalizedDistance;
                            if (vonavpilot_DEBUG.promptMatch) console.log("Levenshtein Normalized Distance:", normalizedDistance, "userSaid:", cleanedUserSaid, "phrase:", cleanedPhrase);
                        } else {
                            if (vonavpilot_DEBUG.promptMatch) console.log(`Discarded Levenshtein match on "${command}" due to no meaningful word overlap.`);
                        }
                    }
                }
            }
        }

        // 3. Word-level fallback (match partial phrases)
        if (!bestMatchCommand) {
            const userWords = userSaid.toLowerCase().split(/\s+/);
            let bestAvgDistance = Infinity;
            // Define filter function once, outside loops
            for (const [command, phrases] of Object.entries(controlCommandMap)) {
                for (const phrase of phrases) {
                    const phraseWords = phrase.toLowerCase().split(/\s+/);
                    // Use the external function here instead of inline arrow function
                    const filteredUserWords = userWords.filter(vonavpilot_isNotStopword);
                    const filteredPhraseWords = phraseWords.filter(vonavpilot_isNotStopword);
                    let totalDistance = 0;
                    let matchedPairs = 0;
                    let hasMeaningfulMatch = false;
                    for (const userWord of filteredUserWords) {
                        if (userWord.length <= 2) continue;
                        for (const phraseWord of filteredPhraseWords) {
                            if (phraseWord.length <= 2) continue;
                            const distance = Levenshtein.get(userWord, phraseWord);
                            const norm = distance / Math.max(userWord.length, phraseWord.length);
                            if (vonavpilot_DEBUG.promptMatch) console.log(`[Word-level-Match] "${userWord}" vs "${phraseWord}" — dist: ${distance}, norm: ${norm}`);
                            if (norm <= 0.2) {
                                totalDistance += norm;
                                matchedPairs++;
                                if (!vonavpilot_stopwords.has(userWord) && !vonavpilot_stopwords.has(phraseWord)) {
                                    hasMeaningfulMatch = true;
                                }
                            }
                        }
                    }

                    const shortInput = filteredUserWords.length <= 2;
                    const strongSingleMatch = matchedPairs === 1 && totalDistance <= 0.05;
                    const isSafeShortMatch = shortInput && strongSingleMatch && filteredUserWords.length === 1;

                    if ((matchedPairs >= 2 || isSafeShortMatch || (matchedPairs === 1 && totalDistance <= 0.1)) && hasMeaningfulMatch) {
                        const avgDistance = totalDistance / matchedPairs;
                        if (avgDistance < bestAvgDistance) {
                            bestMatchCommand = command;
                            bestAvgDistance = avgDistance;
                            if (vonavpilot_DEBUG.promptMatch) console.log("[Word-Match][Final] Avg:", avgDistance, "Matched Words:", matchedPairs, "=>", command);
                        }
                    }
                }
            }

        }

        return bestMatchCommand || "no match found!";
    }

    function vonavpilot_isNotStopword(word) {
        return !vonavpilot_stopwords.has(word);
    }

    function vonavpilot_getPageTextForTTS() {
        const seenTexts = new Set();
        const textParts = [];
        const elementMap = [];
        document.querySelectorAll(".vonavpilot-vonavpilot_DEBUG-line").forEach(el => el.remove());
        const isVisible = el => {
            const style = window.getComputedStyle(el);
            const rect = el.getBoundingClientRect();
            return rect.bottom > 0 &&
                style.display !== "none" &&
                style.visibility !== "hidden" &&
                el.offsetParent !== null;
        };

        const isExcluded = el =>
            el.closest(".vonavpilot-info") ||
            el.closest(".vonavpilot-content-analyse-output") ||
            el.closest("header");

        const walker = document.createTreeWalker(
            document.body,
            NodeFilter.SHOW_ELEMENT | NodeFilter.SHOW_TEXT,
            {
                acceptNode: node => {
                    if (node.nodeType === Node.TEXT_NODE) {
                        const trimmed = node.textContent.trim();
                        const parent = node.parentElement;
                        if (!trimmed || trimmed.length < 2 || !parent) return NodeFilter.FILTER_SKIP;
                        if (isExcluded(parent) || !isVisible(parent)) return NodeFilter.FILTER_SKIP;
                        const elTop = parent.getBoundingClientRect().top + window.scrollY;
                        if (elTop < window.scrollY - 50) return NodeFilter.FILTER_SKIP;
                        return NodeFilter.FILTER_ACCEPT;
                    }
                    if (
                        ["SCRIPT", "STYLE"].includes(node.tagName) ||
                        isExcluded(node) ||
                        !isVisible(node)
                    ) {
                        return NodeFilter.FILTER_SKIP;
                    }
                    return NodeFilter.FILTER_ACCEPT;
                }
            },
            false
        );

        const addUniqueText = (text, el) => {
            if (text && !seenTexts.has(text)) {
                seenTexts.add(text);
                textParts.push(text);
                elementMap.push({ el });
            }
        };

        const addImageAttr = (attr, node) => {
            const value = node[attr]?.trim();
            if (value) addUniqueText(value, node);
        };

        let currentNode;
        while ((currentNode = walker.nextNode())) {
            if (currentNode.nodeType === Node.TEXT_NODE) {
                addUniqueText(currentNode.textContent.trim(), currentNode.parentElement);
            } else if (currentNode.tagName === "A") {
                addUniqueText(currentNode.innerText.trim(), currentNode);
            } else if (currentNode.tagName === "IMG") {
                addImageAttr("alt", currentNode);
                addImageAttr("title", currentNode);
            } else if (
                currentNode.classList.contains("caption") ||
                currentNode.classList.contains("figcaption")
            ) {
                addUniqueText(currentNode.innerText.trim(), currentNode);
            }
        }
        if (vonavpilot_DEBUG.readPage) console.log("textParts:", textParts);
        return { textParts, elementMap };
    }

    const vonavpilot_actionMap = {
        scrollDown: () => window.scrollBy({
            top: 500,
            behavior: "smooth"
        }),
        scrollUp: () => window.scrollBy({
            top: -500,
            behavior: "smooth"
        }),
        scrollTop: () => window.scrollTo({
            top: 0,
            behavior: "smooth"
        }),
        scrollBottom: () => window.scrollTo({
            top: document.body.scrollHeight,
            behavior: "smooth"
        }),
        backToPreviousPage: () => {
            vonavpilot_speakBlock("text");
            let curPath = location.pathname.split("/").filter(Boolean).pop();
            if (curPath !== undefined) {
                curPath = curPath.replace(vonavpilot_language, "");
                if (curPath !== "") {
                    vonavpilot_homeIS = false;
                    window.history.back();
                } else {
                    vonavpilot_homeIS = true;
                }
            } else {
                vonavpilot_homeIS = true;
            }
        },
        refreshPage: () => {
            vonavpilot_speakBlock("text");
            location.reload();
        },
        stopBot: () => {
            vonavpilot_stop();
            vonavpilot_synth.cancel();
        },
        showCookie: () => {
            const cookieBanner = vonavpilot_cookieBannerIds.find(id => document.getElementById(id));
            if (cookieBanner) {
                document.getElementById(cookieBanner).style.display = "block";
            }
        },
        hideCookie: () => {
            const cookieBanner = vonavpilot_cookieBannerIds.find(id => document.getElementById(id));
            if (cookieBanner) {
                document.getElementById(cookieBanner).style.display = "none";
            }
        },
        switchLan: async (transcript) => {
            vonavpilot_speakBlock("text");
            const cleaned = transcript.toLowerCase();
            const supportedLangs = Object.values(vonavpilot_languageMap);
            for (const [key, code] of Object.entries(vonavpilot_languageMap)) {
                if (cleaned.includes(key)) {
                    const currentUrl = window.location.href;
                    const url = new URL(currentUrl);
                    const pathSegments = url.pathname.split("/");
                    // If URL already starts with a supported lang, replace it
                    if (supportedLangs.includes(pathSegments[1])) {
                        pathSegments[1] = code;
                    } else {
                        // Insert the new vonavpilot_language code after the first slash
                        pathSegments.splice(1, 0, code);
                    }
                    // Rebuild and redirect
                    url.pathname = pathSegments.join("/");
                    window.location.href = url.toString();
                    return;
                }
            }
            if (vonavpilot_DEBUG.main) console.log("switch_lan: No matching vonavpilot_language found in transcript.");
            vonavpilot_speakFree();
        },
        openInternalUrl: async (transcript) => {
            vonavpilot_speakBlock("text");
            if (vonavpilot_DEBUG.main) console.log("openInternalUrl - vonavpilot_linkCommandMap: ", vonavpilot_linkCommandMap);
            const commandWords = Array.isArray(vonavpilot_linkCommandMap.openLinkCommandWords) ?
                vonavpilot_linkCommandMap.openLinkCommandWords :
                Object.values(vonavpilot_linkCommandMap.openLinkCommandWords || {}).flat();
            //Clean up the transcript by removing common command words
            if (vonavpilot_DEBUG.main) console.log("transcript: ", transcript);
            const cleanedTranscript = transcript
                .replace(/seite|page|pagina|pag[eè]|página/gi, "") //remove unnecessary open link words
                .split(" ")
                .filter(word => !commandWords.includes(word.toLowerCase()))
                .join(" ")
                .trim() || transcript;
            if (vonavpilot_DEBUG.main) console.log("openInternalUrl: cleaned Transcript:", cleanedTranscript);
            if (vonavpilot_DEBUG.main) console.log("openInternalUrl: vonavpilot_internalLinksMap: ", vonavpilot_internalLinksMap);
            const results = fuzzysort.go(cleanedTranscript, vonavpilot_internalLinksMap.map(item => item.name), {
                threshold: vonavpilot_fuzzysortThresholdForLinks
            });
            if (vonavpilot_DEBUG.main) console.log("openInternalUrl: Results: ", results);
            if (results.length > 0) {
                const bestMatch = results[0];
                if (vonavpilot_DEBUG.main) console.log("openInternalUrl: Best_Match:", bestMatch.target);
                const matchedLink = vonavpilot_internalLinksMap.find(item => item.name === bestMatch.target);
                if (matchedLink) {
                    const elm = document.querySelector(`a[href="${matchedLink.link}"]`);
                    if (elm) {
                        const targetId = matchedLink.link.split("#")[1];
                        if (matchedLink.link.includes("#")) {

                            const targetElement = document.getElementById(targetId);
                            if (targetElement) {
                                targetElement.scrollIntoView({
                                    behavior: "smooth"
                                });
                                if (vonavpilot_DEBUG.main) console.log("openInternalUrl: Scrolled to element with ID:", targetId);
                            } else {
                                if (vonavpilot_DEBUG.main) console.log("openInternalUrl: ", targetId, "not found!");
                            }
                        }
                    } else {
                        if (vonavpilot_DEBUG.main) console.log("openInternalUrl: Element not Found:", matchedLink.link);
                    }
                } else {
                    if (vonavpilot_DEBUG.main) console.log("openUrl: No matching link found");
                }
            } else {
                vonavpilot_updateInfoBoxText("No match found<span class='vonavpilot-fail-symbol'>✗</span>");
            }
        },
        openUrl: async (transcript) => {
            vonavpilot_speakBlock("text");
            if (vonavpilot_DEBUG.main) console.log("openUrl - vonavpilot_linkCommandMap: ", vonavpilot_linkCommandMap);
            const commandWords = Array.isArray(vonavpilot_linkCommandMap.openLinkCommandWords) ?
                vonavpilot_linkCommandMap.openLinkCommandWords :
                Object.values(vonavpilot_linkCommandMap.openLinkCommandWords || {}).flat();
            //Clean up the transcript by removing common command words
            if (vonavpilot_DEBUG.main) console.log("transcript: ", transcript);
            const cleanedTranscript = transcript
                .replace(/seite|page|pagina|pag[eè]|página/gi, "") //remove unnecessary open link words
                .split(" ")
                .filter(word => !commandWords.includes(word.toLowerCase()))
                .join(" ")
                .trim() || transcript;
            if (vonavpilot_DEBUG.main) console.log("openUrl: cleaned Transcript:", cleanedTranscript);
            if (vonavpilot_DEBUG.main) console.log("openUrl: vonavpilot_externalLinksMap: ", vonavpilot_externalLinksMap);
            const results = fuzzysort.go(cleanedTranscript, vonavpilot_externalLinksMap.map(item => item.name), {
                threshold: vonavpilot_fuzzysortThresholdForLinks
            });
            if (vonavpilot_DEBUG.main) console.log("openUrl: Results: ", results);
            if (results.length > 0) {
                const bestMatch = results[0];
                if (vonavpilot_DEBUG.main) console.log("openUrl: Best_Match:", bestMatch.target);
                const matchedLink = vonavpilot_externalLinksMap.find(item => item.name === bestMatch.target);
                if (matchedLink) {
                    let linkDomain = null;
                    try {
                        linkDomain = new URL(matchedLink.link, window.location.origin).hostname;
                    } catch {
                        if (vonavpilot_DEBUG.main) console.warn("openUrl: Invalid_URL:", matchedLink.link);
                        vonavpilot_updateInfoBoxText("Invalid URL<span class='vonavpilot-fail-symbol'>✗</span>");
                    }
                    if (vonavpilot_DEBUG.main) console.log("openUrl: Opening Link:", matchedLink.link);
                        if (linkDomain === vonavpilot_currentDomain) {
                            window.open(matchedLink.link, "_self");
                        } else {
                            setTimeout(()=>{
                                vonavpilot_updateInfoBoxText(vonavpilot_messages.externalLinkNotOpened[vonavpilot_language] + "<span class='vonavpilot-fail-symbol'>✗</span>");
                                const utterance = vonavpilot_createUtterance(vonavpilot_messages.externalLinkNotOpened[vonavpilot_language]);
                                vonavpilot_vonavPilotSpeak = true;
                                vonavpilot_synth.speak(utterance);
                                utterance.onend = function() {
                                    setTimeout(() => {
                                        vonavpilot_vonavPilotSpeak = false;
                                    }, vonavpilot_ttsDelayTime);
                                };

                            }, vonavpilot_vonavpilotDelayTime);
                        }
                    } else {
                    if (vonavpilot_DEBUG.main) console.log("openUrl: No matching link found");
                }
            } else {
                setTimeout(()=>{
                    vonavpilot_updateInfoBoxText("No match found<span class='vonavpilot-fail-symbol'>✗</span>");
                }, vonavpilot_vonavpilotDelayTime);
            }
        },
        pageZoomIn: () => {
            if (vonavpilot_fontSizePercent < vonavpilot_maxFontSizePercent) {
                vonavpilot_fontSizePercent += vonavpilot_fontSizePercentStep;
                vonavpilot_applyPageFontSize();
                vonavpilot_storedVnp.zoom = vonavpilot_fontSizePercent;
                vonavpilot_saveSettings(vonavpilot_storedVnp);
            }
        },
        pageZoomOut: () => {
            if (vonavpilot_fontSizePercent > vonavpilot_minFontSizePercent) {
                vonavpilot_fontSizePercent -= vonavpilot_fontSizePercentStep;
                vonavpilot_applyPageFontSize();
                vonavpilot_storedVnp.zoom = vonavpilot_fontSizePercent;
                vonavpilot_saveSettings(vonavpilot_storedVnp);
            }
        },
        pageZoomReset: () => {
            vonavpilot_fontSizePercent = vonavpilot_defaultFontSizePercent;
            vonavpilot_applyPageFontSize();
            vonavpilot_storedVnp.zoom = vonavpilot_fontSizePercent;
            vonavpilot_saveSettings(vonavpilot_storedVnp);
        },
        enableHighContrast: () => {
            vonavpilot_enableHighContrast();
            vonavpilot_storedVnp.highContrast = true;
            vonavpilot_saveSettings(vonavpilot_storedVnp);
        },
        disableHighContrast: () => {
            document.body.classList.toggle("vonavpilot-high-contrast-mode", false);
            vonavpilot_storedVnp.highContrast = false;
            vonavpilot_saveSettings(vonavpilot_storedVnp);
        },
        enablePlainView: () => {
            vonavpilot_enablePlainView();
            vonavpilot_storedVnp.plainView = true;
            vonavpilot_saveSettings(vonavpilot_storedVnp);
        },
        disablePlainView: () => {
            vonavpilot_disabledStylesheets.forEach(link => link.disabled = false);
            vonavpilot_disabledStyleTags.forEach(style => style.media = "");
            vonavpilot_disabledInlineStyles.forEach(({
                el,
                style
            }) => {
                el.setAttribute("style", style);
            });
            vonavpilot_disabledStylesheets = [];
            vonavpilot_disabledStyleTags = [];
            vonavpilot_disabledInlineStyles = [];
            setTimeout(() => {
               vonavpilot_setWidgetDynamicDesign();
            }, vonavpilot_vonavpilotDelayTime);
            vonavpilot_storedVnp.plainView = false;
            vonavpilot_saveSettings(vonavpilot_storedVnp);
        },
        enableGrayscale: () => {
            vonavpilot_enableGrayscale();
            vonavpilot_storedVnp.grayScale = true;
            vonavpilot_saveSettings(vonavpilot_storedVnp);
        },
        disableGrayscale: () => {
            document.documentElement.style.filter = "";
            document.documentElement.style.transition = "";
            vonavpilot_storedVnp.grayScale = false;
            vonavpilot_saveSettings(vonavpilot_storedVnp);
        },
        hideImages: () => {
            vonavpilot_hideImages();
            vonavpilot_storedVnp.images = false;
            vonavpilot_saveSettings(vonavpilot_storedVnp);
        },
        showImages: () => {
            document.querySelectorAll("img").forEach(img => {
                if (img.dataset.wasVisible === "true") {
                    img.style.display = "";
                }
                delete img.dataset.wasVisible;
            });
            vonavpilot_storedVnp.images = true;
            vonavpilot_saveSettings(vonavpilot_storedVnp);
        },
        readPage: () => {
            vonavpilot_makeReadTextVisible();
            if (!vonavpilot_isSpeechSynthesisSupported) return;
            const {
                textParts,
                elementMap
            } = vonavpilot_getPageTextForTTS();
            let index = 0;
            let currentEl = null;
            function isElementFullyInView(el, offset = 10) {
                const rect = el.getBoundingClientRect();
                return rect.top >= offset && rect.bottom <= (window.innerHeight - offset);
            }
            function speakNextReadPage() {
                if (vonavpilot_cancelPageReadingRequested) return;
                if (index >= textParts.length) {
                    vonavpilot_vonavPilotSpeak = false;
                    vonavpilot_resetReadTextVisible();
                    return vonavpilot_cancelPageReading?.();
                }
                const text = textParts[index];
                const el = elementMap[index].el;
                if (vonavpilot_DEBUG.readPage) console.log(index, "from", textParts.length - 1, "text:", text);
                if (currentEl) currentEl.classList.remove("vonavpilot-read-highlight");
                if (el) {
                    el.classList.add("vonavpilot-read-highlight");
                    currentEl = el;
                    if (!isElementFullyInView(el, 20)) {
                    const rect = el.getBoundingClientRect();
                    const scrollTop = window.scrollY || window.pageYOffset;
                    const elementTopInDoc = rect.top + scrollTop;
                    const scrollTarget = elementTopInDoc - 100; // Add some padding above
                    window.scrollTo({
                        top: scrollTarget,
                        behavior: "smooth"
                    });
                    setTimeout(() => speakCurrent(text), vonavpilot_ttsDelayTime * 2);
                    vonavpilot_vonavPilotSpeak = true;
                    return;
                }
                }
                speakCurrent(text);
            }

            function speakCurrent(text) {
                const utterance = vonavpilot_createUtterance(text);
                utterance.onend = () => {
                    index++;
                    speakNextReadPage();
                };
                vonavpilot_vonavPilotSpeak = true;
                vonavpilot_synth.speak(utterance);
            }
            speakNextReadPage();
        },
        cancelPageRead: () => {
            if (speechSynthesis.speaking) {
                speechSynthesis.cancel(); // stop TTS immediately
            }
            vonavpilot_cancelPageReading();
        },
        openHome: () => {
            const lang = location.pathname.split('/')[1];
            const homeUrl = /^[a-z]{2}$/.test(lang) ? `/${lang}/` : (window.wpHomeUrl || '/');
            window.location.href = homeUrl;
        },
        fillForm: async () => {
            if (vonavpilot_DEBUG.form) console.log("[voNavPilot] Starting voice-based form filling…");
            const forms = Array.from(document.querySelectorAll("form")).filter(form => {
                if (form.offsetParent === null || form.getAttribute("aria-hidden") === "true") return false;
                const id = form.id?.toLowerCase() || "";
                const classes = form.className?.toLowerCase() || "";
                const action = form.getAttribute("action")?.toLowerCase() || "";
                const innerText = form.innerText?.toLowerCase() || "";
                const looksLikeContact =
                    id.includes("contact") || id.includes("form") ||
                    classes.includes("contact") || classes.includes("form") ||
                    action.includes("contact") || action.includes("form") ||
                    innerText.includes("contact") || innerText.includes("message") || innerText.includes("form");
                const hasSubmitButton = !!form.querySelector("button[type='submit'], input[type='submit']");
                return looksLikeContact && hasSubmitButton;
            });
            if (vonavpilot_DEBUG.form) console.log(`[voNavPilot] Found ${forms.length} contact-like forms.`);
            if (forms.length === 0) {
                setTimeout(()=>{vonavpilot_updateInfoBoxText(vonavpilot_messages.formNotFound[vonavpilot_language] + "<span class='vonavpilot-fail-symbol'>✗</span>");}, vonavpilot_ttsDelayTime);
                if (vonavpilot_synth && vonavpilot_isSpeechSynthesisSupported) {
                    const utterance = vonavpilot_createUtterance(vonavpilot_messages.formNotFound[vonavpilot_language]);
                    vonavpilot_vonavPilotSpeak = true;
                    if (vonavpilot_DEBUG.tts) console.log("forms.length === 0 - start speaking page text");
                    vonavpilot_synth.speak(utterance);
                    utterance.onend = function() {
                        setTimeout(() => {
                            vonavpilot_vonavPilotSpeak = false;
                            if (vonavpilot_DEBUG.tts) console.log("forms.length === 0 - stop speaking page text");
                        }, vonavpilot_ttsDelayTime * 2);
                    };
                }
                return;
            }
            form = forms[0];
            setTimeout(() => {
                if (vonavpilot_DEBUG.form) console.log("vonavpilot_killVoNavPilotRecognition");
                if (vonavpilot_voNavPilotRecognition) {
                    vonavpilot_killVoNavPilotRecognition();
                }
                if (vonavpilot_DEBUG.form) console.log("initReadFillFormInstrunctionsRecognition");
                initReadFillFormInstrunctionsRecognition();
                vonavpilot_curUserSaid = undefined; //Hide Command in Widget InfoBox, otherwise set to transcript
                vonavpilot_updateInfoBoxText("<br>" + vonavpilot_messages.formFillingInit[vonavpilot_language] + "<br><br>" + vonavpilot_messages.formFillingCommands[vonavpilot_language].replace(/,/g, ""));
                vonavpilot_domElements.InfoBox.focus();
                vonavpilot_speakBlock("text");
            }, vonavpilot_ttsDelayTime);
            if (vonavpilot_DEBUG.form) console.log("fill_form(formSpeak): enable form speaking reading fill form instructions");
            formSpeak = true;
            const text = "<br>" + vonavpilot_messages.formFillingInit[vonavpilot_language] + "<br><br>" + vonavpilot_messages.formFillingCommands[vonavpilot_language].replace(/,/g, "");
            setTimeout(() => {
                if (vonavpilot_storedVnp.deactive === false) {
                    setTimeout(() => {
                        vonavpilot_speakFree();
                    }, vonavpilot_ttsDelayTime);
                    if (vonavpilot_DEBUG.form) console.log("fill_form(formSpeak): disable form speaking to hear skip command");
                    formSpeak = false;
                    vonavpilot_readFillFormInstrunctionsRecognition.start();
                    vonavpilot_startReadingFormInstructionsRecognition(form);
                }
            }, vonavpilot_dynamicLanguageDelay(vonavpilot_language, text)); //enable hearing after a dynamic delay du to vonavpilot_language to skip the instructions...
            const rawMessage = vonavpilot_messages.formFillingInit[vonavpilot_language] + " " + vonavpilot_messages.formFillingCommands[vonavpilot_language];
            const sanitizedMessage = rawMessage
                .replace(/<\/?strong>/gi, "")
                .replace(/<\/?h4>/gi, "")
                .replace(/<\/?i>/gi, "")
                .replace(/©/g, "")
                .trim();
            const rawSentences = sanitizedMessage
                .split(/<br\s*\/?>|[,:.]/i)
                .map(s => s.trim())
                .filter(Boolean);
            const sentences = rawSentences.map(s =>
                s.replace(/<\/?[^>]+(>|$)/g, "").trim()
            );
            let vonavpilot_currentIndex = 0;

            function speakNextFormIntro() {
                if (vonavpilot_currentIndex >= sentences.length) return; // No more sentences to speak
                const currentSentence = sentences[vonavpilot_currentIndex].trim();
                if (currentSentence === "") {
                    vonavpilot_currentIndex++;
                    speakNextFormIntro(); // Skip empty sentences
                    return;
                }
                const utterance = vonavpilot_createUtterance(currentSentence);
                utterance.rate = vonavpilot_utteranceRate;
                utterance.onend = function() {
                    vonavpilot_currentIndex++;
                    if (vonavpilot_currentIndex < sentences.length) {
                        setTimeout(speakNextFormIntro, vonavpilot_readTextPauseTime * 5);
                    } else {
                        if (vonavpilot_readFillFormInstrunctionsRecognition) {
                            killReadFillFormInstrunctionsRecognition();
                            if (vonavpilot_DEBUG.form) console.log("vonavpilot_initFillFormRecognition");
                            if (vonavpilot_DEBUG.form) console.log("fill_form(formSpeak): disable form speaking after form filling instructions finished");
                            formSpeak = false;
                            if (vonavpilot_DEBUG.form) console.log("vonavpilot_fillFormRecognition...");
                            vonavpilot_initFillFormRecognition();
                            setTimeout(() => {
                                vonavpilot_startFormFilling(form, "start form filling after instructions finished!");
                                vonavpilot_initSpellingTable();
                            }, vonavpilot_ttsDelayTime);
                        }
                    }
                };
                // Speak the current sentence
                vonavpilot_synth.speak(utterance);
            }
            // start speaking the first sentence
            speakNextFormIntro();
        },
        showInstruction: () => {
            vonavpilot_synth.cancel();
            vonavpilot_killVoNavPilotRecognition();
            vonavpilot_initReadVoNavPilotIntroRecognition();
            let startMessage;
            startMessage = "<span class='intro-message'><i>" + vonavpilot_messages.startCommands[vonavpilot_language] + "</i></span>";
            setTimeout(() => {
                vonavpilot_speakBlock("text");
                document.querySelector(".vonavpilot-info-text").innerHTML = startMessage.replace(/,/g, "");
                setTimeout(() => {
                    vonavpilot_readVoNavPilotIntroRecognition.start();
                    vonavpilot_readVoNavPilotIntroRecognitionTriggers();
                    vonavpilot_speakFree();
                }, vonavpilot_dynamicLanguageDelay(vonavpilot_language, startMessage));
                if (vonavpilot_synth && vonavpilot_isSpeechSynthesisSupported) {
                    const sentences = sanitizeIntroText(startMessage);
                    // start speaking the first sentence
                    vonavpilot_currentIndex = 0;
                    vonavpilot_speakNext(sentences);
                }
            }, vonavpilot_ttsDelayTime);
        },
        resetAll: () => {
            vonavpilot_storedVnp.images = true;
            vonavpilot_storedVnp.plainView = false;
            vonavpilot_storedVnp.zoom = vonavpilot_defaultFontSizePercent;
            vonavpilot_storedVnp.grayScale =  false;
            vonavpilot_storedVnp.highContrast = false;
            vonavpilot_saveSettings(vonavpilot_storedVnp);
            location.reload();
        },
        UNKOWN: () => {
            if (vonavpilot_DEBUG.main) console.log("vonavpilot_actionMap: Unknown Command!");
        }
    };

    function vonavpilot_makeReadTextVisible(){
        const hiddenElements = document.querySelectorAll('.hidden');
        hiddenElements.forEach(el => {
            el.classList.add('vonavpilot-read-highlight-visible'); // Temporarily add a class to make it visible
        });
    }

    function vonavpilot_resetReadTextVisible(){
        const hiddenElements = document.querySelectorAll('.hidden');
        hiddenElements.forEach(el => {
            el.classList.remove('vonavpilot-read-highlight-visible'); // remove all Temporarily classes
        });
    }

    function vonavpilot_hideImages(){
        document.querySelectorAll("img").forEach(img => {
            const style = window.getComputedStyle(img);
            if (style.display !== "none") { // only hide if currently visible
                img.dataset.wasVisible = "true"; // mark it was visible
                img.style.setProperty('display', 'none', 'important');
            } else {
                img.dataset.wasVisible = "false"; // was already hidden
            }
        });
    }

    function vonavpilot_enableGrayscale() {
        document.documentElement.style.transition = "filter 0.5s ease";
        document.documentElement.style.filter = "grayscale(100%)";
    }

    function vonavpilot_enableHighContrast() {
        document.body.classList.toggle("vonavpilot-high-contrast-mode", true);
    }

    function vonavpilot_enablePlainView () {
        vonavpilot_disabledStylesheets = Array.from(document.querySelectorAll("link[rel='stylesheet']"));
        vonavpilot_disabledStylesheets.forEach(link => link.disabled = true);
        vonavpilot_disabledStyleTags = Array.from(document.querySelectorAll("style"));
        vonavpilot_disabledStyleTags.forEach(style => style.media = "none");
        vonavpilot_disabledInlineStyles = [];
        const allElements = document.querySelectorAll("*");
        allElements.forEach(el => {
            if (el.hasAttribute("style")) {
                vonavpilot_disabledInlineStyles.push({
                    el,
                    style: el.getAttribute("style")
                });
                el.removeAttribute("style");
            }
        });
    }

    function vonavpilot_speakNext(sentences) {
        if (vonavpilot_currentIndex >= sentences.length) return; // No more sentences to speak
        const currentSentence = sentences[vonavpilot_currentIndex].trim();
        if (currentSentence === "") {
            vonavpilot_currentIndex++;
            vonavpilot_speakNext(); // Skip empty sentences
            return;
        }
        const utterance = vonavpilot_createUtterance(currentSentence);
        utterance.rate = vonavpilot_utteranceRate;
        utterance.onend = function() {
            vonavpilot_currentIndex++;
            if (vonavpilot_currentIndex < sentences.length) {
                setTimeout(vonavpilot_speakNext(sentences), vonavpilot_readTextPauseTime);
            } else {
                if (vonavpilot_readVoNavPilotIntroRecognition) {
                    vonavpilot_updateInfoBoxText(vonavpilot_messages.startShort[vonavpilot_language]);
                    setTimeout(() => {
                        vonavpilot_killReadVoNavPilotIntroRecognition();
                        vonavpilot_startRecognition();
                    }, vonavpilot_ttsDelayTime - 300);
                }
            }
        };
        // Speak the current sentence
        vonavpilot_synth.speak(utterance);
    }

    function vonavpilot_dynamicLanguageDelay(lang, text) {
        // Default words per second (adjust these experimentally)
        const wordsPerSecond = {
            en: 2.8,  // English
            de: 2, // German (slower due to longer words)
            es: 2.6,  // Spanish
            fr: 3,  // French
            it: 2.6,  // Italian
        };
        // Get the number of words in the text
        const words = text.trim().split(/\s+/).length;
        // Scale words per second based on the vonavpilot_utteranceRate (global variable)
        const wps = (wordsPerSecond[lang] || 2) * vonavpilot_utteranceRate;
        // Calculate total delay (ms) + 500ms buffer
        const totalDelayMs = (words / wps) * 1000 + 500;
        // 50% of the total delay to start recognition early
        const partialDelay = totalDelayMs * 0.5;
        if(vonavpilot_DEBUG.tts) console.log('vonavpilot_dynamicLanguageDelay vonavpilot_utteranceRate:', vonavpilot_utteranceRate);
        if(vonavpilot_DEBUG.tts) console.log('vonavpilot_dynamicLanguageDelay words:', words);
        if(vonavpilot_DEBUG.tts) console.log('vonavpilot_dynamicLanguageDelay totalDelay:', totalDelayMs*0.001, 'partialDelay (50%):', partialDelay*0.001);
        return partialDelay;
    }

    function vonavpilot_startFormFilling(form, message) {
        if (vonavpilot_DEBUG.form) console.log("vonavpilot_startFormFilling: ", message);
        vonavpilot_fillFormRecognition.start();
        vonavpilot_startFillingFormRecognition();
        setTimeout(() => {
            vonavpilot_startFillFormProcess(form);
        }, vonavpilot_ttsDelayTime);
    }

    async function vonavpilot_startReadingFormInstructionsRecognition(form) {
        vonavpilot_readFillFormInstrunctionsRecognition.onresult = async (event) => {
            const transcript = event.results[0][0].transcript.toLowerCase().replace(/[.\s]+$/g, "").trim();
            if (vonavpilot_DEBUG.form) console.log("vonavpilot_readFillFormInstrunctionsRecognition userSaid:", transcript);
            if (vonavpilot_DEBUG.form) console.log("vonavpilot_readFillFormInstrunctionsRecognition formSpeak:", formSpeak);
            if (vonavpilot_DEBUG.form) console.log("vonavpilot_readFillFormInstrunctionsRecognition formSpelling:", formSpelling);
            if (formSpelling) {
                formSpeak = false;
            }
            // Prevent processing if speech is ongoing
            if (formSpeak) return;
            if (Object.values(formControlCommandsMap.skipInstruction).flat().some(w => transcript.toLowerCase().includes(w.toLowerCase()))) {
                if (vonavpilot_DEBUG.form) console.log("skip reading instructions...");
                vonavpilot_playBeep();
                vonavpilot_speakBlock("icon");
                if (vonavpilot_readFillFormInstrunctionsRecognition) {
                    if (vonavpilot_DEBUG.form) console.log("killReadFillFormInstrunctionsRecognition...");
                    killReadFillFormInstrunctionsRecognition();
                }
                vonavpilot_synth.cancel();
                vonavpilot_curUserSaid = undefined; //Hide Command in Widget InfoBox, otherwise set to transcript
                vonavpilot_updateInfoBoxText(vonavpilot_messages.formFillingStarted[vonavpilot_language] + "<br><br>" + vonavpilot_messages.formFillingCommands[vonavpilot_language].replace(/,/g, ""));
                vonavpilot_domElements.InfoBox.focus();
                const utterance = vonavpilot_createUtterance(vonavpilot_sanitizeMessageText(vonavpilot_messages.formFillingStarted[vonavpilot_language]).split("<br>")[0]);
                formSpeak = true;
                if (vonavpilot_DEBUG.form) console.log("form_filling_start: enable form speaking to say form filling start");
                vonavpilot_speakBlock("icon");
                vonavpilot_synth.speak(utterance);
                utterance.onend = function() {
                    setTimeout(() => {
                        vonavpilot_synth.cancel();
                        if (vonavpilot_DEBUG.form) console.log("form_filling_start(formSpeak): disable form speaking to hear skip command");
                        if (vonavpilot_DEBUG.form) console.log("form_filling_start(formSpelling): allow form filling commands to fillout the form");
                        formSpeak = false;
                        formSpelling = true;
                        vonavpilot_initFillFormRecognition();
                        vonavpilot_startFormFilling(form, "start form filling after skip command!");
                        vonavpilot_initSpellingTable();
                        vonavpilot_updateInfoBoxText(vonavpilot_messages.formFillingStarted[vonavpilot_language] + "<br><br>" + vonavpilot_messages.formFillingCommands[vonavpilot_language].replace(/,/g, ""));
                        vonavpilot_speakFree();
                    }, vonavpilot_ttsDelayTime - 700);
                };
            } else {
                if (vonavpilot_DEBUG.form) console.log("ignored what user said...");
            }
        };

        vonavpilot_readFillFormInstrunctionsRecognition.onend = async () => {
            vonavpilot_readFillFormInstrunctionsRecognition.start();
        };

        vonavpilot_readFillFormInstrunctionsRecognition.onerror = async () => {
            //vonavpilot_domElements.InfoBox.style.display = "none";
        };
    }

    function vonavpilot_createSpellingListText() {
        const punctuationMap = {
            "en": {
                ".": "dot",
                "-": "dash",
                "space": "space"
            },
            "de": {
                ".": "Punkt",
                "-": "Bindestrich",
                "space": "Leerzeichen"
            },
            "fr": {
                ".": "point",
                "-": "tiret",
                "space": "espace"
            },
            "es": {
                ".": "punto",
                "-": "guion",
                "space": "espacio"
            },
            "it": {
                ".": "punto",
                "-": "trattino",
                "space": "spazio"
            },
        };
        // Select all the letter-row elements
        const letterRows = document.querySelectorAll(".letter-row");
        // Create an empty array to hold the speaking list
        const speakingList = [];
        // Iterate over each letter-row
        letterRows.forEach(row => {
            // Get the letter and word from the row
            let letter = row.querySelector(".letter").textContent.trim();
            const word = row.querySelector(".word").textContent.trim().replace("|", "").trim(); // Remove the "|" and trim spaces
            // Format and push the speaking phrase to the list
            Object.keys(punctuationMap[vonavpilot_language]).forEach(punctuation => {
                const replacement = punctuationMap[vonavpilot_language][punctuation];
                letter = letter.replace(punctuation, replacement); // Replace punctuation with its word equivalent^
            });
            speakingList.push(`${word} for ${letter}`);
        });
        if (vonavpilot_DEBUG.form) console.log(speakingList);
        return speakingList;
    }

    async function vonavpilot_startFillingFormRecognition() {
        vonavpilot_fillFormRecognition.onresult = (event) => {
            const transcript = event.results[0][0].transcript.toLowerCase().replace(/[.\s]+$/g, "").trim();
            vonavpilot_curUserSaid = transcript;
            if (vonavpilot_DEBUG.form) console.log("vonavpilot_fillFormRecognition userSaid:", transcript);
            if (vonavpilot_DEBUG.form) console.log("vonavpilot_fillFormRecognition formSpeak:", formSpeak);
            if (vonavpilot_DEBUG.form) console.log("vonavpilot_fillFormRecognition formSpelling:", formSpelling);
            if (formSpelling) {
                formSpeak = false;
            }
            // Prevent processing if speech is ongoing
            if (formSpeak) return;
            // Spelling mode takes priority but respects TTS
            if (formSpelling && !formSpeak) {
                vonavpilot_handleSpellingInput(transcript);
                return;
            }
        };

        vonavpilot_fillFormRecognition.onend = async () => {
            vonavpilot_fillFormRecognition.start();
        };

        vonavpilot_fillFormRecognition.onerror = async () => {
            //vonavpilot_domElements.InfoBox.style.display = "none";
        };
    }

    function vonavpilot_startFillFormProcess(form) {
        if (vonavpilot_DEBUG.form) console.log("vonavpilot_startFillFormProcess...");
        formFields = Array.from(form.elements).filter(el => {
            const tag = el.tagName;
            const type = el.type ? el.type.toLowerCase() : "";
            const isFillableTag = ["input", "textarea", "select"].includes(tag.toLowerCase());
            const isIgnoredType = ["hidden", "submit", "button", "file", "image", "reset"].includes(type);
            const isVisible = el.offsetParent !== null &&
                el.getAttribute("aria-hidden") !== "true" &&
                el.style.display !== "none" &&
                el.style.visibility !== "hidden" &&
                window.getComputedStyle(el).visibility !== "hidden";
            const isPluginWrapperHidden = el.closest("[style*='display: none'], [style*='visibility: hidden'], [aria-hidden='true']");
            return isFillableTag && !isIgnoredType && !el.disabled && isVisible && !isPluginWrapperHidden;
        });
        if (vonavpilot_DEBUG.form) console.log(`[voNavPilot] Found ${formFields.length} visible, fillable fields in the selected form.`);
        if (formFields.length === 0) {
            if (vonavpilot_synth && vonavpilot_isSpeechSynthesisSupported) {
                const utterance = vonavpilot_createUtterance(vonavpilot_messages.formFieldNotFound[vonavpilot_language]);
                if (vonavpilot_DEBUG.form) console.log("start_fill_form_process: enable form speaking to say form not found");
                formSpeak = true;
                vonavpilot_synth.speak(utterance);
                utterance.onend = function() {
                    setTimeout(() => {
                        if (vonavpilot_DEBUG.form) console.log("start_fill_form_process: disable form speaking to hear form filling commands");
                        formSpeak = false;
                    }, vonavpilot_ttsDelayTime - 700);
                };
            }
            return;
        } else {
            if (vonavpilot_DEBUG.form) console.log("Form field found, start Spelling...");
            formSpelling = true; //allow form filling commands to fillout the form
            vonavpilot_activeFormField = formFields[vonavpilot_fieldCont];
            vonavpilot_addSpellingBorder(vonavpilot_fieldCont);
            vonavpilot_askForLabel(vonavpilot_activeFormField);
        }
    }

    function vonavpilot_askForLabel(vonavpilot_activeFormField) {
        const askPrompt = vonavpilot_messages.formAskFor[vonavpilot_language] + " " + vonavpilot_getLabelForField(vonavpilot_activeFormField).replace(/\*/g, "") + "?";
        const utterance = vonavpilot_createUtterance(askPrompt);
        vonavpilot_domElements.InfoBox.classList.add("vonavpilot-form-filling");
        if (vonavpilot_DEBUG.form) console.log("ask_for_label: enable form speaking to say filed label");
        formSpeak = true;
        formSpelling = false;
        vonavpilot_speakBlock("icon");
        vonavpilot_synth.speak(utterance);
        utterance.onend = () => {
            if (vonavpilot_DEBUG.form) console.log("ask_for_label: disable form speaking to hear form filling commands");
            setTimeout(() => {
                formSpeak = false;
                formSpelling = true;
                vonavpilot_speakFree();
                if (vonavpilot_activeFormField.tagName.toLowerCase() === "select") {
                    const options = vonavpilot_activeFormField.options;
                    if (options.length > 0) {
                        // Find the first non-disabled option starting from index 0 or 1
                        let indexToSelect = 0;
                        if (options[0].disabled) {
                            if (options.length > 1 && !options[1].disabled) {
                                indexToSelect = 1;
                            } else {
                                // fallback if option 1 also disabled or doesn't exist
                                indexToSelect = 0;
                            }
                        }
                        vonavpilot_activeFormField.selectedIndex = indexToSelect;
                        const autoSelectListText = vonavpilot_getLabelForField(vonavpilot_activeFormField).replace(/\*/g, "") + vonavpilot_messages.formSelectAutoSet[vonavpilot_language] + options[indexToSelect].text;
                        const utterance = vonavpilot_createUtterance(autoSelectListText);
                        formSpeak = true;
                        vonavpilot_synth.speak(utterance);
                        utterance.onend = () => {
                            formSpeak = false;
                        };
                    }
                }
                if (
                    vonavpilot_activeFormField.tagName.toLowerCase() === "input" &&
                    vonavpilot_activeFormField.type === "checkbox"
                    ) {
                    const label = document.querySelector(`label[for="${vonavpilot_activeFormField.id}"]`) || vonavpilot_activeFormField.closest("label");
                    const isRequired = vonavpilot_activeFormField.required || (label?.innerText.includes("*"));
                    if (isRequired && !vonavpilot_activeFormField.checked) {
                        vonavpilot_activeFormField.addEventListener("change", () => vonavpilot_DEBUG.form && console.log("✓ change fired"));
                        vonavpilot_activeFormField.addEventListener("input", () => vonavpilot_DEBUG.form && console.log("✓ input fired"));
                        vonavpilot_activeFormField.click();
                        const autoCheckText = vonavpilot_getLabelForField(vonavpilot_activeFormField).replace(/\*/g, "") + vonavpilot_messages.formCheckboxAutoSet[vonavpilot_language];
                        const utterance = vonavpilot_createUtterance(autoCheckText);
                        formSpeak = true;
                        vonavpilot_synth.speak(utterance);
                        utterance.onend = () => { 
                            vonavpilot_removeSpellingBorder();
                            formSpeak = false; 
                        };
                    }
                }
            }, vonavpilot_ttsDelayTime);
        };
    }

    function vonavpilot_addSpellingBorder(count) {
        formFields.forEach((field, index) => {
            field.classList.remove("vonavpilot-spelling-border");
            if (index === count) {
                field.classList.add("vonavpilot-spelling-border");
                field.scrollIntoView({
                    behavior: "smooth",
                    block: "center"
                });
            }
        });
    }

    function vonavpilot_removeSpellingBorder() {
        formFields.forEach((field) => {
            field.classList.remove("vonavpilot-spelling-border");
        });
    }

    async function vonavpilot_handleSpellingInput(transcript) {
        if (vonavpilot_DEBUG.form) console.log("vonavpilot_handleSpellingInput, transcript: ", transcript);
        if (vonavpilot_DEBUG.form) console.log("vonavpilot_handleSpellingInput, formControlCommandsMap: ", formControlCommandsMap);
        let matchedLetter = await vonavpilot_getCommandResponse(transcript, "SPELLING-MAP");
        if (vonavpilot_DEBUG.form) console.log("matchedLetter: ", matchedLetter);
        if (matchedLetter !== "no match found!") {
            if (matchedLetter === "space") {
                vonavpilot_activeFormField.value += " ";
            } else {
                vonavpilot_activeFormField.value += matchedLetter;
            }
            vonavpilot_updateInfoBoxText(vonavpilot_messages.formFillingStarted[vonavpilot_language] + "<br><br>" + vonavpilot_messages.formFillingCommands[vonavpilot_language].replace(/,/g, ""));
            vonavpilot_domElements.InfoBox.focus();
            const mapping = {
                ".": vonavpilot_messages.point[vonavpilot_language],
                "-": vonavpilot_messages.dash[vonavpilot_language]
            };
            let pronouncingMatchedLetter = mapping[matchedLetter] || matchedLetter;
            const utterance = vonavpilot_createUtterance(pronouncingMatchedLetter + " " + vonavpilot_messages.formSpellingEntered[vonavpilot_language]);
            if (vonavpilot_DEBUG.form) console.log("handle_spelling-input(letterMatched): enable form speaking last matched letter");
            formSpeak = true;
            formSpelling = false;
            vonavpilot_speakBlock("icon");
            vonavpilot_synth.speak(utterance);
            utterance.onend = function() {
                setTimeout(() => {
                    if (vonavpilot_DEBUG.form) console.log("handle_spelling-input(letterMatched): disable form speaking to hear form filling commands");
                    formSpeak = false;
                    formSpelling = true;
                    vonavpilot_speakFree();
                }, vonavpilot_ttsDelayTime - 300);

            };
        }
        if (Object.values(formControlCommandsMap.closeSpellingTable).flat().some(w => transcript.toLowerCase().includes(w.toLowerCase()))) {
            formSpeak = true;
            formSpelling = false;
            vonavpilot_domElements.SpellingMap.classList.add("slide-out");
            if (vonavpilot_DEBUG.form) console.log("closeSpellingTable");
            setTimeout(() => {
                formSpeak = false;
                formSpelling = true;
            }, vonavpilot_ttsDelayTime);
            return;
        }
        if (Object.values(formControlCommandsMap.openSpellingTable).flat().some(w => transcript.toLowerCase().includes(w.toLowerCase()))) {
            formSpeak = true;
            formSpelling = false;
            vonavpilot_domElements.SpellingMap.classList.remove("slide-out");
            if (vonavpilot_DEBUG.form) console.log("openSpellingTable");
            setTimeout(() => {
                formSpeak = false;
                formSpelling = true;
            }, vonavpilot_ttsDelayTime);
            return;
        }
        if (Object.values(formControlCommandsMap.readSpellingTable).flat().some(w => transcript.toLowerCase().includes(w.toLowerCase()))) {
            if (vonavpilot_DEBUG.form) console.log("start reading letter spelling Table ...");
            vonavpilot_speakBlock("text");
            vonavpilot_synth.cancel();
            vonavpilot_domElements.button.innerHTML = vonavpilot_waitPointsAnimated;
            setTimeout(() => {
                const utterance = vonavpilot_createUtterance(vonavpilot_createSpellingListText());
                if (vonavpilot_DEBUG.form) console.log("read_letter_spelling_table: enable form speaking to say spelling table content");
                formSpeak = true;
                vonavpilot_synth.speak(utterance);
                vonavpilot_domElements.SpellingMap.classList.remove("slide-out");
                utterance.onend = function() {
                    setTimeout(() => {
                        if (vonavpilot_fillFormRecognition) {
                            vonavpilot_killFillFormRecognition();
                        }
                        vonavpilot_synth.cancel();
                        if (vonavpilot_DEBUG.form) console.log("read_letter_spelling_table(formSpeak): disable form speaking after spelling table content finished");
                        if (vonavpilot_DEBUG.form) console.log("read_letter_spelling_table(formSpelling): allow form filling commands to fillout the form");
                        formSpeak = false;
                        formSpelling = true;
                        vonavpilot_domElements.SpellingMap.classList.add("slide-out");
                        vonavpilot_initFillFormRecognition();
                        vonavpilot_startFormFilling(form, "start form filling after spelling Table finished!");
                        vonavpilot_updateInfoBoxText(vonavpilot_messages.formFillingStarted[vonavpilot_language] + "<br><br>" + vonavpilot_messages.formFillingCommands[vonavpilot_language].replace(/,/g, ""));
                        vonavpilot_speakFree();
                    }, vonavpilot_ttsDelayTime - 700);
                };
            }, vonavpilot_ttsDelayTime - 700);

            return;
        }
        if (Object.values(formControlCommandsMap.stopFormFilling).flat().some(w => transcript.toLowerCase().includes(w.toLowerCase()))) {
            if (vonavpilot_DEBUG.form) console.log("vonavpilot_handleSpellingInput - stop form filling");
            if (vonavpilot_fillFormRecognition) {
                vonavpilot_killFillFormRecognition();
            }
            vonavpilot_domElements.InfoBox.classList.remove("vonavpilot-form-filling");
            document.querySelector(".vonavpilot-contact-form-help").remove();
            vonavpilot_curUserSaid = undefined; //Hide Command in Widget InfoBox, otherwise set to transcript
            vonavpilot_updateInfoBoxText(vonavpilot_messages.formFillingStop[vonavpilot_language]);
            vonavpilot_domElements.InfoBox.focus();
            const utterance = vonavpilot_createUtterance(vonavpilot_messages.formFillingStop[vonavpilot_language]);
            if (vonavpilot_DEBUG.form) console.log("handle_spelling-input(stop): enable form speaking to say form filling stoped");
            formSpeak = true;
            formSpelling = false;
            vonavpilot_speakBlock("icon");
            vonavpilot_synth.speak(utterance);
            vonavpilot_removeSpellingBorder();
            utterance.onend = function() {
                if (vonavpilot_DEBUG.form) console.log("handle_spelling-input(stop): disable form speaking to hear vonavpilot command");
                if (vonavpilot_DEBUG.form) console.log("handle_spelling-input(stop): disable form filling commands");
                formSpeak = false;
                vonavpilot_initVoNavPilotRecognition();
                setTimeout(() => {
                    vonavpilot_speakFree();
                    vonavpilot_voNavPilotRecognition.start();
                    vonavpilot_voNavPilotRecognitionTriggers();
                }, vonavpilot_ttsDelayTime - 700);
            };
        }
        if (Object.values(formControlCommandsMap?.fillTheMessage || {}).flat().some(w => transcript.toLowerCase().includes(w.toLowerCase()))) {
            if (vonavpilot_DEBUG.form) console.log("fill message command - fill out the subject and message");
            formFields.forEach((field, index) => {
                const label = vonavpilot_getLabelForField(field).toLowerCase();
                // Handle subject field
                if (field.value.trim() === "" && Object.values(formControlCommandsMap?.subjectFieldLabelDetection || {}).flat().some(w => label.includes(w.toLowerCase()))) {
                    field.value = vonavpilot_messages.formFillingHelpSubject?.[vonavpilot_language] || "Default Subject";
                    vonavpilot_fieldCont = index;
                    vonavpilot_addSpellingBorder(vonavpilot_fieldCont);
                }
                // Handle message field
                if (field.value.trim() === "" && Object.values(formControlCommandsMap?.messageFieldLabelDetection || {}).flat().some(w => label.includes(w.toLowerCase()))) {
                    field.value = vonavpilot_messages.formFillingHelpMessage?.[vonavpilot_language] || "Default Message";
                    vonavpilot_fieldCont = index;
                    vonavpilot_addSpellingBorder(vonavpilot_fieldCont);
                    vonavpilot_updateInfoBoxText(
                        `${vonavpilot_messages.formMessageFilled?.[vonavpilot_language] || ""}<br>` +
                        `${vonavpilot_messages.formFillingStarted?.[vonavpilot_language] || ""}<br><br>` +
                        `${(vonavpilot_messages.formFillingCommands?.[vonavpilot_language] || "").replace(/,/g, "")}`
                    );
                    vonavpilot_domElements.InfoBox.focus();
                    const utterance = vonavpilot_createUtterance(
                        `${vonavpilot_messages.formMessageFilled?.[vonavpilot_language] || ""}, ${vonavpilot_messages.formFillingHelpMessage?.[vonavpilot_language] || ""}`
                    );
                    if (vonavpilot_DEBUG.form) {console.log("form-handle_spelling-input(fillTheMessage): speaking pre-defined vonavpilot_messages");}
                    formSpeak = true;
                    formSpelling = false;
                    vonavpilot_synth.speak(utterance);
                    utterance.onend = function () {
                        setTimeout(() => {
                            if (vonavpilot_DEBUG.form) {console.log("form-handle_spelling-input(fillTheMessage): finished speaking, enabling input");}
                            formSpeak = false;
                            formSpelling = true;
                        }, vonavpilot_ttsDelayTime);
                    };
                }
            });
            return;
        }

        if (Object.values(formControlCommandsMap.redo).flat().some(w => transcript.toLowerCase().includes(w.toLowerCase()))) {
            if (vonavpilot_DEBUG.form) console.log("back command - remove last character");
            const lastChar = vonavpilot_activeFormField.value.slice(-1);
            vonavpilot_activeFormField.value = vonavpilot_activeFormField.value.slice(0, -1);
            vonavpilot_updateInfoBoxText(vonavpilot_messages.formSpellingDeleted[vonavpilot_language] + "<br>" + vonavpilot_messages.formFillingStarted[vonavpilot_language] + "<br><br>" + vonavpilot_messages.formFillingCommands[vonavpilot_language].replace(/,/g, ""));
            vonavpilot_domElements.InfoBox.focus();
            const utterance = vonavpilot_createUtterance(lastChar + " " + vonavpilot_messages.formSpellingDeleted[vonavpilot_language]);
            if (vonavpilot_DEBUG.form) console.log("form-handle_spelling-input(redo): enable form speaking to say last entry deleted");
            formSpeak = true;
            formSpelling = false;
            vonavpilot_speakBlock("icon");
            vonavpilot_synth.speak(utterance);
            utterance.onend = function() {
                setTimeout(() => {
                    if (vonavpilot_DEBUG.form) console.log("form-handle_spelling-input(redo): disable form speaking to hear form filling commands");
                    formSpeak = false;
                    formSpelling = true;
                    vonavpilot_speakFree();
                }, vonavpilot_ttsDelayTime * 2);
            };
            return;
        }
        if (Object.values(formControlCommandsMap.delete).flat().some(w => transcript.toLowerCase().includes(w.toLowerCase()))) {
            if (vonavpilot_DEBUG.form) console.log("delete command - clear the field");
            vonavpilot_activeFormField.value = ""; // clear whole field
            vonavpilot_updateInfoBoxText(vonavpilot_messages.formSpellingDeleted[vonavpilot_language] + "<br>" + vonavpilot_messages.formFillingStarted[vonavpilot_language] + "<br><br>" + vonavpilot_messages.formFillingCommands[vonavpilot_language].replace(/,/g, ""));
            vonavpilot_domElements.InfoBox.focus();
            const utterance = vonavpilot_createUtterance(vonavpilot_messages.formSpellingDeleted[vonavpilot_language]);
            if (vonavpilot_DEBUG.form) console.log("form-handle_spelling-input(delete): enable form speaking to say field deleted");
            formSpeak = true;
            formSpelling = false;
            vonavpilot_speakBlock("icon");
            vonavpilot_synth.speak(utterance);
            utterance.onend = function() {
                setTimeout(() => {
                    if (vonavpilot_DEBUG.form) console.log("form-handle_spelling-input(delete): disable form speaking to hear form filling commands");
                    formSpeak = false;
                    formSpelling = true;
                    vonavpilot_speakFree();
                }, vonavpilot_ttsDelayTime * 2);
            };
            return;
        }
        if (Object.values(formControlCommandsMap.previous).flat().some(w => transcript.toLowerCase().includes(w.toLowerCase()))) {
            if (vonavpilot_fieldCont > 0) {
                vonavpilot_fieldCont--;
                vonavpilot_activeFormField = formFields[vonavpilot_fieldCont];
                vonavpilot_addSpellingBorder(vonavpilot_fieldCont);
                vonavpilot_askForLabel(vonavpilot_activeFormField);
            }
            if (vonavpilot_DEBUG.form) console.log("goto previous field", vonavpilot_fieldCont, formFields.length);
            return;
        }
        if (Object.values(formControlCommandsMap.next).flat().some(w => transcript.toLowerCase().includes(w.toLowerCase()))) {
            if (vonavpilot_fieldCont < formFields.length - 1) {
                vonavpilot_fieldCont++;
                vonavpilot_activeFormField = formFields[vonavpilot_fieldCont];
                vonavpilot_addSpellingBorder(vonavpilot_fieldCont);
                vonavpilot_askForLabel(vonavpilot_activeFormField);
            }
            if (vonavpilot_DEBUG.form) console.log("goto next field", vonavpilot_fieldCont, formFields.length);
            return;
        }
        if (Object.values(formControlCommandsMap.send).flat().some(w => transcript.toLowerCase().includes(w.toLowerCase()))) {
            if (checkIfFormCompleted()) {
                vonavpilot_playBeep();
                vonavpilot_updateInfoBoxText(vonavpilot_messages.formWillSent[vonavpilot_language] + "<br>" + vonavpilot_messages.formFillingStarted[vonavpilot_language] + "<br><br>" + vonavpilot_messages.formFillingCommands[vonavpilot_language].replace(/,/g, ""));
                const utterance = vonavpilot_createUtterance(vonavpilot_messages.formWillSent[vonavpilot_language]);
                if (vonavpilot_DEBUG.form) console.log("form-handle_spelling-input(send): enable form speaking to say form will send");
                formSpeak = true;
                vonavpilot_speakBlock("text");
                vonavpilot_synth.speak(utterance);
                utterance.onend = function() {
                    document.querySelector(".vonavpilot-contact-form-help").remove();
                    if (form) {
                        setTimeout(() => {
                            const event = new Event("submit", {
                                cancelable: true
                            });
                            if (vonavpilot_DEBUG.form) console.log("init form submit event", form.dispatchEvent(event));
                            if (form.dispatchEvent(event)) {
                                if (vonavpilot_DEBUG.form) console.log("trig form submit!");
                                form.submit();
                                if (vonavpilot_fillFormRecognition) {
                                    vonavpilot_killFillFormRecognition();
                                }
                                setTimeout(() => {
                                    if (vonavpilot_DEBUG.form) console.log("form-handle_spelling-input(send): disable form speaking before send form");
                                    formSpeak = false;
                                    if (formFields) {
                                        vonavpilot_resetForm(); // reset form after form sent
                                    }
                                    setTimeout(() => {
                                        location.reload(); //refresh, back to form page
                                    }, vonavpilot_ttsDelayTime);
                                }, vonavpilot_dynamicLanguageDelay(vonavpilot_language, vonavpilot_messages.formWillSent[vonavpilot_language]));
                            } else {
                                if (vonavpilot_DEBUG.form) console.log("form Submit cancelled- maybe because of a reCaptcha!");
                                setTimeout(() => {
                                    if (vonavpilot_DEBUG.form) console.log("check if captcha Task Iframe detected!");
                                    const recaptchaIframe = document.querySelector("iframe[src*='recaptcha']");
                                    if (vonavpilot_DEBUG.form) console.log("recaptchaIframe", recaptchaIframe);
                                    if (recaptchaIframe !== null) {
                                        const utterance = vonavpilot_createUtterance(vonavpilot_messages.formRecaptchaDetected[vonavpilot_language]);
                                        if (vonavpilot_DEBUG.form) console.log("form-handle_spelling-input(send): enable form speaking to say captcha detected");
                                        formSpeak = true;
                                        formSpelling = false;
                                        vonavpilot_speakBlock("text");
                                        vonavpilot_synth.speak(utterance);
                                        vonavpilot_detectFormSendConfirmationMessage();
                                        utterance.onend = () => {
                                            formSpeak = false;
                                            if (vonavpilot_fillFormRecognition) {
                                                vonavpilot_killFillFormRecognition();
                                            }
                                            let timeLeft = 2 * 60 * 1000; // 2 minutes in milliseconds
                                            const updateTimer = () => {
                                                const m = Math.floor(timeLeft / 60000),
                                                    s = Math.floor((timeLeft % 60000) / 1000);
                                                vonavpilot_domElements.InfoBox.classList.remove("vonavpilot-form-filling");
                                                vonavpilot_domElements.InfoBox.innerHTML = `<span class="vonavpilot-timer">${m}:${s < 10 ? "0" : ""}${s}</span>`;
                                                document.querySelector(".vonavpilot-info").style.padding = "30px";
                                                if (timeLeft <= 0) {
                                                    clearInterval(interval);
                                                    location.reload();
                                                }
                                                timeLeft -= 1000;
                                            };
                                            const interval = setInterval(updateTimer, 1000);
                                            updateTimer(); // Initial call to display the timer immediately
                                            if (vonavpilot_DEBUG.form) console.log("form-handle_spelling-input(send): disbale form speaking + wait 2 Minutes for Captcha Handling");
                                        };
                                    } else {
                                        if (vonavpilot_DEBUG.form) console.log("form unknow vonavpilot_status! refresh page in 10 Seconds");
                                        formSpeak = false;
                                        setTimeout(() => {
                                            location.reload();
                                        }, vonavpilot_ttsDelayTime * 10); // refresh, back to form page after 10 Seconds
                                    }
                                }, vonavpilot_ttsDelayTime * 3);
                            }
                        }, vonavpilot_ttsDelayTime * 2);
                    }
                };
            }
            return;
        }
    }

    function vonavpilot_detectFormSendConfirmationMessage() {
        if (vonavpilot_DEBUG.form) console.log("Starting confirmation check...");
        const confirmationSelectors = [
            ".wpforms-confirmation-container",
            ".wpforms-confirmation-container-full",
            ".wpforms-confirmation-scroll",
            ".wpcf7-response-output",
            ".nf-response-msg",
            ".nf-form-errors",
            ".ninja-forms-success-msg",
            ".gform_confirmation_message",
            ".gform_confirmation_wrapper",
            ".frm_message",
            ".frm_success_style",
            ".ff-message-success",
            ".ff-message-warning",
            ".everest-forms-message",
            ".evf-message-success",
            ".happyforms-message",
            ".happyforms-success",
            ".quform-success-message",
            ".quform-error-message",
            ".caldera-grid .console.log-success",
            ".caldera-grid .console.log-danger"
        ];

        const successClasses = [
            "wpcf7-mail-sent-ok",
            "wpforms-confirmation-container",
            "wpforms-confirmation-container-full",
            "frm_success_style",
            "ff-message-success",
            "happyforms-success",
            "quform-success-message",
            "gform_confirmation_message",
            "evf-message-success"
        ];

        const errorClasses = [
            "wpcf7-validation-errors",
            "wpcf7-not-valid",
            "wpcf7-response-output", // needs context
            "nf-error",
            "ff-message-warning",
            "quform-error-message",
            "caldera-grid .console.log-danger"
        ];

        const interval = setInterval(() => {
            let found = false;
            for (const selector of confirmationSelectors) {
                const el = document.querySelector(selector);
                if (el && el.offsetParent !== null && el.textContent.trim().length > 0) {
                    const text = el.textContent.trim();
                    const classList = [...el.classList];
                    // Check for known success
                    const isSuccess = successClasses.some(cls => classList.includes(cls) || el.matches(`.${cls}`));
                    const isError = errorClasses.some(cls => classList.includes(cls) || el.matches(`.${cls}`));
                    // Special handling for .wpcf7-response-output
                    if (selector === ".wpcf7-response-output" && !isSuccess) {
                        if (classList.includes("wpcf7-mail-sent-ok")) {
                            // Confirmed success
                            speakSuccess(text);
                        } else {
                            // It's an error
                            speakError(text);
                        }
                        clearInterval(interval);
                        found = true;
                        break;
                    }
                    if (isSuccess) {
                        speakSuccess(text);
                        clearInterval(interval);
                        found = true;
                        break;
                    }
                    if (isError) {
                        speakError(text);
                        clearInterval(interval);
                        found = true;
                        break;
                    }
                }
            }
        }, 1000);

        function speakSuccess(text) {
            const message = vonavpilot_messages.formSentSuccess[vonavpilot_language] + " " + text;
            if (vonavpilot_DEBUG.form) console.log("✓ Form confirmation message (success):", message);
            const utterance = vonavpilot_createUtterance(message);
            formSpeak = true;
            vonavpilot_speakBlock("text");
            vonavpilot_synth.speak(utterance);
            utterance.onend = () => {
                location.reload();
            };
        }

        function speakError(text) {
            const message = vonavpilot_messages.formSentError[vonavpilot_language] + " " + text;
            if (vonavpilot_DEBUG.form) console.log("✗ Form error message:", message);
            const utterance = vonavpilot_createUtterance(message);
            formSpeak = true;
            vonavpilot_speakBlock("text");
            vonavpilot_synth.speak(utterance);
            utterance.onend = () => {
                formSpeak = false;
                vonavpilot_speakFree();
            };
        }
    }

    function vonavpilot_getLabelForField(field) {
        const prettify = (text) => {
            return vonavpilot_prettifyName(text.trim());
        };
        // 1. Labels attached to the field
        if (field.labels?.length) {
            const labelTextSet = new Set(
            Array.from(field.labels)
                .filter(label => label.getAttribute("aria-hidden") !== "true")
                .map(label => {
                const labelCopy = label.cloneNode(true);
                labelCopy.querySelectorAll("select").forEach(sel => sel.remove());
                return labelCopy.innerText.trim();
                })
                .filter(text => text)
            );
            const labelText = Array.from(labelTextSet).join(" ");
            if (labelText) {
                if(vonavpilot_DEBUG.form) console.log("Return from labels attached to field:", prettify(labelText));
                return prettify(labelText);
            }
        }
        // 2. aria-label
        const ariaLabel = field.getAttribute("aria-label");
        if (ariaLabel) {
            if(vonavpilot_DEBUG.form) console.log("Return from aria-label:", prettify(ariaLabel));
            return prettify(ariaLabel);
        }
        // 3. Placeholder
        if (field.placeholder) {
            if(vonavpilot_DEBUG.form) console.log("Return from placeholder:", prettify(field.placeholder));
            return prettify(field.placeholder);
        }
        // 4. Search in container
        const container = field.closest(
            ".wpforms-field, .wpforms-field-row-block, .gfield, .wpcf7-form-control-wrap, .form-group, .field, .form-field, " +
            ".nf-field-container, .frm_form_field, .ff-el-group, .evf-field, .happyforms-field, .quform-element, .caldera-grid-cell"
        ) || field.parentElement;
        if (container) {
            const pluginLabel = container.querySelector(
                "label, .gfield_label, .wpforms-field-label, .form-label, .field-label, .nf-field-label, " +
                ".frm_primary_label, .ff-el-label, .evf-label, .happyforms-label, .quform-label, .caldera-label"
            );
            if (pluginLabel?.innerText.trim()) {
                if(vonavpilot_DEBUG.form) console.log("Return from container pluginLabel:", prettify(pluginLabel.innerText.trim()));
                return prettify(pluginLabel.innerText);
            }
            const describedById = field.getAttribute("aria-describedby");
            if (describedById) {
                const describedElem = document.getElementById(describedById);
                if (describedElem?.innerText.trim()) {
                    if(vonavpilot_DEBUG.form) console.log("Return from aria-describedby element:", prettify(describedElem.innerText.trim()));
                    return prettify(describedElem.innerText);
                }
            }
        }
        // 5 & 6 combined with select handling
        if (field.name) {
            if (field.tagName?.toLowerCase() === "select") {
                const selectedOption = field.selectedOptions?.[0] || field.options[field.selectedIndex];
                if (selectedOption?.text.trim()) {
                    const result = prettify(field.name) + ": " + prettify(selectedOption.text);
                    if(vonavpilot_DEBUG.form) console.log("Return from select with selected option:", result);
                    return result;
                }
            }
            if(vonavpilot_DEBUG.form) console.log("Return from fallback name:", prettify(field.name));
            return prettify(field.name);
        }
        if(vonavpilot_DEBUG.form) console.log("Return default: Field");
        return "Field";
    }

    function vonavpilot_prettifyName(name) {
        return name
            .replace(/[_-]/g, " ") // Replace underscores and hyphens with spaces
            .replace(/\bemail\s*(address)?\b/gi, "Mail Address") // Normalize "email" or "email address"
            .split(/\s+/) // Split by one or more spaces
            .filter(Boolean) // Remove empty strings
            .map(word => word.charAt(0).toUpperCase() + word.slice(1)) // Capitalize each word
            .join(" ") // Join words back with a single space
            .replace(/[ *]/g, ''); // Remove space and asterisks from end 
    }

    function checkIfFormCompleted() {
        let messageSpoken = false;
        let allFilled = true;
        formFields.forEach((field, index) => {
            const isRequiredAttr = field.hasAttribute("required") || field.getAttribute("aria-required") === "true";
            const requiredClassPatterns = [
                "required",                       // General/common
                "is-required",                   // General/common
                "frm_required_field",            // Formidable Forms
                "ff-el-is-required",             // Fluent Forms
                "evf-required",                  // Everest Forms
                "happyforms-required",           // HappyForms
                "quform-required",               // Quform
                "caldera-required",             // Caldera Forms
                "wpforms-field-required",        // WPForms
                "wpcf7-validates-as-required",   // Contact Form 7
                "gfield_required",               // Gravity Forms
                "form-required",                 // General
                "field-required"                 // General
            ];
            const isRequiredClass = [...field.classList].some(cls =>
                requiredClassPatterns.includes(cls.toLowerCase()) ||
                requiredClassPatterns.some(reqCls => cls.toLowerCase().includes(reqCls))
            );

            // check if checkbox is required by asterisk in label text
            let isRequiredByLabelAsterisk = false;
            if (field.type === "checkbox") {
                const labelElem = field.closest("label");
                if (labelElem) {
                    // Check if label text ends with '*', trimming whitespace
                    isRequiredByLabelAsterisk = labelElem.textContent.trim().endsWith("*");
                }
            }

            if (isRequiredAttr || isRequiredClass || isRequiredByLabelAsterisk) {
                //if (field.type === "checkbox" && isRequiredByLabelAsterisk) {
                if (field.type === "checkbox" && (isRequiredAttr || isRequiredClass || isRequiredByLabelAsterisk)) {
                    // For required acceptance checkboxes, check if checked
                    if (!field.checked && !messageSpoken) {
                        if (vonavpilot_DEBUG.form) console.log("check_form_completed: enable form speaking to say acceptance checkbox not checked");
                        allFilled = false;
                        formSpeak = true;
                        messageSpoken = true;
                        const utterance = vonavpilot_createUtterance(vonavpilot_messages.formFillingMissing[vonavpilot_language]);
                        vonavpilot_speakBlock("icon");
                        vonavpilot_synth.speak(utterance);
                        utterance.onend = () => {
                            setTimeout(() => {
                                if (vonavpilot_DEBUG.form) console.log("check_form_completed: disbale form speaking to hear form filling commands");
                                formSpeak = false;
                                messageSpoken = false;
                                allFilled = false;
                                vonavpilot_speakFree();
                            }, vonavpilot_ttsDelayTime * 4);
                        };
                    }
                } else {
                    // Existing logic for other field types
                    const value = field.value?.trim();
                    // Validate email if present
                    if (field.type === "email" && value) {
                        const emailPattern = /^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$/;
                        if (!emailPattern.test(value)) {
                            allFilled = false;
                            if (!messageSpoken) {
                                messageSpoken = true;
                                if (vonavpilot_DEBUG.form) console.log("check_form_completed: enable form speaking to say invalid email");
                                vonavpilot_removeSpellingBorder();
                                setTimeout(() => {
                                    vonavpilot_fieldCont = index;
                                    vonavpilot_activeFormField = formFields[vonavpilot_fieldCont];
                                    field.classList.add("vonavpilot-spelling-border");
                                    field.scrollIntoView({
                                        behavior: "smooth",
                                        block: "center"
                                    });
                                }, vonavpilot_ttsDelayTime - 700);
                                const utterance = vonavpilot_createUtterance(vonavpilot_messages.formInvalidEmail[vonavpilot_language]);
                                formSpeak = true;
                                formSpelling = false;
                                vonavpilot_speakBlock("icon");
                                vonavpilot_synth.speak(utterance);
                                utterance.onend = () => {
                                    setTimeout(() => {
                                        if (vonavpilot_DEBUG.form) console.log("check_form_completed: disbale form speaking to hear form filling commands");
                                        formSpeak = false;
                                        formSpelling = true;
                                        messageSpoken = false;
                                        vonavpilot_speakFree();
                                    }, vonavpilot_ttsDelayTime * 2);
                                };
                            }
                        }
                    }

                    // Check if field is filled
                    if (!value && !messageSpoken) {
                        if (vonavpilot_DEBUG.form) console.log("check_form_completed: enable form speaking to say some field not filled");
                        allFilled = false;
                        formSpeak = true;
                        messageSpoken = true;
                        const utterance = vonavpilot_createUtterance(vonavpilot_messages.formFillingMissing[vonavpilot_language]);
                        vonavpilot_speakBlock("icon");
                        vonavpilot_synth.speak(utterance);
                        utterance.onend = () => {
                            setTimeout(() => {
                                if (vonavpilot_DEBUG.form) console.log("check_form_completed: disbale form speaking to hear form filling commands");
                                formSpeak = false;
                                messageSpoken = false;
                                vonavpilot_speakFree();
                            }, vonavpilot_ttsDelayTime * 4);
                        };
                    }
                }
            }
        });
        if (allFilled) {
            vonavpilot_removeSpellingBorder();
            const utterance = vonavpilot_createUtterance(vonavpilot_messages.formFillingDone[vonavpilot_language]);
            if (vonavpilot_DEBUG.form) console.log("check_form_completed: enable form speaking to say all fields filled, send or cancel form");
            formSpeak = true;
            formSpelling = false;
            vonavpilot_speakBlock("icon");
            vonavpilot_domElements.button.innerHTML = vonavpilot_waitPointsAnimated;
            vonavpilot_synth.speak(utterance);
            utterance.onend = () => {
                setTimeout(() => {
                    if (vonavpilot_DEBUG.form) console.log("check_form_completed: disbale form speaking to hear form filling commands");
                    formSpeak = false;
                    formSpelling = true;
                    vonavpilot_domElements.button.innerHTML = vonavpilot_micSVGAnimated;
                    vonavpilot_speakFree();
                }, vonavpilot_ttsDelayTime * 5);
            };
        }
        return allFilled;
    }

    function vonavpilot_resetForm() {
        formFields.forEach(field => {
            const tag = field.tagName.toLowerCase();
            if (tag === "select") {
                field.selectedIndex = 0;
            } else if (tag === "input" && (field.type === "checkbox" || field.type === "radio")) {
                field.checked = false;
            } else {
                field.value = "";
            }
        });
    }
})();