<?php
/*
 * Plugin Name:     Voice Search Optimization
 * Description:     A plugin to enable voice search for page/post titles and content.
 * Plugin URI:      https://profiles.wordpress.org/iflairwebtechnologies
 * Version:         1.1.1
 * Author:          iFlair Web Technologies Pvt. Ltd.
 * Author URI:      https://www.iflair.com/
 * Text Domain:     voice-search-optimization
 * License:         GPLv2 or later
*/

if (!defined('ABSPATH'))
{
    exit; // EXIT IF ACCESSED DIRECTLY
}

// START - DEFINE AND REQUIRE PLUGIN DIRECTORY PATH/FILE
define('VOICSEOP_STR_DIR_PATH', plugin_dir_path(__FILE__)); // PLUGIN DIRECTORY
define('VOICSEOP_STR_DIR_URL', plugin_dir_url(__FILE__)); // PLUGIN URL
define('VOICSEOP_STR_PLUGIN_IMG_DIR_URL', VOICSEOP_STR_DIR_URL . "assets/images/");

// START - ENQUEUE FRONT SCRIPTS AND STYLES
function voicseop_enqueue_scripts()
{
    $voicseop_version = wp_rand(0, 99);

    wp_enqueue_script('jquery');

    // ENQUEUE THE VOICE SEARCH JS FILE
    wp_enqueue_script('voicseop-front-script', plugin_dir_url(__FILE__) . 'assets/frontend/js/vso-front-script.js', array('jquery'), $voicseop_version, true);

    $voicseop_voice_search_timeout = get_option('voicseop_voice_search_timeout', 20);
    $voicseop_voice_command_timeout = get_option('voicseop_voice_command_timeout', 20);

    // LOCALIZE THE SCRIPT WITH HOME_URL AND AJAX_URL
    wp_localize_script('voicseop-front-script', 'voicseop_ajax', array(
        'ajax_url' => admin_url('admin-ajax.php'),
        'home_url' => home_url(),
        'nonce' => wp_create_nonce('voicseop_nonce'), // Add nonce
        'nonce_voice_cm' => wp_create_nonce('voicseop_nonce_voice_cm'), // Add nonce
        'voicseop_voice_search_timeout' => $voicseop_voice_search_timeout,
        'voicseop_voice_command_timeout' => $voicseop_voice_command_timeout
    ));

    // ENQUEUE THE CSS FILE FOR THE PLUGIN
    wp_enqueue_style('voicseop-front-style', plugin_dir_url(__FILE__) . 'assets/frontend/css/vso-front-style.css', array(), $voicseop_version);
}
add_action('wp_enqueue_scripts', 'voicseop_enqueue_scripts');
// END - ENQUEUE FRONT SCRIPTS AND STYLES

// START - ENQUEUE BACKEND SCRIPTS / FILES
function voicseop_enqueue_admin_scripts()
{
    $voicseop_version = wp_rand(0, 99);

    wp_enqueue_script('jquery');

    // CHECK IF WE ARE IN ADMIN AREA
    if (is_admin())
    {
        // ENQUEUE BACKEND CSS
        wp_enqueue_style('voicseop-admin-style', plugin_dir_url(__FILE__) . 'assets/backend/css/vso-admin-style.css', array(), $voicseop_version);

        wp_enqueue_script('voicseop-admin-script', plugin_dir_url(__FILE__) . 'assets/backend/js/vso-admin-script.js', array('jquery'), $voicseop_version, true);
    }
}
add_action('admin_enqueue_scripts', 'voicseop_enqueue_admin_scripts');
// END - ENQUEUE BACKEND SCRIPTS / FILES

// START - ADD ADMIN MENU
function voicseop_add_admin_menu()
{
    add_menu_page('Voice Search Settings', 'Voice Search', 'manage_options', 'vso-settings', 'voicseop_render_settings_page', 'dashicons-microphone', 6);
    add_action('admin_init', 'voicseop_func_plugin_settings');
}
add_action('admin_menu', 'voicseop_add_admin_menu');
// END - ADD ADMIN MENU

// START - CALLBACK FUNCTION FOR PLUGIN MAIN MENU PAGE
function voicseop_render_settings_page()
{
    include_once VOICSEOP_STR_DIR_PATH . 'backend/vso-settings-page.php';
}
// END - CALLBACK FUNCTION FOR PLUGIN MAIN MENU PAGE

// START - FUNCTION PLUGIN ACTIVATION HOOK
function voicseop_plugin_activate()
{
    // BY DEFAULT WHEN ACTIVE PLUGIN THEN FILL DEFAULT SETTING
    $user_form_success_msg = array(
        'voicseop_enable_text_search' => 'yes',
        'voicseop_enable_voice_search' => 'yes',
        'voicseop_voice_search_timeout' => '20',
        'voicseop_enable_voice_command' => 'yes',
        'voicseop_voice_command_timeout' => '20',
        'voicseop_voice_command_position' => 'bottom-right',
    );
    foreach ($user_form_success_msg as $key => $success_value)
    {
        update_option($key, $success_value);
    }
}
register_activation_hook(__FILE__, 'voicseop_plugin_activate');
// END - FUNCTION PLUGIN ACTIVATION HOOK

// REGISTER SETTING
function voicseop_func_plugin_settings()
{
    register_setting('voicseop_plugin_settings', 'voicseop_enable_text_search' ,'voicseop_fields_validate_settings');
    register_setting('voicseop_plugin_settings', 'voicseop_enable_voice_search' ,'voicseop_fields_validate_settings');
    register_setting('voicseop_plugin_settings', 'voicseop_voice_search_timeout' ,'voicseop_fields_validate_settings');
    register_setting('voicseop_plugin_settings', 'voicseop_enable_voice_command' ,'voicseop_fields_validate_settings');
    register_setting('voicseop_plugin_settings', 'voicseop_voice_command_timeout' ,'voicseop_fields_validate_settings');
    register_setting('voicseop_plugin_settings', 'voicseop_voice_command_position' ,'voicseop_fields_validate_settings');
}
// END, REGISTER SETTING

// START CALLBACK FUNCTION TO VALIDATE AND SANITIZE SETTINGS INPUT.
function voicseop_fields_validate_settings($input)
{
    // Validate and sanitize input if necessary
    return sanitize_text_field($input);
}

// START - FUNCTION PLUGIN DEACTIVATION HOOK
function voicseop_plugin_deactivate()
{
    // CLEANUP TASKS WHEN THE PLUGIN IS DEACTIVATED, LIKE CLEARING SCHEDULED EVENTS
    // NO PERMANENT DATA DELETION HERE
}
register_deactivation_hook(__FILE__, 'voicseop_plugin_deactivate');
// END - FUNCTION PLUGIN DEACTIVATION HOOK

// START - FUNCTION PLUGIN UNINSTALLATION HOOK
function voicseop_plugin_uninstall()
{
    // CLEANUP TASKS WHEN THE PLUGIN IS UNINSTALLED
    // REMOVE OPTIONS, DATABASE ENTRIES, OR OTHER PERSISTENT DATA
    delete_option('voicseop_plugin_settings');
}
register_uninstall_hook(__FILE__, 'voicseop_plugin_uninstall');
// END - FUNCTION PLUGIN UNINSTALLATION HOOK

// START - CREATE THE SEARCH FORM WITH SHORTCODE
// START - CREATE THE SEARCH FORM WITH SHORTCODE
function voicseop_display_search_form()
{
    // GET PLUGIN SETTINGS
    $vso_search_bar_show = get_option('voicseop_enable_text_search');
    $vso_voice_search = get_option('voicseop_enable_voice_search');

    ob_start();
    if ($vso_search_bar_show == 'yes' || $vso_voice_search == 'yes') { 
        ?>
        <div >
            <form id="vso-search-form" method="get" action="<?php echo esc_url(home_url('/')); ?>">
                <?php wp_nonce_field('voicseop_nonce', 'voicseop_nonce_field'); // Add nonce field ?>
                <div id="vso-search-container" class="search-filter-head">
                    <?php 
                    // CHECK IF TEXT SEARCH IS ENABLED
                    if ($vso_search_bar_show == 'yes') { ?>
                        <div class="search-input-text-type">
                            <input type="text" id="vso-search-input" class="vso-search-input" name="s" placeholder="Type to search..." value="<?php echo isset($_GET['s']) ? esc_attr(sanitize_text_field(wp_unslash($_GET['s']))) : ''; ?>" />
                        </div>
                    <?php } ?>

                    <!-- Button Group -->
                    <div class="search-btn-group">
                        <?php if ($vso_search_bar_show == 'yes') { ?>
                            <button id="vso-clear-button" class="vso-clear-button" title="Clear">
                                <img src="<?php echo esc_url(VOICSEOP_STR_PLUGIN_IMG_DIR_URL . 'clear-icon.png'); ?>" alt="Clear" />
                            </button>
                            <span class="separator-icon">|</span>
                            <button id="vso-search-button" class="vso-search-button" title="Search">
                                <img src="<?php echo esc_url(VOICSEOP_STR_PLUGIN_IMG_DIR_URL . 'search-icon.png'); ?>" alt="Search" />
                            </button>
                            <input type="hidden" id="vso-nonce" value="<?php echo esc_attr(wp_create_nonce('voicseop_nonce')); ?>" />
                        <?php } ?>

                        <?php 
                        // ADD A HIDDEN INPUT FIELD FOR VOICE SEARCH IF ENABLED
                        if ($vso_voice_search == 'yes') {
                            if ($vso_search_bar_show != 'yes') {
                                // IF TEXT SEARCH IS DISABLED, PROVIDE A HIDDEN INPUT FOR VOICE SEARCH
                                echo '<input type="hidden" id="vso-search-input" class="vso-search-input" />';
                            } ?>
                            <button id="vso-voice-search-button" class="vso-voice-search-button" title="Voice Search">
                                <img src="<?php echo esc_url(VOICSEOP_STR_PLUGIN_IMG_DIR_URL . 'voice-icon.png'); ?>" alt="Voice Search" />
                            </button>
                        <?php } ?>
                    </div>
                </div>
            </form>
        </div>

       
        <?php
    }
    return ob_get_clean();
}
add_shortcode('voice_search', 'voicseop_display_search_form');
// END - CREATE THE SEARCH FORM WITH SHORTCODE

// START - CREATE THE 'VOICE COMMAND' SEARCH WITH SHORTCODE
function voicseop_voice_command_ui()
{
    ob_start(); 
    $vso_voice_command = get_option('voicseop_enable_voice_command');

    if (isset($vso_voice_command) && $vso_voice_command == 'yes')
    {
        ?>
        <div id="vso-command-container">
            <button id="vso-voice-command-button">
                <img src="<?php echo esc_url(VOICSEOP_STR_PLUGIN_IMG_DIR_URL . 'voice-command-icon.png'); ?>" alt="Voice Command" />
            </button>
            <p id="vso-command-status"><?php esc_html_e('Press button & Speak: "Go to [post title]"', 'voice-search-optimization'); ?></p>
        </div>
        <?php
    }
    return ob_get_clean();
}
add_shortcode('voice_command', 'voicseop_voice_command_ui');
// END - CREATE THE 'VOICE COMMAND' SEARCH WITH SHORTCODE

// START - AJAX HANDLER FOR VOICE COMMAND
add_action('wp_ajax_voicseop_search_post', 'voicseop_search_post');
add_action('wp_ajax_nopriv_voicseop_search_post', 'voicseop_search_post');
function voicseop_search_post()
{

    if (!isset($_POST['nonce_voice_cm']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce_voice_cm'])), 'voicseop_nonce_voice_cm')) {
        wp_send_json_error(['message' => 'Security check faileda.']);
    }


    $post_name = isset($_POST['post_name']) ? sanitize_text_field(wp_unslash($_POST['post_name'])) : '';

    if (empty($post_name)) {
        wp_send_json_error(['message' => 'Please provide a valid post name.']);
    }


    // DEFINE WOOCOMMERCE ENDPOINTS AND HOME DYNAMICALLY 
    $endpoints = [
        'my account' => wc_get_page_permalink('myaccount'),
        'downloads' => wc_get_endpoint_url('downloads', '', wc_get_page_permalink('myaccount')),
        'edit address' => wc_get_endpoint_url('edit-address', '', wc_get_page_permalink('myaccount')),
        'orders' => wc_get_endpoint_url('orders', '', wc_get_page_permalink('myaccount')),
        'payment methods' => wc_get_endpoint_url('payment-methods', '', wc_get_page_permalink('myaccount')),
        'account details' => wc_get_endpoint_url('edit-account', '', wc_get_page_permalink('myaccount')),
        'logout' => wc_get_endpoint_url('customer-logout', '', wc_get_page_permalink('myaccount')),
        'home' => home_url(),
    ];

    // CHECK IF THE COMMAND MATCHES ANY OF THE ENDPOINTS
    foreach ($endpoints as $key => $url)
    {
        if (stripos($post_name, $key) !== false)
        {
            wp_send_json_success(['url' => $url]);
        }
    }

    // SEARCH ACROSS ALL PUBLIC POST TYPES WITH TITLES
    $args = [
        'post_type'      => 'any',
        'title'          => $post_name,
        'post_status'    => 'publish',
        'posts_per_page' => 1,
    ];

    $query = new WP_Query($args);

    if ($query->have_posts())
    {
        $query->the_post();
        $post_url = get_permalink(get_the_ID());
        wp_send_json_success(['url' => $post_url]);
    }
    else
    {
        wp_send_json_error(['message' => 'No matching page or post found. Please try again.']);
    }

    wp_reset_postdata();
}
// END - AJAX HANDLER FOR VOICE COMMAND

// START - APPEND VOICE COMMAND SHORTCODE IN FOOTER
function voicseop_display_voice_command_button()
{
    $vso_voicseop_voice_command_position = get_option('voicseop_voice_command_position');
    // CHECK THE POSITION FROM SETTINGS
    $position_class = ($vso_voicseop_voice_command_position === 'bottom-left') ? 'bottom-left' : 'bottom-right'; ?>

    <div class="vso-voice-command-button-sec <?php echo esc_attr($position_class); ?>">
        <?php echo do_shortcode('[voice_command]'); ?>
    </div>
    <?php 
}
add_action('wp_footer', 'voicseop_display_voice_command_button');
// END - APPEND VOICE COMMAND SHORTCODE IN FOOTER