<?php
/**
 * Gallery Slider Template
 *
 * @var int $gallery_id
 * @var object $gallery
 * @var string $gallery_type
 * @var string $display_style
 * @var array $settings
 * @var array $videos
 * @var int $total_videos
 * @var int $per_page
 * @var int $current_page
 * @var int $current_offset
 * @var bool $has_more
 */

if ( empty( $videos ) ) {
	echo '<p class="vimeify-gallery-empty">' . esc_html__( 'No videos found in this gallery.', 'vimeify' ) . '</p>';
	return;
}

$show_title = isset( $settings['show_title'] ) ? $settings['show_title'] : true;
$show_date = isset( $settings['show_date'] ) ? $settings['show_date'] : false;
$show_description = isset( $settings['show_description'] ) ? $settings['show_description'] : false;
$wrapper_class = 'vimeify-gallery vimeify-gallery-slider';
$wrapper_class = apply_filters( 'vimeify_gallery_wrapper_class', $wrapper_class, $gallery_id, $display_style );
?>

<div class="<?php echo esc_attr( $wrapper_class ); ?>" data-gallery-id="<?php echo esc_attr( $gallery_id ); ?>" data-style="slider">
	<!-- Main Video Player -->
	<div class="vimeify-gallery-main-player">
		<?php if ( ! empty( $videos[0] ) ) : ?>
			<?php
			$main_video = $videos[0];
			do_action( 'vimeify_gallery_before_main_player', $main_video, $gallery_id );
			?>
			<div class="vimeify-gallery-player-wrapper" data-video-id="<?php echo esc_attr( $main_video['ID'] ); ?>">
				<?php if ( ! empty( $main_video['embed_link'] ) ) : ?>
					<iframe
						src="<?php echo esc_url( $main_video['embed_link'] ); ?>"
						frameborder="0"
						allow="autoplay; fullscreen; picture-in-picture"
						allowfullscreen
						class="vimeify-gallery-iframe"
					></iframe>
				<?php else : ?>
					<img src="<?php echo esc_url( $main_video['thumbnail'] ); ?>" alt="<?php echo esc_attr( $main_video['title'] ); ?>" class="vimeify-gallery-placeholder">
				<?php endif; ?>
			</div>

			<?php if ( $show_description && ! empty( $main_video['description'] ) ) : ?>
				<div class="vimeify-gallery-video-info">
					<div class="vimeify-gallery-video-description"><?php echo wp_kses_post( $main_video['description'] ); ?></div>
				</div>
			<?php endif; ?>

			<?php do_action( 'vimeify_gallery_after_main_player', $main_video, $gallery_id ); ?>
		<?php endif; ?>
	</div>

	<!-- Thumbnails -->
	<div class="vimeify-gallery-thumbnails-wrapper">
		<?php do_action( 'vimeify_gallery_before_thumbnails', $videos, $gallery_id ); ?>

		<div class="vimeify-gallery-carousel"
			data-gallery-id="<?php echo esc_attr( $gallery_id ); ?>"
			data-current-offset="<?php echo esc_attr( $current_offset ); ?>"
			data-per-page="<?php echo esc_attr( $per_page ); ?>"
			data-has-more="<?php echo $has_more ? '1' : '0'; ?>">

			<!-- Swiper -->
			<div class="swiper vimeify-gallery-swiper">
				<div class="swiper-wrapper">
					<?php foreach ( $videos as $index => $video ) : ?>
						<div class="swiper-slide">
							<div class="vimeify-gallery-thumbnail <?php echo $index === 0 ? 'active' : ''; ?>"
								data-video-id="<?php echo esc_attr( $video['ID'] ); ?>"
								data-vimeo-id="<?php echo esc_attr( $video['vimeo_id'] ); ?>"
								data-embed-url="<?php echo esc_attr( $video['embed_link'] ); ?>"
								data-title="<?php echo esc_attr( $video['title'] ); ?>"
								data-description="<?php echo esc_attr( wp_strip_all_tags( $video['description'] ) ); ?>">

								<img src="<?php echo esc_url( $video['thumbnail'] ); ?>" alt="<?php echo esc_attr( $video['title'] ); ?>">
								<?php if ( $show_title || $show_date ) : ?>
									<div class="vimeify-gallery-thumbnail-info">
										<?php if ( $show_title ) : ?>
											<div class="vimeify-gallery-thumbnail-title"><?php echo esc_html( $video['title'] ); ?></div>
										<?php endif; ?>
										<?php if ( $show_date ) : ?>
											<div class="vimeify-gallery-thumbnail-date"><?php echo esc_html( get_the_date( '', $video['ID'] ) ); ?></div>
										<?php endif; ?>
									</div>
								<?php endif; ?>
							</div>
						</div>
					<?php endforeach; ?>
				</div>

				<!-- Navigation buttons -->
				<div class="swiper-button-prev"></div>
				<div class="swiper-button-next"></div>
			</div>

			<div class="vimeify-carousel-loading" style="display:none;">
				<span><?php esc_html_e( 'Loading more videos...', 'vimeify' ); ?></span>
			</div>
		</div>

		<?php do_action( 'vimeify_gallery_after_thumbnails', $videos, $gallery_id ); ?>
	</div>
</div>

<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/swiper@11/swiper-bundle.min.css" />

<style>
.vimeify-gallery-slider {
	width: 100%;
	max-width: 100%;
	margin: 0 auto;
}

.vimeify-gallery-main-player {
	margin-bottom: 20px;
}

.vimeify-gallery-player-wrapper {
	position: relative;
	padding-bottom: 56.25%; /* 16:9 aspect ratio */
	height: 0;
	overflow: hidden;
	background: #000;
}

.vimeify-gallery-iframe,
.vimeify-gallery-placeholder {
	position: absolute;
	top: 0;
	left: 0;
	width: 100%;
	height: 100%;
}

.vimeify-gallery-video-info {
	margin-top: 15px;
}

.vimeify-gallery-video-title {
	margin: 0 0 10px 0;
	font-size: 24px;
	font-weight: 600;
}

.vimeify-gallery-video-description {
	color: #666;
	line-height: 1.6;
}

.vimeify-gallery-carousel {
	position: relative;
	margin: 20px 0;
}

.vimeify-gallery-swiper {
	width: 100%;
	padding: 0 50px;
}

.vimeify-gallery-swiper .swiper-slide {
	width: auto;
}

.vimeify-gallery-swiper .swiper-button-prev,
.vimeify-gallery-swiper .swiper-button-next {
	width: 40px;
	height: 40px;
	background: rgba(0, 0, 0, 0.7);
	border-radius: 50%;
	color: #fff;
	transition: all 0.3s ease;
}

.vimeify-gallery-swiper .swiper-button-prev:hover,
.vimeify-gallery-swiper .swiper-button-next:hover {
	background: rgba(0, 0, 0, 0.9);
}

.vimeify-gallery-swiper .swiper-button-prev::after,
.vimeify-gallery-swiper .swiper-button-next::after {
	font-size: 20px;
	font-weight: bold;
}

.vimeify-gallery-swiper .swiper-button-prev {
	left: 0;
}

.vimeify-gallery-swiper .swiper-button-next {
	right: 0;
}

.vimeify-gallery-swiper .swiper-button-disabled {
	opacity: 0.3;
	cursor: not-allowed;
}

.vimeify-gallery-thumbnail {
	cursor: pointer;
	border: 2px solid transparent;
	border-radius: 4px;
	overflow: hidden;
	transition: all 0.3s ease;
	width: 200px;
	display: block;
}

.vimeify-gallery-thumbnail:hover {
	border-color: #6366F1;
	transform: translateY(-2px);
	box-shadow: 0 4px 8px rgba(0,0,0,0.1);
}

.vimeify-gallery-thumbnail.active {
	border-color: #8B5CF6;
	box-shadow: 0 0 10px rgba(139, 92, 246, 0.3);
}

.vimeify-gallery-thumbnail img {
	width: 100%;
	height: auto;
	display: block;
}

.vimeify-gallery-thumbnail-info {
	width: 100%;
}

.vimeify-gallery-thumbnail-title {
	padding: 8px;
	font-size: 14px;
	background: #f9f9f9;
	white-space: nowrap;
	overflow: hidden;
	text-overflow: ellipsis;
}

.vimeify-gallery-thumbnail-date {
	padding: 4px 8px;
	font-size: 12px;
	color: #666;
	background: #f9f9f9;
}

.vimeify-carousel-loading {
	text-align: center;
	padding: 20px 0;
	color: #666;
	font-style: italic;
}

.vimeify-gallery-empty {
	text-align: center;
	padding: 40px 20px;
	color: #666;
	font-size: 16px;
}

@media (max-width: 768px) {
	.vimeify-gallery-thumbnail {
		width: 150px;
	}

	.vimeify-gallery-swiper {
		padding: 0 40px;
	}

	.vimeify-gallery-swiper .swiper-button-prev,
	.vimeify-gallery-swiper .swiper-button-next {
		width: 35px;
		height: 35px;
	}

	.vimeify-gallery-swiper .swiper-button-prev::after,
	.vimeify-gallery-swiper .swiper-button-next::after {
		font-size: 16px;
	}
}
</style>

<script src="https://cdn.jsdelivr.net/npm/swiper@11/swiper-bundle.min.js"></script>

<script>
(function() {
	'use strict';

	// Wait for both DOM and Swiper to be loaded
	if (document.readyState === 'loading') {
		document.addEventListener('DOMContentLoaded', initGalleries);
	} else {
		initGalleries();
	}

	function initGalleries() {
		// Initialize all gallery carousels
		var galleries = document.querySelectorAll('.vimeify-gallery-slider');
		galleries.forEach(function(gallery) {
			initGallerySwiper(gallery);
		});
	}

	// Initialize Swiper for a single gallery
	function initGallerySwiper(gallery) {
		var swiperElement = gallery.querySelector('.vimeify-gallery-swiper');
		if (!swiperElement) return;

		var carousel = gallery.querySelector('.vimeify-gallery-carousel');
		var galleryId = carousel.getAttribute('data-gallery-id');

		// Initialize Swiper
		var swiper = new Swiper(swiperElement, {
			slidesPerView: 'auto',
			spaceBetween: 15,
			navigation: {
				nextEl: '.swiper-button-next',
				prevEl: '.swiper-button-prev',
			},
			freeMode: true,
			watchSlidesProgress: true,
			on: {
				reachEnd: function() {
					// Trigger custom event for Pro plugin lazy loading
					var event = new CustomEvent('vimeifyGalleryScrollEnd', {
						detail: {
							gallery: gallery,
							galleryId: galleryId,
							carousel: carousel,
							swiper: swiper
						}
					});
					document.dispatchEvent(event);
				}
			}
		});

		// Store swiper instance for later use
		swiperElement.swiperInstance = swiper;

		// Bind thumbnail click handlers
		bindThumbnailClickHandlers(gallery);
	}

	// Thumbnail click handler
	function handleThumbnailClick(event) {
		var thumbnail = event.currentTarget;
		var gallery = thumbnail.closest('.vimeify-gallery-slider');
		var embedUrl = thumbnail.getAttribute('data-embed-url');
		var title = thumbnail.getAttribute('data-title');
		var description = thumbnail.getAttribute('data-description');

		// Update active state
		var allThumbnails = gallery.querySelectorAll('.vimeify-gallery-thumbnail');
		allThumbnails.forEach(function(t) {
			t.classList.remove('active');
		});
		thumbnail.classList.add('active');

		// Update main player
		var playerWrapper = gallery.querySelector('.vimeify-gallery-player-wrapper');
		playerWrapper.innerHTML = '<iframe src="' + embedUrl + '" frameborder="0" allow="autoplay; fullscreen; picture-in-picture" allowfullscreen class="vimeify-gallery-iframe"></iframe>';

		// Update video info
		if (title) {
			var titleElement = gallery.querySelector('.vimeify-gallery-video-title');
			if (titleElement) {
				titleElement.textContent = title;
			}
		}
		if (description) {
			var descElement = gallery.querySelector('.vimeify-gallery-video-description');
			if (descElement) {
				descElement.textContent = description;
			}
		}

		// Smooth scroll to player
		var targetPosition = gallery.offsetTop - 20;
		window.scrollTo({
			top: targetPosition,
			behavior: 'smooth'
		});
	}

	// Bind click handlers to thumbnails
	function bindThumbnailClickHandlers(gallery) {
		var thumbnails = gallery.querySelectorAll('.vimeify-gallery-thumbnail');
		thumbnails.forEach(function(thumbnail) {
			thumbnail.removeEventListener('click', handleThumbnailClick);
			thumbnail.addEventListener('click', handleThumbnailClick);
		});
	}

	// Expose functions globally for Pro plugin
	window.vimeifyGallery = window.vimeifyGallery || {};
	window.vimeifyGallery.bindThumbnailClickHandlers = function() {
		var galleries = document.querySelectorAll('.vimeify-gallery-slider');
		galleries.forEach(bindThumbnailClickHandlers);
	};
	window.vimeifyGallery.initGallerySwiper = initGallerySwiper;
})();
</script>
