<?php
/**
 * Gallery Videos Metabox
 *
 * @var \Vimeify\Core\Plugin $plugin
 * @var \WP_Post $post
 * @var array $selected_videos
 */

// Get all available videos
$all_videos = get_posts( [
	'post_type'      => \Vimeify\Core\Components\Database::POST_TYPE_VIDEO,
	'posts_per_page' => -1,
	'post_status'    => 'publish',
	'orderby'        => 'title',
	'order'          => 'ASC',
] );

?>

<div class="vimeify-gallery-videos-picker">
	<div class="vimeify-gallery-videos-search">
		<input type="text" id="vimeify-video-search" class="widefat" placeholder="<?php esc_attr_e( 'Search videos...', 'vimeify' ); ?>" />
	</div>

	<div class="vimeify-gallery-videos-list">
		<div class="vimeify-videos-available">
			<h4><?php esc_html_e( 'Available Videos', 'vimeify' ); ?></h4>
			<div class="vimeify-video-grid" id="vimeify-available-videos">
				<?php foreach ( $all_videos as $video ) : ?>
					<?php
					$thumbnail = get_post_meta( $video->ID, 'dgv_vimeo_thumbnail', true );
					$is_selected = in_array( $video->ID, $selected_videos );
					?>
					<div class="vimeify-video-item <?php echo $is_selected ? 'selected' : ''; ?>" data-id="<?php echo esc_attr( $video->ID ); ?>" data-title="<?php echo esc_attr( strtolower( $video->post_title ) ); ?>">
						<div class="vimeify-video-thumb">
							<?php if ( $thumbnail ) : ?>
								<img src="<?php echo esc_url( $thumbnail ); ?>" alt="<?php echo esc_attr( $video->post_title ); ?>" />
							<?php else : ?>
								<div class="vimeify-video-placeholder">
									<span class="dashicons dashicons-video-alt3"></span>
								</div>
							<?php endif; ?>
							<div class="vimeify-video-overlay">
								<button type="button" class="button vimeify-add-video" <?php echo $is_selected ? 'style="display:none;"' : ''; ?>>
									<?php esc_html_e( 'Add', 'vimeify' ); ?>
								</button>
								<button type="button" class="button vimeify-remove-video" <?php echo ! $is_selected ? 'style="display:none;"' : ''; ?>>
									<?php esc_html_e( 'Remove', 'vimeify' ); ?>
								</button>
							</div>
						</div>
						<div class="vimeify-video-title">
							<?php echo esc_html( $video->post_title ); ?>
						</div>
					</div>
				<?php endforeach; ?>
			</div>
		</div>

		<div class="vimeify-videos-selected">
			<h4><?php esc_html_e( 'Selected Videos', 'vimeify' ); ?> (<span id="vimeify-selected-count"><?php echo count( $selected_videos ); ?></span>)</h4>
			<div id="vimeify-selected-videos" class="vimeify-video-grid-sortable">
				<?php foreach ( $selected_videos as $video_id ) : ?>
					<?php
					$video = get_post( $video_id );
					if ( ! $video ) continue;
					$thumbnail = get_post_meta( $video->ID, 'dgv_vimeo_thumbnail', true );
					?>
					<div class="vimeify-video-item selected" data-id="<?php echo esc_attr( $video->ID ); ?>">
						<div class="vimeify-video-thumb">
							<?php if ( $thumbnail ) : ?>
								<img src="<?php echo esc_url( $thumbnail ); ?>" alt="<?php echo esc_attr( $video->post_title ); ?>" />
							<?php else : ?>
								<div class="vimeify-video-placeholder">
									<span class="dashicons dashicons-video-alt3"></span>
								</div>
							<?php endif; ?>
							<div class="vimeify-video-overlay">
								<span class="dashicons dashicons-move vimeify-drag-handle"></span>
								<button type="button" class="button vimeify-remove-video">
									<?php esc_html_e( 'Remove', 'vimeify' ); ?>
								</button>
							</div>
						</div>
						<div class="vimeify-video-title">
							<?php echo esc_html( $video->post_title ); ?>
						</div>
						<input type="hidden" name="vimeify_gallery_videos[]" value="<?php echo esc_attr( $video->ID ); ?>" />
					</div>
				<?php endforeach; ?>
			</div>
		</div>
	</div>
</div>

<style>
.vimeify-gallery-videos-picker {
	padding: 10px 0;
}

.vimeify-gallery-videos-search {
	margin-bottom: 20px;
}

.vimeify-gallery-videos-list {
	display: flex;
	gap: 20px;
}

.vimeify-videos-available,
.vimeify-videos-selected {
	flex: 1;
}

.vimeify-videos-available h4,
.vimeify-videos-selected h4 {
	margin: 0 0 10px 0;
	padding-bottom: 10px;
	border-bottom: 1px solid #ddd;
}

.vimeify-video-grid,
.vimeify-video-grid-sortable {
	display: grid;
	grid-template-columns: repeat(auto-fill, minmax(150px, 1fr));
	gap: 15px;
	max-height: 500px;
	overflow-y: auto;
	padding: 10px;
	background: #f9f9f9;
	border: 1px solid #ddd;
}

.vimeify-video-item {
	background: white;
	border: 2px solid #ddd;
	border-radius: 4px;
	overflow: hidden;
	cursor: pointer;
	transition: all 0.2s;
}

.vimeify-video-item.selected {
	border-color: #2271b1;
}

.vimeify-video-item:hover {
	box-shadow: 0 2px 8px rgba(0,0,0,0.1);
}

.vimeify-video-thumb {
	position: relative;
	padding-bottom: 56.25%;
	background: #f0f0f0;
}

.vimeify-video-thumb img {
	position: absolute;
	top: 0;
	left: 0;
	width: 100%;
	height: 100%;
	object-fit: cover;
}

.vimeify-video-placeholder {
	position: absolute;
	top: 0;
	left: 0;
	width: 100%;
	height: 100%;
	display: flex;
	align-items: center;
	justify-content: center;
	background: #e0e0e0;
}

.vimeify-video-placeholder .dashicons {
	font-size: 48px;
	color: #999;
}

.vimeify-video-overlay {
	position: absolute;
	top: 0;
	left: 0;
	width: 100%;
	height: 100%;
	background: rgba(0,0,0,0.7);
	display: flex;
	align-items: center;
	justify-content: center;
	opacity: 0;
	transition: opacity 0.2s;
	gap: 5px;
}

.vimeify-video-item:hover .vimeify-video-overlay {
	opacity: 1;
}

.vimeify-video-overlay .button {
	font-size: 11px;
	padding: 4px 8px;
	height: auto;
	line-height: 1.2;
}

.vimeify-drag-handle {
	color: white;
	font-size: 24px;
	cursor: move;
}

.vimeify-video-title {
	padding: 8px;
	font-size: 12px;
	text-align: center;
	overflow: hidden;
	text-overflow: ellipsis;
	white-space: nowrap;
}

.vimeify-video-grid-sortable .vimeify-video-item {
	cursor: move;
}

.ui-sortable-helper {
	opacity: 0.8;
	transform: scale(1.05);
}
</style>

<script>
jQuery(document).ready(function($) {
	// Search functionality
	$('#vimeify-video-search').on('keyup', function() {
		var searchTerm = $(this).val().toLowerCase();
		$('#vimeify-available-videos .vimeify-video-item').each(function() {
			var title = $(this).data('title');
			if (title.indexOf(searchTerm) > -1) {
				$(this).show();
			} else {
				$(this).hide();
			}
		});
	});

	// Add video
	$(document).on('click', '.vimeify-add-video', function(e) {
		e.preventDefault();
		var $item = $(this).closest('.vimeify-video-item');
		var videoId = $item.data('id');
		var $clone = $item.clone();

		// Update UI
		$item.addClass('selected');
		$(this).hide();
		$item.find('.vimeify-remove-video').show();

		// Add to selected
		$clone.find('.vimeify-add-video').remove();
		$clone.find('.vimeify-video-overlay').prepend('<span class="dashicons dashicons-move vimeify-drag-handle"></span>');
		$clone.append('<input type="hidden" name="vimeify_gallery_videos[]" value="' + videoId + '" />');
		$('#vimeify-selected-videos').append($clone);

		updateSelectedCount();
	});

	// Remove video
	$(document).on('click', '.vimeify-remove-video', function(e) {
		e.preventDefault();
		var $item = $(this).closest('.vimeify-video-item');
		var videoId = $item.data('id');

		// Update available videos UI
		$('#vimeify-available-videos .vimeify-video-item[data-id="' + videoId + '"]')
			.removeClass('selected')
			.find('.vimeify-add-video').show().end()
			.find('.vimeify-remove-video').hide();

		// Remove from selected
		$('#vimeify-selected-videos .vimeify-video-item[data-id="' + videoId + '"]').remove();

		updateSelectedCount();
	});

	// Make selected videos sortable
	if (typeof $.fn.sortable !== 'undefined') {
		$('#vimeify-selected-videos').sortable({
			handle: '.vimeify-drag-handle',
			placeholder: 'ui-state-highlight',
			update: function() {
				updateSelectedCount();
			}
		});
	}

	function updateSelectedCount() {
		var count = $('#vimeify-selected-videos .vimeify-video-item').length;
		$('#vimeify-selected-count').text(count);
	}
});
</script>
