// Copyright Darko Gjorgjijoski <info@codeverve.com>
// 2020. All Rights Reserved.
// This file is licensed under the GPLv2 License.
// License text available at https://opensource.org/licenses/gpl-2.0.php

(function () {

    'use strict';

    // Hook types
    window.Vimeify = window.Vimeify || {};
    window.Vimeify.Hook_Type_Backend = 1;
    window.Vimeify.Hook_Type_Frontend = 2;

    /**
     * Helper to dispatch custom events
     *
     * @param {string} eventName Event name
     * @param {*} detail Event detail data
     * @param {Element} target Target element (defaults to window)
     */
    function dispatchEvent(eventName, detail, target) {
        target = target || window;
        var event;

        if (typeof CustomEvent === 'function') {
            event = new CustomEvent(eventName, {
                detail: detail,
                bubbles: true,
                cancelable: true
            });
        } else {
            event = document.createEvent('CustomEvent');
            event.initCustomEvent(eventName, true, true, detail);
        }

        target.dispatchEvent(event);
    }

    /**
     * Helper to find closest ancestor matching selector
     */
    function closest(el, selector) {
        if (el.closest) {
            return el.closest(selector);
        }
        while (el) {
            if (el.matches && el.matches(selector)) {
                return el;
            }
            el = el.parentElement;
        }
        return null;
    }

    /**
     * Helper to create video list item HTML
     */
    function createVideoListItem(video) {
        var uri = video.uri || video.vimeo_id;
        var title = video.name || video.title;
        var thumbnail = video.pictures && video.pictures.sizes ? video.pictures.sizes[0].link : '';
        var duration = video.duration ? formatDuration(video.duration) : '';

        return '<div class="vimeify-video-item" data-video-uri="' + uri + '">' +
            (thumbnail ? '<div class="vimeify-video-thumbnail"><img src="' + thumbnail + '" alt="' + title + '"></div>' : '') +
            '<div class="vimeify-video-info">' +
                '<div class="vimeify-video-title">' + title + '</div>' +
                (duration ? '<div class="vimeify-video-duration">' + duration + '</div>' : '') +
            '</div>' +
            '<button type="button" class="button button-primary vimeify-insert-video-btn" data-video-uri="' + uri + '" title="Insert video">' +
                '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">' +
                    '<polyline points="9 18 15 12 9 6"></polyline>' +
                '</svg>' +
            '</button>' +
        '</div>';
    }

    /**
     * Helper to format duration in seconds to MM:SS
     */
    function formatDuration(seconds) {
        var minutes = Math.floor(seconds / 60);
        var secs = seconds % 60;
        return minutes + ':' + (secs < 10 ? '0' : '') + secs;
    }

    /**
     * Helper for AJAX requests
     */
    function ajax(options) {
        return new Promise(function (resolve, reject) {
            var xhr = new XMLHttpRequest();
            var method = options.method || options.type || 'GET';
            var url = options.url;
            var data = options.data;

            if (method === 'GET' && data) {
                var params = [];
                for (var key in data) {
                    if (data.hasOwnProperty(key)) {
                        params.push(encodeURIComponent(key) + '=' + encodeURIComponent(data[key]));
                    }
                }
                if (params.length > 0) {
                    url += (url.indexOf('?') > -1 ? '&' : '?') + params.join('&');
                }
            }

            xhr.open(method, url, true);
            xhr.setRequestHeader('X-Requested-With', 'XMLHttpRequest');

            xhr.onload = function () {
                if (xhr.status >= 200 && xhr.status < 300) {
                    try {
                        var response = JSON.parse(xhr.responseText);
                        if (options.success) {
                            options.success(response);
                        }
                        resolve(response);
                    } catch (e) {
                        if (options.error) {
                            options.error(xhr);
                        }
                        reject(xhr);
                    }
                } else {
                    if (options.error) {
                        options.error(xhr);
                    }
                    reject(xhr);
                }
            };

            xhr.onerror = function () {
                if (options.error) {
                    options.error(xhr);
                }
                reject(xhr);
            };

            if (method === 'POST' && data) {
                xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
                var params = [];
                for (var key in data) {
                    if (data.hasOwnProperty(key)) {
                        params.push(encodeURIComponent(key) + '=' + encodeURIComponent(data[key]));
                    }
                }
                xhr.send(params.join('&'));
            } else {
                xhr.send();
            }
        });
    }

    /**
     * Vimeify Modal System
     * Pure JavaScript modal without external dependencies
     */
    window.Vimeify.Modal = (function () {

        var activeModal = null;

        function Modal(options) {
            this.options = options || {};
            this.element = null;
            this.overlay = null;
        }

        Modal.prototype.show = function (content, options) {
            options = options || {};

            // Close existing modal (unless keepExisting is true)
            if (activeModal && !options.keepExisting) {
                activeModal.close();
            }

            // Create overlay
            this.overlay = document.createElement('div');
            this.overlay.className = 'vimeify-modal-overlay';

            // Create modal container
            this.element = document.createElement('div');
            this.element.className = 'vimeify-modal';
            if (options.className) {
                this.element.className += ' ' + options.className;
            }

            // Create modal dialog
            var dialog = document.createElement('div');
            dialog.className = 'vimeify-modal-dialog';

            // Create modal content
            var modalContent = document.createElement('div');
            modalContent.className = 'vimeify-modal-content';

            // Add content
            if (typeof content === 'string') {
                modalContent.innerHTML = content;
            } else {
                modalContent.appendChild(content);
            }

            dialog.appendChild(modalContent);
            this.element.appendChild(dialog);

            // Add modal to overlay, then overlay to body
            this.overlay.appendChild(this.element);
            document.body.appendChild(this.overlay);

            // Add show class for animation
            setTimeout(function () {
                this.overlay.classList.add('vimeify-modal-show');
            }.bind(this), 10);

            activeModal = this;

            // Handle close button
            var closeBtn = this.element.querySelector('.vimeify-close-modal');
            if (closeBtn) {
                closeBtn.addEventListener('click', function(e) {
                    // Skip if upload in progress - document handler shows confirmation
                    if (window.VimeoUploaderModal && window.VimeoUploaderModal.working) {
                        return;
                    }
                    this.close();
                }.bind(this));
            }

            // Handle outside click
            if (!options.allowOutsideClick && options.allowOutsideClick !== false) {
                this.overlay.addEventListener('click', this.close.bind(this));
            }

            // Handle escape key
            if (!options.allowEscapeKey && options.allowEscapeKey !== false) {
                this.escapeHandler = function (e) {
                    if (e.key === 'Escape' || e.keyCode === 27) {
                        this.close();
                    }
                }.bind(this);
                document.addEventListener('keydown', this.escapeHandler);
            }

            return this;
        };

        Modal.prototype.close = function () {
            if (!this.element) {
                return;
            }

            // Remove show class for animation
            this.overlay.classList.remove('vimeify-modal-show');

            // Remove from DOM after animation
            setTimeout(function () {
                if (this.overlay && this.overlay.parentNode) {
                    this.overlay.parentNode.removeChild(this.overlay);
                }
                this.overlay = null;
                this.element = null;
                if (activeModal === this) {
                    activeModal = null;
                }
            }.bind(this), 300);

            // Remove escape handler
            if (this.escapeHandler) {
                document.removeEventListener('keydown', this.escapeHandler);
                this.escapeHandler = null;
            }
        };

        // Static methods for common modals
        Modal.alert = function (title, message, type) {
            type = type || 'info';
            var iconMap = {
                'success': '✓',
                'error': '✕',
                'warning': '⚠',
                'info': 'ℹ'
            };

            var content = '<div class="vimeify-modal-alert vimeify-modal-' + type + '">' +
                '<div class="vimeify-modal-icon">' + (iconMap[type] || iconMap.info) + '</div>' +
                '<h3 class="vimeify-modal-title">' + title + '</h3>' +
                '<div class="vimeify-modal-message">' + message + '</div>' +
                '<button class="button vimeify-modal-ok">' + (window.Vimeify_Modal_Config && window.Vimeify_Modal_Config.words.ok || 'OK') + '</button>' +
                '</div>';

            var modal = new Modal();
            modal.show(content, {className: 'vimeify-modal-alert-wrapper'});

            // Handle OK button
            var okBtn = modal.element.querySelector('.vimeify-modal-ok');
            if (okBtn) {
                okBtn.addEventListener('click', function () {
                    modal.close();
                });
            }

            return modal;
        };

        Modal.confirm = function (title, message, onConfirm, onCancel, options) {
            options = options || {};
            var content = '<div class="vimeify-modal-confirm">' +
                '<h3 class="vimeify-modal-title">' + title + '</h3>' +
                '<div class="vimeify-modal-message">' + message + '</div>' +
                '<div class="vimeify-modal-buttons">' +
                '<button class="button vimeify-modal-cancel">' + (options.cancelLabel || (window.Vimeify_Modal_Config && window.Vimeify_Modal_Config.words.cancel) || 'Cancel') + '</button>' +
                '<button class="button button-primary vimeify-modal-confirm-btn">' + (options.confirmLabel || (window.Vimeify_Modal_Config && window.Vimeify_Modal_Config.words.confirm) || 'Confirm') + '</button>' +
                '</div>' +
                '</div>';

            var modal = new Modal();
            modal.show(content, {className: 'vimeify-modal-confirm-wrapper', keepExisting: true});

            // Handle buttons
            var confirmBtn = modal.element.querySelector('.vimeify-modal-confirm-btn');
            var cancelBtn = modal.element.querySelector('.vimeify-modal-cancel');

            if (confirmBtn) {
                confirmBtn.addEventListener('click', function () {
                    if (onConfirm) {
                        onConfirm();
                    }
                    modal.close();
                });
            }

            if (cancelBtn) {
                cancelBtn.addEventListener('click', function () {
                    if (onCancel) {
                        onCancel();
                    }
                    modal.close();
                });
            }

            return modal;
        };

        return Modal;
    })();

    /**
     * Byte formatting utility
     */
    var units = ['bytes', 'KB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB'];
    function niceBytes(x) {
        var l = 0, n = parseInt(x, 10) || 0;
        while (n >= 1024 && ++l) {
            n = n / 1024;
        }
        return (n.toFixed(n < 10 && l > 0 ? 1 : 0) + ' ' + units[l]);
    }

    /**
     * Uploader modal constructor
     *
     * @param {string} context Context identifier
     * @param {object} config Configuration options
     * @constructor
     */
    window.Vimeify.UploaderModal = function (context, config) {
        this.context = context;
        this.config = config || {};
        this.modal = null;
    };

    /**
     * Generate the modal form HTML
     * @returns {string}
     */
    window.Vimeify.UploaderModal.prototype.form = function () {

        var opts = [];
        if (window.Vimeify_Modal_Config.enable_local_search && (!this.config.hasOwnProperty('local') || this.config.local)) {
            opts.push('<label><input type="radio" class="vimeify-field-row vimeify-insert-type" name="insert_type" value="local">' + window.Vimeify_Modal_Config.methods.local + '</label>');
        }
        if (window.Vimeify_Modal_Config.enable_vimeo_search && (!this.config.hasOwnProperty('search') || this.config.search)) {
            opts.push('<label><input type="radio" class="vimeify-field-row vimeify-insert-type" name="insert_type" value="search">' + window.Vimeify_Modal_Config.methods.search + '</label>');
        }

        var opts_str;
        var method_upload_style;
        if (opts.length === 0) {
            opts_str = '<input type="hidden" name="insert_type" value="upload">';
            method_upload_style = 'display:block;';
        } else {
            opts = ['<label><input type="radio" class="vimeify-field-row vimeify-insert-type" name="insert_type" value="upload">' + window.Vimeify_Modal_Config.methods.upload + '</label>'].concat(opts);
            opts_str = '<div class="vimeify-vimeo-form-row">' + opts.join("\n") + '</div>';
            method_upload_style = 'display: none;';
        }

        var source = this.config.hasOwnProperty('source') ? this.config.source : 'Default';
        var hooks = this.config.hasOwnProperty('hook_type') ? this.config.hook_type : window.Vimeify.Hook_Type_Backend;

        var privacy_option = '';

        if (window.Vimeify_Modal_Config.upload_form_options.enable_view_privacy && null !== window.Vimeify_Modal_Config.upload_form_options.privacy_view) {

            var privacy_view_options = '';

            for (var key in window.Vimeify_Modal_Config.upload_form_options.privacy_view) {
                var name = window.Vimeify_Modal_Config.upload_form_options.privacy_view[key].name;
                var is_available = window.Vimeify_Modal_Config.upload_form_options.privacy_view[key].available;
                var is_default = window.Vimeify_Modal_Config.upload_form_options.privacy_view[key].default;
                var disabled = is_available ? '' : 'disabled';
                var selected = is_default ? 'selected' : '';
                privacy_view_options += '<option ' + disabled + ' ' + selected + ' value="' + key + '">' + name + '</option>';
            }

            privacy_option = '<div class="vimeify-vimeo-form-row">\n' +
                '<label for="vimeo_view_privacy">' + window.Vimeify_Modal_Config.words.privacy_view + '</label>' +
                '<select name="vimeo_view_privacy" class="vimeify-field-row vimeify-w-100">' + privacy_view_options + '</select>\n' +
                '</div>\n';
        }

        var modal_title = (this.config.hasOwnProperty('title') && this.config.title) ? this.config.title : window.Vimeify_Modal_Config.phrases.title;

        var data_attr = '';
        if (this.config.hasOwnProperty('meta') && this.config.meta) {
            data_attr = "data-meta='" + JSON.stringify(this.config.meta) + "'";
        }

        return '<div id="' + this.context + '" class="vimeify-vimeo-upload-form vimeify-vimeo-plain-form vimeify-text-left">\n' +
            '\n' +
            '    <span class="vimeify-close-modal">&#215;</span>\n' +
            '    <h4 class="vimeify-mt-0">' + modal_title + '</h4>\n' +
            '\n' + opts_str + '\n' +
            '    <div class="vimeify-insert-wrapper vimeify-insert-type-upload" style="' + method_upload_style + '">\n' +
            '        <form id="vimeify-vimeo-upload-modal" ' + data_attr + '>\n' +
            '            <input type="hidden" name="vimeo_source" value="' + source + '">' +
            '            <input type="hidden" name="vimeo_hook_type" value="' + hooks + '">' +
            '            <div class="vimeify-vimeo-form-row">\n' +
            '                <label for="vimeo_title">' + window.Vimeify_Modal_Config.words.title + '</label>' +
            '                <input type="text" name="vimeo_title" class="vimeify-field-row">\n' +
            '            </div>\n' +
            '            <div class="vimeify-vimeo-form-row">\n' +
            '                <label for="vimeo_description">' + window.Vimeify_Modal_Config.words.desc + '</label>' +
            '                <textarea rows="5" name="vimeo_description" class="vimeify-field-row"></textarea>\n' +
            '            </div>\n' + privacy_option +
            '            <div class="vimeify-vimeo-form-row">\n' +
            '                <label for="vimeo_video">' + window.Vimeify_Modal_Config.words.file + '</label>' +
            '                <div class="vimeify-dropzone">\n' +
            '                    <input type="file" name="vimeo_video" class="vimeify-field-row vimeify-file-input" id="vimeo_video" accept="video/*" style="display: none;">\n' +
            '                    <div class="vimeify-dropzone-area">\n' +
            '                        <svg class="vimeify-dropzone-icon" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">\n' +
            '                            <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path>\n' +
            '                            <polyline points="17 8 12 3 7 8"></polyline>\n' +
            '                            <line x1="12" y1="3" x2="12" y2="15"></line>\n' +
            '                        </svg>\n' +
            '                        <p class="vimeify-dropzone-text">Drag and drop your video here or <span class="vimeify-dropzone-browse">browse</span></p>\n' +
            '                        <p class="vimeify-dropzone-filename"></p>\n' +
            '                    </div>\n' +
            '                </div>\n' +
            '            </div>\n' +
            '            <div class="vimeify-vimeo-form-row">\n' +
            '                <div class="vimeify-progress-bar" style="display: none;">\n' +
            '                    <div class="vimeify-progress-bar-inner" style="width: 0;"></div>\n' +
            '                    <div class="vimeify-progress-bar-value">0%</div>\n' +
            '                </div>\n' +
            '            </div>\n' +
            '            <div class="vimeify-vimeo-form-row">\n' +
            '                <button type="submit" class="button button-primary submitUpload" data-waiting="' + window.Vimeify_Modal_Config.words.uploading3d + '" data-finished="Upload">\n' +
            '                    Upload\n' +
            '                </button>\n' +
            '            </div>\n' +
            '        </form>\n' +
            '    </div>\n' +
            '\n' +
            '    <div class="vimeify-insert-wrapper vimeify-insert-type-local" style="display:none;">\n' +
            '        <div class="vimeify-local-search-wrapper">\n' +
            '            <input type="text" class="vimeify-local-search-input" placeholder="' + window.Vimeify_Modal_Config.words.search + ' videos..." />\n' +
            '        </div>\n' +
            '        <div class="vimeify-video-list vimeify-local-video-list"></div>\n' +
            '        <div class="vimeify-local-loading">\n' +
            '            <svg class="vimeify-loader" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><circle cx="12" cy="12" r="10" stroke="currentColor" stroke-opacity="0.25" stroke-width="3"></circle><path d="M12 2a10 10 0 0 1 10 10" stroke="currentColor" stroke-width="3" stroke-linecap="round"></path></svg>\n' +
            '            <span>Loading videos...</span>\n' +
            '        </div>\n' +
            '        <div class="vimeify-load-more-wrapper" style="display: none;">\n' +
            '            <button type="button" class="button button-secondary vimeify-load-more-btn">Load More</button>\n' +
            '        </div>\n' +
            '        <div class="vimeify-vimeo-form-row vimeify-videos-404" style="display: none;">\n' +
            '            <p>' + window.Vimeify_Modal_Config.phrases.videos_not_found + '</p>\n' +
            '        </div>\n' +
            '    </div>\n' +
            '\n' +
            '    <div class="vimeify-insert-wrapper vimeify-insert-type-search" style="display: none;">\n' +
            '        <form id="vimeify-vimeo-search" class="vimeify-vimeo-form-row">\n' +
            '            <div class="vimeify-vimeo-form-row">\n' +
            '                <input type="text" name="video_title" placeholder="' + window.Vimeify_Modal_Config.words.title + '" class="vimeify-field-row">\n' +
            '                <button type="submit" class="button button-primary" data-waiting="' + window.Vimeify_Modal_Config.words.searching3d + '" data-finished="' + window.Vimeify_Modal_Config.words.upload + '">' + window.Vimeify_Modal_Config.words.search + '</button>\n' +
            '            </div>\n' +
            '        </form>\n' +
            '        <div class="vimeify-video-list vimeify-videos-found" style="display: none;"></div>\n' +
            '        <div class="vimeify-vimeo-form-row vimeify-videos-404" style="display: none;">\n' +
            '            <p>' + window.Vimeify_Modal_Config.phrases.search_not_found + '</p>\n' +
            '        </div>\n' +
            '    </div>\n' +
            '\n' +
            '</div>';
    };

    /**
     * Open the uploader modal
     */
    window.Vimeify.UploaderModal.prototype.open = function () {
        var form = this.form();
        this.modal = new window.Vimeify.Modal();
        this.modal.show(form, {
            allowOutsideClick: false,
            allowEscapeKey: false,
            className: 'vimeify-uploader-modal'
        });

        // Initialize dropzone after modal is shown
        setTimeout(function() {
            this.initDropzone();
        }.bind(this), 100);
    };

    /**
     * Initialize dropzone functionality
     */
    window.Vimeify.UploaderModal.prototype.initDropzone = function () {
        var dropzone = document.querySelector('.vimeify-dropzone');
        var dropzoneArea = document.querySelector('.vimeify-dropzone-area');
        var fileInput = document.querySelector('.vimeify-file-input');
        var dropzoneText = document.querySelector('.vimeify-dropzone-text');
        var dropzoneFilename = document.querySelector('.vimeify-dropzone-filename');

        if (!dropzone || !fileInput || !dropzoneArea) {
            return;
        }

        // Click to browse
        dropzoneArea.addEventListener('click', function(e) {
            e.preventDefault();
            fileInput.click();
        });

        // File selected via browse
        fileInput.addEventListener('change', function(e) {
            if (fileInput.files && fileInput.files.length > 0) {
                var file = fileInput.files[0];
                dropzoneText.style.display = 'none';
                dropzoneFilename.textContent = file.name;
                dropzoneFilename.classList.add('vimeify-show');
                dropzone.classList.add('vimeify-dropzone-has-file');
            }
        });

        // Prevent default drag behaviors
        ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(function(eventName) {
            dropzoneArea.addEventListener(eventName, function(e) {
                e.preventDefault();
                e.stopPropagation();
            });
        });

        // Highlight dropzone when item is dragged over
        ['dragenter', 'dragover'].forEach(function(eventName) {
            dropzoneArea.addEventListener(eventName, function() {
                dropzone.classList.add('vimeify-dropzone-highlight');
            });
        });

        ['dragleave', 'drop'].forEach(function(eventName) {
            dropzoneArea.addEventListener(eventName, function() {
                dropzone.classList.remove('vimeify-dropzone-highlight');
            });
        });

        // Handle dropped files
        dropzoneArea.addEventListener('drop', function(e) {
            var files = e.dataTransfer.files;
            if (files && files.length > 0) {
                fileInput.files = files;
                var file = files[0];
                dropzoneText.style.display = 'none';
                dropzoneFilename.textContent = file.name;
                dropzoneFilename.classList.add('vimeify-show');
                dropzone.classList.add('vimeify-dropzone-has-file');
            }
        });
    };

    /**
     * Close the uploader modal
     */
    window.Vimeify.UploaderModal.prototype.close = function () {
        if (this.modal) {
            this.modal.close();
        }
    };

    /**
     * Global uploader state
     */
    window.VimeoUploaderModal = {
        working: false,
        uploader: null
    };

    /**
     * Local library state management
     */
    var localLibraryState = {
        page: 1,
        perPage: 6,
        search: '',
        loading: false,
        hasMore: true,
        total: 0
    };

    /**
     * Initialize local library with pagination
     */
    function initLocalLibrary() {
        // Reset state
        localLibraryState.page = 1;
        localLibraryState.search = '';
        localLibraryState.hasMore = true;
        localLibraryState.total = 0;

        var elList = document.querySelector('.vimeify-local-video-list');
        var searchInput = document.querySelector('.vimeify-local-search-input');
        var elLoading = document.querySelector('.vimeify-local-loading');
        var elLoadMore = document.querySelector('.vimeify-load-more-wrapper');

        if (elList) {
            elList.innerHTML = '';
        }
        if (searchInput) {
            searchInput.value = '';
        }
        if (elLoading) {
            elLoading.classList.remove('vimeify-show');
        }
        if (elLoadMore) {
            elLoadMore.style.display = 'none';
        }

        loadLocalVideos();
    }

    /**
     * Load local videos with pagination
     */
    function loadLocalVideos(append) {
        if (localLibraryState.loading) {
            return;
        }

        if (!localLibraryState.hasMore) {
            return;
        }

        localLibraryState.loading = true;

        var elList = document.querySelector('.vimeify-local-video-list');
        var el404 = document.querySelector('.vimeify-insert-type-local .vimeify-videos-404');
        var elLoading = document.querySelector('.vimeify-local-loading');

        if (elLoading) {
            elLoading.classList.add('vimeify-show');
        }

        ajax({
            url: window.Vimeify_Modal_Config.ajax_url + '?action=vimeify_get_uploads',
            data: {
                _wpnonce: window.Vimeify_Modal_Config.nonce,
                page: localLibraryState.page,
                per_page: localLibraryState.perPage,
                search: localLibraryState.search
            },
            type: 'GET',
            success: function (response) {
                localLibraryState.loading = false;
                if (elLoading) {
                    elLoading.classList.remove('vimeify-show');
                }

                if (response.success) {
                    var uploads = response.data.uploads || [];
                    var total = response.data.total || uploads.length;
                    localLibraryState.total = total;

                    var elLoadMore = document.querySelector('.vimeify-load-more-wrapper');

                    if (uploads.length > 0) {
                        var listHTML = '';
                        for (var i in uploads) {
                            listHTML += createVideoListItem(uploads[i]);
                        }

                        if (append && elList) {
                            elList.insertAdjacentHTML('beforeend', listHTML);
                        } else if (elList) {
                            elList.innerHTML = listHTML;
                        }

                        if (elList) {
                            elList.style.display = 'block';
                        }
                        if (el404) {
                            el404.style.display = 'none';
                        }

                        // Check if there are more videos
                        // If we received fewer items than requested, there are no more
                        localLibraryState.hasMore = uploads.length >= localLibraryState.perPage;
                        localLibraryState.page++;

                        // Show/hide load more button
                        if (elLoadMore) {
                            elLoadMore.style.display = localLibraryState.hasMore ? 'block' : 'none';
                        }
                    } else {
                        if (!append) {
                            if (elList) {
                                elList.style.display = 'none';
                            }
                            if (el404) {
                                el404.style.display = 'block';
                            }
                        }
                        localLibraryState.hasMore = false;

                        // Hide load more button
                        if (elLoadMore) {
                            elLoadMore.style.display = 'none';
                        }
                    }
                } else {
                    window.Vimeify.Modal.alert(
                        window.Vimeify_Modal_Config.words.sorry,
                        response.data.message,
                        'error'
                    );
                }
            },
            error: function () {
                localLibraryState.loading = false;
                if (elLoading) {
                    elLoading.classList.remove('vimeify-show');
                }
                window.Vimeify.Modal.alert(
                    window.Vimeify_Modal_Config.words.sorry,
                    window.Vimeify_Modal_Config.phrases.http_error,
                    'error'
                );
            }
        });
    }

    /**
     * Handle local library search
     */
    var localSearchTimeout;
    document.addEventListener('input', function(e) {
        if (!e.target.classList.contains('vimeify-local-search-input')) {
            return;
        }

        clearTimeout(localSearchTimeout);
        localSearchTimeout = setTimeout(function() {
            localLibraryState.search = e.target.value;
            localLibraryState.page = 1;
            localLibraryState.hasMore = true;
            loadLocalVideos(false);
        }, 500);
    });

    /**
     * Handle Load More button click
     */
    document.addEventListener('click', function(e) {
        if (!e.target.classList.contains('vimeify-load-more-btn')) {
            return;
        }

        e.preventDefault();
        loadLocalVideos(true);
    });


    // Handle insert dropdown
    document.addEventListener('change', function (e) {
        if (!e.target.classList.contains('vimeify-insert-type')) {
            return;
        }

        var form = closest(e.target, '.vimeify-vimeo-upload-form');
        if (!form) {
            return;
        }

        var context = form.getAttribute('id');
        var wrappers = document.querySelectorAll('.vimeify-insert-wrapper');
        for (var i = 0; i < wrappers.length; i++) {
            wrappers[i].style.display = 'none';
        }

        var value = e.target.value;
        var wrapper = document.querySelector('.vimeify-insert-type-' + value);

        if (value === 'local') {
            // Initialize local library with pagination
            initLocalLibrary();
        }

        if (wrapper) {
            wrapper.style.display = 'block';
        }
    });

    // Handle vimeo search
    document.addEventListener('submit', function (e) {
        var form = e.target;
        if (form.id !== 'vimeify-vimeo-search') {
            return;
        }

        e.preventDefault();

        var uploadForm = closest(form, '.vimeify-vimeo-upload-form');
        var context = uploadForm ? uploadForm.getAttribute('id') : null;

        var searchInput = form.querySelector('input[name=video_title]');
        var search_phrase = searchInput ? searchInput.value : '';
        var searchButton = form.querySelector('button[type="submit"]');

        if (search_phrase !== '') {
            var searchWrapper = closest(form, '.vimeify-insert-type-search');
            var wrapperFound = searchWrapper.querySelector('.vimeify-videos-found');
            var wrapper404 = searchWrapper.querySelector('.vimeify-videos-404');
            var vimeoProfile = new window.Vimeify.Profile(window.Vimeify_Modal_Config.access_token);

            // Show loading state
            if (searchButton) {
                searchButton.classList.add('loading');
                searchButton.disabled = true;
                var originalText = searchButton.textContent;
                searchButton.innerHTML = '<svg class="vimeify-loader" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><circle cx="12" cy="12" r="10" stroke="currentColor" stroke-opacity="0.25" stroke-width="3"></circle><path d="M12 2a10 10 0 0 1 10 10" stroke="currentColor" stroke-width="3" stroke-linecap="round"></path></svg>' + originalText;
            }

            // Hide previous results
            wrapperFound.style.display = 'none';
            wrapper404.style.display = 'none';

            vimeoProfile.search({
                'page': 1,
                'per_page': 100,
                'query': search_phrase,
                'sort': 'date',
                'direction': 'desc',
                'onSuccess': function (response) {
                    // Remove loading state
                    if (searchButton) {
                        searchButton.classList.remove('loading');
                        searchButton.disabled = false;
                        searchButton.textContent = originalText;
                    }

                    if (response.data.length > 0) {
                        var listHTML = '';
                        for (var i in response.data) {
                            listHTML += createVideoListItem(response.data[i]);
                        }
                        wrapper404.style.display = 'none';
                        wrapperFound.innerHTML = listHTML;
                        wrapperFound.style.display = 'block';
                    } else {
                        wrapperFound.style.display = 'none';
                        wrapper404.style.display = 'block';
                    }
                },
                'onError': function (response) {
                    // Remove loading state
                    if (searchButton) {
                        searchButton.classList.remove('loading');
                        searchButton.disabled = false;
                        searchButton.textContent = originalText;
                    }

                    window.Vimeify.Modal.alert(
                        window.Vimeify_Modal_Config.words.sorry,
                        window.Vimeify_Modal_Config.phrases.invalid_search_phrase,
                        'error'
                    );
                }
            });
        } else {
            window.Vimeify.Modal.alert(
                window.Vimeify_Modal_Config.words.sorry,
                window.Vimeify_Modal_Config.phrases.invalid_search_phrase,
                'error'
            );
        }
    });

    // Handle vimeo insert from video list
    document.addEventListener('click', function (e) {
        var target = e.target;

        // Check if click is on insert button or its child elements (like SVG)
        if (!target.classList.contains('vimeify-insert-video-btn')) {
            target = closest(target, '.vimeify-insert-video-btn');
        }

        if (!target) {
            return;
        }

        e.preventDefault();

        var uploadForm = closest(target, '.vimeify-vimeo-upload-form');
        var context = uploadForm ? uploadForm.getAttribute('id') : null;

        var uri = target.getAttribute('data-video-uri');

        if (!uri) {
            window.Vimeify.Modal.alert('Please select a video first.', 'warning');
            return;
        }

        if (uri) {
            uri = uri.replace('/videos/', '');
            dispatchEvent('vimeify.events.insert', {uri: uri, context: context});

            // Close modal
            var modalElement = closest(e.target, '.vimeify-modal');
            if (modalElement && modalElement.__vimeifyModal) {
                modalElement.__vimeifyModal.close();
            } else {
                // Fallback
                var modals = document.querySelectorAll('.vimeify-modal');
                if (modals.length > 0) {
                    var overlay = document.querySelector('.vimeify-modal-overlay');
                    if (overlay && overlay.parentNode) {
                        overlay.parentNode.removeChild(overlay);
                    }
                    for (var i = 0; i < modals.length; i++) {
                        if (modals[i].parentNode) {
                            modals[i].parentNode.removeChild(modals[i]);
                        }
                    }
                }
            }
        }
    });

    // Handle vimeo upload
    document.addEventListener('submit', function (e) {
        var form = e.target;
        if (form.id !== 'vimeify-vimeo-upload-modal') {
            return;
        }

        e.preventDefault();

        var uploadForm = closest(form, '.vimeify-vimeo-upload-form');
        var context = uploadForm ? uploadForm.getAttribute('id') : null;

        var submit = form.querySelector('button[type=submit]');
        var progressBar = form.querySelector('.vimeify-progress-bar');

        var formData = new FormData(form);
        var videoFile = formData.get('vimeo_video');
        var notify_meta = form.dataset.meta ? JSON.parse(form.dataset.meta) : null;
        var notify_endpoint_enabled = true;

        if (!window.Vimeify.Uploader.validateVideo(videoFile)) {
            window.Vimeify.Modal.alert(
                window.Vimeify_Modal_Config.words.sorry,
                window.Vimeify_Modal_Config.phrases.upload_invalid_file,
                'error'
            );
            return false;
        }

        var title = formData.get('vimeo_title');
        var description = formData.get('vimeo_description');
        var privacy = formData.get('vimeo_view_privacy');
        var source = formData.get('vimeo_source') || 'UploadModal';
        var hook_type = formData.get('vimeo_hook_type') || 1;

        if (!privacy) {
            privacy = window.Vimeify_Modal_Config.default_privacy;
        }

        var errorHandler = function (error) {
            var message = error;
            var type = 'error';

            // Reset button state
            if (submit) {
                submit.classList.remove('loading');
                submit.disabled = false;
                submit.textContent = submit.getAttribute('data-finished') || 'Upload';
            }

            // Reset progress bar
            if (progressBar) {
                updateProgressBar(progressBar, 0);
                progressBar.style.display = 'none';
            }

            window.Vimeify.Modal.alert(window.Vimeify_Modal_Config.words.sorry, message, type);
        };

        var updateProgressBar = function (pbar, value) {
            if (pbar.style.display === 'none') {
                pbar.style.display = 'block';
            }
            var inner = pbar.querySelector('.vimeify-progress-bar-inner');
            var valueEl = pbar.querySelector('.vimeify-progress-bar-value');
            if (inner) {
                inner.style.width = value + '%';
            }
            if (valueEl) {
                valueEl.textContent = value + '%';
            }
        };

        var params = {
            action: 'vimeify_store_upload',
            source: source,
            hook_type: hook_type,
            _wpnonce: window.Vimeify_Modal_Config.nonce
        };

        var esc = encodeURIComponent;
        var query = Object.keys(params)
            .map(function (k) { return esc(k) + '=' + esc(params[k]); })
            .join('&');

        window.VimeoUploaderModal.uploader = new window.Vimeify.Uploader(window.Vimeify_Modal_Config.access_token, videoFile, {
            'title': title,
            'description': description,
            'privacy': privacy,
            'wp': {
                'notify_endpoint': notify_endpoint_enabled ? (window.Vimeify_Modal_Config.ajax_url + '?' + query) : false,
                'notify_meta': notify_meta || null,
            },
            'beforeStart': function () {
                if (submit) {
                    submit.classList.add('loading');
                    submit.disabled = true;
                    var originalText = submit.textContent;
                    submit.innerHTML = '<svg class="vimeify-loader" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><circle cx="12" cy="12" r="10" stroke="currentColor" stroke-opacity="0.25" stroke-width="3"></circle><path d="M12 2a10 10 0 0 1 10 10" stroke="currentColor" stroke-width="3" stroke-linecap="round"></path></svg>' + originalText;
                }
                window.VimeoUploaderModal.working = true;
            },
            'onProgress': function (bytesUploaded, bytesTotal) {
                var percentage = (bytesUploaded / bytesTotal * 100).toFixed(2);
                updateProgressBar(progressBar, percentage);
            },
            'onSuccess': function (response, currentUpload) {
                var type = response.success ? 'success' : 'error';
                var message = response.data.message;

                window.setTimeout(function () {
                    form.reset();
                    if (submit) {
                        submit.classList.remove('loading');
                        submit.disabled = false;
                        submit.textContent = submit.getAttribute('data-finished') || 'Upload';
                    }
                    updateProgressBar(progressBar, 0);
                    progressBar.style.display = 'none';
                }, 1000);

                window.Vimeify.Modal.alert(window.Vimeify_Modal_Config.words.success, message, type);

                var uri = currentUpload.uri;
                uri = uri.replace('/videos/', '');
                var data = {
                    name: currentUpload.hasOwnProperty('name') ? currentUpload.name : '',
                    description: currentUpload.hasOwnProperty('description') ? currentUpload.description : '',
                    context: context,
                    uri: uri,
                    full_uri: currentUpload.uri,
                    size_formatted: niceBytes(currentUpload.upload.size),
                    root: currentUpload
                };

                window.VimeoUploaderModal.working = false;
                dispatchEvent('vimeify.events.upload', data);
            },
            'onError': function (error) {
                window.VimeoUploaderModal.working = false;
                errorHandler(error);
            },
            'onVideoCreateError': function (error) {
                window.VimeoUploaderModal.working = false;
                errorHandler(error);
            },
            'onWPNotifyError': function (error) {
                window.VimeoUploaderModal.working = false;
                errorHandler(error);
            }
        });

        window.VimeoUploaderModal.uploader.start();
    });

    // Close the modal
    document.addEventListener('click', function (e) {
        if (!e.target.classList.contains('vimeify-close-modal')) {
            return;
        }

        e.preventDefault();

        if (window.VimeoUploaderModal.working) {
            window.Vimeify.Modal.confirm(
                window.Vimeify_Modal_Config.phrases.confirm_title || 'Confirm',
                window.Vimeify_Modal_Config.phrases.cancel_upload_confirm,
                function () {
                    // Confirmed
                    window.VimeoUploaderModal.uploader.abort();
                    window.VimeoUploaderModal.working = false;

                    // Close upload modal
                    var modals = document.querySelectorAll('.vimeify-modal');
                    var overlay = document.querySelector('.vimeify-modal-overlay');
                    if (overlay && overlay.parentNode) {
                        overlay.parentNode.removeChild(overlay);
                    }
                    for (var i = 0; i < modals.length; i++) {
                        if (modals[i].parentNode) {
                            modals[i].parentNode.removeChild(modals[i]);
                        }
                    }
                },
                null,
                {
                    cancelLabel: window.Vimeify_Modal_Config.words.keep_uploading || 'Keep Uploading',
                    confirmLabel: window.Vimeify_Modal_Config.words.cancel_upload || 'Cancel Upload'
                }
            );
        } else {
            // Close modal
            var modals = document.querySelectorAll('.vimeify-modal');
            var overlay = document.querySelector('.vimeify-modal-overlay');
            if (overlay && overlay.parentNode) {
                overlay.parentNode.removeChild(overlay);
            }
            for (var i = 0; i < modals.length; i++) {
                if (modals[i].parentNode) {
                    modals[i].parentNode.removeChild(modals[i]);
                }
            }
        }
    });

})();
